/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <search.h>
#include <assert.h>
#undef DEBUG
#define NDEBUG
#include "tbox.h"
#include "rbtree.h"


/*
** The tree routines in this module, rbtsearch(), rbtfind(), rbtwalk() and
** rbtdelete(), were written to behave like the standard C library tree
** functions tsearch(), tfind(), twalk() and tdelete(), but use red-black
** tree algorithms. Red-black binary trees are designed to balance
** themselves, regardless of the order nodes are added. A red-black tree
** with n nodes has height at most 2*log2(n+1).
**
** Manual pages for the standard C library tree functions can serve as
** documentation for these red-black tree routines.
**
** NOTES:
**
**	1. rbtdelete() returns the address to the key/data that was
**	   contained in the deleted node.  This makes freeing malloced
**	   key/data space easier in calling program. Standard function
**	   tdelete() returns the address of the deleted node's parent.
**
**	2. Prototypes for external functions are defined in rbtree.h.
**	   rbtwalk() calls require standard header file search.h.
**
**	3. Red-black trees should not be deleted by calling rbdelete() from
**	   the rbtwalk() function. Ancestor nodes are not static in a
**	   rbdelete() call. Rotations occur that confuse rbtwalk(). To
**	   delete an entire tree, call rbtfree().
**
** For a detailed explanation of these red-black tree algorithms, there is
** an entire chapter devoted to them in "Introduction to Algorithms,"
** Thomas H. Cormen, Charles E. Leiserson and Ronald L. Rivest, MIT Press,
** McGraw Hill, 1989.
**
** Red-black tree properties:
**	1. Every node is either red or black.
**	2. Every leaf (null_node) is black.
**	3. If a node is red, then both its children are black.
**	4. Every simple path from a node to a descendant leaf
**	   contains the same number of black nodes.
**	5. Root node is black.
*/

typedef struct rbtree_s	rbtree_t;

typedef enum { Red, Black } rbtcolor_t;

struct rbtree_s {
	void		*datum;
	rbtree_t	*parent;
	rbtree_t	*left;
	rbtree_t	*right;
	rbtcolor_t	color;
};

/*
** null_node is a sentinel node used to mark the end of all simple paths.
*/
static rbtree_t	null_node = { 0, &null_node, &null_node, &null_node, Black };

static void node_step( const rbtree_t *,  void (*)( const void *, VISIT, int ), int );
static rbtree_t *tree_successor( rbtree_t * );
static void left_rotate( rbtree_t **, rbtree_t * );
static void right_rotate( rbtree_t **, rbtree_t * );
static rbtree_t *tree_minimum( rbtree_t * );

/*
** This function searches for a matching key in the tree and returns an
** address to an existing node's datum pointer. If an existing node is
** not found, a new node is added. The address of the new node's datum
** pointer is then returned. A null pointer is returned for fatal errors.
**
** If the root pointer points to a null address, rbtsearch() starts a new
** tree by populating the root pointer.
*/
void *rbtsearch( void *datum, void **r, int (*compar)( const void *, const void * ) )
{
	register rbtree_t	*node;
	register rbtree_t	*parent = &null_node;
	rbtree_t	**root = (rbtree_t **) r;
	rbtree_t	*uncle, *grandparent;
	int	compar_value;
	void	*ret;
#if 0
struct mallinfo	minfo;
#endif

	assert( root != (rbtree_t **) 0 );
	assert( compar != (int (*)(const void *, const void *)) 0 );

	DEBUG_FUNC_START;

	if ( *root == (rbtree_t *) 0 ) {
		/*
		** About to add first node to new tree.
		*/
		*root = &null_node;
	}

	node = *root;

	/*
	** Look for existing node.
	*/
	while ( node != &null_node ) {
		parent = node;

		/*
		** Use calling function's supplied routine
		** for comparing key values.
		*/
		compar_value = (*compar)( datum, node->datum );
		if ( compar_value == 0 ) {
			RETURN_POINTER( (void *) &node->datum );
		}
		if ( compar_value < 0 ) {
			node = node->left;
		} else {
			node = node->right;
		}
	}

	/*
	** Did not find node existing in tree.
	** Create a new node.
	*/
#if 0
	minfo = mallinfo();
	fprintf( stderr, "MALLOC before, arena %d, #blks %d, #used blks %d, #free blks %d\n", minfo.arena, minfo.ordblks, minfo.uordblks, minfo.fordblks );
#endif

	if ( ( node = (rbtree_t *) malloc( sizeof( rbtree_t ) ) ) == (rbtree_t *) 0 ) {
		RETURN_POINTER( (void *) 0 );
	}
#if 0
	minfo = mallinfo();
	fprintf( stderr, "MALLOC after, arena %d, #blks %d, #used blks %d, #free blks %d\n", minfo.arena, minfo.ordblks, minfo.uordblks, minfo.fordblks );
#endif

	node->parent = parent;
	node->datum = datum;
	node->left = node->right = &null_node;
	node->color = Red;

	if ( parent == &null_node ) {
		/*
		** Start new tree.
		*/
		*root = node;
	} else {
		/*
		** Use last return value from key compare routine
		** for deciding which side of the parent to place
		** new node.
		*/
		if ( compar_value < 0 ) {
			parent->left = node;
		} else {
			parent->right = node;
		}
	}

	ret = (void *) &node->datum;

	/*
	** New node has been added to binary tree.
	** Time to make red-black balancing adjustments.
	**
	** The purpose of the following while loop is to resolve
	** property 3, which is initially not true, and maintain
	** property 4 (see above for red-black tree properties).
	*/
	assert( node->parent == parent );
	while ( node != *root && parent->color == Red ) {
		assert( node->parent == parent );
		grandparent = parent->parent;
		if ( parent == grandparent->left ) {
			/*
			** Parent node is on the left side of
			** the grandparent node.
			*/
			uncle = grandparent->right;
			if ( uncle->color == Red ) {
				assert( grandparent->color == Black );
				/*
				** The uncle is red and grandparent is black.
				** To resolve property 3 and maintain property
				** 4 for the given subtree, parent of the red
				** node must be changed to black. When the
				** parent becomes black, the uncle must also
				** become black (property 4). With both parent
				** and uncle black, the grandparent becomes
				** red (property 3). 
				*/
				parent->color = Black;
				uncle->color = Black;
				grandparent->color = Red;
				node = grandparent;
				parent = node->parent;
				continue;
			}

			/*
			** The uncle and grandparent are both
			** black (property 3). The parent and new
			** node are red.
			*/
			if ( node == parent->right ) {
				/*
				** New node is on the right side of
				** its parent. Left rotate new node
				** and its parent node. This will
				** reposition the new node to become 
				** the parent of the orignal parent.
				*/
				node = parent;
				left_rotate( root, node );
				parent = node->parent;
			}
			/*
			** Right rotate parent and grandparent to
			** preserve property 4. This will reposition
			** the parent node to become the parent of
			** the grandparent node. Since the original
			** grandparent node will continue being the
			** parent of the black uncle, as well as the
			** new parent of one his decendent's black
			** nodes, change his color to red (property 3).
			** Since the original parent will become the
			** parent to the red grandparent, he must be
			** changed to black (property 3).
			*/
			parent->color = Black;
			grandparent->color = Red;
			right_rotate( root, grandparent );
			parent = node->parent;
			continue;
		}

		/*
		** Parent node is on the right side of
		** the grandparent node.
		*/
		assert( parent == grandparent->right );
		uncle = grandparent->left;
		if ( uncle->color == Red ) {
			assert( grandparent->color == Black );
			/*
			** The uncle is red and grandparent is black.
			** To resolve property 3 and maintain property
			** 4 for the given subtree, parent of the red
			** node must be changed to black. When the
			** parent becomes black, the uncle must also
			** become black (property 4). With both parent
			** and uncle black, the grandparent becomes
			** red (property 3). 
			*/
			parent->color = Black;
			uncle->color = Black;
			grandparent->color = Red;
			node = grandparent;
			parent = node->parent;
			continue;
		}

		/*
		** The uncle and grandparent are both
		** black (property 3). The parent and new
		** node are red.
		*/
		assert( uncle->color == Black );
		assert( grandparent->color == Black );
		if ( node == parent->left ) {
			/*
			** New node is on the left side of
			** its parent. Right rotate new node
			** and its parent node. This will
			** reposition the new node to become 
			** the parent of the orignal parent.
			*/
			node = parent;
			right_rotate( root, node );
			parent = node->parent;
		}

		/*
		** Left rotate parent and grandparent to
		** preserve property 4. This will reposition
		** the parent node to become the parent of
		** the grandparent node. Since the original
		** grandparent node will continue being the
		** parent of the black uncle, as well as the
		** new parent of one his decendent's black
		** nodes, change his color to red (property 3).
		** Since the original parent will become the
		** parent to the red grandparent, he must be
		** changed to black (property 3).
		*/
		parent->color = Black;
		grandparent->color = Red;
		left_rotate( root, grandparent );
		parent = node->parent;
	}

	/*
	** Root node is always black (property 5).
	*/
	( *root )->color = Black;

	RETURN_POINTER( ret );
}

/*
** This function initiates a tree walk that uses a calling function's
** supplied routine for preorder, postorder and endorder traversals.
*/
void rbtwalk( const void *root, void (*action)( const void *, VISIT, int ) )
{
	assert( action != (void (*)( const void *, VISIT, int )) 0 );

	DEBUG_FUNC_START;

	if ( root == (void *)0 ) {
		RETURN_VOID;
	}

	node_step( (const rbtree_t *) root, action, 0 );

	RETURN_VOID;
}

static void node_step( const rbtree_t *node,  void (*action)( const void *, VISIT, int ), int depth )
{
	if ( node == &null_node ) {
		/*
		** Sentinel null node.
		*/
		return;
	}

#if 0
printf( "%s, %s\n", (char *)node->datum, node->color == Red ? "RED" : "BLACK" );
#endif

	if ( node->left == &null_node && node->right == &null_node ) {
		/*
		** Leaf node (as defined on the twalk man page).
		*/
		(*action)( &node->datum, leaf, depth );
		return;
	}

	(*action)( &node->datum, preorder, depth );

	node_step( node->left, action, depth + 1 );

	(*action)( &node->datum, postorder, depth );

	node_step( node->right, action, depth + 1 );

	(*action)( &node->datum, endorder, depth );
}

/*
** This function will search the tree for one particular key value
** and delete the associated node. Upon completion, the address of
** the node's datum is returned. The calling function can then easily
** free malloced space. This is incompatible with standard tdelete().
** tdelete() returns the address of the deleted node's parent.
** If the key value does not have a tree match, a null pointer is returned.
*/
void *rbtdelete( const void *datum, void **r, int (*compar)( const void *, const void * ) )
{
	register rbtree_t	*node, *sibling, *parent;
	rbtree_t	**root = (rbtree_t **) r;
	rbtree_t	*splice_out, *child;
	void	*ret;
	int	compar_value;
#if 0
struct mallinfo	minfo;
#endif

	assert( root != (rbtree_t **) 0 );
	assert( compar != (int (*)(const void *, const void *)) 0 );

	DEBUG_FUNC_START;

	node = *root;

	if ( node == (rbtree_t *) 0 ) {
		RETURN_POINTER( (void *) 0 );
	}

	/*
	** Search for key value in tree.
	*/
	while ( node != &null_node ) {
		/*
		** Use calling function's supplied routine
		** for comparing key values.
		*/
		compar_value = (*compar)( datum, node->datum );
		if ( compar_value == 0 ) {
			break;
		}
		if ( compar_value < 0 ) {
			node = node->left;
		} else {
			node = node->right;
		}
	}

	if ( node == &null_node ) {
		/*
		** Did not find node to delete.
		*/
		RETURN_POINTER( (void *) 0 );
	}

	ret = node->datum;

	if ( node->left == &null_node || node->right == &null_node ) {
		splice_out = node;
	} else {
		splice_out = tree_successor( node );
	}

	if ( splice_out->left != &null_node ) {
		child = splice_out->left;
	} else {
		child = splice_out->right;
	}

	parent = splice_out->parent;

	/*
	** Splice around splice_out node.
	*/
	child->parent = parent;

	if ( parent == &null_node ) {
		*root = child;
	} else {
		if ( splice_out == parent->left ) {
			parent->left = child;
		} else {
			parent->right = child;
		}
	}

	if ( splice_out != node ) {
		node->datum = splice_out->datum;
	}

	/*
	** Node has been deleted.
	*/

	if ( splice_out->color == Red ) {
		/*
		** Do not need to make red-black adjustments.
		*/
#if 0
		minfo = mallinfo();
		fprintf( stderr, "FREE #1 before, arena %d, #blks %d, #used blks %d, #free blks %d\n", minfo.arena, minfo.ordblks, minfo.uordblks, minfo.fordblks );
#endif
		free( (void *)splice_out );
#if 0
		minfo = mallinfo();
		fprintf( stderr, "FREE #1 after, arena %d, #blks %d, #used blks %d, #free blks %d\n", minfo.arena, minfo.ordblks, minfo.uordblks, minfo.fordblks );
#endif
		RETURN_POINTER( ret );
	}

	/*
	** Time to make red-black adjustments.
	**
	** When the spliced out node is black, its removal causes
	** any path that previously contained the deleted node to
	** have one fewer black node. Property 4 is violated by
	** all of its ancestors (see above for red-black tree properties).
	*/
	node = child;
	while ( node != *root && node->color == Black ) {
		parent = node->parent;
		if ( node == parent->left ) {
			/*
			** Node is on left side of parent.
			*/
			sibling = parent->right;
			if ( sibling->color == Red ) {
				/*
				** Since the sibling must have black
				** children (property 3), we can switch
				** the colors of the sibling and parent
				** and perform a left rotation.
				** The purpose of this maneuvering is to
				** rearrange the current situation into a
				** similar situation in which the sibling
				** is black. Subsequent code expects the
				** sibling to be black.
				*/
#if DEBUG
				(void) fputs( "Case 1A\n", stderr );
#endif
				assert( sibling->left->color == Black );
				assert( sibling->right->color == Black );
				sibling->color = Black;
				parent->color = Red;
				left_rotate( root, parent );
				parent = node->parent;
				sibling = parent->right;
			}

			/*
			** At this point, we expect sibling to be black.
			*/
			assert( sibling->color == Black );

			if ( sibling->left->color == Black && sibling->right->color == Black ) {
				/*
				** Both of sibling's children are also black.
				** Make sibling red (property 3).
				*/
#if DEBUG
				(void) fputs( "Case 2A\n", stderr );
#endif
				sibling->color = Red;
				node = parent;
				continue;
			}

			if ( sibling->right->color == Black ) {
				/*
				** Sibling and his right child are
				** black. The sibling's left child
				** is red. We need to transform
				** this situation into one in which
				** the sibling is black and the
				** sibling's right child is red.
				*/
#if DEBUG
				(void) fputs( "Case 3A\n", stderr );
#endif
				assert( sibling->color == Black );
				assert( sibling->left->color == Red );
				sibling->left->color = Black;
				sibling->color = Red;
				right_rotate( root, sibling );
				parent = node->parent;
				sibling = parent->right;
				assert( sibling->color == Black );
				assert( sibling->right->color == Red );
			}

			/*
			** At this point, the sibling is expected
			** to be black, and the sibling's right
			** child is expected to red.
			*/
#if DEBUG
			(void) fputs( "Case 4A\n", stderr );
#endif
			assert( sibling->color == Black );
			assert( sibling->right->color == Red );
			sibling->color = parent->color;
			parent->color = Black;
			sibling->right->color = Black;
			left_rotate( root, parent );
			/*
			** Cause the while loop to terminate.
			*/
			node = *root;
			continue;
		}

		/*
		** Node is on right side of parent.
		*/
		assert( node == parent->right );
		sibling = parent->left;
		if ( sibling->color == Red ) {
			/*
			** Since the sibling must have black
			** children (property 3), we can switch
			** the colors of the sibling and parent
			** and perform a right rotation.
			** The purpose of this maneuvering is to
			** rearrange the current situation into a
			** similar situation in which the sibling
			** is black. Subsequent code expects the
			** sibling to be black.
			*/
#if DEBUG
			(void) fputs( "Case 1B\n", stderr );
#endif
			assert( sibling->left->color == Black );
			assert( sibling->right->color == Black );
			sibling->color = Black;
			parent->color = Red;
			right_rotate( root, parent );
			parent = node->parent;
			sibling = parent->left;
		}

		/*
		** At this point, we expect sibling to be black.
		*/
		assert( sibling->color == Black );

		if ( sibling->right->color == Black && sibling->left->color == Black ) {
			/*
			** Both of sibling's children are also black.
			** Make sibling red (property 3).
			*/
#if DEBUG
			(void) fputs( "Case 2B\n", stderr );
#endif
			sibling->color = Red;
			node = parent;
			continue;
		}

		if ( sibling->left->color == Black ) {
			/*
			** Sibling and his left child are
			** black. The sibling's right child
			** is red. We need to transform
			** this situation into one in which
			** the sibling is black and the
			** sibling's left child is red.
			*/
#if DEBUG
			(void) fputs( "Case 3B\n", stderr );
#endif
			assert( sibling->color == Black );
			assert( sibling->right->color == Red );
			sibling->right->color = Black;
			sibling->color = Red;
			left_rotate( root, sibling );
			parent = node->parent;
			sibling = parent->left;
			assert( sibling->color == Black );
			assert( sibling->left->color == Red );
		}

		/*
		** At this point, the sibling is expected
		** to be black, and the sibling's left
		** child is expected to red.
		*/
#if DEBUG
		(void) fputs( "Case 4B\n", stderr );
#endif
		assert( sibling->color == Black );
		assert( sibling->left->color == Red );
		sibling->color = parent->color;
		parent->color = Black;
		sibling->left->color = Black;
		right_rotate( root, parent );
		/*
		** Cause the while loop to terminate.
		*/
		node = *root;
	}

	node->color = Black;

#if 0
	minfo = mallinfo();
	fprintf( stderr, "FREE #2 before, arena %d, #blks %d, #used blks %d, #free blks %d\n", minfo.arena, minfo.ordblks, minfo.uordblks, minfo.fordblks );
#endif
	free( (void *)splice_out );
#if 0
	minfo = mallinfo();
	fprintf( stderr, "FREE #2 after, arena %d, #blks %d, #used blks %d, #free blks %d\n", minfo.arena, minfo.ordblks, minfo.uordblks, minfo.fordblks );
#endif

	RETURN_POINTER( ret );
}

/*
** This function will free an entire tree.
*/
void rbtfree( void **r, void (*free_datum)( void * ) )
{
	register rbtree_t	*node, *parent;
	rbtree_t	**root = (rbtree_t **) r;
#if 0
struct mallinfo	minfo;
#endif

	assert( root != (rbtree_t **) 0 );

	DEBUG_FUNC_START;

	node = *root;

	if ( node == (rbtree_t *) 0 ) {
		RETURN_VOID;
	}

	/*
	** Do an endorder tree walk to destroy tree.
	** Since entire tree is going to be freed, red-black
	** properties are ignored.
	*/
	while ( node != &null_node ) {
		if ( node->left != &null_node ) {
			node = tree_minimum( node );
		} else {
			if ( node->right != &null_node ) {
				node = node->right;
			}
		}

		if ( node->left != &null_node || node->right != &null_node ) {
			continue;
		}

		/*
		** At this point, all children of current
		** node have been freed; free this node.
		** Detach current node from parent.
		*/
		parent = node->parent;
		if ( parent->left == node ) {
			parent->left = &null_node;
		} else {
			parent->right = &null_node;
		}

		if ( free_datum != (void (*)( void * )) 0 ) {
			/*
			** Use calling function's supplied routine for
			** freeing any malloced datum variables.
			*/
			( *free_datum )( node->datum );
		}
		free( (void *)node );
		node = parent;
	}

	*root = (void *) 0;

	RETURN_VOID;
}

/*
** This function will locate a key value in the tree and return the address
** of the node's datum address. A null pointer is returned if the key was
** not found.
*/
void *rbtfind( const void *datum, void * const *root, int (*compar)( const void *, const void * ) )
{
	register const rbtree_t	*node;
	int	compar_value;

	assert( root != (void **) 0 );
	assert( compar != (int (*)(const void *, const void *)) 0 );

	DEBUG_FUNC_START;

	node = *root;

	if ( node == (rbtree_t *) 0 ) {
		RETURN_POINTER( (void *) 0 );
	}

	/*
	** Search for key value in tree.
	*/
	while ( node != &null_node ) {
		/*
		** Use calling function's supplied routine
		** for comparing key values.
		*/
		compar_value = (*compar)( datum, node->datum );
		if ( compar_value == 0 ) {
			RETURN_POINTER( (void *) &node->datum );
		}
		if ( compar_value < 0 ) {
			node = node->left;
		} else {
			node = node->right;
		}
	}

	RETURN_POINTER( (void *) 0 );
}

/*
** This function is used to pick a successor node.
** If the right subtree of starting node is nonempty, the successor
** is the left-most node in the right subtree. Otherwise, the
** successor is the lowest ancestor whose left child is also an
** ancestor of the starting node.
*/
static rbtree_t *tree_successor( rbtree_t *node )
{
	rbtree_t	*successor;

	if ( node->right != &null_node ) {
		return tree_minimum( node->right );
	}

	successor = node->parent;

	while ( successor != &null_node && node == successor->right ) {
		node = successor;
		successor = successor->parent;
	}

	return successor;
}

/*
** This function uses variables x and y for identifying the two
** tree nodes to rotate. Variable x starts as the parent to y.
** The rotation operation makes variable y becomes the parent of x.
** The left nodes of y also become right nodes of x.
*/
static void left_rotate( rbtree_t **root, register rbtree_t *x )
{
	register rbtree_t	*y;

	y = x->right;

	/*
	** Turn y's left subtree into x's right subtree.
	*/
	x->right = y->left;

	if ( y->left != &null_node ) {
		y->left->parent = x;
	}

	/*
	** Link x's parent to y.
	*/
	y->parent = x->parent;

	if ( x->parent == &null_node ) {
		*root = y;
	} else {
		if ( x == x->parent->left ) {
			x->parent->left = y;
		} else {
			x->parent->right = y;
		}
	}

	/*
	** Put x on y's left.
	*/
	y->left = x;

	x->parent = y;
}

/*
** This function uses variables x and y for identifying the two
** tree nodes to rotate. Variable x starts as the parent to y.
** The rotation operation makes variable y becomes the parent of x.
** The right nodes of y also become left nodes of x.
*/
static void right_rotate( rbtree_t **root, register rbtree_t *x )
{
	register rbtree_t	*y;

	y = x->left;

	/*
	** Turn y's right subtree into x's left subtree.
	*/
	x->left = y->right;

	if ( y->right != &null_node ) {
		y->right->parent = x;
	}

	/*
	** Link x's parent to y.
	*/
	y->parent = x->parent;

	if ( x->parent == &null_node ) {
		*root = y;
	} else {
		if ( x == x->parent->right ) {
			x->parent->right = y;
		} else {
			x->parent->left = y;
		}
	}

	/*
	** Put x on y's right.
	*/
	y->right = x;

	x->parent = y;
}

/*
** This function will return the left-most node.
*/
static rbtree_t *tree_minimum( register rbtree_t *node )
{
	assert( node != (rbtree_t *) 0 );

	while ( node->left != &null_node ) {
		node = node->left;
	}

	return node;
}

#ifdef MT_rbtree
/*
** This main() function is used to regression test routines in this module.
** The following command is used to compile:
** 	x=rbtree; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/
static short	node_cnt = 0;
static void	print_tree( const void *, VISIT, int );
int		compare_colors( const void *, const void * );

int main( void )

{
	static const char	blank_line[] = ">>>\n";
	static char	*keys[] = {
		"yellow", "green", "brown", "orange", "black",
		"red", "blue", "purple", "gray", "white", 0
	};
	char	**key_ptr, **key_find, *key_delete;
	void	*root;

	setbuf( stdout, (char *) 0 );

	(void) fprintf( stderr, ">>> Start red-black tree module test.\n" );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Build tree with color names as keys.\n", stderr );
	(void) fputs( blank_line, stderr );

	root = (void *) 0;
	for ( key_ptr = keys; *key_ptr != (char *) 0; ++key_ptr ) {
		(void) fprintf( stderr, ">>> Adding color [%s] to tree.\n", *key_ptr );
		if ( ( key_find = (char **)rbtsearch( (void *) *key_ptr, &root, compare_colors ) ) == (char **) 0 ) {
			(void) fputs( ">>> rbtsearch() failed.\n", stderr );
			return 10;
		}
		if ( strcmp( *key_ptr, *key_find ) != 0 ) {
			(void) fprintf( stderr, ">>> rbtsearch() incorrectly return [%s] when adding [%s].\n", *key_find, *key_ptr );
			return 12;
		}
		(void) fprintf( stderr, ">>> Color [%s] was correctly added to tree.\n", *key_ptr );
		(void) fputs( blank_line, stderr );
		(void) fputs( ">>> Print current tree in ascending order.\n", stderr );
		node_cnt = 0;
		rbtwalk( root, print_tree );
		(void) fputs( blank_line, stderr );
	}

	(void) fputs( ">>> Find the color orange using rbtfind().\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_find = (char **) rbtfind( (void *) "orange", &root, compare_colors ) ) == (char **) 0 ) {
		(void) fputs( ">>> rbtfind() failed to find the color orange.\n", stderr );
		return 15;
	}

	if ( strcmp( *key_find, "orange" ) != 0 ) {
		(void) fprintf( stderr, ">>> rbtfind() incorrectly return [%s] when searching for orange.\n", *key_find );
		return 17;
	}
	(void) fputs( ">>> rbtfind() correctly found the color orange.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Delete orange node from tree.\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_delete = (char *) rbtdelete( (void *) "orange", &root, compare_colors ) ) == (char *) 0 ) {
		(void) fputs( ">>> rbtdelete() failed to delete the color orange.\n", stderr );
		return 20;
	}

	if ( strcmp( key_delete, "orange" ) != 0 ) {
		(void) fprintf( stderr, ">>> rbtdelete() incorrectly deleted [%s] while attempting to delete orange.\n", key_delete );
		return 25;
	}
	(void) fputs( ">>> rbtdelete() correctly deleted the color orange.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Attempt invalid search for the color orange using rbtfind().\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_find = (char **) rbtfind( (void *) "orange", &root, compare_colors ) ) != (char **) 0 ) {
		(void) fputs( ">>> rbtfind() incorrectly returned a color from tree.\n", stderr );
		return 27;
	}
	(void) fputs( ">>> rbtfind() correctly failed to find the color orange.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Attempt to delete orange node from tree.\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_delete = (char *) rbtdelete( (void *) "orange", &root, compare_colors ) ) != (char *) 0 ) {
		(void) fputs( ">>> rbtdelete() incorrectly deleted a node from the tree.\n", stderr );
		return 30;
	}
	(void) fputs( ">>> rbtdelete() correctly did not delete a node from the tree.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Print current tree in ascending order.\n", stderr );
	(void) fputs( blank_line, stderr );
	node_cnt = 0;
	rbtwalk( root, print_tree );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Add color orange to tree.\n", stderr );
	(void) fputs( blank_line, stderr );
	if ( ( key_find = (char **)rbtsearch( (void *) "orange", &root, compare_colors ) ) == (char **) 0 ) {
		(void) fputs( ">>> rbtsearch() failed.\n", stderr );
		return 33;
	}
	if ( strcmp( "orange", *key_find ) != 0 ) {
		(void) fprintf( stderr, ">>> rbtsearch() incorrectly returned [%s] when adding orange.\n", *key_find );
		return 35;
	}
	(void) fputs( ">>> Color orange was correctly added to tree.\n", stderr );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Print current tree in ascending order.\n", stderr );
	(void) fputs( blank_line, stderr );
	node_cnt = 0;
	rbtwalk( root, print_tree );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Purge entire tree.\n", stderr );
	rbtfree( &root, ( void (*)( void * ) ) 0 );

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> Red-black tree module testing is complete.\n", stderr );
	return 0;
}

int compare_colors( const void *x, const void *y )
{
	return strcmp( (char *) x, (char *) y );
}

/*
** This function prints all tree nodes in sorted order.
*/
static void print_tree( const void *k, VISIT visit, int depth )
{
	char	* const *key = k;

	if ( visit == postorder || visit == leaf ) {
		(void) printf( "%d. [%s], depth is %d\n", ++node_cnt, *key, depth );
	}
}
#endif
