/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "nsplit.h"

static ssize_t write_block( int, char *, size_t );

int copy_data( char **remaining, size_t *buffer_length, int output_fd, int input_fd, char *buffer, size_t buffer_size, size_t output_file_remaining_cnt )
{
	static int	eof_flag = 0;	/* initialize to false */
	size_t	copy_cnt, len, write_len;
	ssize_t	write_status, read_cnt;
	char	*ptr, *eol_ptr;

	assert( remaining != (char **)0 );
	assert( buffer_length != (size_t *)0 );
	assert( output_fd >= 0 );
	assert( input_fd >= 0 );
	assert( buffer != (char *)0 );
	assert( buffer_size >= (size_t)512 );

	DEBUG_FUNC_START;

	copy_cnt = (size_t)0;

	for ( ;; ) {
		if ( *buffer_length == (size_t)0 ) {
			/*
			** Refill empty buffer.
			*/
			if ( eof_flag ) {
				break;
			}
			read_cnt = read( input_fd, buffer, buffer_size );
			if ( read_cnt < 0 ) {
				UNIX_ERROR( "read() failed" );
				RETURN_INT( -1 );
			}

			if ( read_cnt == (ssize_t)0 ) {
				/*
				** EOF
				** Set eof_flag to true.
				*/
				eof_flag = 1;
				break;
			}

			*buffer_length = read_cnt;
			*remaining = buffer;
		}

		if ( output_file_remaining_cnt >= *buffer_length ) {
			assert( ( *remaining + *buffer_length ) <= ( buffer + buffer_size ) );
			write_status = write_block( output_fd, *remaining, *buffer_length );

			if ( write_status != *buffer_length ) {
				RETURN_INT( -1 );
			}

			output_file_remaining_cnt -= *buffer_length;
			copy_cnt += *buffer_length;
			*buffer_length = (size_t)0;
			continue;
		}

		/*
		** Look for end of line.
		*/
		ptr = *remaining + output_file_remaining_cnt;
		len = *buffer_length - output_file_remaining_cnt;

		assert( ( ptr + len ) <= ( buffer + buffer_size ) );
		eol_ptr = (char *)memchr( (void *)ptr, '\n', len );

		if ( eol_ptr != (char *)0 ) {
			/*
			** Found end of line.
			*/
			assert( eol_ptr >= *remaining );
			write_len = (size_t)( eol_ptr - *remaining ) + 1;
			assert( ( *remaining + write_len ) <= ( buffer + buffer_size ) );
			write_status = write_block( output_fd, *remaining, write_len );

			if ( write_status != write_len ) {
				RETURN_INT( -1 );
			}

			copy_cnt += write_len;
			*remaining += write_len;
			assert( *buffer_length >= write_len );
			*buffer_length -= write_len;
			break;
		}

		assert( ( *remaining + *buffer_length ) <= ( buffer + buffer_size ) );
		write_status = write_block( output_fd, *remaining, *buffer_length );

		if ( write_status != *buffer_length ) {
			RETURN_INT( -1 );
		}

		output_file_remaining_cnt = (size_t)0;
		copy_cnt += *buffer_length;
		*buffer_length = (size_t)0;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "copied %u, remaining %u\n", copy_cnt, *buffer_length );
	}

	RETURN_INT( 0 );
}

static ssize_t write_block( int fd, char *buffer, size_t content_size )
{
	ssize_t	write_cnt;

	write_cnt = write( fd, buffer, content_size );

	if ( write_cnt < 0 ) {
		UNIX_ERROR( "write() failed" );
		return write_cnt;
	}

	if ( write_cnt != content_size ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could only write ", stderr );
		(void) fput_long( write_cnt, stderr );
		(void) fputs( " bytes to output file.\n", stderr );
		return write_cnt;
	}

	return write_cnt;
}
