/*
 *   This file is part of AkariXB
 *   Copyright 2015-2016  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "connectiondialog.h"


ConnectionDialog::ConnectionDialog(QString jid, QString password,
                                   bool autoconnect, QString resource,
                                   int priority,
                                   QWidget *parent) : QWidget(parent)
{
    this->setWindowTitle(tr("Connection Settings") + " - AkariXB");
    this->setWindowIcon(QIcon::fromTheme("network-connect",
                                         QIcon(":/images/button-online.png")));
    this->setWindowFlags(Qt::Dialog);
    this->setWindowModality(Qt::WindowModal);
    this->setMinimumSize(460, 240);


    QString jidHelp = tr("Like user@server");
    this->jidLineEdit = new QLineEdit(jid, this);
    jidLineEdit->setPlaceholderText(jidHelp);
    jidLineEdit->setToolTip(jidHelp);
    connect(jidLineEdit, SIGNAL(textChanged(QString)),
            this, SLOT(validateJid(QString)));


    this->passwordLineEdit = new QLineEdit(password, this);
    passwordLineEdit->setEchoMode(QLineEdit::Password);
    connect(jidLineEdit, SIGNAL(returnPressed()),
            passwordLineEdit, SLOT(setFocus()));


    this->autoconnectCheckbox = new QCheckBox(this);
    autoconnectCheckbox->setChecked(autoconnect);


    this->resourceLineEdit = new QLineEdit(resource, this);


    this->prioritySpinbox = new QSpinBox(this);
    prioritySpinbox->setRange(-1, 100);
    prioritySpinbox->setValue(priority);


    this->connectButton = new QPushButton(QIcon::fromTheme("network-connect",
                                                           QIcon(":/images/button-online.png")),
                                          tr("&Connect"),
                                          this);
    connect(connectButton, SIGNAL(clicked()),
            this, SLOT(startConnection()));
    connect(passwordLineEdit, SIGNAL(returnPressed()),
            connectButton, SLOT(click()));


    this->cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel",
                                                          QIcon(":/images/button-cancel.png")),
                                         tr("Cancel"),
                                         this);
    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(hide()));


    this->cancelAction = new QAction(this);
    cancelAction->setShortcut(QKeySequence(Qt::Key_Escape));
    connect(cancelAction, SIGNAL(triggered()),
            this, SLOT(hide()));
    this->addAction(cancelAction);


    this->topLayout = new QFormLayout();
    topLayout->addRow(tr("XMPP Address (Jabber ID)"), jidLineEdit);
    topLayout->addRow(tr("Password"),                 passwordLineEdit);
    topLayout->addRow(tr("Connect Automatically"),    autoconnectCheckbox);
    topLayout->addRow(tr("Resource"),                 resourceLineEdit);
    topLayout->addRow(tr("Priority"),                 prioritySpinbox);

    this->buttonLayout = new QHBoxLayout();
    buttonLayout->setAlignment(Qt::AlignRight);
    buttonLayout->addWidget(connectButton);
    buttonLayout->addWidget(cancelButton);

    this->mainLayout = new QVBoxLayout();
    mainLayout->addLayout(topLayout);
    mainLayout->addSpacing(16);
    mainLayout->addStretch(0);
    mainLayout->addLayout(buttonLayout);
    this->setLayout(mainLayout);


    this->validateJid(jid);

    qDebug() << "ConnectionDialog created";
}


ConnectionDialog::~ConnectionDialog()
{
    qDebug() << "ConnectionDialog destroyed";
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////




void ConnectionDialog::validateJid(QString jid)
{
    this->connectButton->setEnabled(Helpers::isValidJid(jid));
}



void ConnectionDialog::startConnection()
{
    if (resourceLineEdit->text().trimmed().isEmpty())
    {
        resourceLineEdit->setText("AkariXB"); // Use default
    }

    emit connectionRequested(this->jidLineEdit->text().trimmed(),
                             this->passwordLineEdit->text(),
                             this->autoconnectCheckbox->isChecked(),
                             this->resourceLineEdit->text().trimmed(),
                             this->prioritySpinbox->value());

    this->hide();
}
