/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "post.h"

Post::Post(PumpController *pumpController,
           QMap<QString,QString> postData,
           QWidget *parent)  :  QFrame(parent)
{
    this->pController = pumpController;

    this->setFrameStyle(QFrame::Box | QFrame::Raised);
    //this->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken); // alternative?

    this->setSizePolicy(QSizePolicy::MinimumExpanding,
                        QSizePolicy::Maximum);


    leftColumnLayout = new QVBoxLayout();
    leftColumnLayout->setAlignment(Qt::AlignLeft | Qt::AlignTop);


    this->postID = postData.value("id");


    /////////////////////////////////////////////////// Left column, post Meta info


    // Author avatar
    postAuthorAvatar = new QLabel();

    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(postData.value("authorAvatar"));

    //qDebug() << "Avatar remote file:" << postData.value("authorAvatar");
    //qDebug() << "Avatar local file:" << avatarFilename;

    if (QFile::exists(avatarFilename))
    {
        // Load avatar if already cached
        postAuthorAvatar->setPixmap(QPixmap(avatarFilename)
                                    .scaled(48, 48,
                                            Qt::KeepAspectRatio,
                                            Qt::SmoothTransformation));
        qDebug() << "Using cached avatar";
    }
    else
    {
        // Placeholder image
        postAuthorAvatar->setPixmap(QIcon::fromTheme("user-identity").pixmap(48,48));
        qDebug() << "Using placeholder, downloading real avatar now";

        // Download avatar for next time
        pumpController->getAvatar(postData.value("authorAvatar"));
    }

    leftColumnLayout->addWidget(postAuthorAvatar);


    QFont authorFont;
    authorFont.setBold(true);
    authorFont.setUnderline(true);

    // Author name
    postAuthorName = new QLabel(postData.value("author"));
    postAuthorName->setTextFormat(Qt::PlainText);
    postAuthorName->setWordWrap(true);
    postAuthorName->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postAuthorName->setFont(authorFont);
    postAuthorName->setToolTip(postData.value("authorId"));

    leftColumnLayout->addWidget(postAuthorName);


    QFont detailsFont;
    detailsFont.setPointSize(detailsFont.pointSize() - 2); // FIXME: check size first

    // Post creation time
    QString createdAt = postData.value("createdAt");
    // Format is "Combined date and time in UTC", like "2012-02-07T01:32:02Z"
    // As specified in ISO 8601 http://en.wikipedia.org/wiki/ISO_8601
    QString exactPostTime = Timestamp::localTimeDate(createdAt);
    createdAt = Timestamp::fuzzyTime(createdAt);

    postCreatedAt = new QLabel(createdAt + "\n");
    postCreatedAt->setToolTip(exactPostTime);
    postCreatedAt->setTextFormat(Qt::PlainText);
    postCreatedAt->setWordWrap(true);
    postCreatedAt->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCreatedAt->setFont(detailsFont);
    leftColumnLayout->addWidget(postCreatedAt);


    // Like, comment, share buttons
    likeButton = new QPushButton(QIcon::fromTheme("emblem-favorite"), "");
    likeButton->setToolTip(tr("Like this post"));
    likeButton->setCheckable(true);
    connect(likeButton, SIGNAL(clicked()),
            this, SLOT(likePost()));

    commentButton = new QPushButton(QIcon::fromTheme("mail-reply-sender"), "");
    commentButton->setToolTip(tr("Comment on this post"));
    connect(commentButton, SIGNAL(clicked()),
            this, SLOT(commentPost()));

    shareButton = new QPushButton(QIcon::fromTheme("mail-forward"), "");
    shareButton->setToolTip(tr("Share this post"));
    connect(shareButton, SIGNAL(clicked()),
            this, SLOT(sharePost()));


    buttonsLayout = new QHBoxLayout();
    buttonsLayout->addWidget(likeButton);
    buttonsLayout->addWidget(commentButton);
    buttonsLayout->addWidget(shareButton);

    leftColumnLayout->addLayout(buttonsLayout);



    // Post type: StatusMessage or Reshare -- TODO: fix for pump.io
    if (postData.value("postType") == "Reshare")
    {
        postType = new QLabel(tr("Reshare of a post by:"));
        postType->setWordWrap(true);
        postType->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postType->setFont(detailsFont);
        leftColumnLayout->addWidget(postType);

        postRootAuthor = new QLabel(postData.value("rootAuthor"));
        postRootAuthor->setTextFormat(Qt::PlainText);
        postRootAuthor->setWordWrap(true);
        postRootAuthor->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postRootAuthor->setFont(detailsFont);
        leftColumnLayout->addWidget(postRootAuthor);
    }


    QString likesCount = postData.value("likesCount");
    if (likesCount != "0")
    {
        postLikesCount = new QLabel(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("%1 likes").arg(likesCount));
        postLikesCount->setWordWrap(true);
        postLikesCount->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postLikesCount->setFont(detailsFont);
        leftColumnLayout->addWidget(postLikesCount);
    }

    QString commentsCount = postData.value("commentsCount");
    if (commentsCount != "0")
    {
        postCommentsCount = new QLabel(tr("%1 comments").arg(commentsCount));
        postCommentsCount->setWordWrap(true);
        postCommentsCount->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postCommentsCount->setFont(detailsFont);
        leftColumnLayout->addWidget(postCommentsCount);
    }

    QString resharesCount = postData.value("resharesCount");
    if (resharesCount != "0")
    {
        postResharesCount = new QLabel(QString::fromUtf8("\342\231\272 ") // recycle symbol
                                     + tr("%1 shares").arg(resharesCount));
        postResharesCount->setWordWrap(true);
        postResharesCount->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postResharesCount->setFont(detailsFont);
        leftColumnLayout->addWidget(postResharesCount);
    }




    ////////////////////////////////////////////////// Right column, content
    ///////////////////////// Post text

    postText = new QTextBrowser();
    postText->setAlignment(Qt::AlignLeft | Qt::AlignTop);

    postText->setWordWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);
    postText->setOpenExternalLinks(true); // don't open HTTP links, open in browser
    //postText->setWordWrapMode(QTextOption::NoWrap);
    //postText->setWordWrapMode(QTextOption::WordWrap);

    postText->setReadOnly(true); // it's default with QTextBrowser, but still...


    /****
    // Insert thumbnail of embedded stuff (mainly videos)
    QString postOembedThumbnailUrl = postData.value("oembedThumbnailUrl");
    qDebug() << "oEmbed Thumbnail URL:" << postOembedThumbnailUrl;
    if (!postOembedThumbnailUrl.isEmpty())
    {
        postOembedThumbnailUrl = QString("![embeddedStuff](%1)\n\n")
                                 .arg(postOembedThumbnailUrl);
    }
    */


    // Get URL of post image, if it's "image" type of post
    QString postTitle;
    QString postImage;
    if (postData.value("postType") == "image")
    {
        postImage = postData.value("image");

        if (QFile::exists(MiscHelpers::getCachedImageFilename(postImage)))
        {
            qDebug() << "Using cached image";
        }
        else
        {
            qDebug() << "Image not cached, downloading";
            pumpController->getImage(postImage);
        }

        postTitle = "<h2>" + postData.value("title") + "</h2><br>";
        postImage = "<img src=\"" + MiscHelpers::getCachedImageFilename(postImage) + "\" width=\"480\"></img><br><br>";
    }



    // Add the text content of the post
    postText->setHtml(postTitle
                    + postImage
                    + postData.value("text"));


    int postLines = postText->document()->lineCount();
    qDebug() << "Post lines:" << postLines;

    int postMinimumHeight = postLines * 3 * postText->fontInfo().pixelSize();
    if (postMinimumHeight > 400)  // Don't allow a post to be higher than 400 px
    {
        postMinimumHeight = 400;  // Scrollbars will appear
        qDebug() << "Post would require more space: scrollbars might appear";
    }
    qDebug() << "Post minimum height:" << postMinimumHeight;
    postText->setMinimumHeight(postMinimumHeight);


    // NSFW button, to replace #nsfw posts until clicked
    nsfwCoveringButton = new QPushButton(tr("This post has been marked Not Safe For Work (#NSFW).\n"
                                            "Click here to show it."));
    nsfwCoveringButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                      QSizePolicy::MinimumExpanding);
    connect(nsfwCoveringButton, SIGNAL(clicked()),
            this, SLOT(showNsfwPost()));


    mainLayout = new QHBoxLayout();

    mainLayout->addLayout(leftColumnLayout, 1); // stretch 1/6
    mainLayout->addWidget(postText,         5); // stretch 5/6
    if (postData.value("nsfw") == "true")
    {
        postText->hide();
        mainLayout->addWidget(nsfwCoveringButton, 5);
    }

    mainLayout->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    this->setLayout(mainLayout);

    qDebug() << "Post created";
}



Post::~Post()
{
    qDebug() << "Post destroyed";
}


void Post::resizeEvent(QResizeEvent *event)
{
    //postText->adjustSize();
    //postText->resize(postText->contentsRect().size());
    //postText->resize(postText->frameSize());

    //this->adjustSize();

    postText->scroll(0, 0);
    event->accept();
}



//////////////////////////////// SLOTS ///////////////////////////////////


void Post::showNsfwPost()
{
    this->nsfwCoveringButton->hide();
    this->postText->show();

    qDebug() << "Shown NSFW post";
}


void Post::likePost()
{
    qDebug() << "Liking this post";

    this->pController->likePost(this->postID);
}


void Post::commentPost()
{
    qDebug() << "Commenting on this post";
}


void Post::sharePost()
{
    qDebug() << "Sharing this post";
}
