/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "composer.h"


Composer::Composer()
{
    this->setAcceptRichText(true);
    this->setTabChangesFocus(true);
    //this->setCursorWidth(2);

    this->setToolTip(tr("Type a message here to post it"));

    mainLayout = new QGridLayout();

    startConversationLabel = new QLabel(tr("Click here or press Control+N to post a note..."));


    toolsMenu = new QMenu("tools-menu");
    toolsMenu->addAction(QIcon::fromTheme(""),
                         tr("Normal"),
                         this,
                         SLOT(makeNormal()));
    toolsMenu->addAction(QIcon::fromTheme("format-text-bold"),
                         tr("Bold"),
                         this,
                         SLOT(makeBold()));
    toolsMenu->addAction(QIcon::fromTheme("format-text-italic"),
                         tr("Italic"),
                         this,
                         SLOT(makeItalic()));
    toolsMenu->addAction(QIcon::fromTheme(""),
                         tr("Bold + Italic"),
                         this,
                         SLOT(makeBoldAndItalic()));
    toolsMenu->addAction(QIcon::fromTheme("format-text-underline"),
                         tr("Underline"),
                         this,
                         SLOT(makeUnderline()));
    toolsMenu->addSeparator();
    toolsMenu->addAction(QIcon::fromTheme("insert-link"),
                         tr("Make a link"),
                         this,
                         SLOT(makeLink()));
    /*
    toolsMenu->addAction(QIcon::fromTheme("insert-image"),
                         tr("Insert an image from a web site"),
                         this,
                         SLOT(insertImage()));
    */
    toolsMenu->addSeparator();
    toolsMenu->addAction(QIcon::fromTheme("dialog-cancel"),
                         tr("Cancel Post"),
                         this,
                         SLOT(cancelPost()));

    toolsButton = new QPushButton(QIcon::fromTheme("format-list-ordered"), tr("&F", "F for formatting, translate accordingly"));
    toolsButton->setMenu(toolsMenu);
    toolsButton->setToolTip(tr("Text Formatting Options"));
    toolsButton->setHidden(true); // Hidden on startup (Publisher in miniMode)

    mainLayout->addWidget(startConversationLabel, 0, 0, 1, 1, Qt::AlignLeft | Qt::AlignVCenter);
    mainLayout->addWidget(toolsButton,            0, 1, 1, 1, Qt::AlignRight | Qt::AlignTop);


    this->setLayout(mainLayout);

    qDebug() << "Composer box created";
}

Composer::~Composer()
{
    qDebug() << "Composer box destroyed";
}



void Composer::erase()
{
    this->clear();

    startConversationLabel->show();

    toolsButton->hide();  /// FIXME: maybe move this line to focusOutEvent
}


/*
 * Send a signal when getting focus
 *
 */
void Composer::focusInEvent(QFocusEvent *event)
{
    emit focusReceived(); // inform Publisher() that we are focused

    startConversationLabel->hide(); // hide placeholder message
    toolsButton->show();

    QTextEdit::focusInEvent(event); // process standard event: allows context menu


    qDebug() << "Composer box got focus";
}



void Composer::keyPressEvent(QKeyEvent *event)
{
    //qDebug() << "Composer::keyPressEvent()";
    //qDebug() << event->modifiers();
    //qDebug() << event->key();

    // Control+Enter = Send message (post)
    if ((event->key() == Qt::Key_Enter || event->key() == Qt::Key_Return)
      && event->modifiers() == Qt::ControlModifier)
    {
        qDebug() << "Control+Enter was pressed";
        emit editingFinished();
    }
    else if (event->key() == Qt::Key_Escape)
    {
        qDebug() << "Escape was pressed";
        if (this->toPlainText().isEmpty())
        {
            qDebug() << "There was no text, cancelling post";
            this->cancelPost();
        }
    }
    else
    {
        QTextEdit::keyPressEvent(event);
    }
}



/****************************** SLOTS ************************************/



/*
 * Remove text formatting from selection, bold, italic, etc.
 *
 */
void Composer::makeNormal()
{
    this->setCurrentCharFormat(QTextCharFormat());
}



/*
 * Make selected text bold, by adding <b> and </b>
 *
 */
void Composer::makeBold()
{
    //qDebug() << this->textCursor().selectionStart() << " -> "  << this->textCursor().selectionEnd();

    QString selectedText = this->textCursor().selectedText();

    if (!selectedText.isEmpty())
    {
        this->textCursor().removeSelectedText();
        this->insertHtml("<b>" + selectedText + "</b>");
    }

    this->setFocus(); // give focus back to text editor
}


/*
 * Make selected text italic, by adding <i>
 *
 */
void Composer::makeItalic()
{
    QString selectedText = this->textCursor().selectedText();

    if (!selectedText.isEmpty())
    {
        this->textCursor().removeSelectedText();
        this->insertHtml("<i>" + selectedText + "</i>");
    }

    this->setFocus();
}



/*
 * Make selected text bold AND italic, by adding <b><i>
 *
 */
void Composer::makeBoldAndItalic()
{
    QString selectedText = this->textCursor().selectedText();

    if (!selectedText.isEmpty())
    {
        this->textCursor().removeSelectedText();
        this->insertHtml("<b><i>" + selectedText + "</i></b>");
    }

    this->setFocus(); // give focus back to text editor
}


/*
 * Underline selected text italic, by adding <u> </u>
 *
 */
void Composer::makeUnderline()
{
    QString selectedText = this->textCursor().selectedText();

    if (!selectedText.isEmpty())
    {
        this->textCursor().removeSelectedText();
        this->insertHtml("<u>" + selectedText + "</u>");
    }

    this->setFocus();
}




/*
 * Convert selected text into a link
 *
 */
void Composer::makeLink()
{
    QString selectedText = this->textCursor().selectedText();
    QString link;


    if (selectedText.isEmpty())
    {
        link = QInputDialog::getText(this,
                                     tr("Insert a link"),
                                     tr("Type or paste a web address here.\n"),
                                     QLineEdit::Normal,
                                     "http://");
        selectedText = link;
    }
    else
    {
        link = QInputDialog::getText(this,
                                     tr("Make a link from selected text"),
                                     tr("Type or paste a web address here.\n"
                                        "The selected text (%1) will be converted to a link.").arg(selectedText),
                                     QLineEdit::Normal,
                                     "http://");
    }



    if (!link.isEmpty())
    {
        this->textCursor().removeSelectedText();
        this->insertHtml("<a href=\"" + link + "\">" + selectedText + "</a>");
    }
    else
    {
        qDebug() << "makeLink(): Invalid URL";
    }

    this->setFocus();
}



/*
 * Insert an image from a URL
 *
 */
void Composer::insertImage()
{
    QString selectedText = this->textCursor().selectedText();
    QString imageURL;

    if (selectedText.isEmpty())
    {
        imageURL = QInputDialog::getText(this,
                                         tr("Insert and image from a URL"),
                                         tr("Type or paste the image address here.\n"),
                                         QLineEdit::Normal,
                                         "http://");
        selectedText = "image/url";
    }
    else
    {
        imageURL = QInputDialog::getText(this,
                                         tr("Insert and image from a URL"),
                                         tr("Type or paste the image address here.\n"
                                            "The selected text (%1) will be used as description.").arg(selectedText),
                                         QLineEdit::Normal,
                                         "http://");
    }



    if (!imageURL.isEmpty())
    {
        this->textCursor().removeSelectedText();
        this->insertHtml("<img alt=\"" + selectedText + "\" src=\"" + imageURL + "\"></img>");
    }
    else
    {
        qDebug() << "insertImage(): Invalid image URL";
    }

    this->setFocus();
}




/*
 * Cancel editing of the post, clear it, return to minimum mode
 *
 */
void Composer::cancelPost()
{
    this->erase();

    // emit signal to make Publisher go back to minimum mode
    emit editingCancelled();

    qDebug() << "Post cancelled";
}
