/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "post.h"

Post::Post(PumpController *pumpController,
           QMap<QString,QString> postData,
           QVariantList commentsVariantList,
           QWidget *parent)  :  QFrame(parent)
{
    this->pController = pumpController;

    this->setFrameStyle(QFrame::Box | QFrame::Raised);
    //this->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken); // alternative?

    this->setSizePolicy(QSizePolicy::MinimumExpanding,
                        QSizePolicy::Maximum);


    leftColumnLayout = new QVBoxLayout();
    leftColumnLayout->setAlignment(Qt::AlignLeft | Qt::AlignTop);


    this->postID = postData.value("id");
    this->postType = postData.value("postType");


    /////////////////////////////////////////////////// Left column, post Meta info

    QString authorTooltipInfo = postData.value("authorId");
    authorTooltipInfo.append("\n");
    authorTooltipInfo.append(tr("Hometown") + ": " + postData.value("authorHometown"));
    authorTooltipInfo.append("\n\n");
    authorTooltipInfo.append(postData.value("authorBio"));

    if (postData.value("authorId") == pController->currentUserID())
    {
        postIsOwn = true;
        qDebug() << "post is our own!";
    }
    else
    {
        postIsOwn = false;
    }

    // Author avatar
    postAuthorAvatar = new QLabel();

    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(postData.value("authorAvatar"));

    //qDebug() << "Avatar remote file:" << postData.value("authorAvatar");
    //qDebug() << "Avatar local file:" << avatarFilename;

    if (QFile::exists(avatarFilename))
    {
        // Load avatar if already cached
        postAuthorAvatar->setPixmap(QPixmap(avatarFilename)
                                    .scaled(48, 48,
                                            Qt::KeepAspectRatio,
                                            Qt::SmoothTransformation));
        qDebug() << "Using cached avatar";
    }
    else
    {
        // Placeholder image
        postAuthorAvatar->setPixmap(QIcon::fromTheme("user-identity").pixmap(48,48));
        qDebug() << "Using placeholder, downloading real avatar now";

        // Download avatar for next time
        pumpController->getAvatar(postData.value("authorAvatar"));
    }
    postAuthorAvatar->setToolTip(authorTooltipInfo);

    leftColumnLayout->addWidget(postAuthorAvatar);


    QFont authorFont;
    authorFont.setBold(true);
    authorFont.setUnderline(true);
    authorFont.setPointSize(authorFont.pointSize()-1);

    // Author name
    postAuthorName = new QLabel(postData.value("author"));
    postAuthorName->setTextFormat(Qt::PlainText);
    postAuthorName->setWordWrap(true);
    postAuthorName->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postAuthorName->setFont(authorFont);
    postAuthorName->setToolTip(authorTooltipInfo);

    leftColumnLayout->addWidget(postAuthorName);


    QFont detailsFont;
    detailsFont.setPointSize(detailsFont.pointSize() - 2); // FIXME: check size first

    // Post creation time
    QString createdAt = postData.value("createdAt");
    // Format is "Combined date and time in UTC", like "2012-02-07T01:32:02Z"
    // As specified in ISO 8601 http://en.wikipedia.org/wiki/ISO_8601
    QString exactPostTime = Timestamp::localTimeDate(createdAt);
    createdAt = Timestamp::fuzzyTime(createdAt);

    postCreatedAt = new QLabel(createdAt + "\n");
    postCreatedAt->setToolTip(exactPostTime + " " + tr("using") + " " + postData.value("generator"));
    postCreatedAt->setTextFormat(Qt::PlainText);
    postCreatedAt->setWordWrap(true);
    postCreatedAt->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCreatedAt->setFont(detailsFont);
    leftColumnLayout->addWidget(postCreatedAt);


    // Like, comment, share buttons
    likeButton = new QPushButton(QIcon::fromTheme("emblem-favorite"), tr("Like"));
    likeButton->setToolTip(tr("Like this post"));
    likeButton->setCheckable(true);
    likeButton->setFlat(true);
    connect(likeButton, SIGNAL(clicked()),
            this, SLOT(likePost()));

    commentButton = new QPushButton(QIcon::fromTheme("mail-reply-sender"), tr("Comment"));
    commentButton->setToolTip(tr("Comment on this post"));
    commentButton->setFlat(true);
    connect(commentButton, SIGNAL(clicked()),
            this, SLOT(commentOnPost()));

    shareButton = new QPushButton(QIcon::fromTheme("mail-forward"), tr("Share"));
    shareButton->setToolTip(tr("Share this post"));
    shareButton->setFlat(true);
    connect(shareButton, SIGNAL(clicked()),
            this, SLOT(sharePost()));

    /* // Type-A button arrangement
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->addWidget(likeButton);
    buttonsLayout->addWidget(commentButton);
    buttonsLayout->addWidget(shareButton);
    buttonsLayout->setContentsMargins(0,0,0,0);
    buttonsLayout->setAlignment(Qt::AlignLeft);
    buttonsLayout->setMargin(0);
    buttonsLayout->setSpacing(0);

    leftColumnLayout->addLayout(buttonsLayout);
    /**/



    // Is the post a reshare? // FIXME
    if (postData.value("postIsShared") == "true")
    {
        postTypeLabel = new QLabel(tr("Reshared post"));
        postTypeLabel->setWordWrap(true);
        postTypeLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postTypeLabel->setFont(detailsFont);
        leftColumnLayout->addWidget(postTypeLabel);
/*
        postRootAuthor = new QLabel(postData.value("rootAuthor"));
        postRootAuthor->setTextFormat(Qt::PlainText);
        postRootAuthor->setWordWrap(true);
        postRootAuthor->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postRootAuthor->setFont(detailsFont);
        leftColumnLayout->addWidget(postRootAuthor);*/
    }


    QString likesCount = postData.value("likesCount");
    if (likesCount != "0")
    {
        postLikesCount = new QLabel(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("%1 likes").arg(likesCount));
        postLikesCount->setWordWrap(true);
        postLikesCount->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postLikesCount->setFont(detailsFont);
        leftColumnLayout->addWidget(postLikesCount);
    }

    QString commentsCount = postData.value("commentsCount");
    if (commentsCount != "0")
    {
        postCommentsCount = new QLabel(tr("%1 comments").arg(commentsCount));
        postCommentsCount->setWordWrap(true);
        postCommentsCount->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postCommentsCount->setFont(detailsFont);
        leftColumnLayout->addWidget(postCommentsCount);
    }

    QString resharesCount = postData.value("resharesCount");
    if (resharesCount != "0")
    {
        postResharesCount = new QLabel(QString::fromUtf8("\342\231\272 ") // recycle symbol
                                     + tr("%1 shares").arg(resharesCount));
        postResharesCount->setWordWrap(true);
        postResharesCount->setAlignment(Qt::AlignLeft | Qt::AlignTop);
        postResharesCount->setFont(detailsFont);
        leftColumnLayout->addWidget(postResharesCount);
    }




    ////////////////////////////////////////////////// Right column, content

    rightColumnLayout = new QVBoxLayout();
    rightColumnLayout->setAlignment(Qt::AlignRight | Qt::AlignTop);


    ///////////////////////////////////// Post text
    postText = new QTextBrowser();
    postText->setAlignment(Qt::AlignLeft | Qt::AlignTop);

    postText->setWordWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);
    postText->setOpenExternalLinks(true); // don't open HTTP links, open in browser
    postText->setReadOnly(true); // it's default with QTextBrowser, but still...


    // Add like, comment, share and, if post is our own, delete buttons
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignHCenter | Qt::AlignBottom);
    buttonsLayout->setContentsMargins(0, 0, 0, 0);
    buttonsLayout->setMargin(0);
    buttonsLayout->setSpacing(0);
    buttonsLayout->addWidget(likeButton,    0, Qt::AlignLeft);
    buttonsLayout->addWidget(commentButton, 0, Qt::AlignLeft);
    buttonsLayout->addWidget(shareButton,   0, Qt::AlignLeft);
    buttonsLayout->addStretch(1); // so the (optional) Delete buttons gets separated

    if (postIsOwn)
    {
        deleteButton = new QPushButton(QIcon::fromTheme("edit-delete"), tr("Delete"));
        deleteButton->setToolTip(tr("Delete this post"));
        deleteButton->setFlat(true);
        connect(deleteButton, SIGNAL(clicked()),
                this, SLOT(deletePost()));

        buttonsLayout->addWidget(deleteButton, 0, Qt::AlignRight);
    }

    // Type-B button arrangement
    // postText->setLayout(buttonsLayout);



    // Get URL of post image, if it's "image" type of post
    QString postTitle;
    QString postImage;
    if (this->postType == "image")
    {
        postImage = postData.value("image");

        if (QFile::exists(MiscHelpers::getCachedImageFilename(postImage)))
        {
            qDebug() << "Using cached image";
        }
        else
        {
            qDebug() << "Image not cached, downloading";
            pumpController->getImage(postImage);
        }

        postTitle = "<h2>" + postData.value("title") + "</h2><br>";
        postImage = "<img src=\"" + MiscHelpers::getCachedImageFilename(postImage) + "\" width=\"400\"></img><br><br>";
    }



    // Add the text content of the post
    postText->setHtml(postTitle
                    + postImage
                    + postData.value("text"));


    int postLines = postText->document()->lineCount();
    qDebug() << "Post lines:" << postLines;

    int postMinimumHeight = postLines * 3 * postText->fontInfo().pixelSize();
    if (postMinimumHeight > 400)  // Don't allow a post to be higher than 400 px
    {
        postMinimumHeight = 400;  // Scrollbars will appear
        qDebug() << "Post would require more space: scrollbars might appear";
    }
    qDebug() << "Post minimum height:" << postMinimumHeight;
    postText->setMinimumHeight(postMinimumHeight);


    // NSFW button, to replace #nsfw posts until clicked
    nsfwCoveringButton = new QPushButton(tr("This post has been marked Not Safe For Work (#NSFW).\n"
                                            "Click here to show it."));
    nsfwCoveringButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                      QSizePolicy::MinimumExpanding);
    connect(nsfwCoveringButton, SIGNAL(clicked()),
            this, SLOT(showNsfwPost()));


    /////////////////////////////////////////////////////// Comments block
    this->commenter = new Commenter();
    connect(commenter, SIGNAL(commentSent(QString)),
            this, SLOT(sendComment(QString)));


    rightColumnLayout->addWidget(postText,  7);
    rightColumnLayout->addLayout(buttonsLayout);
    rightColumnLayout->addWidget(commenter, 1);

    QString commentsString;
    foreach (QVariant commentMap, commentsVariantList)
    {
        commentsString = "<" + commentMap.toMap().value("author").toMap().value("displayName").toString() + "> "
                + commentMap.toMap().value("content").toString() + "\n"
                + commentsString;
    }

    this->commenter->setCommments(commentsString);



    mainLayout = new QHBoxLayout();
    mainLayout->addLayout(leftColumnLayout,  1); // stretch 1/6
    mainLayout->addLayout(rightColumnLayout, 5); // stretch 5/6

    if (postData.value("nsfw") == "true")
    {
        postText->hide();
        mainLayout->addWidget(nsfwCoveringButton, 5);
    }

    mainLayout->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    this->setLayout(mainLayout);


    qDebug() << "Post created";
}



Post::~Post()
{
    qDebug() << "Post destroyed";
}


void Post::resizeEvent(QResizeEvent *event)
{
    //postText->adjustSize();
    //postText->resize(postText->contentsRect().size());
    //postText->resize(postText->frameSize());

    //this->adjustSize();

    postText->scroll(0, 0);
    event->accept();
}



//////////////////////////////// SLOTS ///////////////////////////////////


void Post::showNsfwPost()
{
    this->nsfwCoveringButton->hide();
    this->postText->show();

    qDebug() << "Shown NSFW post";
}


/*
 * Like (favorite) a post
 *
 */
void Post::likePost()
{
    qDebug() << "Liking this post";

    this->pController->likePost(this->postID, this->postType);
}


/*
 * Make the commenter widget visible, so user can type the comment
 *
 */
void Post::commentOnPost()
{
    qDebug() << "Commenting on this post";

    this->commenter->setFullMode();
}


/*
 * The actual sending of the comment to the Pump controller
 *
 */
void Post::sendComment(QString comment)
{
    qDebug() << "About to publish this comment:" << comment;

    this->pController->addComment(comment, this->postID, this->postType);
}


/*
 * Re-share a post
 *
 */
void Post::sharePost()
{
    int confirmation = QMessageBox::question(this, tr("Share post?"),
                                             tr("Do you want to share %1's post?").arg(this->postAuthorName->text()),
                                             tr("&Yes, share it"), tr("&No"));

    if (confirmation == 0)
    {
        qDebug() << "Sharing this post";
        this->pController->sharePost(this->postID, this->postType);
    }
    else
    {
        qDebug() << "Confirmation cancelled, not sharing";
    }
}



/*
 * Delete a post
 *
 */
void Post::deletePost()
{
    int confirmation = QMessageBox::question(this, tr("Delete post?"),
                                             tr("Are you sure you want to delete this post?"),
                                             tr("&Yes, delete it"), tr("&No"));

    if (confirmation == 0)
    {
        qDebug() << "Deleting post";
        this->pController->deletePost(this->postID, this->postType);
    }
    else
    {
        qDebug() << "Confirmation cancelled, not deleting the post";
    }
}
