/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "pumpcontroller.h"

PumpController::PumpController(QObject *parent) :  QObject(parent)
{
    this->userAgentString = "Dianara/0.5";

    this->postsPerPage = 20;

    qoauth = new QOAuth::Interface();
    qoauth->setRequestTimeout(10000); // 10 sec timeout


    QSettings settings;
    this->clientID = settings.value("clientID", "").toString();
    this->clientSecret = settings.value("clientSecret", "").toString();
    qoauth->setConsumerKey(clientID.toLocal8Bit());
    qoauth->setConsumerSecret(clientSecret.toLocal8Bit());


    this->isApplicationAuthorized = settings.value("isApplicationAuthorized", false).toBool();

    if (isApplicationAuthorized)
    {
        qDebug() << "Dianara is already authorized for user ID:" << this->userID;

        this->token = settings.value("token", "").toString().toLocal8Bit();
        this->tokenSecret = settings.value("tokenSecret", "").toString().toLocal8Bit();

        qDebug() << "Using token" << token;
        qDebug() << "And token secret" << tokenSecret;
    }
    emit this->authorizationStatusChanged(isApplicationAuthorized);




    connect(&nam, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)));

    // FIXME: setting this up for now, to at least have debug messages just in case
    connect(&nam, SIGNAL(sslErrors(QNetworkReply*,QList<QSslError>)),
            this, SLOT(sslErrorsHandler(QNetworkReply*,QList<QSslError>)));

    this->initialDataStep = 0;

    initialDataTimer = new QTimer(this);
    initialDataTimer->setSingleShot(false); // Triggered constantly until stopped
    connect(initialDataTimer, SIGNAL(timeout()),
            this, SLOT(getInitialData()));



    qDebug() << "PumpController created";
}


PumpController::~PumpController()
{
    qDebug() << "PumpController destroyed";
}



void PumpController::setPostsPerPage(int ppp)
{
    this->postsPerPage = ppp;
    qDebug() << "PumpController: setting postsPerPage to" << this->postsPerPage;
}



/*
 * Set new user ID (user@domain.tld) and clear OAuth-related tokens/secrets
 *
 *
 */
void PumpController::setNewUserID(QString userID)
{
    this->userID = userID;
    QStringList splittedUserID = this->userID.split("@");
    this->userName = splittedUserID.at(0); // get username, before @
    this->serverURL = splittedUserID.at(1); // get URL, after @

    qDebug() << "Server URL to connect:" << serverURL << "; username:" << userName;

    this->clientID.clear();
    this->clientSecret.clear();
    this->token.clear();
    this->tokenSecret.clear();

    this->isApplicationAuthorized = false;
    emit this->authorizationStatusChanged(isApplicationAuthorized);
}




/*
 * Get "pumpserver.org" and "user" from "user@pumpserver.org", set OAuth token from Account dlg
 *
 */
void PumpController::setUserCredentials(QString userID)
{
    this->initialDataTimer->stop(); // Just in case it was running before


    this->userID = userID;
    QStringList splittedUserID = this->userID.split("@");
    this->userName = splittedUserID.at(0);
    this->serverURL = splittedUserID.at(1);
    qDebug() << "New userID is:" << this->userID;


    this->getUserProfile(this->userID);


    // This will call getContactList() and getTimeLine(), etc.
    this->initialDataStep = 0;
    this->initialDataTimer->start(100);  // start immediately
}



QString PumpController::currentUserID()
{
    return this->userID;
}





/*
 * Get any user's profile (not only our own)
 *
 * GET https://pumpserver.example/api/user/username
 *
 */
void PumpController::getUserProfile(QString userID)
{
    QStringList splittedUserID = userID.split("@");

    QString url = "https://" + splittedUserID.at(1) + "/api/user/" + splittedUserID.at(0);

    QNetworkRequest userProfileRequest = this->prepareRequest(url, QOAuth::GET, UserProfileRequest);
    nam.get(userProfileRequest);

    qDebug() << "Requested user profile:" << userProfileRequest.url();
}




void PumpController::getAvatar(QString avatarURL)
{
    qDebug() << "Getting avatar";


    QNetworkRequest avatarRequest(QUrl((const QString)avatarURL));
    avatarRequest.setRawHeader("User-Agent", userAgentString);
    avatarRequest.setAttribute(QNetworkRequest::User,
                               QVariant(AvatarRequest));

    nam.get(avatarRequest);
}



void PumpController::getImage(QString imageURL)
{
    // FIXME: now we get BadRequest on public images on other hosts
    // Probably need to use "proxyURL" image

    QNetworkRequest imageRequest = this->prepareRequest(imageURL, QOAuth::GET, ImageRequest);

    nam.get(imageRequest);
    qDebug() << "imageRequest sent";
}




/*
 * GET https://pumpserver.example/api/......
 *
 */
void PumpController::getContactList()
{
    emit currentJobChanged(tr("Getting contact list..."));
    qDebug() << "Getting contact list";

    QString url = "https://" + this->serverURL + "/api/user/" +this->userName + "/following";
                                                                                // also /followers?

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "100");  // TMP/FIXME get 100 contacts

    QNetworkRequest contactListRequest = this->prepareRequest(url, QOAuth::GET,
                                                              ContactListRequest, paramMap);
    nam.get(contactListRequest);
}




/*
 * GET https://pumpserver.example/api/username/inbox/major
 *
 */
void PumpController::getTimeline(int timelineOffset)
{
    emit currentJobChanged(tr("Getting timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/major";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPage).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           TimelineRequest, paramMap);


    qDebug() << "===================================";
    qDebug() << "Getting timeline" << url;
    qDebug() << "with params:" << paramMap;
    qDebug() << "\nAuthorization Header:" << timelineRequest.rawHeader("Authorization");
    qDebug() << "\n\nFinal URL to retrieve:" << timelineRequest.url();
    qDebug() << "===================================";

    nam.get(timelineRequest);
}



/*
 * Get one specific post
 *
 * GET https://pumpserver.example/api/posts/.....
 *
 */
void PumpController::getPost(int id)
{
    qDebug() << "Getting post" << id;


    QNetworkRequest postRequest(QUrl("https://" + this->serverURL
                                     + "/api/posts/" + QString("%1").arg(id) ));
    postRequest.setRawHeader("User-Agent", userAgentString);
    postRequest.setAttribute(QNetworkRequest::User,
                             QVariant(PostRequest));

    nam.get(postRequest);
}



/*
 * Prepare a QNetworkRequest with OAuth header, content type and user agent.
 *
 */
QNetworkRequest PumpController::prepareRequest(QString url, QOAuth::HttpMethod method,
                                               int requestType, QOAuth::ParamMap paramMap)
{
    QByteArray authorizationHeader = qoauth->createParametersString(url,
                                                                 method,
                                                                 this->token,
                                                                 this->tokenSecret,
                                                                 QOAuth::HMAC_SHA1,
                                                                 paramMap,
                                                                 QOAuth::ParseForHeaderArguments);

    QNetworkRequest request;
    // Don't append inline parameters if it's an image request, it messes up the filename
    if (requestType != ImageRequest)
    {
        url.append(qoauth->inlineParameters(paramMap, QOAuth::ParseForInlineQuery));
    }
    request.setUrl(QUrl(url));
    request.setRawHeader("Authorization", authorizationHeader);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
    request.setRawHeader("User-Agent", userAgentString);


    request.setAttribute(QNetworkRequest::User,  QVariant(requestType));

    return request;
}



/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/*********************************** SLOTS *********************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/




void PumpController::requestFinished(QNetworkReply *reply)
{
    int httpCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    int requestType = reply->request().attribute(QNetworkRequest::User).toInt();

    qDebug() << "Request finished. HTTP code:" << httpCode;
    qDebug() << "Size:" << reply->size() << "bytes; URL:" << reply->url();
    qDebug() << "isFinished()?" << reply->isFinished() << "; Request type:" << requestType;

    switch (httpCode)
    {
    case 503:
        emit showNotification("Service Unavailable (503)\n" + reply->url().toString());
        qDebug() << "HTTP 503: Service Unavailable.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 500:
        emit showNotification("Internal server error (500)\n" + reply->url().toString());
        qDebug() << "HTTP 500: Internal server error.";
        qDebug() << "Data:  " << reply->readAll();
        return;


    case 404:
        emit showNotification("Not Found (404)\n" + reply->url().toString());
        qDebug() << "HTTP 404: Not Found.";
        qDebug() << "Data:  " << reply->readAll();
        return;

    case 403:
        emit showNotification("Forbidden (403)\n" + reply->url().toString());
        qDebug() << "HTTP 403: Forbidden.";
        qDebug() << "Data:  " << reply->readAll();
        return;


    case 400:
        emit showNotification("Bad Request (400)\n" + reply->url().toString());
        qDebug() << "HTTP 400: Bad Request.";
        qDebug() << "Data:  " << reply->readAll();
        //qDebug() << reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute);
        return;


    case 200:
        qDebug() << "HTTP 200: OK!";
    }


    if (httpCode == 200) // If OK received
    {
        switch (requestType) // Inform in status bar which operation was successful
        {
        case PublishPostRequest:
            emit currentJobChanged(tr("Post published successfully."));
            break;

        case LikePostRequest:
            emit currentJobChanged(tr("Post liked successfully."));
            break;

        case CommentPostRequest:
            emit currentJobChanged(tr("Comment posted successfully."));
            break;

        case SharePostRequest:
            emit currentJobChanged(tr("Post shared successfully."));
            break;

        case DeletePostRequest:
            emit currentJobChanged(tr("Post deleted successfully."));
            break;

        case FollowContactRequest:
            emit currentJobChanged(tr("Following successfully."));

            // Re-request the contact list, now with a new contact (not very optimized...)
            this->getContactList();
            break;

        case UnfollowContactRequest:
            emit currentJobChanged(tr("Stopped following successfully."));

            // Re-request the contact list, now with one contact less (not very optimized...)
            this->getContactList();
            break;

        }
    }


    // Read all received data
    QByteArray packetData = reply->readAll();

    // Prepare the JSON parser
    QJson::Parser jsonParser;
    bool jsonParsedOK = false;
    QVariantMap jsonData;
    QVariantList jsonDataList;


    switch (requestType)
    {

    case ClientRegistrationRequest:
        qDebug() << "Client Registration was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << jsonData;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            this->clientID = jsonData["client_id"].toString();
            this->clientSecret = jsonData["client_secret"].toString();

            // FIXME: error control, etc.
            // check if jsonData.keys().contains("client_id") !!

            QSettings settings;
            settings.setValue("clientID",     this->clientID);
            settings.setValue("clientSecret", this->clientSecret);


            this->getToken();
        }


        break;



    case UserProfileRequest:
        qDebug() << "A user profile was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << jsonData;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            QString profId = jsonData["profile"].toMap()["id"].toString();
            if (profId == "acct:" + this->userID)
            {
                qDebug() << "Received OWN profile";

                QString profDisplayName = jsonData["profile"].toMap()["displayName"].toString();
                QString profImageUrl = jsonData["profile"].toMap()["image"].toMap()["url"].toString();

                emit profileReceived(profDisplayName, profImageUrl);
            }
        }

        break;




    ////////////////////////////////////////////////// If a timeline was requested
    case TimelineRequest:
        qDebug() << "A Timeline was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();
        qDebug() << jsonData.keys();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("Timeline received. Updating post list..."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in inbox timeline:" << jsonDataList.size();

            QString previousLink = jsonData["links"].toMap().value("prev").toMap().value("href").toString();
            QString nextLink = jsonData["links"].toMap().value("next").toMap().value("href").toString();

            emit timeLineReceived(jsonDataList, this->postsPerPage, previousLink, nextLink);
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }
        break;



    ///////////////////////////////////////// If a single post was requested, by ID
    case PostRequest:
        qDebug() << "A single post was requested";

        break;




    case ContactListRequest:
        qDebug() << "The contact list was requested";

        jsonData = jsonParser.parse(packetData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant contactsVariant = jsonData.value("items");
            if (contactsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing contacts...";
                emit currentJobChanged(tr("Contact list received."));
                emit contactListReceived(contactsVariant.toList());
            }
            else
            {
                qDebug() << "Expected a list of contacts, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << packetData; // JSON directly
        }

        break;



    case AvatarRequest:
        qDebug() << "Received AVATAR data, from " << reply->url();
        // FIXME: take care of possible redirections

        //this->avatarData.append(packetData);
        if (reply->isFinished())
        {
            qDebug() << "Avatar received 100%";
            emit avatarPictureReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Avatar not complete yet";
        }
        break;


    case ImageRequest:
        qDebug() << "Received IMAGE data, from " << reply->url();
        //this->imageData.append(packetData);
        if (reply->isFinished())
        {
            qDebug() << "Image received 100%";
            emit imageReceived(packetData, reply->url());
        }
        else
        {
            qDebug() << "Image not complete yet";
        }
        break;


    }
    // end switch (requestType)
}




void PumpController::sslErrorsHandler(QNetworkReply *reply, QList<QSslError> errorList)
{
    qDebug() << "\n\n==== SSL errors!! ====";
    qDebug() << "At:" << reply->url();
    qDebug() << "Error list:" << errorList << "\n\n";

}




void PumpController::getToken()
{
    // If we do not have client_id or client_secret, do dynamic client registration
    if (this->clientID.isEmpty() || this->clientSecret.isEmpty())
    {
        qDebug() << "PumpController::getToken()";
        qDebug() << "We do not have client_id/client_secret yet; doing Dynamic Client Registration";

        // POST to https://hotpump.net/api/client/register, f.e.
        QNetworkRequest postRequest(QUrl("https://" + this->serverURL + "/api/client/register"));


        postRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
        postRequest.setRawHeader("User-Agent", userAgentString);
        postRequest.setAttribute(QNetworkRequest::User,
                                 QVariant(ClientRegistrationRequest));

        QByteArray data("{"
                        " \"type\": \"client_associate\",  "
                        " \"application_type\": \"native\", "
                        " \"application_name\": \"Dianara\" "
                        "}");

        qDebug() << "About to POST:" << data;


        nam.post(postRequest, data); // upon receiving data (id+secret), will execute getToken() again
    }
    else
    {
        qDebug() << "Using saved client_id and client_secret:" << this->clientID << this->clientSecret;

        // OAuth stuff.....
        // 1. obtaining an unauthorized Request Token from the Service Provider,
        // 2. asking the User to authorize the Request Token,
        // 3. exchanging the Request Token for the Access Token

        qDebug() << "Doing OAuth token stuff...";


        qoauth->setConsumerKey(this->clientID.toLocal8Bit());
        qoauth->setConsumerSecret(this->clientSecret.toLocal8Bit());


        QString requestTokenURL = "https://" + this->serverURL + "/oauth/request_token";
        qDebug() << "GET: " << requestTokenURL << "with" << qoauth->consumerKey() << qoauth->consumerSecret();


        QOAuth::ParamMap oAuthParams;
        oAuthParams.insert("oauth_callback", "oob");


        QOAuth::ParamMap reply = qoauth->requestToken(requestTokenURL,
                                                      QOAuth::GET,
                                                      QOAuth::HMAC_SHA1,   // or PLAINTEXT?
                                                      oAuthParams);


        if (qoauth->error() == QOAuth::NoError)
        {
            qDebug() << "requestToken OK:" << reply.keys();

            token = reply.value(QOAuth::tokenParameterName());
            tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

            qDebug() << "Token:" << token;
            qDebug() << "Token Secret:" << tokenSecret;

            QUrl oAuthAuthorizeURL("https://" + this->serverURL + "/oauth/authorize");
            oAuthAuthorizeURL.addQueryItem("oauth_token", token);

            QDesktopServices::openUrl(oAuthAuthorizeURL);

            // Now, user should enter VERIFIER in AccountDialog to authorize the program
        }
        else
        {
            qDebug() << "QOAuth error" << qoauth->error() << "!";

            qDebug() << reply.keys();
        }

    }
}



void PumpController::authorizeApplication(QString verifierCode)
{
    qDebug() << "Verifier code entered by user:" << verifierCode;

    QOAuth::ParamMap moreParams;
    moreParams.insert("oauth_verifier", verifierCode.toUtf8()); // verifier as QByteArray

    QString requestAuthorizationURL = "https://" + this->serverURL + "/oauth/access_token";

    QOAuth::ParamMap reply = qoauth->accessToken(requestAuthorizationURL,
                                                 QOAuth::GET,
                                                 token,
                                                 tokenSecret,
                                                 QOAuth::HMAC_SHA1,
                                                 moreParams);

    if (qoauth->error() == QOAuth::NoError) // Woooohooo!!
    {
        qDebug() << "Got authorized token; Dianara is authorized to access the account";
        token = reply.value(QOAuth::tokenParameterName());
        tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

        this->isApplicationAuthorized = true;

        QSettings settings;
        settings.setValue("isApplicationAuthorized", this->isApplicationAuthorized);
        settings.setValue("token",       this->token);
        settings.setValue("tokenSecret", this->tokenSecret);

        qDebug() << "Token:" << token;
        qDebug() << "TokenSecret:" << tokenSecret;

        emit this->authorizationStatusChanged(isApplicationAuthorized);
    }
    else
    {
        qDebug() << "OAuth error while authorizing application" << qoauth->error();
    }

}




/*
 * Called by a QTimer, get initial data (contacts, timeline...) one step at a time
 *
 */
void PumpController::getInitialData()
{
    qDebug() << "PumpController::getInitialData() step" << initialDataStep;

    initialDataTimer->setInterval(2000);  // Every 2 sec


    switch (this->initialDataStep)
    {
    case 0:
        this->getContactList();
        break;

    case 1:
        this->getTimeline();
        break;


    default:
        emit currentJobChanged(tr("Ready.")); // FIXME
        initialDataTimer->stop();

        qDebug() << "All initial data loaded ------";
        qDebug() << "------------------------------";
    }

    ++initialDataStep;
}





/*
 * Send a post ("note") to the server
 *
 */
void PumpController::post(QString postText)
{
    qDebug() << "PumpController::post()";



    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST, PublishPostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "note");
    jsonVariantObject.insert("content", postText);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    bool ok;
    QByteArray data = serializer.serialize(jsonVariant, &ok);


    qDebug() << "About to POST:" << data;

    nam.post(postRequest, data);
}



/*
 * Like (favorite) a post, by its URL
 *
 */
void PumpController::likePost(QString postID, QString postType)
{
    qDebug() << "PumpController::likePost() liking post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest likeRequest = this->prepareRequest(url, QOAuth::POST, LikePostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "favorite");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    bool ok;
    QByteArray data = serializer.serialize(jsonVariant, &ok);

    qDebug() << "about to POST:" << data;

    nam.post(likeRequest, data);
}



void PumpController::addComment(QString comment, QString postID, QString postType)
{
    qDebug() << "PumpController::addComment() sending comment to this post:" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest commentRequest = this->prepareRequest(url, QOAuth::POST, CommentPostRequest);



    QVariantMap jsonVariantInReplyTo;
    jsonVariantInReplyTo.insert("id", postID);
    jsonVariantInReplyTo.insert("objectType", postType);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "comment");
    jsonVariantObject.insert("content", comment);
    jsonVariantObject.insert("inReplyTo", jsonVariantInReplyTo);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    bool ok;
    QByteArray data = serializer.serialize(jsonVariant, &ok);

    qDebug() << "about to POST:" << data;

    nam.post(commentRequest, data);
}




void PumpController::sharePost(QString postID, QString postType)
{
    qDebug() << "PumpController::sharePost() sharing post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest shareRequest = this->prepareRequest(url, QOAuth::POST, SharePostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "share");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    bool ok;
    QByteArray data = serializer.serialize(jsonVariant, &ok);

    qDebug() << "about to POST:" << data;

    nam.post(shareRequest, data);
}



void PumpController::deletePost(QString postID, QString postType)
{
    qDebug() << "PumpController::deletePost() deleting post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest deleteRequest = this->prepareRequest(url, QOAuth::POST, DeletePostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "delete");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    bool ok;
    QByteArray data = serializer.serialize(jsonVariant, &ok);

    qDebug() << "about to POST:" << data;

    nam.post(deleteRequest, data);
}




/*
 * Add a contact to the /following list with their webfinger address
 *
 * This will trigger a re-request of the contact list, upon receiving the HTTP 200
 * confirming the addition of the contact to /following
 *
 */
void PumpController::followContact(QString address)
{
    qDebug() << "PumpController::followContact()" << address;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest followRequest = this->prepareRequest(url, QOAuth::POST, FollowContactRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", "acct:" + address);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "follow");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    bool ok;
    QByteArray data = serializer.serialize(jsonVariant, &ok);

    qDebug() << "about to POST:" << data;

    nam.post(followRequest, data);
}



/*
 * Remove a contact from the /following list with their webfinger address
 *
 * This will trigger a re-request of the contact list, upon receiving the HTTP 200
 * confirming the removal of the contact from /following
 *
 */
void PumpController::unfollowContact(QString address)
{
    qDebug() << "PumpController::unfollowContact()" << address;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest unfollowRequest = this->prepareRequest(url, QOAuth::POST, UnfollowContactRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", "acct:" + address);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "stop-following");
    jsonVariant.insert("object", jsonVariantObject);

    QJson::Serializer serializer;
    bool ok;
    QByteArray data = serializer.serialize(jsonVariant, &ok);

    qDebug() << "about to POST:" << data;

    nam.post(unfollowRequest, data);
}




void PumpController::setCurrentTimeline(QString link)
{
    qDebug() << "PumpController::setCurrentTimeline()" << link;
}
