/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "timeline.h"

TimeLine::TimeLine(PumpController *pumpController, QWidget *parent) :  QWidget(parent)
{
    this->pController = pumpController;

    mainLayout = new QVBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);


    QMap<QString,QString> demoPostData;
    demoPostData.insert("author", "DemoUser");
    demoPostData.insert("text",   "<h2><b>Demo Post</b></h2><br>"
                                  "Press <b>F5</b> or <i>Update timelines</i> in the menu.<br><br>"
                                  "Upon startup, timeline should update automatically.<br><br>"
                                  "<a href=\"http://jancoding.wordpress.com/dianara\">Dianara's blog</a><br>"
                                  "<a href=\"http://dianara.nongnu.org\">Dianara's website</a><br><br>");
    demoPostData.insert("createdAt", "1970-01-01T01:02:03Z");
    demoPostData.insert("likesCount", "32");
    demoPostData.insert("commentsCount", "128");
    demoPostData.insert("resharesCount", "256");


    splitter = new QSplitter(Qt::Vertical, this);
    splitter->setChildrenCollapsible(false);
    mainLayout->addWidget(splitter);


    this->timelineOffset = 0;

    firstPageButton = new QPushButton(QIcon::fromTheme("go-first"), tr("&First Page"));
    connect(firstPageButton, SIGNAL(clicked()),
            this, SLOT(goToFirstPage()));
    previousPageButton = new QPushButton(QIcon::fromTheme("go-previous"), tr("&Previous Page"));
    connect(previousPageButton, SIGNAL(clicked()),
            this, SLOT(goToPreviousPage()));
    nextPageButton = new QPushButton(QIcon::fromTheme("go-next"), tr("&Next Page"));
    connect(nextPageButton, SIGNAL(clicked()),
            this, SLOT(goToNextPage()));

    bottomLayout = new QHBoxLayout();
    bottomLayout->addWidget(firstPageButton);
    bottomLayout->addSpacing(16);
    bottomLayout->addWidget(previousPageButton);
    bottomLayout->addWidget(nextPageButton);
    mainLayout->addLayout(bottomLayout);

    // Add the default "demo" post
    splitter->addWidget(new Post(pController, demoPostData, QVariantList(), this));


    this->setLayout(mainLayout);


    QSettings settings;
    this->previousNewestPostID = settings.value("previousNewestPostID", "").toString();

    qDebug() << "TimeLine created";
}


TimeLine::~TimeLine()
{
    QSettings settings;
    settings.setValue("previousNewestPostID", this->previousNewestPostID);

    qDebug() << "TimeLine destroyed";
}



/*
 * Remove all widgets (Post *) from the timeline
 *
 */
void TimeLine::clearTimeLineContents()
{
    QList<QWidget*> previousPostList;

    // Build list of widgets contained in the layout
    for (int counter = 0; counter < splitter->count(); ++counter)
    {
        qDebug() << "Adding for deletion:" << splitter->widget(counter);

        previousPostList.append(splitter->widget(counter));
    }

    qDebug() << "List to delete:" << previousPostList;


    foreach (QWidget *oldPost, previousPostList)
    {
        //delete mainLayout->itemAt(counter)->widget();
        //mainLayout->itemAt(counter)->widget()->deleteLater();

        qDebug() << "will deleteLater:" << oldPost;
        oldPost->deleteLater(); // delete should be safe too, but still...
    }


    /* This is to avoid the splitter getting bigger and bigger everytime the user
     * updates the timeline (F5, etc)
     */
    this->splitter->setMinimumHeight(0);
    this->splitter->resize(1,1);
}



/************************ SLOTS **************************/


void TimeLine::setTimeLineContents(QVariantList postList, int postsPerPage,
                                   QString previousLink, QString nextLink)
{
    qDebug() << "TimeLine::setTimeLineContents()";

    // Remove all previous posts in timeline
    qDebug() << "Removing previous posts from timeline";
    this->clearTimeLineContents();


    this->postsPerPage = postsPerPage;

    int newPostCount = 0;
    bool allNewPostsCounted = false;
    QString newestPostID; // Here we'll store the postID for the first (newest) post in the timeline
                          // With it, we can know how many new posts (if any) we receive next time

    // Fill timeline with new contents
    foreach (QVariant singlePost, postList)
    {
        if (singlePost.type() == QVariant::Map)
        {
            this->previousPageLink = previousLink;   // Useless at the moment
            this->nextPageLink = nextLink;

            QVariantMap singlePostMap = singlePost.toMap();

            //QString itemVerb = singlePostMap.value("verb").toString();

            QString postIsDeleted = singlePostMap.value("object").toMap().value("deleted").toString();


            QString postID;
            QString postType;
            QString postAuthor;
            QString postAuthorAvatar;
            QString postAuthorId;
            QString postAuthorHometown;
            QString postAuthorBio;
            QString postGenerator;
            QString postTitle;
            QString postImage;
            QString postText;
            QString postIsNsfw;
            QString postCreatedAt;
            QString postRootAuthor;
            QString postLikesCount;
            QString postCommentsCount;
            QString postResharesCount;
            QString postIsShared;


            postID = singlePostMap.value("object").toMap().value("id").toString();
            if (newestPostID.isEmpty()) // only first time, for newest post
            {
                if (this->timelineOffset == 0)
                {
                    newestPostID = postID;
                }
                else
                {
                    newestPostID = this->previousNewestPostID;
                    allNewPostsCounted = true;
                }
            }

            if (!allNewPostsCounted)
            {
                if (postID == this->previousNewestPostID)
                {
                    allNewPostsCounted = true;
                }
                else
                {
                    ++newPostCount;
                }
            }


            postType = singlePostMap.value("object").toMap().value("objectType").toString();
            postIsShared = singlePostMap.value("object").toMap().value("pump_io").toMap().value("shared").toString();
            //qDebug() << "post type:" << postType << "post is shared:" << postIsShared;

            postAuthor = singlePostMap.value("actor").toMap().value("displayName").toString();
            postAuthorAvatar = singlePostMap.value("actor").toMap().value("image").toMap().value("url").toString();
            // for Author ID, remove the first 5 characters from the field, "acct:"
            postAuthorId = singlePostMap.value("actor").toMap().value("id").toString().remove(0,5);

            postAuthorBio = singlePostMap.value("actor").toMap().value("summary").toString();
            postAuthorHometown = singlePostMap.value("actor").toMap().value("location").toMap().value("displayName").toString();

            postGenerator = singlePostMap.value("generator").toMap().value("displayName").toString();

            postCreatedAt = singlePostMap.value("object").toMap().value("published").toString();


            ///
            /// End of "meta"; Start of content
            ///

            if (postType == "image")
            {
                postTitle = singlePostMap.value("object").toMap().value("displayName").toString();

                // See if there's a proxy URL for the image first (for private images)
                postImage = singlePostMap.value("object").toMap().value("image").toMap().value("pump_io").toMap().value("proxyURL").toString();

                // If that's empty, use regular image->url field
                if (postImage.isEmpty())
                {
                    postImage = singlePostMap.value("object").toMap().value("image").toMap().value("url").toString();
                }
            }


            postText = singlePostMap.value("object").toMap().value("content").toString();

            postLikesCount = singlePostMap.value("object").toMap().value("likes").toMap().value("totalItems").toString();
            postCommentsCount = singlePostMap.value("object").toMap().value("replies").toMap().value("totalItems").toString();
            postResharesCount = singlePostMap.value("object").toMap().value("shares").toMap().value("totalItems").toString();

            // get comments list here, pass to the Post()
            QVariantList commentsList = singlePostMap.value("object").toMap().value("replies").toMap().value("items").toList();



            QMap<QString,QString> postData;
            postData.insert("id",                 postID);
            postData.insert("postType",           postType);
            postData.insert("author",             postAuthor);
            postData.insert("authorAvatar",       postAuthorAvatar);
            postData.insert("authorId",           postAuthorId);
            postData.insert("authorBio",          postAuthorBio);
            postData.insert("authorHometown",     postAuthorHometown);
            postData.insert("generator",          postGenerator);
            postData.insert("title",              postTitle);
            postData.insert("image",              postImage);
            postData.insert("text",               postText);
            postData.insert("createdAt",          postCreatedAt);
            postData.insert("nsfw",               postIsNsfw);
            postData.insert("rootAuthor",         postRootAuthor);
            postData.insert("likesCount",         postLikesCount);
            postData.insert("commentsCount",      postCommentsCount);
            postData.insert("resharesCount",      postResharesCount);
            postData.insert("postIsShared",       postIsShared);



            if (postIsDeleted.isEmpty())  // if post was NOT deleted
            {
                this->splitter->addWidget(new Post(pController, postData, commentsList, this));
            }
            else
            {
                // if there's a "deleted" key, ignore this post
                qDebug() << "This post was deleted on" << postIsDeleted << " / Not adding.";
            }


            /*
            this->splitter->setStretchFactor(splitter->count()-1,
                                             splitter->widget(splitter->count()-1)->height()/200);
            */

            // TMP FIXME, avoid unresizable posts (not enough height on splitter area)
            this->splitter->setMinimumHeight(this->splitter->height() + 200); // was 400
        }
        else  // singlePost.type() is not a QVariant::Map
        {
            qDebug() << "Expected a Map, got something else";
            qDebug() << postList;
        }
    }

    emit scrollToTop(); // ask mainWindow to scroll the QScrollArea containing the timeline to the top

    this->previousNewestPostID = newestPostID;

    qDebug() << "New posts:" << newPostCount << "; Newest post ID:" << previousNewestPostID;

    emit timelineRendered(newPostCount);

    qDebug() << "setTimeLineContents() /END";
}





void TimeLine::goToFirstPage()
{
    qDebug() << "TimeLine::goToFirstPage()";
    this->pController->getTimeline();

    this->timelineOffset = 0; // Keep timeline offset coherent
}



void TimeLine::goToPreviousPage()
{
    qDebug() << "TimeLine::goToPreviousPage()";

    this->timelineOffset -= this->postsPerPage;
    if (timelineOffset < 0)
    {
        timelineOffset = 0;
    }
    this->pController->getTimeline(this->timelineOffset);
}



void TimeLine::goToNextPage()
{
    qDebug() << "TimeLine::goToNextPage()";

    this->timelineOffset += this->postsPerPage;

    this->pController->getTimeline(this->timelineOffset);
}
