/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "account.h"

AccountDialog::AccountDialog(PumpController *pumpController, QWidget *parent) : QWidget(parent)
{
    this->setWindowTitle("Dianara - " + tr("Account configuration"));
    this->setWindowIcon(QIcon::fromTheme("user-properties"));
    this->setMinimumSize(540, 150);
    this->resize(800, 150);

    this->pController = pumpController;

    connect(pController, SIGNAL(authorizationStatusChanged(bool)),
            this, SLOT(showAuthorizationStatus(bool)));


    userIDIconLabel = new QLabel();
    userIDIconLabel->setPixmap(QIcon::fromTheme("preferences-desktop-user").pixmap(32,32));
    userIDLabel = new QLabel(tr("Your Pump.io address:"));
    userIDLineEdit = new QLineEdit();
    userIDLineEdit->setPlaceholderText(tr("Webfinger ID, like username@pumpserver.org"));

    getVerifierButton = new QPushButton(QIcon::fromTheme("object-unlocked"),
                                        tr("Get &Verifier Code"));
    connect(getVerifierButton, SIGNAL(clicked()),
            this, SLOT(askForToken()));

    verifierIconLabel = new QLabel();
    verifierIconLabel->setPixmap(QIcon::fromTheme("dialog-password").pixmap(32,32));
    verifierLabel = new QLabel(tr("Verifier code:"));
    verifierLineEdit = new QLineEdit();
    verifierLineEdit->setPlaceholderText(tr("Enter the verification code provided by your Pump server here"));


    authorizeApplicationButton = new QPushButton(QIcon::fromTheme("security-high"),
                                                 tr("&Authorize Application"));
    connect(authorizeApplicationButton, SIGNAL(clicked()),
            this, SLOT(setVerifierCode()));



    // To notify invalid ID or empty verifier code
    // Cleared when typing in any of the two fields
    errorsLabel = new QLabel("");
    connect(userIDLineEdit, SIGNAL(textChanged(QString)),
            errorsLabel, SLOT(clear()));
    connect(verifierLineEdit, SIGNAL(textChanged(QString)),
            errorsLabel, SLOT(clear()));


    authorizationStatusLabel = new QLabel("");


    saveButton = new QPushButton(QIcon::fromTheme("document-save"),
                                 tr("&Save details"));
    connect(saveButton, SIGNAL(clicked()),
            this, SLOT(saveDetails()));
    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                   tr("&Cancel"));
    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(close()));

    mainLayout = new QGridLayout();
    mainLayout->setAlignment(Qt::AlignLeft);
    mainLayout->addWidget(userIDIconLabel,   0, 0);
    mainLayout->addWidget(userIDLabel,       0, 1);
    mainLayout->addWidget(userIDLineEdit,    0, 2);
    mainLayout->addWidget(getVerifierButton, 0, 3);

    mainLayout->addWidget(verifierIconLabel,          1, 0);
    mainLayout->addWidget(verifierLabel,              1, 1);
    mainLayout->addWidget(verifierLineEdit,           1, 2);
    mainLayout->addWidget(authorizeApplicationButton, 1, 3);

    mainLayout->addWidget(authorizationStatusLabel,   2, 0, 1, 4, Qt::AlignCenter);
    mainLayout->addWidget(errorsLabel,       3, 0, 1, 4, Qt::AlignCenter);

    mainLayout->addWidget(saveButton,        4, 2, 1, 1, Qt::AlignRight);
    mainLayout->addWidget(cancelButton,      4, 3, 1, 1, Qt::AlignLeft);
    this->setLayout(mainLayout);


    QSettings settings;  // Load saved User ID
    userIDLineEdit->setText(settings.value("userID", "").toString());

    this->showAuthorizationStatus(settings.value("isApplicationAuthorized", false).toBool());


    // Disable verifier input field and button initially
    // They will be used after requesting a token
    verifierLineEdit->setDisabled(true);
    authorizeApplicationButton->setDisabled(true);


    qDebug() << "Account dialog created";
}


AccountDialog::~AccountDialog()
{
    qDebug() << "Account dialog destroyed";
}





//////////////////////////////////////////////////////////////////////
///////////////////////////////// SLOTS //////////////////////////////
//////////////////////////////////////////////////////////////////////



void AccountDialog::askForToken()
{
    // Don't fail if there're spaces before or after the ID, they're only human ;)
    userIDLineEdit->setText(userIDLineEdit->text().trimmed());

    // Match username@server.tld AND check for only 1 @ sign
    if (QRegExp("\\S+@\\S+\\.\\S+").exactMatch(this->userIDLineEdit->text())
     && userIDLineEdit->text().count("@") == 1)
    {
        // Clear verifier field and re-enable it and the button
        this->verifierLineEdit->clear();
        this->verifierLineEdit->setEnabled(true);
        this->authorizeApplicationButton->setEnabled(true);


        // Show message about the web browser that will be started
        this->errorsLabel->setText("[ " + tr("A web browser will start now, where you can get the verifier code") + " ]");

        this->pController->setNewUserID(userIDLineEdit->text());
        this->pController->getToken();
    }
    else // userID does not match user@hostname.domain
    {
        this->errorsLabel->setText("[ " + tr("Your Pump address is invalid") + " ]");
        qDebug() << "userID is invalid!";
    }
}



void AccountDialog::setVerifierCode()
{
    qDebug() << "AccountDialog::setVerifierCode()" << this->verifierLineEdit->text();

    if (!this->verifierLineEdit->text().isEmpty())
    {
        this->pController->authorizeApplication(this->verifierLineEdit->text());
    }
    else
    {
        this->errorsLabel->setText("[ " + tr("Verifier code is empty") + " ]");
    }
}




void AccountDialog::showAuthorizationStatus(bool authorized)
{
    if (authorized)
    {
        this->authorizationStatusLabel->setText("[[ " + tr("Dianara is authorized to access your data") + " ]]");
    }
    else
    {
        this->authorizationStatusLabel->clear();
    }
}





/*
 *
 *
 */
void AccountDialog::saveDetails()
{

    QSettings settings;
    settings.setValue("userID", this->userIDLineEdit->text());


    emit userIDChanged(userIDLineEdit->text());


    this->errorsLabel->clear(); // Clear previous error messages, if any
    this->close();
}
