/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "contactcard.h"

ContactCard::ContactCard(PumpController *pumpController,
                         QMap<QString, QString> contactData,
                         QWidget *parent)  :  QFrame(parent)
{
    this->pController = pumpController;

    this->setFrameStyle(QFrame::Box | QFrame::Raised);

    this->setSizePolicy(QSizePolicy::MinimumExpanding,
                        QSizePolicy::Maximum);

    mainLayout = new QHBoxLayout();

    avatarLabel = new QLabel();

    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(contactData.value("avatar"));

    //qDebug() << "Avatar remote file:" << contactData.value("avatar");
    //qDebug() << "Avatar local file:" << avatarFilename;

    QFile avatarFile(avatarFilename);
    if (avatarFile.exists(avatarFilename) && !contactData.value("avatar").isEmpty())
    {
        // Load avatar if already cached
        avatarLabel->setPixmap(QPixmap(avatarFilename).scaled(48, 48,
                                                              Qt::KeepAspectRatio,
                                                              Qt::SmoothTransformation));
        qDebug() << "ContactCard: Using cached avatar";
    }
    else
    {
        // Placeholder image
        avatarLabel->setPixmap(QIcon::fromTheme("user-identity").pixmap(48,48));
        qDebug() << "ContactCard: Using placeholder";

        // Download avatar for next time
        pController->getAvatar(contactData.value("avatar"));
    }


    centerLayout = new QVBoxLayout();

    QFont nameFont;
    nameFont.setBold(true);
    nameFont.setUnderline(true);

    nameLabel = new QLabel();
    nameLabel->setText(contactData.value("name"));
    nameLabel->setFont(nameFont);


    this->contactID = contactData.value("id");

    tmpLabel = new QLabel();
    tmpLabel->setText(QString("<%1>\n").arg(contactID)
                      + tr("Hometown") + QString(": %2").arg(contactData.value("hometown")));
    tmpLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    tmpLabel->setToolTip(contactData.value("bio")); // kinda tmp...


    centerLayout->addWidget(nameLabel);
    centerLayout->addWidget(tmpLabel);

    followButton = new QPushButton("--");
    followButton->setFlat(true);    
    if (contactData.value("following") == "false")
    {
        this->setButtonToFollow();
    }
    else
    {
        this->setButtonToUnfollow();
    }


    mainLayout->addWidget(avatarLabel,  1);  // Stretch 1/8
    mainLayout->addLayout(centerLayout, 6);  // Stretch 6/8
    mainLayout->addWidget(followButton, 1);  // 1/8


    this->setLayout(mainLayout);

    qDebug() << "ContactCard created";
}


ContactCard::~ContactCard()
{
    qDebug() << "ContactCard destroyed";
}



void ContactCard::setButtonToFollow()
{
    followButton->setIcon(QIcon::fromTheme("list-add"));
    followButton->setText(tr("Follow"));
    connect(followButton, SIGNAL(clicked()),
            this, SLOT(followContact()));
}


void ContactCard::setButtonToUnfollow()
{
    followButton->setIcon(QIcon::fromTheme("list-remove"));
    followButton->setText(tr("Stop following"));
    connect(followButton, SIGNAL(clicked()),
            this, SLOT(unfollowContact()));
}


/**************************************************************************/
/******************************** SLOTS ***********************************/
/**************************************************************************/



void ContactCard::followContact()
{
    this->pController->followContact(this->contactID);
    this->setButtonToUnfollow();
}


void ContactCard::unfollowContact()
{
    this->pController->unfollowContact(this->contactID);
    this->setButtonToFollow();
}
