/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "publisher.h"

Publisher::Publisher(PumpController *pumpController,
                     int publisherType,
                     QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;

    this->setFocusPolicy(Qt::StrongFocus); // To keep the publisher from getting focus by accident


    mainLayout = new QGridLayout();


    pictureLabel = new QLabel();
    pictureLabel->setScaledContents(true);
    pictureLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    pictureLabel->hide();

    titleLineEdit = new QLineEdit();
    titleLineEdit->setSizePolicy(QSizePolicy::MinimumExpanding,
                                 QSizePolicy::Maximum);
    titleLineEdit->setPlaceholderText(tr("Title for the picture (optional)"));
    titleLineEdit->setToolTip(tr("Title for the picture"));
    titleLineEdit->hide();

    selectPictureButton = new QPushButton(QIcon::fromTheme("insert-image"),
                                          tr("Select Picture..."));
    selectPictureButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                       QSizePolicy::Maximum);
    selectPictureButton->setToolTip(tr("Find the picture in your folders"));
    connect(selectPictureButton, SIGNAL(clicked()),
            this, SLOT(findPictureFile()));
    selectPictureButton->hide();

    // Set default pixmap and "picture not set" message
    this->setEmptyPictureData();




    composerBox = new Composer(true); // forPublisher = true
    connect(composerBox, SIGNAL(focusReceived()),    this, SLOT(setFullMode()));
    connect(composerBox, SIGNAL(editingFinished()),  this, SLOT(sendPost()));
    connect(composerBox, SIGNAL(editingCancelled()), this, SLOT(setMinimumMode()));


    aspectList = new QMenu("to-list");
    aspectList->addAction(tr("Public"))->setCheckable(true);
    aspectList->addAction(tr("Followers"))->setCheckable(true);


    // Default = "followers"
    aspectList->actions().at(1)->setChecked(true); // VERY, VERY tmp


    aspectSelector = new QPushButton(QIcon::fromTheme("system-users"),
                                     tr("&To...")); // FIXME
    aspectSelector->setToolTip(tr("Select who will see this post"));
    aspectSelector->setMenu(aspectList);


    pictureButton = new QPushButton(QIcon::fromTheme("camera-photo"),
                                    tr("Add &Picture"));
    pictureButton->setToolTip(tr("Upload photo"));
    connect(pictureButton, SIGNAL(clicked()),
            this, SLOT(setPictureMode()));


    postButton = new QPushButton(QIcon::fromTheme("mail-send"),
                                  tr("&Post"));
    postButton->setToolTip(tr("Hit Control+Enter to post with the keyboard"));
    connect(postButton, SIGNAL(clicked()),
            this, SLOT(sendPost()));


    // Now the layout
    mainLayout->addWidget(pictureLabel,        0, 0, 2, 3);
    mainLayout->addWidget(titleLineEdit,       0, 3, 1, 1, Qt::AlignTop);
    mainLayout->addWidget(selectPictureButton, 1, 3, 1, 1, Qt::AlignBottom | Qt::AlignRight);

    // The rest depends on the publisher type
    switch (publisherType)
    {
    case 1: // second layout, buttons around
        mainLayout->addWidget(aspectSelector, 2, 0, 1, 1);
        mainLayout->addWidget(pictureButton,  3, 0, 1, 1);
        mainLayout->addWidget(composerBox,    2, 1, 3, 4); // 3 rows, 4 columns
        mainLayout->addWidget(postButton,     2, 5, 2, 1); // 2 rows
        postButton->setSizePolicy(QSizePolicy::Maximum,
                                   QSizePolicy::MinimumExpanding);
        break;

    case 2: // Third layout, buttons on right side
        mainLayout->addWidget(composerBox,    2, 0, 2, 3); // 2 rows, 3 columns
        mainLayout->addWidget(aspectSelector, 2, 3, 1, 1);
        mainLayout->addWidget(pictureButton,  3, 3, 1, 1);
        mainLayout->addWidget(postButton,     2, 4, 2, 1); // 2 rows, 1 column
        postButton->setSizePolicy(QSizePolicy::Maximum,
                                  QSizePolicy::MinimumExpanding);
        break;


    case 0: // First (default) layout
        // just let it jump to default, so incorrect values go to default

    default:
        mainLayout->addWidget(composerBox,    2, 0, 3, 4); // 3 rows, 4 columns
        mainLayout->addWidget(aspectSelector, 5, 0, 1, 1);
        mainLayout->addWidget(pictureButton,  5, 1, 1, 1, Qt::AlignLeft);
        mainLayout->addWidget(postButton,     5, 3, 1, 1, Qt::AlignRight);

        break;

    }



    this->setLayout(mainLayout);

    this->setMinimumMode();

    qDebug() << "Publisher created";
}


Publisher::~Publisher()
{
    qDebug() << "Publisher destroyed";
}


/*
 * Set default "no photo" pixmap and "picture not set" message
 *
 * Clear the filename and content type variables too
 */
void Publisher::setEmptyPictureData()
{
    titleLineEdit->clear();

    pictureLabel->setPixmap(QIcon::fromTheme("image-x-generic")
                            .pixmap(200, 150)
                            .scaled(200, 150,
                                    Qt::IgnoreAspectRatio,
                                    Qt::SmoothTransformation));
    pictureLabel->setToolTip(tr("Picture not set"));


    this->pictureFilename.clear();
    this->pictureContentType.clear();
}


/*
 * Create a key:value map, listing who will receive a post, like:
 *
 * "collection" : "http://activityschema.org/collection/public"
 * "collection" : "https://pump.example/api/user/followers"
 * "person"     : "acct:somecontact@pumpserver.example"
 *
 */
QMap<QString,QString> Publisher::getAudienceMap()
{
    QMap<QString,QString> audienceMap;

    if (aspectList->actions().at(0)->isChecked())  // Public is checked
    {
        audienceMap.insertMulti("collection", "http://activityschema.org/collection/public");
    }


    // Last check: if Followers is checked, or nothing is, add Followers
    if (aspectList->actions().at(1)->isChecked()
     || audienceMap.isEmpty())
    {
        audienceMap.insertMulti("collection", this->pController->currentFollowersURL());
    }


    return audienceMap;
}



/********************************************************************/
/***************************** SLOTS ********************************/
/********************************************************************/



void Publisher::setMinimumMode()
{
    qDebug() << "setting Publisher to minimum mode";

    this->postButton->setFocus(); // give focus to button,
                                   // in case user shared with Ctrl+Enter

    // disable possible scrollbars
    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    // ~1 row
    this->setMaximumHeight((this->composerBox->fontInfo().pixelSize()+4) * 3);

    this->aspectSelector->hide();
    this->pictureButton->hide();
    this->postButton->hide();

    // Clear formatting options like bold, italic and underline
    this->composerBox->setCurrentCharFormat(QTextCharFormat());

    // Hide "picture mode" controls
    this->pictureLabel->hide();
    this->titleLineEdit->hide();
    this->selectPictureButton->hide();

    this->setEmptyPictureData();
}




void Publisher::setFullMode()
{
    qDebug() << "setting Publisher to full mode";

    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->setMaximumHeight(2048);  // i.e. "unlimited"

    this->aspectSelector->show();
    // Avoid re-enabling the picture button when re-focusing publisher, but still in picture mode...
    if (pictureButton->isHidden())
    {
        this->pictureButton->setEnabled(true); // If it wasn't hidden, don't re-enable
    }
    this->pictureButton->show();
    this->postButton->show();

    this->composerBox->setFocus(); // In case user used menu or shortcut
    // instead of clicking on it
}




void Publisher::setPictureMode()
{
    this->pictureButton->setDisabled(true);

    this->pictureLabel->show();

    this->titleLineEdit->show();
    this->selectPictureButton->show();

    this->findPictureFile(); // Show the open file dialog directly
}




void Publisher::findPictureFile()
{
    QString filename;

    filename = QFileDialog::getOpenFileName(this, tr("Select one image"),
                                            QDir::homePath(),
                                            tr("Image files") + "(*.png *.jpg *.jpeg *.gif);;"
                                            + tr("All files") + " (*)");

    if (!filename.isEmpty())
    {
        qDebug() << "Selected" << filename << "for upload";
        this->pictureLabel->setToolTip(filename);

        QString imageFormat = QString::fromLocal8Bit(QImageReader::imageFormat(filename));
        qDebug() << "Image format:" << imageFormat;

        if (imageFormat == "png" || imageFormat == "jpeg" || imageFormat == "gif")
        {
            this->pictureContentType = "image/" + imageFormat;
            qDebug() << pictureContentType;
            this->pictureFilename = filename;

            this->pictureLabel->setPixmap(QPixmap(filename).scaled(200, 150,
                                                                   Qt::IgnoreAspectRatio,
                                                                   Qt::SmoothTransformation));
        }
        else
        {
            // In the future, avoid this by using libmagic or similar
            qDebug() << "Unknown image format; Extension is probably wrong";
            QMessageBox::warning(this, tr("Invalid image"),
                                 tr("The image format cannot be detected.\n"
                                    "The extension might be wrong, like a GIF "
                                    "image renamed to image.jpg or similar."));

            this->pictureContentType.clear();
            this->pictureFilename.clear();
        }


    }
}



void Publisher::sendPost()
{
    qDebug() << "Publisher character count:" << composerBox->textCursor().document()->characterCount();


    // If there's some text in the post, or attached picture, send it
    if (composerBox->textCursor().document()->characterCount() > 1     // kinda tmp
      || !pictureFilename.isEmpty())
    {
        QString cleanHtml = composerBox->toHtml();
        cleanHtml.replace("\n", " ");  // Remove line breaks, as that results in server error 500


        // Kinda tmp cleanup of HTML stuff
        QRegExp headRE("<html><head>.*</head>");
        headRE.setMinimal(true);
        cleanHtml.remove(headRE);
        cleanHtml.remove("</body></html>");

        QRegExp bodyRE("<body style.*>");
        bodyRE.setMinimal(true);
        cleanHtml.remove(bodyRE);


        QMap<QString,QString> audienceMap = this->getAudienceMap();


        if (this->pictureFilename.isEmpty())
        {
            this->pController->postNote(audienceMap, cleanHtml);
        }
        else
        {
            this->pController->postImage(audienceMap,
                                         cleanHtml,
                                         this->titleLineEdit->text(), // Title
                                         pictureFilename,
                                         pictureContentType);
        }


        this->composerBox->erase();

        // Done composing message, hide buttons until we get focus again
        setMinimumMode();
    }
    else
    {
        qDebug() << "Can't send post: text is empty, and no picture";
    }
}
