/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "mainwindow.h"


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    this->setWindowTitle("Dianara");
    this->setWindowIcon(QIcon(":/icon/64x64/dianara.png"));
    this->setMinimumSize(400, 400);

    QSettings settings;

    firstRun = true;
    prepareDataDirectory();

    reallyQuitProgram = false;
    trayIconAvailable = false;


    // Network control
    pumpController = new PumpController();


    ////// GUI
    mainSplitter = new QSplitter(Qt::Horizontal, this);


    // Left side
    leftSideWidget = new QWidget();
    leftLayout = new QVBoxLayout();

    avatarIconLabel = new QLabel();
    avatarIconLabel->setPixmap(QPixmap(":/images/no-avatar.png")
                             .scaled(64, 64,
                                     Qt::KeepAspectRatio,
                                     Qt::SmoothTransformation));
    leftLayout->addWidget(avatarIconLabel);
    leftLayout->addSpacing(4);

    fullNameLabel = new QLabel("[----------]");
    leftLayout->addWidget(fullNameLabel);
    leftLayout->addSpacing(8);


    leftPanel = new QToolBox(); // For now, only one "section"...

    meanwhileFeed = new MinorFeed(pumpController);
    leftPanel->addItem(meanwhileFeed,
                       QIcon::fromTheme("clock"),
                       tr("Meanwhile..."));

    leftLayout->addWidget(leftPanel);

    this->leftSideWidget->setLayout(leftLayout);



    // Right side
    rightSideWidget = new QWidget();
    rightLayout = new QVBoxLayout();

    this->publisherType = settings.value("Configuration/publisherType", 0).toInt(); // TMP/FIXME?
    publisher = new Publisher(pumpController, publisherType);


    mainTimeline = new TimeLine(TimeLine::TimelineTypeMain,
                                pumpController,
                                this);
    mainTimelineScrollArea = new QScrollArea();
    mainTimelineScrollArea->setWidget(mainTimeline);  // Make timeline scrollable
    mainTimelineScrollArea->setWidgetResizable(true); // Seems to work in opposite way
    connect(mainTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollMainTimelineToTop()));
    connect(mainTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));

    connect(mainTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(notifyTimelineUpdate(int,int)));


    directTimeline = new TimeLine(TimeLine::TimelineTypeDirect,
                                  pumpController,
                                  this);
    directTimelineScrollArea = new QScrollArea();
    directTimelineScrollArea->setWidget(directTimeline);
    directTimelineScrollArea->setWidgetResizable(true);
    connect(directTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollDirectTimelineToTop()));
    connect(directTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));


    activityTimeline = new TimeLine(TimeLine::TimelineTypeActivity,
                                    pumpController,
                                    this);
    activityTimelineScrollArea = new QScrollArea();
    activityTimelineScrollArea->setWidget(activityTimeline);  // Make it scrollable
    activityTimelineScrollArea->setWidgetResizable(true);
    connect(activityTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollActivityTimelineToTop()));
    connect(activityTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));


    favoritesTimeline = new TimeLine(TimeLine::TimelineTypeFavorites,
                                     pumpController,
                                     this);
    favoritesTimelineScrollArea = new QScrollArea();
    favoritesTimelineScrollArea->setWidget(favoritesTimeline);
    favoritesTimelineScrollArea->setWidgetResizable(true);
    connect(favoritesTimeline, SIGNAL(scrollToTop()),
            this, SLOT(scrollFavoritesTimelineToTop()));
    connect(favoritesTimeline, SIGNAL(timelineRendered(int,int)),
            this, SLOT(setTimelineTabTitle(int,int)));



    // The contact list has its own tabs with its own scroll areas
    contactList = new ContactList(pumpController, this);


    tabWidget = new QTabWidget();
    tabWidget->addTab(mainTimelineScrollArea,
                      QIcon::fromTheme("view-list-details"),
                      "MAIN TIMELINE TAB");
    tabWidget->setTabToolTip(0, tr("The main timeline"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeMain, 0);
    tabWidget->addTab(directTimelineScrollArea,
                      QIcon::fromTheme("mail-message"),
                      "MESSAGES TAB");
    tabWidget->setTabToolTip(1, tr("Messages sent explicitly to you"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeDirect, 0);
    tabWidget->addTab(activityTimelineScrollArea,
                      QIcon::fromTheme("user-home"),
                      "ACTIVITY TAB");
    tabWidget->setTabToolTip(2, tr("Your own posts"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeActivity, 0);
    tabWidget->addTab(favoritesTimelineScrollArea,
                      QIcon::fromTheme("folder-favorites"),
                      "FAVORITES TAB");
    tabWidget->setTabToolTip(3, tr("Your favorited posts"));
    this->setTimelineTabTitle(TimeLine::TimelineTypeFavorites, 0);
    tabWidget->addTab(contactList,
                      QIcon::fromTheme("system-users"),
                      tr("&Contacts"));
    tabWidget->setTabToolTip(4, tr("The people you follow, and the ones who follow you"));
    connect(tabWidget, SIGNAL(currentChanged(int)),
            this, SLOT(setTitleAndTrayInfo(int)));



    rightLayout->addWidget(publisher, 1); // stretch 1/10
    rightLayout->addWidget(tabWidget, 9); // stretch 9/10

    this->rightSideWidget->setLayout(rightLayout);


    mainSplitter->addWidget(leftSideWidget);
    mainSplitter->addWidget(rightSideWidget);
    mainSplitter->setHandleWidth(4);
    mainSplitter->setChildrenCollapsible(false);
    this->setCentralWidget(mainSplitter);



    ////////////////// Load configuration from disk
    loadSettings();


    // FreeDesktop.org notifications handler
    fdNotifier = new FDNotifications();
    fdNotifier->setNotificationType(showNotifications); // valid since loadSettings()
                                                        // was just called
    connect(fdNotifier, SIGNAL(showFallbackNotification(QString)),
            this, SLOT(showTrayFallbackMessage(QString)));


    updateTimer = new QTimer(this);
    updateTimer->setInterval(this->updateInterval * 1000 * 60); // min > msec
    connect(updateTimer, SIGNAL(timeout()),
            this, SLOT(updateMainDirectMinorTimelines()));
    updateTimer->start();


    profileEditor = new ProfileEditor(pumpController, this);


    accountDialog = new AccountDialog(pumpController, this);
    connect(accountDialog, SIGNAL(userIDChanged(QString)),
            this, SLOT(updateUserID(QString)));

    configDialog = new ConfigDialog(this->updateInterval,
                                    this->postsPerPage,
                                    this->tabsPosition,
                                    this->tabsMovable,
                                    this->publisherType,
                                    this->publicPosts,
                                    this->showNotifications,
                                    this);
    connect(configDialog, SIGNAL(configurationChanged(int,int,int,bool,int,bool,int)),
            this, SLOT(updateSettings(int,int,int,bool,int,bool,int)));



    ///
    //////////////////////////////// Connections for PumpController //////////
    ///

    connect(pumpController, SIGNAL(profileReceived(QString,QString,QString,QString)),
            this, SLOT(updateProfileData(QString,QString,QString,QString)));

    connect(pumpController, SIGNAL(avatarPictureReceived(QByteArray,QUrl)),
            this, SLOT(storeAvatar(QByteArray,QUrl)));
    connect(pumpController, SIGNAL(imageReceived(QByteArray,QUrl)),
            this, SLOT(storeImage(QByteArray,QUrl)));


    // After receiving timeline contents, update corresponding timeline
    connect(pumpController, SIGNAL(mainTimelineReceived(QVariantList,int,QString,QString)),
            mainTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));
    connect(pumpController, SIGNAL(directTimelineReceived(QVariantList,int,QString,QString)),
            directTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));
    connect(pumpController, SIGNAL(activityTimelineReceived(QVariantList,int,QString,QString)),
            activityTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));
    connect(pumpController, SIGNAL(favoritesTimelineReceived(QVariantList,int,QString,QString)),
            favoritesTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString)));


    // After sucessful posting, request updated timeline, with your post included
    connect(pumpController, SIGNAL(postPublished()),
            this, SLOT(updateMainActivityMinorTimelines()));


    // After successful liking, update likes count
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            mainTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            directTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            activityTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    // We don't update likes count in favorites timeline,
    // since it still doesn't know about this post

    // Instead, reload favorites timeline
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            favoritesTimeline, SLOT(goToFirstPage()));


    // After commenting successfully, refresh list of comments in that post
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            mainTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            directTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            activityTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            favoritesTimeline, SLOT(setCommentsInPost(QVariantList,QString)));


    // After successful sharing....

    // TODO***


    // After receiving the minor feed ("Meanwhile"), update it
    connect(pumpController, SIGNAL(minorFeedReceived(QVariantList)),
            meanwhileFeed, SLOT(setFeedContents(QVariantList)));


    // After receiving a contact list, update it
    connect(pumpController, SIGNAL(contactListReceived(QString,QVariantList)),
            contactList, SLOT(setContactListContents(QString,QVariantList)));

    // After receiving the list of lists, update it
    connect(pumpController, SIGNAL(listsListReceived(QVariantList)),
            contactList, SLOT(setListsListContents(QVariantList)));



    // Show notifications for events sent from pumpController
    connect(pumpController, SIGNAL(showNotification(QString)),
            fdNotifier, SLOT(showMessage(QString)));

    // Update statusBar message from pumpController's infos
    connect(pumpController, SIGNAL(currentJobChanged(QString)),
            this, SLOT(setStatusBarMessage(QString)));



    // Don't set PumpController in motion if no User ID is set!
    if (!userID.isEmpty())
    {
        pumpController->setUserCredentials(this->userID);
        // getUserProfile() will be called from setUserCredentials()
    }


    // Add menus
    createMenus();

    // Add the system tray icon
    createTrayIcon();


    // tmp statusBar stuff
    this->statusBar()->showMessage(tr("Initializing..."));


    settings.beginGroup("MainWindow");

    // Now set the "view side panel" checkable menu to its saved state
    // That will also trigger the action to hide/show it
    viewSidePanel->setChecked(settings.value("viewSidePanel", true).toBool());


    // Set the "view status bar" checkable menu to its saved state, which
    // will also trigger the action to hide/show it
    viewStatusBar->setChecked(settings.value("viewStatusBar", true).toBool());

    settings.endGroup();

    qDebug() << "MainWindow created";
}



MainWindow::~MainWindow()
{
    qDebug() << "MainWindow destroyed";
}




void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "MainWindow::closeEvent()";


    if (reallyQuitProgram)
    {
        event->accept(); // really close, if called from Quit menu
        qDebug() << "Quit called from menu, shutting down program";
    }
    else
    {
        this->hide();     // Hide window, app accessible via tray icon
        qDebug() << "Tried to close main window, so hiding to tray";
        event->ignore();  // ignore the closeEvent
    }
}




/*
 * Prepare the data directory. Create if necessary
 *
 */
void MainWindow::prepareDataDirectory()
{
    #if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    dataDirectory = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
    #else
    dataDirectory = QStandardPaths::standardLocations(QStandardPaths::DataLocation).first();
    #endif
    qDebug() << "Data directory:" << this->dataDirectory;

    QDir dataDir;
    if (!dataDir.exists(dataDirectory))
    {
        qDebug() << "Creating data directory";
        if (dataDir.mkpath(dataDirectory))
        {
            qDebug() << "Data directory created";
        }
        else
        {
            qDebug() << "Error creating data directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/images"))
    {
        qDebug() << "Creating images directory";
        if (dataDir.mkpath(dataDirectory + "/images"))
        {
            qDebug() << "Images directory created";
        }
        else
        {
            qDebug() << "Error creating images directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/avatars"))
    {
        qDebug() << "Creating avatars directory";
        if (dataDir.mkpath(dataDirectory + "/avatars"))
        {
            qDebug() << "Avatars directory created";
        }
        else
        {
            qDebug() << "Error creating avatars directory!";
        }
    }

}



/*
 * Populate the menus
 *
 */
void MainWindow::createMenus()
{
    sessionMenu = new QMenu(tr("&Session"));

    sessionUpdateMainTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                            tr("&Update Main Timeline"),
                                            this);
    sessionUpdateMainTimeline->setShortcut(QKeySequence::Refresh);
    connect(sessionUpdateMainTimeline, SIGNAL(triggered()),
            mainTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateMainTimeline);

    sessionUpdateDirectTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                              tr("Update &Messages Timeline"),
                                              this);
    connect(sessionUpdateDirectTimeline, SIGNAL(triggered()),
            directTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateDirectTimeline);

    sessionUpdateActivityTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                                tr("Update &Activity Timeline"),
                                                this);
    connect(sessionUpdateActivityTimeline, SIGNAL(triggered()),
            activityTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateActivityTimeline);


    sessionUpdateFavoritesTimeline = new QAction(QIcon::fromTheme("view-refresh"),
                                                 tr("Update Favorites &Timeline"),
                                                 this);
    connect(sessionUpdateFavoritesTimeline, SIGNAL(triggered()),
            favoritesTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateFavoritesTimeline);


    sessionUpdateMinorFeed = new QAction(QIcon::fromTheme("view-refresh"),
                                         tr("Update Minor &Feed"),
                                         this);
    connect(sessionUpdateMinorFeed, SIGNAL(triggered()),
            meanwhileFeed, SLOT(updateFeed()));
    sessionMenu->addAction(sessionUpdateMinorFeed);


    sessionUpdateAllTimelines = new QAction(QIcon::fromTheme("view-refresh"),
                                         tr("Update All Timelines"),
                                         this);
    sessionUpdateAllTimelines->setShortcut(QKeySequence("Ctrl+F5"));
    connect(sessionUpdateAllTimelines, SIGNAL(triggered()),
            this, SLOT(updateAllTimelines()));
    sessionMenu->addAction(sessionUpdateAllTimelines);

    sessionMenu->addSeparator();

    sessionPostNote = new QAction(QIcon::fromTheme("document-edit"),
                                          tr("&Post a Note"),
                                          this);
    sessionPostNote->setShortcut(QKeySequence("Ctrl+N"));
    connect(sessionPostNote, SIGNAL(triggered()),
            publisher, SLOT(setFullMode()));
    connect(sessionPostNote, SIGNAL(triggered()),
            this, SLOT(show()));  // show window, in case it's hidden and
                                  // Post a Note is used from tray menu
    sessionMenu->addAction(sessionPostNote);

    sessionMenu->addSeparator();

    sessionQuit = new QAction(QIcon::fromTheme("application-exit"),
                              tr("&Quit"), this);
    sessionQuit->setShortcut(QKeySequence::Quit);
    connect(sessionQuit, SIGNAL(triggered()),
            this, SLOT(quitProgram()));
    sessionMenu->addAction(sessionQuit);

    this->menuBar()->addMenu(sessionMenu);


    viewMenu = new QMenu(tr("&View"));

    viewSidePanel = new QAction(QIcon::fromTheme("view-sidetree"),
                                tr("Side &Panel"), this);
    connect(viewSidePanel, SIGNAL(toggled(bool)),
            this, SLOT(toggleSidePanel(bool)));
    viewSidePanel->setCheckable(true);
    viewSidePanel->setChecked(true);
    viewSidePanel->setShortcut(QKeySequence("F9"));

    viewMenu->addAction(viewSidePanel);

    viewStatusBar = new QAction(QIcon::fromTheme("configure-toolbars"),
                                tr("Status &Bar"), this);
    connect(viewStatusBar, SIGNAL(toggled(bool)),
            this, SLOT(toggleStatusBar(bool)));
    viewStatusBar->setCheckable(true);
    viewStatusBar->setChecked(true);

    viewMenu->addAction(viewStatusBar);

    this->menuBar()->addMenu(viewMenu);



    settingsMenu = new QMenu(tr("S&ettings"));

    settingsEditProfile = new QAction(QIcon::fromTheme("user-properties"),
                                      tr("Edit &Profile"), this);
    connect(settingsEditProfile, SIGNAL(triggered()),
            profileEditor, SLOT(show()));
    settingsMenu->addAction(settingsEditProfile);


    settingsAccount = new QAction(QIcon::fromTheme("dialog-password"),
                                  tr("&Account Details"), this);
    connect(settingsAccount, SIGNAL(triggered()),
            accountDialog, SLOT(show()));
    settingsMenu->addAction(settingsAccount);

    settingsMenu->addSeparator();

    settingsConfigure = new QAction(QIcon::fromTheme("configure"),
                                    tr("&Configure Dianara"), this);
    connect(settingsConfigure, SIGNAL(triggered()),
            configDialog, SLOT(show()));
    settingsMenu->addAction(settingsConfigure);
    this->menuBar()->addMenu(settingsMenu);
    this->menuBar()->addSeparator();



    helpMenu = new QMenu(tr("&Help"));

    helpVisitWebsite = new QAction(QIcon::fromTheme("internet-web-browser"),
                            tr("Visit &Website"), this);
    connect(helpVisitWebsite, SIGNAL(triggered()),
            this, SLOT(visitWebSite()));
    helpMenu->addAction(helpVisitWebsite);

    helpAbout = new QAction(QIcon::fromTheme("system-help"),
                            tr("About &Dianara"), this);
    connect(helpAbout, SIGNAL(triggered()),
            this, SLOT(aboutDianara()));
    helpMenu->addAction(helpAbout);
    this->menuBar()->addMenu(helpMenu);



    // Context menu for the tray icon
    trayContextMenu = new QMenu("Tray Context Menu");
    trayContextMenu->addAction(QIcon(":/icon/64x64/dianara.png"),
                               tr("&Show Window"),
                               this, SLOT(show()));
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionUpdateMainTimeline);
    trayContextMenu->addAction(sessionPostNote);
    trayContextMenu->addAction(settingsConfigure);
    trayContextMenu->addAction(helpAbout);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionQuit);

    // FIXME: if mainwindow is hidden, program quits
    // after closing Configure or About window (now partially fixed)


    qDebug() << "Menus created";
}



/*
 * Create an icon in the system tray, define its contextual menu, etc.
 *
 */
void MainWindow::createTrayIcon()
{
    trayIcon = new QSystemTrayIcon(QIcon::fromTheme("dianara",
                                                    QIcon(":/icon/32x32/dianara.png")), this);

    if (trayIcon->isSystemTrayAvailable())
    {
        trayIconAvailable = true;

        this->setTitleAndTrayInfo(this->tabWidget->currentIndex());


        // Catch clicks on icon
        connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
                this, SLOT(trayControl(QSystemTrayIcon::ActivationReason)));


        // Set contextual menu for the icon
        trayIcon->setContextMenu(this->trayContextMenu);


        trayIcon->show();

        qDebug() << "Tray icon created";
    }
    else
    {
        trayIconAvailable = false;

        qDebug() << "System tray not available";
    }
}



/*
 * Load general program settings and state: size, position...
 *
 */
void MainWindow::loadSettings()
{
    QSettings settings;

    firstRun = settings.value("firstRun", true).toBool();
    if (firstRun)
    {
        qDebug() << "This is the first run";
    }


    userID = settings.value("userID", "").toString();
    this->setTitleAndTrayInfo(tabWidget->currentIndex());



    // Main window state
    settings.beginGroup("MainWindow");

    this->resize(settings.value("windowSize", QSize(640, 450)).toSize());
    if (!firstRun)
    {
        this->move(settings.value("windowPosition").toPoint());
    }
    this->mainSplitter->restoreState(settings.value("mainSplitterState",
                                                    mainSplitter->saveState()).toByteArray());
    // Set childrenCollapsible to false AFTER loading state, to make sure state does not mess with it
    mainSplitter->setChildrenCollapsible(false);

    settings.endGroup();


    // General program configuration
    settings.beginGroup("Configuration");

    this->updateInterval = settings.value("updateInterval", 5).toInt();

    this->postsPerPage = settings.value("postsPerPage", 20).toInt();
    this->pumpController->setPostsPerPage(this->postsPerPage);


    this->tabsPosition = settings.value("tabsPosition", QTabWidget::North).toInt();
    tabWidget->setTabPosition((QTabWidget::TabPosition)tabsPosition);

    this->tabsMovable = settings.value("tabsMovable", true).toBool();
    tabWidget->setMovable(tabsMovable);


    // publisherType 0 by default, buttons below
    this->publisherType = settings.value("publisherType", 0).toInt();

    this->publicPosts = settings.value("publicPosts", false).toBool();
    this->publisher->setDefaultPublicPosting(this->publicPosts);


    this->showNotifications = settings.value("showNotifications", 0).toInt();

    settings.endGroup();


    qDebug() << "Settings loaded";
}

/*
 * Save general program settings and state: size, position...
 *
 */
void MainWindow::saveSettings()
{
    QSettings settings;

    settings.setValue("firstRun", false);

    // General main window status
    settings.beginGroup("MainWindow");

    settings.setValue("windowSize", this->size());
    settings.setValue("windowPosition", this->pos());
    settings.setValue("mainSplitterState", this->mainSplitter->saveState());

    settings.setValue("viewSidePanel", this->viewSidePanel->isChecked());
    settings.setValue("viewStatusBar", this->viewStatusBar->isChecked());

    settings.endGroup();


    // From config dialog
    settings.beginGroup("Configuration");

    settings.setValue("updateInterval", this->updateInterval);

    settings.setValue("postsPerPage", this->postsPerPage);

    settings.setValue("tabsPosition", this->tabsPosition);
    settings.setValue("tabsMovable", this->tabsMovable);

    settings.setValue("publisherType", this->publisherType);
    settings.setValue("publicPosts", this->publicPosts);

    settings.setValue("showNotifications", this->showNotifications);

    settings.endGroup();


    qDebug() << "Settings saved";
}



//////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// SLOTS ////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



/*
 * Update UserID string from signal emitted in AccountDialog
 *
 */
void MainWindow::updateUserID(QString newUserID)
{
    this->userID = newUserID;

    // update window title and tray icon tooltip
    this->setTitleAndTrayInfo(tabWidget->currentIndex());

    this->pumpController->setUserCredentials(userID);

    // Remove current user's name and avatar
    this->fullNameLabel->setText("--");
    avatarIconLabel->setPixmap(QPixmap(":/images/no-avatar.png")
                             .scaled(64, 64,
                                     Qt::KeepAspectRatio,
                                     Qt::SmoothTransformation));


    qDebug() << "UserID updated from AccountDialog:" << userID;
}



/*
 * Update settings changed from ConfigDialog()
 *
 */
void MainWindow::updateSettings(int newUpdateInterval, int newPostsPerPage,
                                int newTabsPosition, bool newTabsMovable,
                                int newPublisherType, bool newPublicPosts,
                                int newShowNotifications)
{
    this->updateInterval = newUpdateInterval;
    this->updateTimer->setInterval(updateInterval * 1000 * 60);

    this->postsPerPage = newPostsPerPage;
    this->pumpController->setPostsPerPage(this->postsPerPage);

    this->tabsPosition = newTabsPosition;
    this->tabWidget->setTabPosition((QTabWidget::TabPosition)tabsPosition);

    this->tabsMovable = newTabsMovable;
    this->tabWidget->setMovable(tabsMovable);

    this->publisherType = newPublisherType;

    this->publicPosts = newPublicPosts;
    this->publisher->setDefaultPublicPosting(this->publicPosts);

    this->showNotifications = newShowNotifications;
    fdNotifier->setNotificationType(showNotifications); // FIXME: Safe?


    qDebug() << "updateInterval updated:" << updateInterval << updateInterval*60000;
    qDebug() << "tabsPosition updated:" << tabsPosition << tabWidget->tabPosition();
    qDebug() << "tabsMovable updated:" << tabsMovable;
    qDebug() << "publisherType updated:" << publisherType;
    qDebug() << "public posts updated:" << publicPosts;
    qDebug() << "Notifications updated:" << showNotifications;
}




/*
 * Control interaction with the system tray icon
 *
 */
void MainWindow::trayControl(QSystemTrayIcon::ActivationReason reason)
{
    qDebug() << "Tray icon activation reason:" << reason;

    if (reason != QSystemTrayIcon::Context) // Simple "main button" click in icon
    {
        /*
        qDebug() << "trayControl()";
        qDebug() << "isHidden?"    << this->isHidden();
        qDebug() << "isVisible?"   << this->isVisible();
        qDebug() << "isMinimized?" << this->isMinimized();
        qDebug() << "hasFocus?"    << this->hasFocus();
        */

        // Hide or show the main window
        if (this->isMinimized())
        {
            // hide and show, because raise() wouln't work
            this->hide();
            this->showNormal();
            qDebug() << "RAISING!";
        }
        else if (this->isHidden())
        {
            this->show();
            qDebug() << "SHOWING";
        }
        else
        {
            this->hide();
            qDebug() << "HIDING";
        }
    }
}



/*
 * If FreeDesktop.org notifications are not available,
 * fall back to Qt's balloon ones
 *
 */
void MainWindow::showTrayFallbackMessage(QString message)
{
    this->trayIcon->showMessage(tr("Dianara Notification"),
                                message,
                                QSystemTrayIcon::Information,
                                4000); // 4 secs
}



void MainWindow::updateProfileData(QString avatarURL, QString fullName,
                                   QString hometown, QString bio)
{
    this->fullNameLabel->setText(fullName);
    qDebug() << "Updated profile data from server:" << fullName << " @" << hometown;
    this->avatarURL = avatarURL;
    qDebug() << "Own avatar URL:" << avatarURL;


    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(avatarURL);

    if (QFile::exists(avatarFilename))
    {
        // Load avatar if already cached
        this->avatarIconLabel->setPixmap(QPixmap(avatarFilename)
                                                .scaled(64, 64,
                                                        Qt::KeepAspectRatio,
                                                        Qt::SmoothTransformation));
        qDebug() << "Using cached avatar for user";
    }
    else
    {
        pumpController->getAvatar(avatarURL);
    }


    // Fill/update this info in the profile editor too
    this->profileEditor->setProfileData(avatarURL, fullName,
                                        hometown, bio);
}


/*
 * Update all timelines
 *
 */
void MainWindow::updateAllTimelines()
{
    mainTimeline->goToFirstPage(); // received timeline will come in a SIGNAL()
    directTimeline->goToFirstPage();
    activityTimeline->goToFirstPage();
    favoritesTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated all timelines by menu";
}

/*
 * Update some of the timelines:
 *
 * Main, Direct messages, and Minor feed
 *
 */
void MainWindow::updateMainDirectMinorTimelines()
{
    mainTimeline->goToFirstPage();
    directTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated some timelines by menu or after:" << this->updateInterval << "min";
}


void MainWindow::updateMainActivityMinorTimelines()
{
    mainTimeline->goToFirstPage();
    activityTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated some timelines by menu or after posting";
}



void MainWindow::scrollMainTimelineToTop()
{
    this->mainTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->mainTimelineScrollArea->horizontalScrollBar()->setValue(0);
}

void MainWindow::scrollDirectTimelineToTop()
{
    this->directTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->directTimelineScrollArea->horizontalScrollBar()->setValue(0);
}

void MainWindow::scrollActivityTimelineToTop()
{
    this->activityTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->activityTimelineScrollArea->horizontalScrollBar()->setValue(0);
}

void MainWindow::scrollFavoritesTimelineToTop()
{
    this->favoritesTimelineScrollArea->verticalScrollBar()->setValue(0);
    this->favoritesTimelineScrollArea->horizontalScrollBar()->setValue(0);
}



void MainWindow::notifyTimelineUpdate(int timelineType, int newPostCount)
{
    if (newPostCount > 0)
    {
        QString newPostsString;
        if (newPostCount == 1)
        {
            newPostsString = tr("There is 1 new post.");
        }
        else
        {
            newPostsString = tr("There are %1 new posts.").arg(newPostCount);
        }
        this->setStatusBarMessage(tr("Timeline updated.") + " " + newPostsString);


        if (this->showNotifications != 2) // If some type of notifications are enabled
        {
            this->fdNotifier->showMessage(tr("Timeline updated at %1.").arg(QTime::currentTime().toString())
                                        + " " + newPostsString);
        }
    }
    else
    {
        this->setStatusBarMessage(tr("Timeline updated. No new posts."));
    }


    // In either case, restart update timer, so every manual update of timeline
    this->updateTimer->stop();  // will postpone the auto-update
    this->updateTimer->start();
}


/*
 * Update timelines titles with number of new posts
 *
 */
void MainWindow::setTimelineTabTitle(int timelineType, int newPostCount)
{
    int updatedTab = 0;

    QString messageCountString;
    if (newPostCount > 0)
    {
        messageCountString = QString(" (%1)").arg(newPostCount);
    }


    switch (timelineType)
    {
    case TimeLine::TimelineTypeMain:
        this->tabWidget->setTabText(0,
                                    tr("&Timeline") + messageCountString);
        updatedTab = 0;
        break;

    case TimeLine::TimelineTypeDirect:
        this->tabWidget->setTabText(1,
                                    tr("&Messages") + messageCountString);
        updatedTab = 1;
        break;

    case TimeLine::TimelineTypeActivity:
        this->tabWidget->setTabText(2,
                                    tr("&Activity") + messageCountString);
        updatedTab = 2;
        break;

    case TimeLine::TimelineTypeFavorites:
        this->tabWidget->setTabText(3,
                                    tr("Fav&orites") + messageCountString);
        updatedTab = 3;
        break;
    }

    // If the updated tab is the current one, set also window title, etc.
    if (updatedTab == tabWidget->currentIndex())
    {
        this->setTitleAndTrayInfo(updatedTab);
    }
}


/*
 * Set mainWindow's title based on current tab, and user ID
 *
 * Use that same title as tray icon's tooltip
 *
 */
void MainWindow::setTitleAndTrayInfo(int currentTab)
{
    QString currentTabTitle;
    currentTabTitle = this->tabWidget->tabText(currentTab);
    currentTabTitle.remove("&"); // Remove accelators

    QString title = "Dianara - "
                  + currentTabTitle
                  + " - "
                  + (userID.isEmpty() ?
                     tr("[ Your Pump.io account is not configured ]") : userID);

    this->setWindowTitle(title);
    if (trayIconAvailable)
    {
        this->trayIcon->setToolTip(title);
    }
}




/*
 * Store avatars on disk
 *
 */
void MainWindow::storeAvatar(QByteArray avatarData, QUrl avatarURL)
{
    QString avatarString = avatarURL.toString();
    QString fileName = avatarString.toUtf8().toBase64();


    QString fileExtension = avatarString.remove(QRegExp(".*\\."));
    fileName.append("." + fileExtension);

    qDebug() << "Saving avatar to disk: " << fileName;

    QFile avatarFile(dataDirectory + "/avatars/" + fileName);
    avatarFile.open(QFile::WriteOnly);
    avatarFile.write(avatarData);
    avatarFile.close();

    if (avatarURL == this->avatarURL)
    {
        this->avatarIconLabel->setPixmap(QPixmap(dataDirectory + "/avatars/" + fileName)
                                         .scaled(64, 64,
                                                 Qt::KeepAspectRatio,
                                                 Qt::SmoothTransformation));
    }        

    qDebug() << "avatarData size:" << avatarData.size();
}





/*
 * Store images on disk
 *
 */
void MainWindow::storeImage(QByteArray imageData, QUrl imageURL)
{
    QString imageString = imageURL.toString();

    QString fileName = imageString.toUtf8().toBase64();
    qDebug() << "Saving image to disk: " << fileName;

    QFile imageFile(dataDirectory + "/images/" + fileName);
    imageFile.open(QFile::WriteOnly);
    imageFile.write(imageData);
    imageFile.close();

    qDebug() << "imageData size:" << imageData.size();
}



/*
 * Update status bar message, from pumpController's (and others) signals
 *
 */
void MainWindow::setStatusBarMessage(QString message)
{
    this->statusBar()->showMessage("[" + QTime::currentTime().toString() + "] " + message, 0);
}



/*
 * Hide or show the side panel
 *
 */
void MainWindow::toggleSidePanel(bool shown)
{
    // Unless the publisher (really, the composer) has focus already,
    // give focus to timeline before, to avoid giving focus to the publisher
    if (!this->publisher->hasFocus())
    {
        // FIXME: check if publisher's composer has focus!
        this->mainTimeline->setFocus();
    }

    qDebug() << "Showing side panel:" << shown;
    this->leftSideWidget->setVisible(shown);
}


/*
 * Hide or show the status bar
 *
 */
void MainWindow::toggleStatusBar(bool shown)
{
    qDebug() << "Showing side panel:" << shown;
    this->statusBar()->setVisible(shown);
}





/*
 * Open website in browser
 *
 */
void MainWindow::visitWebSite()
{
    qDebug() << "Opening website in browser";
    QDesktopServices::openUrl(QUrl("http://jancoding.wordpress.com/dianara"));
}




/*
 * About... message
 *
 */
void MainWindow::aboutDianara()
{
    QMessageBox::about(this, tr("About Dianara"),
                       "<b>Dianara v0.8</b><br>"
                       "Copyright 2012-2013  JanKusanagi<br><br>"
                       "<a href=\"http://jancoding.wordpress.com/dianara\">"
                       "http://jancoding.wordpress.com/dianara</a><br><br>"

                       + tr("Dianara is a pump.io social networking client.")
                       + "<br><br>"

                       + tr("With Dianara you can see your timelines, create new posts, "
                            "upload pictures, interact with posts, manage "
                            "your contacts and follow new people.")
                       + "<br><br><br>"

                       + tr("English translation by JanKusanagi.",
                            "TRANSLATORS: Change this with your language and name ;)"));

    if (this->isHidden()) // FIXME: ugly workaround to avoid closing the program
    {                     // after closing About dialog, if mainWindow hidden
        this->show();
        this->hide();     // This hack might be causing problems under LXDE
        qDebug() << "MainWindow was hidden, showing and hiding again";
    }
}



/*
 * Close the program. Needed to quit correctly from context menu
 *
 */
void MainWindow::quitProgram()
{
    // Add more needed shutdown stuff here

    if (this->isHidden())
    {
        this->show();
    }

    saveSettings();

    reallyQuitProgram = true;

    qApp->closeAllWindows();

    qDebug() << "All windows closed, bye!";
}
