/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "post.h"

Post::Post(PumpController *pumpController,
           QMap<QString,QString> postData,
           QVariantList lastLikesVariantList,
           QVariantList lastCommentsVariantList,
           QVariantList lastSharesVariantList,
           QWidget *parent)  :  QFrame(parent)
{
    this->pController = pumpController;

    this->setFrameStyle(QFrame::Box | QFrame::Raised);
    //this->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken); // alternative?

    this->setSizePolicy(QSizePolicy::MinimumExpanding,
                        QSizePolicy::Maximum);


    leftColumnLayout = new QVBoxLayout();


    this->postID = postData.value("id");
    this->postType = postData.value("postType");


    QFont detailsFont;
    detailsFont.setPointSize(detailsFont.pointSize() - 2); // FIXME: check size first


    /////////////////////////////////////////////////// Left column, post Meta info

    // Is the post a reshare? Indicate who shared it
    postIsSharedLabel = new QLabel();
    if (postData.value("postIsShared") == "true")
    {
        postIsSharedLabel->setText(tr("Via %1").arg(postData.value("sharedBy")));
        postIsSharedLabel->setWordWrap(true);
        postIsSharedLabel->setAlignment(Qt::AlignCenter);
        postIsSharedLabel->setFont(detailsFont);
        postIsSharedLabel->setForegroundRole(QPalette::Background);
        postIsSharedLabel->setBackgroundRole(QPalette::Foreground);
        postIsSharedLabel->setAutoFillBackground(true);
        postIsSharedLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
        leftColumnLayout->addWidget(postIsSharedLabel);
    }
    leftColumnLayout->addSpacing(4);


    QString authorTooltipInfo = postData.value("authorId");
    authorTooltipInfo.append("<br>");
    if (!postData.value("authorHometown").isEmpty())
    {
        authorTooltipInfo.append(tr("Hometown") + ": " + postData.value("authorHometown"));
        authorTooltipInfo.append("<br>");
    }
    if (!postData.value("authorBio").isEmpty())
    {
        authorTooltipInfo.append("<br>");
        authorTooltipInfo.append(postData.value("authorBio"));
    }

    if (postData.value("authorId") == pController->currentUserID())
    {
        postIsOwn = true;
        qDebug() << "post is our own!";
    }
    else
    {
        postIsOwn = false;
    }

    // Author avatar
    postAuthorAvatar = new QLabel();

    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(postData.value("authorAvatar"));


    if (QFile::exists(avatarFilename) && !postData.value("authorAvatar").isEmpty())
    {
        // Load avatar if already cached
        postAuthorAvatar->setPixmap(QPixmap(avatarFilename)
                                    .scaled(48, 48,
                                            Qt::KeepAspectRatio,
                                            Qt::SmoothTransformation));
        qDebug() << "Using cached avatar";
    }
    else
    {
        // Placeholder image
        postAuthorAvatar->setPixmap(QIcon::fromTheme("user-identity").pixmap(48,48));
        qDebug() << "Using placeholder, downloading real avatar now";

        // Download avatar for next time
        pumpController->getAvatar(postData.value("authorAvatar"));
    }
    postAuthorAvatar->setToolTip(authorTooltipInfo);

    leftColumnLayout->addWidget(postAuthorAvatar);
    leftColumnLayout->addSpacing(2);


    QFont authorFont;
    authorFont.setBold(true);
    authorFont.setUnderline(true);
    authorFont.setPointSize(authorFont.pointSize()-1);

    // Author name
    postAuthorName = new QLabel(postData.value("author"));
    postAuthorName->setTextFormat(Qt::PlainText);
    postAuthorName->setWordWrap(true);
    postAuthorName->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postAuthorName->setFont(authorFont);
    postAuthorName->setToolTip(authorTooltipInfo);

    leftColumnLayout->addWidget(postAuthorName);
    leftColumnLayout->addSpacing(4);



    // Post creation time
    QString createdAt = postData.value("createdAt");
    // Format is "Combined date and time in UTC", like "2012-02-07T01:32:02Z"
    // as specified in ISO 8601 http://en.wikipedia.org/wiki/ISO_8601
    QString exactPostTime = Timestamp::localTimeDate(createdAt);
    createdAt = Timestamp::fuzzyTime(createdAt);
    QString generator = postData.value("generator");

    postCreatedAt = new QLabel(createdAt);
    if (generator.isEmpty())
    {
        postCreatedAt->setToolTip(tr("Posted on %1","1=Date")
                                  .arg(exactPostTime));
    }
    else
    {
        postCreatedAt->setToolTip(tr("Posted on %1\n"
                                     "using %2",        "1=Date of post, 2=Program used for posting")
                                  .arg(exactPostTime).arg(generator));
    }
    postCreatedAt->setTextFormat(Qt::PlainText);
    postCreatedAt->setWordWrap(true);
    postCreatedAt->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCreatedAt->setFont(detailsFont);
    leftColumnLayout->addWidget(postCreatedAt);
    leftColumnLayout->addSpacing(4);


    postToLabel = new QLabel(tr("To") + ": " + postData.value("to"));
    postToLabel->setWordWrap(true);
    postToLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postToLabel);

    postCCLabel = new QLabel(tr("CC") + ": " + postData.value("cc"));
    postCCLabel->setWordWrap(true);
    postCCLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postCCLabel);
    leftColumnLayout->addSpacing(4);


    openInBrowserLabel = new QLabel("<a href=\"" + postData.value("url") + "\">" + tr("Open") +" &#9658;</a>");
    openInBrowserLabel->setToolTip(tr("Open this post in your web browser"));
    openInBrowserLabel->setOpenExternalLinks(true);
    openInBrowserLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(openInBrowserLabel);

    leftColumnLayout->addSpacing(12);


    this->postLikesURL = postData.value("likesURL");
    this->postCommentsURL = postData.value("commentsURL");


    // Set this QLabels with parent=this, since we're gonna hide them right away
    // They'll be reparented to the layout, but now having a parent before that
    // would cause visual glitches
    postLikesCountLabel = new QLabel(this);
    postLikesCountLabel->setWordWrap(true);
    postLikesCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postLikesCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postLikesCountLabel);
    postLikesCountLabel->hide();
    int likesCount = postData.value("likesCount").toInt();
    this->setLikesLabel(likesCount);


    postCommentsCountLabel = new QLabel(this);
    postCommentsCountLabel->setWordWrap(true);
    postCommentsCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postCommentsCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postCommentsCountLabel);
    postCommentsCountLabel->hide();
    int commentsCount = postData.value("commentsCount").toInt();
    this->setCommentsLabel(commentsCount);


    postResharesCountLabel = new QLabel(this);
    postResharesCountLabel->setWordWrap(true);
    postResharesCountLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    postResharesCountLabel->setFont(detailsFont);
    leftColumnLayout->addWidget(postResharesCountLabel);
    postResharesCountLabel->hide();
    int resharesCount = postData.value("resharesCount").toInt();
    this->setSharesLabel(resharesCount);


    // Try to use all remaining space, aligning
    // all previous widgets nicely at the top
    // leftColumnLayout->setAlignment(Qt::AlignTop) caused glitches
    leftColumnLayout->addStretch(1);


    //////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////////////////////////////
    //////////////////////////////////////////////////// Right column, content

    rightColumnLayout = new QVBoxLayout();
    rightColumnLayout->setAlignment(Qt::AlignRight | Qt::AlignTop);


    ///////////////////////////////////// Post text
    postText = new QTextBrowser();
    postText->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    //postText->setFrameStyle(QFrame::StyledPanel | QFrame::Raised); // Raised style? :-m
    postText->setSizePolicy(QSizePolicy::MinimumExpanding,
                            QSizePolicy::MinimumExpanding);

    postText->setWordWrapMode(QTextOption::WrapAtWordBoundaryOrAnywhere);
    postText->setOpenLinks(false); // don't open links, manage in openClickedURL()
    postText->setReadOnly(true); // it's default with QTextBrowser, but still...
    connect(postText, SIGNAL(anchorClicked(QUrl)),
            this, SLOT(openClickedURL(QUrl)));
    connect(postText, SIGNAL(highlighted(QUrl)),
            this, SLOT(showHighlightedURL(QUrl)));

    // Add a label in the bottom right corner, to show where links go
    postTextLayout = new QHBoxLayout();

    highlightedUrlLabel = new QLabel();
    highlightedUrlLabel->setAlignment(Qt::AlignLeft);
    highlightedUrlLabel->setFont(detailsFont);
    highlightedUrlLabel->setForegroundRole(QPalette::ToolTipText); // use Tooltip colors
    highlightedUrlLabel->setBackgroundRole(QPalette::ToolTipBase);
    highlightedUrlLabel->setAutoFillBackground(true);
    highlightedUrlLabel->setWordWrap(true);
    highlightedUrlLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);

    postTextLayout->addWidget(highlightedUrlLabel, 0, Qt::AlignRight | Qt::AlignBottom);
    highlightedUrlLabel->hide(); // Hidden initially
    postText->setLayout(postTextLayout);


    // Like, comment, share buttons
    likeButton = new QPushButton(QIcon::fromTheme("emblem-favorite"), "");
    likeButton->setCheckable(true);
    this->fixLikeButton(postData.value("postIsLiked"));
    likeButton->setFlat(true);
    connect(likeButton, SIGNAL(clicked(bool)),
            this, SLOT(likePost(bool)));

    commentButton = new QPushButton(QIcon::fromTheme("mail-reply-sender"), tr("Comment"));
    commentButton->setToolTip(tr("Comment on this post"));
    commentButton->setFlat(true);
    connect(commentButton, SIGNAL(clicked()),
            this, SLOT(commentOnPost()));

    shareButton = new QPushButton(QIcon::fromTheme("mail-forward"), tr("Share"));
    shareButton->setToolTip(tr("Share this post"));
    shareButton->setFlat(true);
    connect(shareButton, SIGNAL(clicked()),
            this, SLOT(sharePost()));



    // Add like, comment, share and, if post is our own, delete buttons
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignHCenter | Qt::AlignBottom);
    buttonsLayout->setContentsMargins(0, 0, 0, 0);
    buttonsLayout->setMargin(0);
    buttonsLayout->setSpacing(0);
    buttonsLayout->addWidget(likeButton,    0, Qt::AlignLeft);
    buttonsLayout->addWidget(commentButton, 0, Qt::AlignLeft);
    buttonsLayout->addWidget(shareButton,   0, Qt::AlignLeft);
    buttonsLayout->addStretch(1); // so the (optional) Delete buttons gets separated

    if (postIsOwn)
    {
        deleteButton = new QPushButton(QIcon::fromTheme("edit-delete"), tr("Delete"));
        deleteButton->setToolTip(tr("Delete this post"));
        deleteButton->setFlat(true);
        connect(deleteButton, SIGNAL(clicked()),
                this, SLOT(deletePost()));

        buttonsLayout->addWidget(deleteButton, 0, Qt::AlignRight);
    }



    // Get URL of post image, if it's "image" type of post
    QString postTitle;
    QString postImage;
    if (this->postType == "image")
    {
        postImage = postData.value("image");

        if (QFile::exists(MiscHelpers::getCachedImageFilename(postImage)))
        {
            qDebug() << "Using cached image";
        }
        else
        {
            qDebug() << "Image not cached, downloading" << postImage;
            pumpController->getImage(postImage);
        }

        this->postImageTitle = postData.value("title");
        // If title's not empty, add the header tags and line break,
        // otherwhise don't add useless <br>
        if (!postImageTitle.isEmpty())
        {
            postTitle = "<h3>" + postImageTitle + "</h3><br>";
        }
        postImage = "<a href=\"image:/" + MiscHelpers::getCachedImageFilename(postImage) + "\">"
                    "<img src=\"" + MiscHelpers::getCachedImageFilename(postImage) + "\" width=\"320\" />"
                    "</a>"
                    "<br><br>";
    }


    QStringList postTextImageList = MiscHelpers::htmlWithReplacedImages(postData.value("text"));
    QString postTextContents = postTextImageList.takeAt(0);


    // Add the text content of the post
    postText->setHtml(postTitle
                    + postImage
                    + postTextContents);


    // If the image list is not empty, get them (unless they're cached already)
    if (!postTextImageList.isEmpty())
    {
        qDebug() << "Post has" << postTextImageList.size() << "images included...";

        foreach (QString imageUrl, postTextImageList)
        {
            if (QFile::exists(MiscHelpers::getCachedImageFilename(imageUrl)))
            {
                qDebug() << "Image already cached, ignoring";
            }
            else
            {
                qDebug() << "Image not cached, downloading";
                this->pController->getImage(imageUrl);
            }
        }
    }


    int postLines = postText->document()->lineCount() + 1;
    qDebug() << "Post lines:" << postLines;

    int postMinimumHeight = postLines * 2 * postText->fontInfo().pixelSize();
    if (postType == "image")
    {
        postMinimumHeight += 100;
    }
    if (postMinimumHeight > 100)  // Don't allow a post to have a minimum height > 200 px
    {
        postMinimumHeight = 100;  // Scrollbars will appear
        qDebug() << "Post would require more space: scrollbars might appear";
    }
    qDebug() << "Post minimum height:" << postMinimumHeight;
    postText->setMinimumHeight(postMinimumHeight);


    // NSFW button, to replace #nsfw posts until clicked
    nsfwCoveringButton = new QPushButton("This post has been marked Not Safe For Work (#NSFW).\n"
                                         "Click here to show it.");
    nsfwCoveringButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                      QSizePolicy::MinimumExpanding);
    connect(nsfwCoveringButton, SIGNAL(clicked()),
            this, SLOT(showNsfwPost()));


    /////////////////////////////////////////////////////// Comments block
    this->commenter = new Commenter(this->pController);
    connect(commenter, SIGNAL(commentSent(QString)),
            this, SLOT(sendComment(QString)));
    connect(commenter, SIGNAL(allCommentsRequested()),
            this, SLOT(getAllComments()));


    rightColumnLayout->addWidget(postText,      2);
    rightColumnLayout->addLayout(buttonsLayout, 0);
    rightColumnLayout->addWidget(commenter,     0);


    // Set the initial likes, comments and shares (4 most recent)
    this->setLikes(lastLikesVariantList);
    this->commenter->setComments(lastCommentsVariantList);
    this->setShares(lastSharesVariantList);


    mainLayout = new QHBoxLayout();
    mainLayout->addLayout(leftColumnLayout,  2); // stretch 2/10
    mainLayout->addLayout(rightColumnLayout, 8); // stretch 8/10

    if (postData.value("nsfw") == "true")
    {
        postText->hide();
        mainLayout->addWidget(nsfwCoveringButton, 5);
    }

    mainLayout->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    this->setLayout(mainLayout);


    qDebug() << "Post created";
}



Post::~Post()
{
    qDebug() << "Post destroyed";
}



/*
 * Return the likes/favorites URL for this post
 *
 */
QString Post::likesURL()
{
    return this->postLikesURL;
}


/*
 * Update the tooltip in "%NUMBER likes" with the names
 * of the people who liked the post
 *
 */
void Post::setLikes(QVariantList likesList)
{
    if (likesList.size() > 0)
    {
        QString peopleString;
        foreach (QVariant likesMap, likesList)
        {
            peopleString = likesMap.toMap().value("displayName").toString() + ", "
                    + peopleString;
        }

        // Remove last comma and add "like this"
        peopleString.remove(-2, 2); // Last 2 characters

        QString likesString;
        if (likesList.size() == 1)
        {
            likesString = tr("%1 likes this", "One person").arg(peopleString);
        }
        else
        {
            likesString = tr("%1 like this", "More than one person").arg(peopleString);
        }
        likesString.append("<a></a>"); // Turn the label into rich text

        this->postLikesCountLabel->setToolTip(likesString);
    }

    // TMP/FIXME this can set the number to lower than initially set
    // if called with the "initial" up-to-4 likes list that comes with the post
    this->setLikesLabel(likesList.size());
}


/*
 * Update the "NUMBER likes" label in left side
 *
 */
void Post::setLikesLabel(int likesCount)
{
    if (likesCount != 0)
    {
        if (likesCount == 1)
        {
            postLikesCountLabel->setText(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("1 like"));
        }
        else
        {
            postLikesCountLabel->setText(QString::fromUtf8("\342\231\245 ") // heart symbol
                                  + tr("%1 likes").arg(likesCount));
        }
        postLikesCountLabel->show();
    }
    else
    {
        postLikesCountLabel->clear();
        postLikesCountLabel->hide();
    }
}


/*
 * Return the comments URL for this post
 *
 */
QString Post::commentsURL()
{
    return this->postCommentsURL;
}


/*
 * Ask the Commenter to set new comments
 *
 */
void Post::setComments(QVariantList commentsList)
{
    this->commenter->setComments(commentsList);

    // update number of comments in left side counter
    this->setCommentsLabel(commentsList.size());
}

/*
 * Update the "NUMBER comments" label in left side
 *
 */
void Post::setCommentsLabel(int commentsCount)
{
    if (commentsCount != 0)
    {
        if (commentsCount == 1)
        {
            postCommentsCountLabel->setText(QString::fromUtf8("\342\234\215 ") // writing hand
                                      + tr("1 comment"));
        }
        else
        {
            postCommentsCountLabel->setText(QString::fromUtf8("\342\234\215 ") // writing hand
                                      + tr("%1 comments").arg(commentsCount));
        }
        postCommentsCountLabel->show();
    }
    else
    {
        postCommentsCountLabel->clear();
        postCommentsCountLabel->hide();
    }
}


/*
 * Return the shares URL for this post,
 * list of people who reshared it
 *
 */
QString Post::sharesURL()
{
    return this->postSharesURL;
}


/*
 * Update the tooltip for "%NUMBER shares" with names
 *
 */
void Post::setShares(QVariantList sharesList)
{
    if (sharesList.size() > 0)
    {
        QString peopleString;
        foreach (QVariant sharesMap, sharesList)
        {
            peopleString = sharesMap.toMap().value("displayName").toString() + ", "
                    + peopleString;
        }

        // Remove last comma and add "shared this"
        peopleString.remove(-2, 2); // Last 2 characters

        QString sharesString;
        if (sharesList.size() == 1)
        {
            sharesString = tr("%1 shared this", "%1 = One person name").arg(peopleString);
        }
        else
        {
            sharesString = tr("%1 shared this", "%1 = Names for more than one person").arg(peopleString);
        }
        sharesString.append("<a></a>");  // So that the label is rich text

        this->postResharesCountLabel->setToolTip(sharesString);
    }

    // TMP/FIXME this can set the number to lower than initially set
    // if called with the "initial" up-to-4 shares list that comes with the post
    this->setSharesLabel(sharesList.size());
}


void Post::setSharesLabel(int resharesCount)
{
    if (resharesCount != 0)
    {
        if (resharesCount == 1)
        {
            postResharesCountLabel->setText(QString::fromUtf8("\342\231\273 ") // recycle symbol
                                      + tr("Shared one time"));
        }
        else
        {
            postResharesCountLabel->setText(QString::fromUtf8("\342\231\273 ") // recycle symbol
                                      + tr("Shared %1 times").arg(resharesCount));
        }

        postResharesCountLabel->show();
    }
    else
    {
        postResharesCountLabel->clear();
        postResharesCountLabel->hide();
    }
}




void Post::resizeEvent(QResizeEvent *event)
{
    //postText->adjustSize();
    //postText->resize(postText->contentsRect().size());
    //postText->resize(postText->frameSize());
    //this->adjustSize();

    postText->moveCursor(QTextCursor::Start);

    event->accept();
}



////////////////////////////////////////////////////////////////////////////
////////////////////////////////// SLOTS ///////////////////////////////////
////////////////////////////////////////////////////////////////////////////


void Post::showNsfwPost()
{
    this->nsfwCoveringButton->hide();
    this->postText->show();

    qDebug() << "Shown NSFW post";
}


/*
 * Like (favorite) a post
 *
 */
void Post::likePost(bool like)
{
    qDebug() << "Post::likePost()" << (like ? "like" : "unlike");

    this->pController->likePost(this->postID, this->postType, like);

    this->fixLikeButton(like ? "true" : "false");

    connect(pController, SIGNAL(likeSet()),
            this, SLOT(getAllLikes()));
}

/*
 * Set the right labels and tooltips to the like button, depending on its state
 *
 */
void Post::fixLikeButton(QString state)
{
    if (state == "true")
    {
        likeButton->setToolTip(tr("You like this"));
        likeButton->setText(tr("Unlike"));
        likeButton->setChecked(true);
    }
    else
    {
        likeButton->setToolTip(tr("Like this post"));
        likeButton->setText(tr("Like"));
        likeButton->setChecked(false);
    }
}


void Post::getAllLikes()
{
    disconnect(pController, SIGNAL(likeSet()),
               this, SLOT(getAllLikes()));

    this->pController->getPostLikes(this->postLikesURL);
}


/*
 * Make the commenter widget visible, so user can type the comment
 *
 */
void Post::commentOnPost()
{
    qDebug() << "Commenting on this post";

    this->commenter->setFullMode();
}


/*
 * The actual sending of the comment to the Pump controller
 *
 */
void Post::sendComment(QString comment)
{
    qDebug() << "About to publish this comment:" << comment;

    this->pController->addComment(MiscHelpers::cleanHtml(comment),
                                  this->postID, this->postType);
}


void Post::getAllComments()
{
    this->pController->getPostComments(this->postCommentsURL);
}


/*
 * Re-share a post
 *
 */
void Post::sharePost()
{
    int confirmation = QMessageBox::question(this, tr("Share post?"),
                                             tr("Do you want to share %1's post?").arg(this->postAuthorName->text()),
                                             tr("&Yes, share it"), tr("&No"),
                                             "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Sharing this post";
        this->pController->sharePost(this->postID, this->postType);
    }
    else
    {
        qDebug() << "Confirmation cancelled, not sharing";
    }
}



/*
 * Delete a post
 *
 */
void Post::deletePost()
{
    int confirmation = QMessageBox::question(this, tr("WARNING: Delete post?"),
                                             tr("Are you sure you want to delete this post?"),
                                             tr("&Yes, delete it"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Deleting post";
        this->pController->deletePost(this->postID, this->postType);

        this->setDisabled(true); // disable... maybe hide?
    }
    else
    {
        qDebug() << "Confirmation cancelled, not deleting the post";
    }
}



void Post::openClickedURL(QUrl url)
{
    qDebug() << "Anchor URL clicked:" << url.toString();

    if (url.scheme() == "image") // Use our own viewer, or maybe a configured external viewer
    {
        qDebug() << "Opening this image in our own viewer...";
        ImageViewer *viewer = new ImageViewer(url.toString(), this->postImageTitle);
        viewer->show();
    }
    else
    {
        qDebug() << "Opening this link in browser";
        QDesktopServices::openUrl(url);
    }


}



void Post::showHighlightedURL(QUrl url)
{
    QString urlString = url.toString();
    if (!urlString.isEmpty())
    {
        if (urlString.startsWith("image://")) // Own image:// URL
        {
            this->highlightedUrlLabel->setText(tr("Click the image to see it in full size"));
        }
        else  // Normal URL
        {
            this->highlightedUrlLabel->setText(tr("Link to: %1").arg(urlString));
            qDebug() << "Highlighted url:" << urlString;
        }
        this->highlightedUrlLabel->show();
    }
    else
    {
        this->highlightedUrlLabel->clear();
        this->highlightedUrlLabel->hide();
    }
}
