/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "publisher.h"

Publisher::Publisher(PumpController *pumpController,
                     int publisherType,
                     QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;

    this->setFocusPolicy(Qt::StrongFocus); // To keep the publisher from getting focus by accident


    this->audienceSelectorTo = new AudienceSelector(pController, "to", this);
    connect(audienceSelectorTo, SIGNAL(audienceSelected(QString,QStringList)),
            this, SLOT(updateToCcFields(QString,QStringList)));
    this->audienceSelectorCC = new AudienceSelector(pController, "cc", this);
    connect(audienceSelectorCC, SIGNAL(audienceSelected(QString,QStringList)),
            this, SLOT(updateToCcFields(QString,QStringList)));


    mainLayout = new QGridLayout();


    pictureLabel = new QLabel();
    pictureLabel->setScaledContents(true);
    pictureLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    pictureLabel->hide();

    titleLineEdit = new QLineEdit();
    titleLineEdit->setSizePolicy(QSizePolicy::MinimumExpanding,
                                 QSizePolicy::Maximum);
    titleLineEdit->setPlaceholderText(tr("Title for the picture (optional)"));
    titleLineEdit->setToolTip(tr("Title for the picture"));
    titleLineEdit->hide();

    selectPictureButton = new QPushButton(QIcon::fromTheme("insert-image"),
                                          tr("Select Picture..."));
    selectPictureButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                       QSizePolicy::Maximum);
    selectPictureButton->setToolTip(tr("Find the picture in your folders"));
    connect(selectPictureButton, SIGNAL(clicked()),
            this, SLOT(findPictureFile()));
    selectPictureButton->hide();

    // Set default pixmap and "picture not set" message
    this->setEmptyPictureData();




    composerBox = new Composer(true); // forPublisher = true
    composerBox->setSizePolicy(QSizePolicy::Minimum,
                               QSizePolicy::Minimum);
    connect(composerBox, SIGNAL(focusReceived()),    this, SLOT(setFullMode()));
    connect(composerBox, SIGNAL(editingFinished()),  this, SLOT(sendPost()));
    connect(composerBox, SIGNAL(editingCancelled()), this, SLOT(setMinimumMode()));


    toSelectorMenu = new QMenu("to-menu");
    toSelectorMenu->addAction(tr("Public"))->setCheckable(true);
    toSelectorMenu->addAction(tr("Followers"))->setCheckable(true);
    toSelectorMenu->addSeparator();
    toSelectorMenu->addAction(tr("More..."), audienceSelectorTo, SLOT(show()));

    toSelectorButton = new QPushButton(QIcon::fromTheme("system-users"),
                                     tr("To..."));
    toSelectorButton->setToolTip(tr("Select who will see this post"));
    toSelectorButton->setMenu(toSelectorMenu);



    ccSelectorMenu = new QMenu("cc-menu");
    ccSelectorMenu->addAction(tr("Public"))->setCheckable(true);
    ccSelectorMenu->addAction(tr("Followers"))->setCheckable(true);
    ccSelectorMenu->addSeparator();
    ccSelectorMenu->addAction(tr("More..."), audienceSelectorCC, SLOT(show()));

    // Default = "CC: followers"
    ccSelectorMenu->actions().at(1)->setChecked(true); // kinda tmp

    ccSelectorButton = new QPushButton(QIcon::fromTheme("system-users"),
                                     tr("CC..."));
    ccSelectorButton->setToolTip(tr("Select who will get a copy of this post"));
    ccSelectorButton->setMenu(ccSelectorMenu);


    QFont audienceLabelsFont;
    audienceLabelsFont.setPointSize(audienceLabelsFont.pointSize()-1);

    // These will hold the names of the *people* selected for the To or CC fields, if any
    toAudienceLabel = new QLabel();
    toAudienceLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    toAudienceLabel->setFont(audienceLabelsFont);
    toAudienceLabel->setWordWrap(true);

    ccAudienceLabel = new QLabel();
    ccAudienceLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    ccAudienceLabel->setFont(audienceLabelsFont);
    ccAudienceLabel->setWordWrap(true);


    // To set the 'picture mode'
    pictureButton = new QPushButton(QIcon::fromTheme("camera-photo"),
                                    tr("Ad&d Picture"));
    pictureButton->setToolTip(tr("Upload photo"));
    connect(pictureButton, SIGNAL(clicked()),
            this, SLOT(setPictureMode()));


    // To send the post
    postButton = new QPushButton(QIcon::fromTheme("mail-send"),
                                  tr("Post"));
    postButton->setToolTip(tr("Hit Control+Enter to post with the keyboard"));
    connect(postButton, SIGNAL(clicked()),
            this, SLOT(sendPost()));


    // Now the layout, starting with "picture mode" stuff
    mainLayout->addWidget(pictureLabel,        0, 0, 2, 3);
    mainLayout->addWidget(titleLineEdit,       0, 3, 1, 3, Qt::AlignTop);
    mainLayout->addWidget(selectPictureButton, 1, 4, 1, 2, Qt::AlignBottom | Qt::AlignRight);

    // The rest depends on the publisher type
    switch (publisherType)
    {
    case 1: // second layout, buttons around
        mainLayout->addWidget(toSelectorButton, 2, 0, 1, 1);
        mainLayout->addWidget(ccSelectorButton, 3, 0, 1, 1);
        mainLayout->addWidget(pictureButton,    4, 0, 1, 1);
        mainLayout->addWidget(composerBox,      2, 1, 3, 4); // 3 rows, 4 columns
        mainLayout->addWidget(postButton,       2, 5, 3, 1); // 3 rows
        postButton->setSizePolicy(QSizePolicy::Maximum,
                                  QSizePolicy::MinimumExpanding);
        break;

    case 2: // Third layout, buttons on right side
        mainLayout->addWidget(composerBox,      2, 0, 3, 4); // 3 rows, 4 columns
        mainLayout->addWidget(toSelectorButton, 2, 4, 1, 1);
        mainLayout->addWidget(ccSelectorButton, 3, 4, 1, 1);
        mainLayout->addWidget(pictureButton,    4, 4, 1, 1);
        mainLayout->addWidget(postButton,       2, 5, 3, 1); // 3 rows, 1 column
        postButton->setSizePolicy(QSizePolicy::Maximum,
                                  QSizePolicy::MinimumExpanding);
        break;


    case 0: // First (default) layout
        // just let it jump to default, so incorrect values go to default

    default:
        mainLayout->addWidget(composerBox,      2, 0, 3, 6); // 3 rows, 6 columns
        mainLayout->addWidget(toSelectorButton, 5, 0, 1, 1, Qt::AlignLeft);
        mainLayout->addWidget(ccSelectorButton, 5, 1, 1, 1, Qt::AlignLeft);
        mainLayout->addWidget(pictureButton,    5, 3, 1, 1, Qt::AlignLeft);
        mainLayout->addWidget(postButton,       5, 5, 1, 1, Qt::AlignRight);

        break;

    }

    // And finally, the 2 labels holding people's names, if any
    mainLayout->addWidget(toAudienceLabel, 6, 0, 1, 1);
    mainLayout->addWidget(ccAudienceLabel, 6, 1, 1, 1);


    this->setLayout(mainLayout);

    this->setMinimumMode();

    qDebug() << "Publisher created";
}


Publisher::~Publisher()
{
    qDebug() << "Publisher destroyed";
}


/*
 * Set if "public" option for audience is checked as default
 *
 */
void Publisher::setDefaultPublicPosting(bool defaultPublicPosts)
{
    this->defaultPublicPosting = defaultPublicPosts;

    // Ensure status is sync'ed
    this->toSelectorMenu->actions().at(0)->setChecked(this->defaultPublicPosting);
}


/*
 * Set default "no photo" pixmap and "picture not set" message
 *
 * Clear the filename and content type variables too
 */
void Publisher::setEmptyPictureData()
{
    titleLineEdit->clear();

    pictureLabel->setPixmap(QIcon::fromTheme("image-x-generic")
                            .pixmap(200, 150)
                            .scaled(200, 150,
                                    Qt::IgnoreAspectRatio,
                                    Qt::SmoothTransformation));
    pictureLabel->setToolTip(tr("Picture not set"));


    this->pictureFilename.clear();
    this->pictureContentType.clear();
}



/*
 * Create a key:value map, listing who will receive a post, like:
 *
 * "collection" : "http://activityschema.org/collection/public"
 * "collection" : "https://pump.example/api/user/followers"
 * "person"     : "acct:somecontact@pumpserver.example"
 *
 */
QMap<QString,QString> Publisher::getAudienceMap()
{
    QMap<QString,QString> audienceMap;


    // To: Public is checked
    if (toSelectorMenu->actions().at(0)->isChecked())
    {
        audienceMap.insertMulti("to|collection", "http://activityschema.org/collection/public");
    }

    // To: List of individual people
    foreach (QString address, this->toAddressStringList)
    {
        audienceMap.insertMulti("to|person", "acct:" + address);
    }

    // To: Followers is checked
    if (toSelectorMenu->actions().at(1)->isChecked())
    {
        audienceMap.insertMulti("to|collection", this->pController->currentFollowersURL());
    }


    // CC: Public is checked
    if (ccSelectorMenu->actions().at(0)->isChecked())
    {
        audienceMap.insertMulti("cc|collection", "http://activityschema.org/collection/public");
    }

    // CC: List of individual people
    foreach (QString address, this->ccAddressStringList)
    {
        audienceMap.insertMulti("cc|person", "acct:" + address);
    }


    // Last check: if CC: Followers is checked, or nothing is, add Followers
    if (ccSelectorMenu->actions().at(1)->isChecked()
     || audienceMap.isEmpty())
    {
        audienceMap.insertMulti("cc|collection", this->pController->currentFollowersURL());
    }


    return audienceMap;
}



/********************************************************************/
/***************************** SLOTS ********************************/
/********************************************************************/



void Publisher::setMinimumMode()
{
    qDebug() << "setting Publisher to minimum mode";

    this->postButton->setFocus(); // give focus to button,
                                   // in case user shared with Ctrl+Enter

    // disable possible scrollbars
    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    // ~1 row
    this->setMaximumHeight((this->composerBox->fontInfo().pixelSize()+4) * 3);

    this->toSelectorButton->hide();
    this->ccSelectorButton->hide();

    // Check "public" if "public posts" is set in the preferences
    this->toSelectorMenu->actions().at(0)->setChecked(this->defaultPublicPosting);
    this->toSelectorMenu->actions().at(1)->setChecked(false);

    this->ccSelectorMenu->actions().at(0)->setChecked(false);
    this->ccSelectorMenu->actions().at(1)->setChecked(true);  // CC: Followers by default

    this->audienceSelectorTo->resetLists();
    this->audienceSelectorCC->resetLists();

    this->toAudienceLabel->setText("...");
    toAudienceLabel->repaint(); // Avoid a flicker-like effect later
    toAudienceLabel->clear();
    toAudienceLabel->hide();
    this->ccAudienceLabel->setText("...");
    ccAudienceLabel->repaint();
    ccAudienceLabel->clear();
    ccAudienceLabel->hide();

    this->toAddressStringList.clear();
    this->ccAddressStringList.clear();


    this->pictureButton->hide();
    this->postButton->hide();

    // Clear formatting options like bold, italic and underline
    this->composerBox->setCurrentCharFormat(QTextCharFormat());

    // Hide "picture mode" controls
    this->pictureLabel->hide();
    this->titleLineEdit->hide();
    this->selectPictureButton->hide();

    this->setEmptyPictureData();
}




void Publisher::setFullMode()
{
    qDebug() << "setting Publisher to full mode";

    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->setMaximumHeight(2048); // i.e. "unlimited"

    this->toSelectorButton->show();
    this->ccSelectorButton->show();

    this->toAudienceLabel->show();
    this->ccAudienceLabel->show();


    // Avoid re-enabling the picture button when re-focusing publisher, but still in picture mode...
    if (pictureButton->isHidden())
    {
        this->pictureButton->setEnabled(true); // If it wasn't hidden, don't re-enable
    }
    this->pictureButton->show();
    this->postButton->show();

    this->composerBox->setFocus(); // In case user used menu or shortcut
                                   // instead of clicking on it
}




void Publisher::setPictureMode()
{
    this->pictureButton->setDisabled(true);

    this->pictureLabel->show();

    this->titleLineEdit->show();
    this->selectPictureButton->show();

    this->findPictureFile(); // Show the open file dialog directly
}




void Publisher::findPictureFile()
{
    QString filename;

    filename = QFileDialog::getOpenFileName(this, tr("Select one image"),
                                            QDir::homePath(),
                                            tr("Image files") + "(*.png *.jpg *.jpeg *.gif);;"
                                            + tr("All files") + " (*)");

    if (!filename.isEmpty())
    {
        qDebug() << "Selected" << filename << "for upload";
        this->pictureLabel->setToolTip(filename);

        QString imageFormat = QString::fromLocal8Bit(QImageReader::imageFormat(filename));
        qDebug() << "Image format:" << imageFormat;

        if (imageFormat == "png" || imageFormat == "jpeg" || imageFormat == "gif")
        {
            this->pictureContentType = "image/" + imageFormat;
            qDebug() << pictureContentType;
            this->pictureFilename = filename;

            this->pictureLabel->setPixmap(QPixmap(filename).scaled(200, 150,
                                                                   Qt::IgnoreAspectRatio,
                                                                   Qt::SmoothTransformation));
        }
        else
        {
            // In the future, avoid this by using libmagic or similar
            qDebug() << "Unknown image format; Extension is probably wrong";
            QMessageBox::warning(this, tr("Invalid image"),
                                 tr("The image format cannot be detected.\n"
                                    "The extension might be wrong, like a GIF "
                                    "image renamed to image.jpg or similar."));

            this->pictureContentType.clear();
            this->pictureFilename.clear();
        }


    }
}



/*
 * Receive a list of addresses for the To or CC fields
 *
 */
void Publisher::updateToCcFields(QString selectorType,
                                 QStringList contactsList)
{
    qDebug() << "Publisher::updateToCcFields()" << selectorType << contactsList;

    QRegExp contactRE("(.+)\\s+\\<(.+@.+)\\>", Qt::CaseInsensitive);
    contactRE.setMinimal(true);

    QString addressesString;
    QStringList addressesStringList;

    foreach (QString contactString, contactsList)
    {
        contactRE.indexIn(contactString);

        addressesString.append(contactRE.cap(1).trimmed());
        addressesString.append(", ");

        addressesStringList.append(contactRE.cap(2).trimmed());
    }
    addressesString.remove(-2, 2); // remove ", " at the end


    if (selectorType == "to")
    {
        this->toAudienceLabel->setText(addressesString);
        this->toAddressStringList = addressesStringList;
    }
    else   // "cc"
    {
        this->ccAudienceLabel->setText(addressesString);
        this->ccAddressStringList = addressesStringList;
    }

    qDebug() << "Names:" << addressesString;
    qDebug() << "Addresses:" << addressesStringList;
}



/*
 * Send the post (note, image...) to the server
 *
 */
void Publisher::sendPost()
{
    qDebug() << "Publisher character count:" << composerBox->textCursor().document()->characterCount();


    // If there's some text in the post, or attached picture, send it
    if (composerBox->textCursor().document()->characterCount() > 1     // kinda tmp
      || !pictureFilename.isEmpty())
    {
        QString cleanHtml = MiscHelpers::cleanHtml(composerBox->toHtml());


        QMap<QString,QString> audienceMap = this->getAudienceMap();


        if (this->pictureFilename.isEmpty())
        {
            this->pController->postNote(audienceMap, cleanHtml);
        }
        else
        {
            this->pController->postImage(audienceMap,
                                         cleanHtml,
                                         this->titleLineEdit->text(), // Title
                                         pictureFilename,
                                         pictureContentType);
        }


        this->composerBox->erase();

        // Done composing message, hide buttons until we get focus again
        setMinimumMode();
    }
    else
    {
        qDebug() << "Can't send post: text is empty, and no picture";
    }
}
