/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "publisher.h"

Publisher::Publisher(PumpController *pumpController,
                     int publisherType,
                     QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;
    connect(pController, SIGNAL(postPublished()),
            this, SLOT(onPublishingOk()));
    connect(pController, SIGNAL(postPublishingFailed()),
            this, SLOT(onPublishingFailed()));

    this->setFocusPolicy(Qt::StrongFocus); // To keep the publisher from getting focus by accident


    this->audienceSelectorTo = new AudienceSelector(pController, "to", this);
    connect(audienceSelectorTo, SIGNAL(audienceSelected(QString,QStringList)),
            this, SLOT(updateToCcFields(QString,QStringList)));
    this->audienceSelectorCC = new AudienceSelector(pController, "cc", this);
    connect(audienceSelectorCC, SIGNAL(audienceSelected(QString,QStringList)),
            this, SLOT(updateToCcFields(QString,QStringList)));


    mainLayout = new QGridLayout();


    pictureLabel = new QLabel();
    pictureLabel->setScaledContents(true);
    pictureLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    pictureLabel->hide();

    titleLineEdit = new QLineEdit();
    titleLineEdit->setSizePolicy(QSizePolicy::MinimumExpanding,
                                 QSizePolicy::Maximum);
    titleLineEdit->setPlaceholderText(tr("Title for the picture (optional)"));
    titleLineEdit->setToolTip(tr("Title for the picture"));
    titleLineEdit->hide();

    pictureInfoLabel = new QLabel();


    selectPictureButton = new QPushButton(QIcon::fromTheme("insert-image"),
                                          tr("Select Picture..."));
    selectPictureButton->setSizePolicy(QSizePolicy::MinimumExpanding,
                                       QSizePolicy::Maximum);
    selectPictureButton->setToolTip(tr("Find the picture in your folders"));
    connect(selectPictureButton, SIGNAL(clicked()),
            this, SLOT(findPictureFile()));
    selectPictureButton->hide();

    // Set default pixmap and "picture not set" message
    this->setEmptyPictureData();
    lastUsedDirectory = QDir::homePath();



    composerBox = new Composer(true); // forPublisher = true
    composerBox->setSizePolicy(QSizePolicy::Minimum,
                               QSizePolicy::Minimum);
    connect(composerBox, SIGNAL(focusReceived()),    this, SLOT(setFullMode()));
    connect(composerBox, SIGNAL(editingFinished()),  this, SLOT(sendPost()));
    connect(composerBox, SIGNAL(editingCancelled()), this, SLOT(setMinimumMode()));


    // To... menu
    toPublicAction = new QAction(tr("Public"), this);
    toPublicAction->setCheckable(true);
    connect(toPublicAction, SIGNAL(toggled(bool)),
            this, SLOT(setToPublic(bool)));

    toFollowersAction = new QAction(tr("Followers"), this);
    toFollowersAction->setCheckable(true);
    connect(toFollowersAction, SIGNAL(toggled(bool)),
            this, SLOT(setToFollowers(bool)));

    toSelectorMenu = new QMenu("to-menu");
    toSelectorMenu->addAction(toPublicAction);
    toSelectorMenu->addAction(toFollowersAction);
    toSelectorMenu->addSeparator();
    toSelectorMenu->addAction(tr("More..."), audienceSelectorTo, SLOT(show()));

    toSelectorButton = new QPushButton(QIcon::fromTheme("system-users"),
                                     tr("To..."));
    toSelectorButton->setToolTip(tr("Select who will see this post"));
    toSelectorButton->setMenu(toSelectorMenu);


    // CC... menu
    ccPublicAction = new QAction(tr("Public"), this);
    ccPublicAction->setCheckable(true);
    connect(ccPublicAction, SIGNAL(toggled(bool)),
            this, SLOT(setCCPublic(bool)));

    ccFollowersAction = new QAction(tr("Followers"), this);
    ccFollowersAction->setCheckable(true);
    connect(ccFollowersAction, SIGNAL(toggled(bool)),
            this, SLOT(setCCFollowers(bool)));

    ccSelectorMenu = new QMenu("cc-menu");
    ccSelectorMenu->addAction(ccPublicAction);
    ccSelectorMenu->addAction(ccFollowersAction);
    ccSelectorMenu->addSeparator();
    ccSelectorMenu->addAction(tr("More..."), audienceSelectorCC, SLOT(show()));

    ccSelectorButton = new QPushButton(QIcon::fromTheme("system-users"),
                                     tr("CC..."));
    ccSelectorButton->setToolTip(tr("Select who will get a copy of this post"));
    ccSelectorButton->setMenu(ccSelectorMenu);


    QFont audienceLabelsFont; // "To" column will be normal, "CC" will be italic
    audienceLabelsFont.setPointSize(audienceLabelsFont.pointSize()-1);

    // These will hold the names of the *people* selected for the To or CC fields, if any
    toAudienceLabel = new QLabel();
    toAudienceLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    toAudienceLabel->setFont(audienceLabelsFont);  // Normal
    toAudienceLabel->setWordWrap(true);

    ccAudienceLabel = new QLabel();
    ccAudienceLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    audienceLabelsFont.setItalic(true);
    ccAudienceLabel->setFont(audienceLabelsFont);  // Italic
    ccAudienceLabel->setWordWrap(true);

    // And these will show "public" or "followers" current selection (in bold)
    toPublicFollowersLabel = new QLabel();
    toPublicFollowersLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    audienceLabelsFont.setBold(true);
    audienceLabelsFont.setItalic(false);
    toPublicFollowersLabel->setFont(audienceLabelsFont);  // Bold

    ccPublicFollowersLabel = new QLabel();
    ccPublicFollowersLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    audienceLabelsFont.setItalic(true);
    ccPublicFollowersLabel->setFont(audienceLabelsFont);  // Bold + italic


    // To set the 'picture mode'
    pictureButton = new QPushButton(QIcon::fromTheme("camera-photo"),
                                    tr("Ad&d Picture"));
    pictureButton->setToolTip(tr("Upload photo"));
    connect(pictureButton, SIGNAL(clicked()),
            this, SLOT(setPictureMode()));


    // Sending status info label
    statusInfoLabel = new QLabel();
    statusInfoLabel->setAlignment(Qt::AlignCenter);
    statusInfoLabel->setWordWrap(true);
    audienceLabelsFont.setBold(false);
    audienceLabelsFont.setItalic(false);
    statusInfoLabel->setFont(audienceLabelsFont);


    // To send the post
    postButton = new QPushButton(QIcon::fromTheme("mail-send"),
                                  tr("Post"));
    postButton->setToolTip(tr("Hit Control+Enter to post with the keyboard"));
    connect(postButton, SIGNAL(clicked()),
            this, SLOT(sendPost()));

    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                   tr("Cancel"));
    cancelButton->setToolTip(tr("Cancel the post"));
    connect(cancelButton, SIGNAL(clicked()),
            composerBox, SLOT(cancelPost()));


    // Now the layout, starting with "picture mode" stuff
    mainLayout->addWidget(pictureLabel,        0, 0, 2, 3);
    mainLayout->addWidget(titleLineEdit,       0, 3, 1, 3, Qt::AlignTop);
    mainLayout->addWidget(pictureInfoLabel,    1, 3, 1, 1, Qt::AlignBottom);
    mainLayout->addWidget(selectPictureButton, 1, 4, 1, 2, Qt::AlignBottom | Qt::AlignRight);

    // The rest depends on the publisher type
    switch (publisherType)
    {
    case 1: // second layout, buttons around
        mainLayout->addWidget(toSelectorButton, 2, 0, 1, 1);
        mainLayout->addWidget(ccSelectorButton, 3, 0, 1, 1);
        mainLayout->addWidget(pictureButton,    4, 0, 1, 1);
        mainLayout->addWidget(composerBox,      2, 1, 3, 4); // 3 rows, 4 columns
        mainLayout->addWidget(postButton,       2, 5, 3, 1); // 3 rows
        postButton->setSizePolicy(QSizePolicy::Maximum,
                                  QSizePolicy::MinimumExpanding);
        break;

    case 2: // Third layout, buttons on right side
        mainLayout->addWidget(composerBox,      2, 0, 3, 4); // 3 rows, 4 columns
        mainLayout->addWidget(toSelectorButton, 2, 4, 1, 1);
        mainLayout->addWidget(ccSelectorButton, 3, 4, 1, 1);
        mainLayout->addWidget(pictureButton,    4, 4, 1, 1);
        mainLayout->addWidget(postButton,       2, 5, 3, 1); // 3 rows, 1 column
        postButton->setSizePolicy(QSizePolicy::Maximum,
                                  QSizePolicy::MinimumExpanding);
        break;


    case 0: // First (default) layout
        // just let it jump to default, so incorrect values go to default

    default:
        mainLayout->addWidget(composerBox,      2, 0, 3, 6); // 3 rows, 6 columns
        mainLayout->addWidget(toSelectorButton, 5, 0, 1, 1, Qt::AlignLeft);
        mainLayout->addWidget(ccSelectorButton, 5, 1, 1, 1, Qt::AlignLeft);
        mainLayout->addWidget(pictureButton,    5, 3, 1, 1, Qt::AlignLeft);
        mainLayout->addWidget(postButton,       5, 5, 1, 1);

        break;

    }

    // The 2 labels holding people's names, if any
    mainLayout->addWidget(toAudienceLabel, 6, 0, 1, 1);
    mainLayout->addWidget(ccAudienceLabel, 6, 1, 1, 1);

    // The 2 labels holding "public/followers", if selected
    mainLayout->addWidget(toPublicFollowersLabel, 7, 0, 1, 1);
    mainLayout->addWidget(ccPublicFollowersLabel, 7, 1, 1, 1);

    // the status info label
    mainLayout->addWidget(statusInfoLabel, 5, 4, 2, 1, Qt::AlignCenter);

    // The "cancel" button
    mainLayout->addWidget(cancelButton,    7, 5, 1, 1);

    this->setLayout(mainLayout);

    this->setMinimumMode();

    qDebug() << "Publisher created";
}


Publisher::~Publisher()
{
    qDebug() << "Publisher destroyed";
}


/*
 * Set if "public" option for audience is checked as default
 *
 */
void Publisher::setDefaultPublicPosting(bool defaultPublicPosts)
{
    this->defaultPublicPosting = defaultPublicPosts;

    // Ensure status is sync'ed
    this->toSelectorMenu->actions().at(0)->setChecked(this->defaultPublicPosting);
}


/*
 * Set default "no photo" pixmap and "picture not set" message
 *
 * Clear the filename and content type variables too
 */
void Publisher::setEmptyPictureData()
{
    pictureLabel->setPixmap(QIcon::fromTheme("image-x-generic")
                            .pixmap(200, 150)
                            .scaled(200, 150,
                                    Qt::IgnoreAspectRatio,
                                    Qt::SmoothTransformation));
    pictureLabel->setToolTip(tr("Picture not set"));

    titleLineEdit->clear();
    pictureInfoLabel->clear();

    this->pictureFilename.clear();
    this->pictureContentType.clear();
}


void Publisher::updatePublicFollowersLabels()
{
    QString toString;
    if (toPublicAction->isChecked())
    {
        toString.append("+" + tr("Public") + "\n");
    }
    if (toFollowersAction->isChecked())
    {
        toString.append("+" + tr("Followers") + "\n");
    }
    toPublicFollowersLabel->setText(toString);


    QString ccString;
    if (ccPublicAction->isChecked())
    {
        ccString.append("+" + tr("Public") + "\n");
    }
    if (ccFollowersAction->isChecked())
    {
        ccString.append("+" + tr("Followers") + "\n");
    }
    ccPublicFollowersLabel->setText(ccString);
}



/*
 * Create a key:value map, listing who will receive a post, like:
 *
 * "collection" : "http://activityschema.org/collection/public"
 * "collection" : "https://pump.example/api/user/followers"
 * "person"     : "acct:somecontact@pumpserver.example"
 *
 */
QMap<QString,QString> Publisher::getAudienceMap()
{
    QMap<QString,QString> audienceMap;


    // To: Public is checked
    if (toPublicAction->isChecked())
    {
        audienceMap.insertMulti("to|collection", "http://activityschema.org/collection/public");
    }

    // To: List of individual people
    foreach (QString address, this->toAddressStringList)
    {
        audienceMap.insertMulti("to|person", "acct:" + address);
    }

    // To: Followers is checked
    if (toFollowersAction->isChecked())
    {
        audienceMap.insertMulti("to|collection", this->pController->currentFollowersURL());
    }


    // CC: Public is checked
    if (ccPublicAction->isChecked())
    {
        audienceMap.insertMulti("cc|collection", "http://activityschema.org/collection/public");
    }

    // CC: List of individual people
    foreach (QString address, this->ccAddressStringList)
    {
        audienceMap.insertMulti("cc|person", "acct:" + address);
    }


    // Last check: if CC: Followers is checked, or nothing is, add Followers
    if (ccFollowersAction->isChecked() || audienceMap.isEmpty())
    {
        audienceMap.insertMulti("cc|collection", this->pController->currentFollowersURL());
    }


    return audienceMap;
}



/********************************************************************/
/***************************** SLOTS ********************************/
/********************************************************************/



void Publisher::setMinimumMode()
{
    qDebug() << "setting Publisher to minimum mode";

    this->postButton->setFocus(); // give focus to button,
                                  // in case user shared with Ctrl+Enter

    // disable possible scrollbars
    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    // ~1 row
    this->setMaximumHeight((this->composerBox->fontInfo().pixelSize() + 6) * 3);


    this->audienceSelectorTo->resetLists();
    this->audienceSelectorCC->resetLists();

    this->toAudienceLabel->setText("...");
    toAudienceLabel->repaint(); // Avoid a flicker-like effect later
    toAudienceLabel->clear();
    toAudienceLabel->hide();

    this->toAddressStringList.clear();

    toPublicFollowersLabel->hide();


    this->ccAudienceLabel->setText("...");
    ccAudienceLabel->repaint();
    ccAudienceLabel->clear();
    ccAudienceLabel->hide();

    this->ccAddressStringList.clear();

    ccPublicFollowersLabel->hide();


    this->toSelectorButton->hide();
    this->ccSelectorButton->hide();

    // Check "public" if "public posts" is set in the preferences
    toPublicAction->setChecked(this->defaultPublicPosting);
    toFollowersAction->setChecked(false);

    ccPublicAction->setChecked(false);
    ccFollowersAction->setChecked(true);  // CC: Followers by default


    this->pictureButton->hide();
    statusInfoLabel->clear();
    this->statusInfoLabel->hide();
    this->postButton->hide();
    this->cancelButton->hide();


    // Clear formatting options like bold, italic and underline
    this->composerBox->setCurrentCharFormat(QTextCharFormat());

    // Hide "picture mode" controls
    this->pictureLabel->hide();
    this->titleLineEdit->hide();
    this->pictureInfoLabel->hide();
    this->selectPictureButton->hide();

    this->setEmptyPictureData();
}




void Publisher::setFullMode()
{
    qDebug() << "setting Publisher to full mode";

    this->composerBox->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->composerBox->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    this->setMaximumHeight(2048); // i.e. "unlimited"

    this->toSelectorButton->show();
    this->ccSelectorButton->show();

    this->toAudienceLabel->show();
    this->toPublicFollowersLabel->show();

    this->ccAudienceLabel->show();
    this->ccPublicFollowersLabel->show();


    // Avoid re-enabling the picture button when re-focusing publisher, but still in picture mode...
    if (pictureButton->isHidden())
    {
        this->pictureButton->setEnabled(true); // If it wasn't hidden, don't re-enable
    }
    this->pictureButton->show();
    this->statusInfoLabel->show();
    this->postButton->show();
    this->cancelButton->show();

    this->composerBox->setFocus(); // In case user used menu or shortcut
                                   // instead of clicking on it
}




void Publisher::setPictureMode()
{
    this->pictureButton->setDisabled(true);

    this->pictureLabel->show();

    this->titleLineEdit->show();
    this->pictureInfoLabel->show();
    this->selectPictureButton->show();

    this->findPictureFile(); // Show the open file dialog directly
}


/*
 * After the post is confirmed to have been received by the server
 * re-enable publisher, clear text, etc.
 *
 */
void Publisher::onPublishingOk()
{
    this->statusInfoLabel->clear();
    this->setEnabled(true);

    this->composerBox->erase();

    // Done composing message, hide buttons until we get focus again
    setMinimumMode();
}

/*
 * If there was a HTTP error while posting...
 *
 */
void Publisher::onPublishingFailed()
{
    qDebug() << "Posting failed, re-enabling Publisher";
    this->statusInfoLabel->setText(tr("Posting failed.\nTry again."));
    this->setEnabled(true);
    this->composerBox->setFocus();
}


/*
 * These are called when selecting Public or Followers in the menus
 *
 * When selecting "CC: Followers", "To: Followers" gets unselected, etc.
 *
 */
void Publisher::setToPublic(bool activated)
{
    if (activated)
    {
        ccPublicAction->setChecked(false);     // Unselect "CC: Public"
    }
    updatePublicFollowersLabels();
}

void Publisher::setToFollowers(bool activated)
{
    if (activated)
    {
        ccFollowersAction->setChecked(false);  // Unselect "CC: Followers"
    }
    updatePublicFollowersLabels();
}

void Publisher::setCCPublic(bool activated)
{
    if (activated)
    {
        toPublicAction->setChecked(false);     // Unselect "To: Public"
    }
    updatePublicFollowersLabels();
}

void Publisher::setCCFollowers(bool activated)
{
    if (activated)
    {
        toFollowersAction->setChecked(false);  // Unselect "To: Followers"
    }
    updatePublicFollowersLabels();
}




void Publisher::findPictureFile()
{
    QString filename;

    filename = QFileDialog::getOpenFileName(this, tr("Select one image"),
                                            this->lastUsedDirectory,
                                            tr("Image files") + "(*.png *.jpg *.jpeg *.gif);;"
                                            + tr("All files") + " (*)");

    if (!filename.isEmpty())
    {
        qDebug() << "Selected" << filename << "for upload";
        this->pictureLabel->setToolTip(filename);

        QString formatString = QString::fromLocal8Bit(QImageReader::imageFormat(filename));
        qDebug() << "Image format:" << formatString;

        if (formatString == "png" || formatString == "jpeg" || formatString == "gif")
        {
            this->pictureContentType = "image/" + formatString;
            this->pictureFilename = filename;

            QPixmap imagePixmap = QPixmap(filename);
            this->pictureLabel->setPixmap(imagePixmap.scaled(200, 150,
                                                             Qt::IgnoreAspectRatio,
                                                             Qt::SmoothTransformation));

            QFileInfo fileInfo(filename);
            int filesize = fileInfo.size();
            this->lastUsedDirectory = fileInfo.path();
            qDebug() << "last used directory:" << lastUsedDirectory;
            QString sizeUnit = tr("bytes");
            if (filesize > 1024) // if > 1024 bytes, transform to KiB
            {
                filesize /= 1024;
                sizeUnit = "KiB";
            }
            if (filesize > 1024) // if > 1024 KiB, transform to MiB
            {
                filesize /= 1024;
                sizeUnit = "MiB";
            }
            pictureInfoLabel->setText(QString("<b>" +tr("Resolution") + ":</b> %1x%2"
                                      "<br>"
                                      "<b>" + tr("Size") + ":</b> %3 %4")
                                      .arg(imagePixmap.width())
                                      .arg(imagePixmap.height())
                                      .arg(filesize)
                                      .arg(sizeUnit));
        }
        else
        {
            // In the future, avoid this by using libmagic or similar
            qDebug() << "Unknown image format; Extension is probably wrong";
            QMessageBox::warning(this, tr("Invalid image"),
                                 tr("The image format cannot be detected.\n"
                                    "The extension might be wrong, like a GIF "
                                    "image renamed to image.jpg or similar."));

            this->pictureContentType.clear();
            this->pictureFilename.clear();
        }
    }
}



/*
 * Receive a list of addresses for the To or CC fields
 *
 */
void Publisher::updateToCcFields(QString selectorType,
                                 QStringList contactsList)
{
    qDebug() << "Publisher::updateToCcFields()" << selectorType << contactsList;

    QRegExp contactRE("(.+)\\s+\\<(.+@.+)\\>", Qt::CaseInsensitive);
    contactRE.setMinimal(true);

    QString addressesString;
    QStringList addressesStringList;

    foreach (QString contactString, contactsList)
    {
        contactRE.indexIn(contactString);

        addressesString.append(contactRE.cap(1).trimmed());
        addressesString.append(", ");

        addressesStringList.append(contactRE.cap(2).trimmed());
    }
    addressesString.remove(-2, 2); // remove ", " at the end


    if (selectorType == "to")
    {
        this->toAudienceLabel->setText(addressesString);
        this->toAddressStringList = addressesStringList;
    }
    else   // "cc"
    {
        this->ccAudienceLabel->setText(addressesString);
        this->ccAddressStringList = addressesStringList;
    }

    qDebug() << "Names:" << addressesString;
    qDebug() << "Addresses:" << addressesStringList;
}



/*
 * Send the post (note, image...) to the server
 *
 */
void Publisher::sendPost()
{
    qDebug() << "Publisher character count:" << composerBox->textCursor().document()->characterCount();


    // If there's some text in the post, or attached picture, send it
    if (composerBox->textCursor().document()->characterCount() > 1     // kinda tmp
      || !pictureFilename.isEmpty())
    {
        QString cleanHtml = MiscHelpers::cleanHtml(composerBox->toHtml());


        QMap<QString,QString> audienceMap = this->getAudienceMap();


        this->statusInfoLabel->setText(tr("Posting..."));
        // Don't erase just yet!! Just disable until we get "200 OK" from the server.
        this->setDisabled(true);


        if (this->pictureFilename.isEmpty())
        {
            this->pController->postNote(audienceMap, cleanHtml);
        }
        else
        {
            this->pController->postImage(audienceMap,
                                         cleanHtml,
                                         this->titleLineEdit->text(), // Title
                                         pictureFilename,
                                         pictureContentType);
        }
    }
    else
    {
        qDebug() << "Can't send post: text is empty, and no picture";
    }
}
