/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "timeline.h"

TimeLine::TimeLine(int timelineType,
                   PumpController *pumpController,
                   QWidget *parent) :  QWidget(parent)
{
    this->timelineType = timelineType;

    this->pController = pumpController;

    mainLayout = new QVBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);


    QMap<QString,QString> demoPostData;
    demoPostData.insert("author",         "Demo User");
    demoPostData.insert("authorId",       "demo@somepump.example");
    demoPostData.insert("authorBio",      "I am not a real user");
    demoPostData.insert("authorHometown", "Demoville");
    demoPostData.insert("generator",      "Dianara");
    demoPostData.insert("createdAt",      "2013-05-01T00:00:00Z");


    QSettings settings; // FIXME: kinda tmp, until posts have "unread" status, etc.
    settings.beginGroup("TimelineStates");

    // Demo post content depends on timeline type
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        demoPostData.insert("text",   "<h2><b>" +tr("Welcome to Dianara") + "</b></h2><br>"

                                      + tr("Dianara is a <b>pump.io</b> client.")
                                      + "<br>"

                                      + tr("If you don't have a Pump account yet, you can get one "
                                           "at the following address:")
                                      + "<br>"
                                        "<a href=\"http://pump.io/tryit.html\">http://pump.io/tryit.html</a>"
                                        "<br><br>"

                                      + tr("First, configure your account from the "
                                           "<b>Settings - Account</b> menu.")
                                      + "<br>"
                                      + tr("After the process is done, your profile "
                                           "and timelines should update automatically.")
                                      + "<br><br>"

                                      + tr("Take a moment to look around the menus and "
                                           "the Configuration window.")
                                      + "<br><br>"

                                      + tr("There are tooltips everywhere, so if you hover "
                                           "a button or a text field with your mouse, you'll "
                                           "probably see some extra information.")
                                      + "<br><br>"

                                      + "<a href=\"http://jancoding.wordpress.com/dianara\">"
                                      + tr("Dianara's blog") + "</a><br><br>"

                                        "<a href=\"https://github.com/e14n/pump.io/wiki/FAQ\">"
                                      + tr("Frequently asked questions about pump.io")
                                      + "</a>"
                                      + "<br>");

        this->previousNewestPostID = settings.value("previousNewestPostIDMain", "").toString();
        break;


    case TimelineTypeDirect:
        demoPostData.insert("text",   "<h2><b>" + tr("Direct Messages Timeline") + "</b></h2><br>"

                                      + tr("Here, you'll see posts specifically directed to you.")

                                      + "<br>");
        this->previousNewestPostID = settings.value("previousNewestPostIDDirect", "").toString();
        break;


    case TimelineTypeActivity:
        demoPostData.insert("text",   "<h2><b>" + tr("Activity Timeline") + "</b></h2><br>"

                                      + tr("You'll see your own posts here.")

                                      +"<br>");
        this->previousNewestPostID = settings.value("previousNewestPostIDActivity", "").toString();
        break;


    case TimelineTypeFavorites:
        demoPostData.insert("text",   "<h2><b>" + tr("Favorites Timeline") + "</b></h2><br>"

                                      + tr("Posts and comments you've liked.")

                                      + "<br>");
        this->previousNewestPostID = settings.value("previousNewestPostIDFavorites", "").toString();
        break;



    default:
        demoPostData.insert("text", "<h2>Empty timeline</h2>");

    }
    settings.endGroup();



    splitter = new QSplitter(Qt::Vertical, this);
    splitter->setChildrenCollapsible(false);
    mainLayout->addWidget(splitter);


    this->unreadPostsCount = 0;
    this->timelineOffset = 0;

    firstPageButton = new QPushButton(QIcon::fromTheme("go-first"), tr("F&irst Page"));
    connect(firstPageButton, SIGNAL(clicked()),
            this, SLOT(goToFirstPage()));

    currentPageLabel = new QLabel();
    currentPageLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    currentPageLabel->setSizePolicy(QSizePolicy::Maximum,
                                    QSizePolicy::Maximum);

    previousPageButton = new QPushButton(QIcon::fromTheme("go-previous"), tr("&Previous Page"));
    connect(previousPageButton, SIGNAL(clicked()),
            this, SLOT(goToPreviousPage()));
    nextPageButton = new QPushButton(QIcon::fromTheme("go-next"), tr("&Next Page"));
    connect(nextPageButton, SIGNAL(clicked()),
            this, SLOT(goToNextPage()));


    bottomLayout = new QHBoxLayout();
    bottomLayout->addWidget(firstPageButton);
    bottomLayout->addSpacing(8);
    bottomLayout->addWidget(currentPageLabel);
    bottomLayout->addSpacing(8);
    bottomLayout->addWidget(previousPageButton);
    bottomLayout->addWidget(nextPageButton);
    mainLayout->addLayout(bottomLayout);


    // Add the default "demo" post
    splitter->addWidget(new Post(pController, demoPostData,
                                 QVariantList(), QVariantList(), QVariantList(),
                                 this));


    this->setLayout(mainLayout);


    qDebug() << "TimeLine created";
}


TimeLine::~TimeLine()
{
    QSettings settings;
    settings.beginGroup("TimelineStates");

    switch (timelineType)
    {
    case TimelineTypeMain:
        settings.setValue("previousNewestPostIDMain", this->previousNewestPostID);
        break;

    case TimelineTypeDirect:
        settings.setValue("previousNewestPostIDDirect", this->previousNewestPostID);
        break;

    case TimelineTypeActivity:
        settings.setValue("previousNewestPostIDActivity", this->previousNewestPostID);
        break;

    case TimelineTypeFavorites:
        settings.setValue("previousNewestPostIDFavorites", this->previousNewestPostID);
        break;
    }
    settings.endGroup();


    qDebug() << "TimeLine destroyed";
}



/*
 * Remove all widgets (Post *) from the timeline
 *
 */
void TimeLine::clearTimeLineContents()
{
    QList<QWidget*> previousPostList;

    // Build list of widgets contained in the layout
    for (int counter = 0; counter < splitter->count(); ++counter)
    {
        //qDebug() << "Adding for deletion:" << splitter->widget(counter);
        previousPostList.append(splitter->widget(counter));
    }

    qDebug() << "List to delete:" << previousPostList;


    foreach (QWidget *oldPost, previousPostList)
    {
        qDebug() << "will deleteLater:" << oldPost;
        oldPost->deleteLater(); // delete should be safe too, but still...
    }

    this->postsInTimeline.clear();

    /*
     * This is to avoid the splitter getting bigger and bigger every time the user
     * updates the timeline (F5, etc)
     *
     */
    this->splitter->setMinimumHeight(0);
    this->splitter->resize(1,1);
}



void TimeLine::requestTimelinePage()
{
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        this->pController->getMainTimeline(this->timelineOffset);
        break;

    case TimelineTypeDirect:
        this->pController->getDirectTimeline(this->timelineOffset);
        break;

    case TimelineTypeActivity:
        this->pController->getActivityTimeline(this->timelineOffset);
        break;

    case TimelineTypeFavorites:
        this->pController->getFavoritesTimeline(this->timelineOffset);
        break;
    }

}


/*
 *  Update the label at the bottom of the page, indicating current "page"
 *
 */
void TimeLine::updateCurrentPageNumber()
{
    QString currentPageString;
    currentPageString = QString("%1").arg((this->timelineOffset / this->postsPerPage) + 1);

    this->currentPageLabel->setText(currentPageString);
}




/*********************************************************/
/*********************************************************/
/************************ SLOTS **************************/
/*********************************************************/
/*********************************************************/



void TimeLine::setTimeLineContents(QVariantList postList, int postsPerPage,
                                   QString previousLink, QString nextLink)
{
    qDebug() << "TimeLine::setTimeLineContents()";

    // Remove all previous posts in timeline
    qDebug() << "Removing previous posts from timeline";
    this->clearTimeLineContents();


    this->postsPerPage = postsPerPage;

    int newPostCount = 0;
    bool allNewPostsCounted = false;
    QString newestPostID; // Here we'll store the postID for the first (newest) post in the timeline
                          // With it, we can know how many new posts (if any) we receive next time

    // Fill timeline with new contents
    foreach (QVariant singlePost, postList)
    {
        if (singlePost.type() == QVariant::Map)
        {
            this->previousPageLink = previousLink;   // Useless at the moment
            this->nextPageLink = nextLink;

            QVariantMap singlePostMap = singlePost.toMap();
            QVariantMap objectMap;
            if (this->timelineType != TimelineTypeFavorites)  // VERY TMP!! Favorites-tmp-stuff
            {
                objectMap = singlePostMap["object"].toMap();
            }
            else
            {
                // Since "favorites" is a collection of objects directly, not activities
                objectMap = singlePostMap;
            }


            QString itemVerb = singlePostMap["verb"].toString();

            // This will hold the date when the post was deleted, or empty if it wasn't
            QString postIsDeleted = objectMap["deleted"].toString();


            QString postID;
            QString postURL;
            QString postType;
            QString postAuthor;
            QString postAuthorAvatar;
            QString postAuthorId;
            QString postAuthorHometown;
            QString postAuthorBio;
            QString postGenerator;
            QString postCreatedAt;
            QString postTo;
            QString postCC;
            QString postTitle;
            QString postImage;
            QString postText;
            QString postIsNsfw;
            QString postLikesCount;
            QString postCommentsCount;
            QString postResharesCount;

            QString postIsLiked;
            QString postIsShared;
            QString postSharedBy;
            QString postSharedById;

            bool postIsNew = false;

            postID = objectMap["id"].toString();
            if (newestPostID.isEmpty()) // only first time, for newest post
            {
                if (this->timelineOffset == 0)
                {
                    newestPostID = postID;
                }
                else
                {
                    newestPostID = this->previousNewestPostID;
                    allNewPostsCounted = true;
                }
            }


            if (!allNewPostsCounted)
            {
                if (postID == this->previousNewestPostID)
                {
                    allNewPostsCounted = true;
                }
                else
                {
                    // If post is NOT deleted, add it to the count
                    if (postIsDeleted.isEmpty())
                    {
                        ++newPostCount;

                        // Mark current post as new
                        postIsNew = true;
                    }
                }
            }

            postURL = objectMap["url"].toString();

            postType = objectMap["objectType"].toString();

            postIsLiked = objectMap["liked"].toString();

            QVariantMap authorMap;
            if (itemVerb == "share")
            {
                postIsShared = "true";

                // Get original author data
                authorMap = objectMap["author"].toMap();

                postSharedBy = singlePostMap["actor"].toMap()["displayName"].toString();
                postSharedById = singlePostMap["actor"].toMap()["id"].toString();
                if (postSharedById.startsWith("acct:"))
                {
                    postSharedById.remove(0,5);
                }
            }
            else
            {
                postIsShared = "false";

                // Not a share, so no original author, just get the actor
                authorMap = singlePostMap["actor"].toMap();
            }

            //// VERY TMP!! Favorites-tmp-stuff
            if (this->timelineType==TimelineTypeFavorites)
            {
                authorMap = objectMap["author"].toMap();
            }

            postAuthor = authorMap["displayName"].toString();
            postAuthorAvatar = authorMap["image"].toMap().value("url").toString();
            postAuthorId = authorMap["id"].toString();
            // For Author ID, remove the first 5 characters from the field, if they are "acct:"
            // (users have acct: in front of the ID, but services like OFG don't)
            if (postAuthorId.startsWith("acct:"))
            {
                postAuthorId.remove(0,5);
            }

            postAuthorBio = authorMap["summary"].toString();
            postAuthorHometown = authorMap["location"].toMap().value("displayName").toString();


            postCreatedAt = objectMap["published"].toString();
            postGenerator = singlePostMap["generator"].toMap().value("displayName").toString();


            // Audience
            foreach (QVariant postToVariant, singlePostMap.value("to").toList())
            {
                if (postToVariant.toMap().value("id").toString()
                  == "http://activityschema.org/collection/public")
                {
                    postTo += tr("Public") + ", ";
                }
                else
                {
                    postTo += postToVariant.toMap().value("displayName").toString() + ", ";
                }
            }
            postTo.remove(-2, 2); // remove last comma and space

            // CC
            foreach (QVariant postCCVariant, singlePostMap.value("cc").toList())
            {
                if (postCCVariant.toMap().value("id").toString()
                  == "http://activityschema.org/collection/public")
                {
                    postCC += tr("Public") + ", ";
                }
                else
                {
                    postCC += postCCVariant.toMap().value("displayName").toString() + ", ";
                }
            }
            postCC.remove(-2, 2);




            ///
            /// End of "meta"; Start of content
            ///

            if (postType == "image")
            {
                postTitle = objectMap["displayName"].toString();


                // See if there's a proxy URL for the image first (for private images)
                postImage = objectMap["fullImage"].toMap().value("pump_io").toMap().value("proxyURL").toString();
                qDebug() << "Trying Proxyed fullImage";

                // if that's empty, try the "small" version
                if (postImage.isEmpty())
                {
                    qDebug() << "Trying Proxyed thumbnail image";
                    postImage = objectMap["image"].toMap().value("pump_io").toMap().value("proxyURL").toString();
                }

                // If that's also empty, use regular fullImage->url field
                if (postImage.isEmpty())
                {
                    qDebug() << "Trying direct fullImage";
                    postImage = objectMap["fullImage"].toMap().value("url").toString();
                }

                // And if that is ALSO empty, use regular image->url
                if (postImage.isEmpty())
                {
                    qDebug() << "Trying direct thumbnail image";
                    postImage = objectMap["image"].toMap().value("url").toString();
                }
                qDebug() << "postImage:" << postImage;
            }


            postText = objectMap["content"].toString();

            postLikesCount = objectMap["likes"].toMap().value("totalItems").toString();
            postCommentsCount = objectMap["replies"].toMap().value("totalItems").toString();
            postResharesCount = objectMap["shares"].toMap().value("totalItems").toString();

            // get last likes, comments and shares list here, pass to the Post()
            QVariantList lastLikesList = objectMap["likes"].toMap().value("items").toList();
            QVariantList lastCommentsList = objectMap["replies"].toMap().value("items").toList();
            QVariantList lastSharesList = objectMap["shares"].toMap().value("items").toList();

            // Get URL for likes; first, proxyURL if it exists
            QString postLikesURL = objectMap["likes"].toMap().value("pump_io").toMap().value("proxyURL").toString();
            // If still empty, get regular URL (that means the post is in the same server we are)
            if (postLikesURL.isEmpty())
            {
                postLikesURL = objectMap["likes"].toMap().value("url").toString();
            }

            // Get URL for comments; first, proxyURL if it exists
            QString postCommentsURL = objectMap["replies"].toMap().value("pump_io").toMap().value("proxyURL").toString();
            if (postCommentsURL.isEmpty()) // If still empty, get regular URL
            {
                postCommentsURL = objectMap["replies"].toMap().value("url").toString();
            }



            QMap<QString,QString> postData;
            postData.insert("id",                 postID);
            postData.insert("url",                postURL);
            postData.insert("postType",           postType);
            postData.insert("author",             postAuthor);
            postData.insert("authorAvatar",       postAuthorAvatar);
            postData.insert("authorId",           postAuthorId);
            postData.insert("authorBio",          postAuthorBio);
            postData.insert("authorHometown",     postAuthorHometown);
            postData.insert("generator",          postGenerator);
            postData.insert("createdAt",          postCreatedAt);

            postData.insert("to",                 postTo);
            postData.insert("cc",                 postCC);

            postData.insert("title",              postTitle);
            postData.insert("image",              postImage);
            postData.insert("text",               postText);
            postData.insert("nsfw",               postIsNsfw);
            postData.insert("likesCount",         postLikesCount);
            postData.insert("likesURL",           postLikesURL);
            postData.insert("commentsCount",      postCommentsCount);
            postData.insert("commentsURL",        postCommentsURL);
            postData.insert("resharesCount",      postResharesCount);

            postData.insert("postIsLiked",        postIsLiked);
            postData.insert("postIsShared",       postIsShared);
            postData.insert("sharedBy",           postSharedBy);
            postData.insert("sharedById",         postSharedById);

            postData.insert("postIsNew",          postIsNew ? "true":"false");


            if (postIsDeleted.isEmpty())  // if post was NOT deleted
            {
                Post *newPost = new Post(pController,
                                         postData,
                                         lastLikesList,
                                         lastCommentsList,
                                         lastSharesList,
                                         this);
                this->postsInTimeline.append(newPost);

                this->splitter->addWidget(newPost);

                this->splitter->setMinimumHeight(this->splitter->height()
                                                 + newPost->sizeHint().height());

                connect(newPost, SIGNAL(postRead()),
                        this, SLOT(decreaseUnreadPostsCount()));
            }
            else
            {
                // if there's a "deleted" key, ignore this post
                qDebug() << "This post was deleted on" << postIsDeleted << " / Not adding.";
            }


            // set stretch factor of 1 to each added post
            this->splitter->setStretchFactor(splitter->count()-1,  1);
        }
        else  // singlePost.type() is not a QVariant::Map
        {
            qDebug() << "Expected a Map, got something else";
            qDebug() << postList;
        }
    } // end foreach

    // Add the "extra" empty widget
    /*
    this->splitter->addWidget(new QWidget());
    this->splitter->setMinimumHeight(this->splitter->minimumHeight()
                                     - splitter->widget(splitter->count()-1)->height());
    */

    emit scrollToTop(); // ask mainWindow to scroll the QScrollArea containing the timeline to the top

    this->previousNewestPostID = newestPostID;
    this->unreadPostsCount = newPostCount;
    qDebug() << "New posts:" << newPostCount << "; Newest post ID:" << previousNewestPostID;

    this->updateCurrentPageNumber();

    emit timelineRendered(this->timelineType, unreadPostsCount);

    qDebug() << "setTimeLineContents() /END";
}





/*
 * Add the full list of likes to a post
 *
 */
void TimeLine::setLikesInPost(QVariantList likesList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setLikesInPost()";

    QString originatingPostCleanURL = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanURL;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->likesURL() << "==" << originatingPostCleanURL;

        if (post->likesURL() == originatingPostCleanURL)
        {
            qDebug() << "Found originating Post; setting likes on it...";
            post->setLikes(likesList);

            break;
        }
    }
}


/*
 * Add the full list of comments to a post
 *
 */
void TimeLine::setCommentsInPost(QVariantList commentsList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setCommentsInPost()";

    QString originatingPostCleanURL = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanURL;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->commentsURL() << "==" << originatingPostCleanURL;

        if (post->commentsURL() == originatingPostCleanURL)
        {
            qDebug() << "Found originating Post; setting comments on it...";
            post->setComments(commentsList);

            // break;
            /* Don't break, so comments get set in copies of the post too,
               like if JohnDoe posted something and JaneDoe shared it soon
               after, so both the original post and its shared copy are visible
               in the timeline. */
        }
    }
}





void TimeLine::goToFirstPage()
{
    qDebug() << "TimeLine::goToFirstPage()";

    this->timelineOffset = 0;

    this->requestTimelinePage();
}



void TimeLine::goToPreviousPage()
{
    qDebug() << "TimeLine::goToPreviousPage()";

    this->timelineOffset -= this->postsPerPage;
    if (timelineOffset < 0)
    {
        timelineOffset = 0;
    }

    this->requestTimelinePage();
}



void TimeLine::goToNextPage()
{
    qDebug() << "TimeLine::goToNextPage()";

    this->timelineOffset += this->postsPerPage;

    this->requestTimelinePage();
}



/*
 * Decrease internal counter of unread posts (by 1), and inform
 * the parent window, so it can update its tab titles
 *
 */
void TimeLine::decreaseUnreadPostsCount()
{
    --unreadPostsCount;

    emit unreadPostsCountChanged(this->timelineType, this->unreadPostsCount);
}
