/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "audienceselector.h"

AudienceSelector::AudienceSelector(PumpController *pumpController,
                                   QString selectorType,
                                   QWidget *parent) : QFrame(parent)
{
    this->selectorType = selectorType;

    QString titlePart;
    if (this->selectorType == "to")
    {
        titlePart = tr("'To' List");
    }
    else
    {
        titlePart = tr("'CC' List");
    }

    this->setWindowTitle("Dianara - " + titlePart);
    this->setWindowIcon(QIcon::fromTheme("system-users"));
    this->setWindowFlags(Qt::Window);
    this->setWindowModality(Qt::WindowModal);
    this->setMinimumSize(200, 200);

    this->pController = pumpController;
    connect(pController, SIGNAL(contactListReceived(QString,QVariantList)),
            this, SLOT(updateAllContactsList(QString,QVariantList)));


    // Left side, all contacts, with filter

    this->searchLabel = new QLabel(tr("&Search:"));
    this->searchLineEdit = new QLineEdit();
    searchLineEdit->setPlaceholderText(tr("Enter a name here to search for it"));
    searchLabel->setBuddy(searchLineEdit);
    connect(searchLineEdit, SIGNAL(textChanged(QString)),
            this, SLOT(filterList(QString)));

    allContactsListWidget = new QListWidget();
    allContactsListWidget->setDragDropMode(QListView::DragDrop);
    allContactsListWidget->setDefaultDropAction(Qt::CopyAction);
    allContactsListWidget->setSelectionMode(QListView::ExtendedSelection);
    allContactsListWidget->setSortingEnabled(true);
    connect(allContactsListWidget, SIGNAL(activated(QModelIndex)),
            this, SLOT(copyClickedToSelected(QModelIndex)));

    addToSelectedButton = new QPushButton(QIcon::fromTheme("list-add"),
                                          tr("&Add to Selected") + " >>");
    connect(addToSelectedButton, SIGNAL(clicked()),
            this, SLOT(copyToSelected()));

    this->allGroupboxLayout = new QVBoxLayout();
    allGroupboxLayout->addWidget(searchLabel);
    allGroupboxLayout->addWidget(searchLineEdit);
    allGroupboxLayout->addSpacing(2);
    allGroupboxLayout->addWidget(allContactsListWidget);
    allGroupboxLayout->addSpacing(4);
    allGroupboxLayout->addWidget(addToSelectedButton);

    allContactsGroupbox = new QGroupBox(tr("All Contacts"));
    allContactsGroupbox->setLayout(allGroupboxLayout);


    // Right side, selected contacts

    explanationLabel = new QLabel(tr("Select people from the list on the left.\n"
                                     "You can drag them with the mouse, click or "
                                     "double-click on them, or select them and "
                                     "use the button below."));
    explanationLabel->setWordWrap(true);

    selectedListWidget = new QListWidget();
    selectedListWidget->setDragDropMode(QListView::DragDrop);
    selectedListWidget->setDefaultDropAction(Qt::MoveAction);
    selectedListWidget->setSelectionMode(QListView::ExtendedSelection);

    this->clearSelectedListButton = new QPushButton(QIcon::fromTheme("edit-clear-list"),
                                                    tr("Clear &List"));
    connect(clearSelectedListButton, SIGNAL(clicked()),
            selectedListWidget, SLOT(clear()));


    doneButton = new QPushButton(QIcon::fromTheme("dialog-ok"),
                                 tr("&Done"));
    connect(doneButton, SIGNAL(clicked()),
            this, SLOT(setAudience()));
    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel"),
                                   tr("&Cancel"));
    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(close()));
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignRight);
    buttonsLayout->addWidget(doneButton);
    buttonsLayout->addWidget(cancelButton);


    selectedGroupboxLayout = new QVBoxLayout();
    selectedGroupboxLayout->addWidget(explanationLabel);
    selectedGroupboxLayout->addSpacing(12);
    selectedGroupboxLayout->addWidget(selectedListWidget);
    selectedGroupboxLayout->addWidget(clearSelectedListButton);
    selectedGroupboxLayout->addSpacing(8);
    selectedGroupboxLayout->addLayout(buttonsLayout);

    this->selectedListGroupbox = new QGroupBox(tr("Selected People"));
    selectedListGroupbox->setLayout(selectedGroupboxLayout);

    this->mainLayout = new QHBoxLayout();
    mainLayout->addWidget(allContactsGroupbox,  3);
    mainLayout->addWidget(selectedListGroupbox, 4);
    this->setLayout(mainLayout);


    this->filterList(""); // Populate the list without filter, initially

    qDebug() << "AudienceSelector created";
}


AudienceSelector::~AudienceSelector()
{
    qDebug() << "AudienceSelector destroyed";
}



/*
 * Reset lists and widgets to default status
 *
 */
void AudienceSelector::resetLists()
{
    this->selectedListWidget->clear();

    this->searchLineEdit->clear(); // might also trigger filterLists()
    this->allContactsListWidget->scrollToTop();
}




void AudienceSelector::keyPressEvent(QKeyEvent *event)
{
    if (event->key() == Qt::Key_Escape)
    {
        this->close();
    }
    else
    {
        event->accept();
    }
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////// SLOTS //////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


/*
 * Filter the list of all contacts based on what
 * the user entered in the search box
 *
 */
void AudienceSelector::filterList(QString searchTerms)
{
    allContactsListWidget->clear();

    allContactsListWidget->addItems(allContactsStringList.filter(searchTerms,
                                                                 Qt::CaseInsensitive));
}


/*
 * Update the list of all contacts (actually, 'following')
 * from the PumpController
 *
 */
void AudienceSelector::updateAllContactsList(QString listType,
                                             QVariantList contactsVariantList)
{
    if (listType != "following")
    {
        return;
    }
    qDebug() << "AudienceSelector: received list of Following; updating...";

    allContactsStringList.clear();

    QString singleContactString;
    foreach (QVariant contact, contactsVariantList)
    {
        singleContactString = contact.toMap().value("displayName").toString();
        singleContactString.append("  <");
        singleContactString.append(contact.toMap().value("id").toString().remove(0,5));
        singleContactString.append(">");

        allContactsStringList.append(singleContactString);
    }


    // Populate the list widget, using whatever filter is set
    this->filterList(searchLineEdit->text());
}



/*
 * Copy a contact to the list of Selected People
 *
 * Used when selecting a row and clicking the "add" button
 */
void AudienceSelector::copyToSelected()
{
    // if something is selected...
    if (this->allContactsListWidget->currentRow() != -1)
    {
        this->selectedListWidget->addItem(allContactsListWidget->currentItem()->data(0).toString());
    }
}


/*
 * Copy a contact to the list of Selected
 *
 * Used when double-clicking on it
 */
void AudienceSelector::copyClickedToSelected(QModelIndex modelIndex)
{
    this->selectedListWidget->addItem(modelIndex.data().toString());
}


/*
 * The "Done" button: emit signal with list of selected people
 *
 */
void AudienceSelector::setAudience()
{
    QStringList addressList;

    int addressCount = this->selectedListWidget->count();

    for (int counter=0; counter < addressCount; ++counter)
    {
        addressList.append(selectedListWidget->item(counter)->text());
    }


    emit audienceSelected(selectorType, addressList);

    this->close();
}
