/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "contactcard.h"

ContactCard::ContactCard(PumpController *pumpController,
                         QMap<QString, QString> contactData,
                         QWidget *parent)  :  QFrame(parent)
{
    this->pController = pumpController;

    this->setFrameStyle(QFrame::Box | QFrame::Raised);

    this->setSizePolicy(QSizePolicy::MinimumExpanding,
                        QSizePolicy::Maximum);

    mainLayout = new QHBoxLayout();
    avatarLabel = new QLabel();

    this->contactAvatarUrl = contactData.value("avatar");

    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(contactAvatarUrl);

    QFile avatarFile(avatarFilename);
    if (avatarFile.exists(avatarFilename) && !contactAvatarUrl.isEmpty())
    {
        // Load avatar if already cached
        avatarLabel->setPixmap(QPixmap(avatarFilename).scaled(48, 48,
                                                              Qt::KeepAspectRatio,
                                                              Qt::SmoothTransformation));
        qDebug() << "ContactCard: Using cached avatar";
    }
    else
    {
        // Placeholder image
        avatarLabel->setPixmap(QIcon::fromTheme("user-identity").pixmap(48,48));
        qDebug() << "ContactCard: Using placeholder, downloading avatar";

        connect(pController, SIGNAL(avatarStored(QString,QString)),
                this, SLOT(redrawAvatar(QString,QString)));

        // Download avatar for next time (if empty, pumpController will do nothing)
        pController->getAvatar(contactAvatarUrl);
    }


    centerLayout = new QVBoxLayout();

    QFont nameFont;
    nameFont.setBold(true);
    nameFont.setUnderline(true);

    QString contactNameString = contactData.value("name");

    nameLabel = new QLabel();
    nameLabel->setText(contactNameString);
    nameLabel->setFont(nameFont);


    this->contactID = contactData.value("id");
    this->contactURL = contactData.value("url");

    userInfoLabel = new QLabel();
    userInfoLabel->setTextInteractionFlags(Qt::TextBrowserInteraction);
    userInfoLabel->setText(QString("<%1>\n").arg(contactID)
                      + tr("Hometown") + QString(": %2").arg(contactData.value("hometown")));
    userInfoLabel->setAlignment(Qt::AlignLeft | Qt::AlignTop);
    userInfoLabel->setWordWrap(true);


    // Bio as tooltip for the whole contact card
    QString contactBio = contactData.value("bio");
    if (!contactBio.isEmpty())
    {
        // Make it rich text, so that it gets wordwrap
        this->setToolTip("<b><u>"
                         + tr("Bio for %1",
                              "Abbreviation for Biography, "
                              "but you can use the full word; "
                              "%1=contact name")
                           .arg(contactNameString)
                         + "</u></b>"
                         "<br><br>"
                         + contactBio);
    }
    else
    {
        if (contactNameString.isEmpty())
        {
            this->setToolTip(tr("This user doesn't have a biography"));
        }
        else
        {
            this->setToolTip(tr("No biography for %1",
                                "%1=contact name").arg(contactNameString));
        }
    }


    centerLayout->addWidget(nameLabel);
    centerLayout->addWidget(userInfoLabel);



    rightLayout = new QVBoxLayout();

    followButton = new QPushButton("*follow*");
    followButton->setFlat(true);    
    if (contactData.value("following") == "false")
    {
        this->setButtonToFollow();
    }
    else
    {
        this->setButtonToUnfollow();
    }


    openProfileAction = new QAction(QIcon::fromTheme("internet-web-browser"),
                                    tr("Open Profile in Web Browser"),
                                    this);
    connect(openProfileAction, SIGNAL(triggered()),
            this, SLOT(openProfileInBrowser()));

    addToListMenu = new QMenu(tr("In Lists..."));
    addToListMenu->setIcon(QIcon::fromTheme("format-list-unordered"));
    addToListMenu->addAction("fake list 1")->setCheckable(true);
    addToListMenu->addAction("fake list 2")->setCheckable(true);
    addToListMenu->addAction("fake list 3")->setCheckable(true);
    addToListMenu->addAction("fake list 4")->setCheckable(true);
    addToListMenu->addAction("fake list 5")->setCheckable(true);

    optionsMenu = new QMenu("*options*");
    optionsMenu->addAction(openProfileAction);
    optionsMenu->addMenu(addToListMenu);
    addToListMenu->setDisabled(true); // TMP/FIXME: disabled until implemented

    optionsButton = new QPushButton(QIcon::fromTheme("user-properties"),
                                    tr("User Options"));
    optionsButton->setFlat(true);
    optionsButton->setMenu(optionsMenu);


    rightLayout->addWidget(followButton);
    rightLayout->addWidget(optionsButton);

    mainLayout->addWidget(avatarLabel,  1);  // Stretch 1/8
    mainLayout->addLayout(centerLayout, 6);  // Stretch 6/8
    mainLayout->addLayout(rightLayout,  1);  // 1/8


    this->setLayout(mainLayout);

    qDebug() << "ContactCard created" << this->contactID;
}


ContactCard::~ContactCard()
{
    qDebug() << "ContactCard destroyed" << this->contactID;
}



void ContactCard::setButtonToFollow()
{
    followButton->setIcon(QIcon::fromTheme("list-add"));
    followButton->setText(tr("Follow"));
    connect(followButton, SIGNAL(clicked()),
            this, SLOT(followContact()));
    disconnect(followButton, SIGNAL(clicked()),
               this, SLOT(unfollowContact()));
}


void ContactCard::setButtonToUnfollow()
{
    followButton->setIcon(QIcon::fromTheme("list-remove"));
    followButton->setText(tr("Stop Following"));
    connect(followButton, SIGNAL(clicked()),
            this, SLOT(unfollowContact()));
    disconnect(followButton, SIGNAL(clicked()),
               this, SLOT(followContact()));
}



/**************************************************************************/
/******************************** SLOTS ***********************************/
/**************************************************************************/



void ContactCard::followContact()
{
    this->pController->followContact(this->contactID);
    this->setButtonToUnfollow();
}


void ContactCard::unfollowContact()
{
    int confirmation = QMessageBox::question(this, tr("Stop following?"),
                                             tr("Are you sure you want to stop following %1?")
                                             .arg(this->contactID),
                                             tr("&Yes, stop following"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        this->pController->unfollowContact(this->contactID);
        this->setButtonToFollow();
    }
}


void ContactCard::openProfileInBrowser()
{
    QDesktopServices::openUrl(this->contactURL);
}



/*
 * Redraw contact's avatar after it's been downloaded and stored
 *
 */
void ContactCard::redrawAvatar(QString avatarUrl, QString avatarFilename)
{
    if (avatarUrl == this->contactAvatarUrl)
    {
        disconnect(pController, SIGNAL(avatarStored(QString,QString)),
                   this, SLOT(redrawAvatar(QString,QString)));

        avatarLabel->setPixmap(QPixmap(avatarFilename).scaled(48, 48,
                                                              Qt::KeepAspectRatio,
                                                              Qt::SmoothTransformation));
    }
}
