/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "imageviewer.h"

ImageViewer::ImageViewer(QString fileURI,
                         QString title,
                         QWidget *parent) : QLabel(parent)
{
    this->setMinimumSize(250, 100);

    if (title.isEmpty())
    {
        title = "--";
    }

    fileURI.remove(0, 7); // remove "image:/" from filename URI
    this->originalFileURI = fileURI;

    QDesktopWidget desktopWidget;
    QPixmap pixmap(originalFileURI);
    QString resolution = QString("%1x%2").arg(pixmap.width()).arg(pixmap.height());

    this->setWindowTitle("Dianara - " + tr("Image") + ": " + title
                         + " (" + resolution + ", "
                         + MiscHelpers::fileSizeString(originalFileURI) +  ")");
    this->setWindowIcon(QIcon::fromTheme("folder-image"));
    this->setAlignment(Qt::AlignCenter);
    this->setToolTip(tr("ESC to close, secondary-click for options"));


    // Resize pixmap according to desktop (screen) size
    if (pixmap.height() > desktopWidget.height())
    {
        pixmap = pixmap.scaledToHeight(desktopWidget.height() - 100,
                                       Qt::SmoothTransformation);
    }

    if (pixmap.width() > desktopWidget.width())
    {
        pixmap = pixmap.scaledToWidth(desktopWidget.width() - 100,
                                      Qt::SmoothTransformation);
    }

    this->setPixmap(pixmap);
    this->resize(pixmap.size());

    this->createContextMenu();


    qDebug() << "ImageViewer created";
}


ImageViewer::~ImageViewer()
{
    qDebug() << "ImageViewer destroyed";
}


void ImageViewer::createContextMenu()
{
    contextMenu = new QMenu("imaveViewerMenu");


    contextMenu->addAction(QIcon::fromTheme("document-save-as"),
                           tr("Save Image..."),
                           this, SLOT(saveImage()));
    contextMenu->addAction(QIcon::fromTheme("window-close"),
                           tr("Close Viewer"),
                           this,
                           SLOT(close()));
}



void ImageViewer::closeEvent(QCloseEvent *event)
{
    qDebug() << "ImageViewer::closeEvent()";
    event->accept();

    this->deleteLater();
}


void ImageViewer::keyPressEvent(QKeyEvent *event)
{
    if (event->key() == Qt::Key_Escape)
    {
        this->close();
    }
    else
    {
        event->accept();
    }
}


void ImageViewer::contextMenuEvent(QContextMenuEvent *event)
{
    this->contextMenu->exec(event->globalPos());
}


/****************************************************************************/
/************************************ SLOTS *********************************/
/****************************************************************************/


void ImageViewer::saveImage()
{
    bool savedCorrectly;

    QString filename;
    filename = QFileDialog::getSaveFileName(this, tr("Save Image As..."),
                                            QDir::homePath(),
                                            tr("Image files") + "(*.jpg *.png);;"
                                            + tr("All files") + " (*)");

    if (!filename.isEmpty())
    {
        // Save pixmap from original file
        savedCorrectly = QPixmap(this->originalFileURI).save(filename);

        if (!savedCorrectly)
        {
            QMessageBox::warning(this,
                                 tr("Error saving image"),
                                 tr("There was a problem while saving %1.\n\n"
                                    "Filename should end in .jpg "
                                    "or .png extensions.").arg(filename));
        }
    }
}
