/*
 *   This file is part of Dianara
 *   Copyright 2012-2013  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#ifndef PUMPCONTROLLER_H
#define PUMPCONTROLLER_H

#include <QObject>
#include <QStringList>
#include <QByteArray>
#include <QUrl>
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
#include <QUrlQuery>
#endif
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QNetworkAccessManager>
#include <QSslError> // TMP
#include <QTimer>
#include <QFile>

#include <QSettings>
#include <QDesktopServices>

#include <QDebug>


/// For JSON parsing
#include <QVariant>
#include <QVariantMap>
#include <qjson/parser.h>
#include <qjson/serializer.h>

// For OAuth authentication
#include <QtOAuth>


class PumpController : public QObject
{
    Q_OBJECT

public:
    explicit PumpController(QObject *parent = 0);
    ~PumpController();

    void setPostsPerPage(int ppp);
    void setUpdatesToTimelineBlocked(bool blocked);

    void setNewUserID(QString userID);
    void setUserCredentials(QString userID);
    QString currentUserID();
    QString currentUsername();
    QString currentFollowersURL();

    void getUserProfile(QString userID);
    void updateUserProfile(QString avatarUrl, QString fullName,
                           QString hometown, QString bio);

    void getAvatar(QString avatarURL);
    void getImage(QString imageURL);
    void notifyAvatarStored(QString avatarUrl, QString avatarFilename);
    void notifyImageStored(QString imageUrl);

    void getContactList(QString listType);
    bool userInFollowing(QString contactId);
    void updateInternalFollowingIdList(QStringList idList);

    void getListsList();
    void createPersonList(QString name, QString description);
    void deletePersonList(QString id);

    void getMainTimeline(int timelineOffset);
    void getDirectTimeline(int timelineOffset);
    void getActivityTimeline(int timelineOffset);
    void getFavoritesTimeline(int timelineOffset);

    void getPostLikes(QString postLikesURL);
    void getPostComments(QString postCommentsURL);
    void getPostShares(QString postSharesURL);

    void getMinorFeed();

    QNetworkRequest prepareRequest(QString url,
                                   QOAuth::HttpMethod method,
                                   int requestType,
                                   QOAuth::ParamMap paramMap = QOAuth::ParamMap(),
                                   QString contentTypeString="application/json");

    void uploadFile(QString filename, QString contentType,
                    int uploadType = UploadFileRequest);

    QList<QVariantList> processAudience(QMap<QString,QString> audienceMap);

    enum requestTypes
    {
        NoRequest,

        ClientRegistrationRequest,
        TokenRequest,

        UserProfileRequest,
        UpdateProfileRequest,

        FollowingListRequest,
        FollowersListRequest,
        ListsListRequest,
        CreatePersonListRequest,
        DeletePersonListRequest,

        MainTimelineRequest,
        DirectTimelineRequest,
        ActivityTimelineRequest,
        FavoritesTimelineRequest,

        PostLikesRequest,
        PostCommentsRequest,
        PostSharesRequest,

        MinorFeedRequest,

        PublishPostRequest,
        LikePostRequest,
        CommentPostRequest,
        SharePostRequest,
        DeletePostRequest,
        UpdatePostRequest,

        FollowContactRequest,
        UnfollowContactRequest,


        AvatarRequest,
        ImageRequest,

        UploadFileRequest,
        UploadPictureRequest,
        UploadAvatarRequest,
        PublishAvatarRequest
    };




signals:
    void openingAuthorizeURL(QUrl url);
    void authorizationStatusChanged(bool authorized);

    void profileReceived(QString avatarURL, QString fullName,
                         QString hometown, QString bio);
    void contactListReceived(QString listType, QVariantList contactList);
    void listsListReceived(QVariantList listsList);


    void mainTimelineReceived(QVariantList postList, int postsPerPage,
                              QString previousLink, QString nextLink);
    void directTimelineReceived(QVariantList postList, int postsPerPage,
                                QString previousLink, QString nextLink);
    void activityTimelineReceived(QVariantList postList, int postsPerPage,
                                  QString previousLink, QString nextLink);
    void favoritesTimelineReceived(QVariantList postList, int postsPerPage,
                                   QString previousLink, QString nextLink);

    void likesReceived(QVariantList likesList, QString originatingPostURL);
    void commentsReceived(QVariantList commentsList, QString originatingPostURL);

    void minorFeedReceived(QVariantList activitiesList);

    void avatarPictureReceived(QByteArray pictureData, QUrl pictureURL);
    void imageReceived(QByteArray pictureData, QUrl pictureURL);
    void avatarStored(QString avatarUrl, QString avatarFilename);
    void imageStored(QString imageUrl);

    void postPublished();
    void postPublishingFailed();
    void likeSet();
    void commentPosted();
    void commentPostingFailed();

    void avatarUploaded(QString url);

    void showNotification(QString message);
    void currentJobChanged(QString message);



public slots:
    void requestFinished(QNetworkReply *reply);

    void sslErrorsHandler(QNetworkReply *reply, QList<QSslError> errorList);

    void getToken();
    void authorizeApplication(QString verifierCode);

    void getInitialData();

    void postNote(QMap<QString,QString> audienceMap,
                  QString postText,
                  QString postTitle);
    void postImage(QMap<QString,QString> audienceMap,
                   QString postText, QString currentImageTitle,
                   QString imageFilename, QString contentType);
    void postImageStepTwo(QString id);
    void postImageStepThree(QString id);

    void postAvatarStepTwo(QString id);


    void updatePost(QString id, QString content, QString title);

    void likePost(QString postID, QString postType, bool like);
    void addComment(QString comment, QString postID, QString postType);
    void sharePost(QString postID, QString postType);
    void deletePost(QString postID, QString postType);

    void followContact(QString address);
    void unfollowContact(QString address);



private:
    QNetworkAccessManager nam;
    QByteArray userAgentString;

    // QOAuth-related
    QOAuth::Interface *qoauth;

    bool isApplicationAuthorized;

    QString clientID;
    QString clientSecret;

    QByteArray token;
    QByteArray tokenSecret;


    QString userID;  // Full webfinger address, user@host.tld
    QString userName;
    QString serverURL;

    QString userFollowersURL;
    QStringList followingIdList;

    QTimer *initialDataTimer;
    int initialDataStep;
    bool haveProfile;
    bool haveFollowing;
    bool haveFollowers;
    bool havePersonLists;


    int postsPerPage;
    bool updatesToTimelineBlocked;


    // For multi-step operations in posts
    QString currentImageTitle;
    QString currentImageDescription;
    QMap<QString, QString> currentAudienceMap;
};

#endif // PUMPCONTROLLER_H
