/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "comment.h"

Comment::Comment(PumpController *pumpController,
                 ASObject *object,
                 QWidget *parent) : QFrame(parent)
{
    this->pController = pumpController;

    object->setParent(this); // reparent the passed object

    QSizePolicy sizePolicy;
    sizePolicy.setHeightForWidth(false);
    sizePolicy.setWidthForHeight(false);
    sizePolicy.setHorizontalPolicy(QSizePolicy::MinimumExpanding);
    sizePolicy.setVerticalPolicy(QSizePolicy::Maximum);
    this->setSizePolicy(sizePolicy);
    this->setMaximumHeight(4096);


    this->commentId = object->getId();
    this->objectType = object->getType();

    QString commentAuthorId = object->author()->getId();;
    if (commentAuthorId == pController->currentUserId())
    {
        commentIsOwn = true; // Comment is ours!

        // Different frame style depending on whether the comment is ours or not
        this->setFrameStyle(QFrame::Sunken | QFrame::StyledPanel);
    }
    else
    {
        commentIsOwn = false;

        this->setFrameStyle(QFrame::Raised | QFrame::StyledPanel);
    }



    // Avatar pixmap
    avatarButton = new AvatarButton(object->author(),
                                    this->pController,
                                    QSize(32,32));


    QFont commentsFont;
    commentsFont.setPointSize(commentsFont.pointSize() - 1); // 1 point less than default


    // Name, with ID as tooltip
    commentsFont.setBold(true);
    if (commentIsOwn)
    {
        commentsFont.setItalic(true);
    }


    fullNameLabel = new QLabel(object->author()->getName());
    if (object->author()->getName().isEmpty())
    {
        fullNameLabel->setText(commentAuthorId); // Use ID if name is empty
    }
    fullNameLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    fullNameLabel->setFont(commentsFont);
    fullNameLabel->setToolTip(commentAuthorId);


    // Timestamps
    createdAt = object->getCreatedAt();
    updatedAt = object->getUpdatedAt();

    commentsFont.setBold(false);
    commentsFont.setItalic(true);

    timestampLabel = new QLabel();
    timestampLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    timestampLabel->setFont(commentsFont);
    QString timestampTooltip = tr("Posted on %1")
                                .arg(Timestamp::localTimeDate(createdAt));
    if (createdAt != updatedAt)
    {
        timestampTooltip.append("<br>"
                                + tr("Modified on %1")
                                   .arg(Timestamp::localTimeDate(updatedAt)));
    }
    timestampLabel->setToolTip(timestampTooltip); // Precise time on tooltip
    this->setFuzzyTimestamps();



    // Like and Delete "buttons"
    commentsFont.setBold(true);
    commentsFont.setItalic(false);

    likeLabel = new QLabel("*like*");
    likeLabel->setContextMenuPolicy(Qt::NoContextMenu);
    likeLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    likeLabel->setFont(commentsFont);
    likeLabel->setToolTip(tr("Like or unlike this comment"));
    connect(likeLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(likeComment(QString)));

    if (object->isLiked() == "true")
    {
        commentIsLiked = true;
    }
    else
    {
        commentIsLiked = false;
    }
    this->fixLikeLabelText();


    quoteLabel = new QLabel("<a href=\"quote://\">"
                            + tr("Quote", "This is a verb, infinitive")
                            + "</a>");
    quoteLabel->setContextMenuPolicy(Qt::NoContextMenu);
    quoteLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    quoteLabel->setFont(commentsFont);
    quoteLabel->setToolTip(tr("Reply quoting this comment"));
    connect(quoteLabel, SIGNAL(linkHovered(QString)),
            this, SLOT(saveCommentSelectedText()));
    connect(quoteLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(quoteComment()));


    editLabel = new QLabel("<a href=\"edit://\">" + tr("Edit") + "</a>");
    editLabel->setContextMenuPolicy(Qt::NoContextMenu);
    editLabel->setAlignment(Qt::AlignTop | Qt::AlignRight);
    editLabel->setFont(commentsFont);
    editLabel->setToolTip(tr("Modify this comment"));
    connect(editLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(editComment()));


    deleteLabel = new QLabel("<a href=\"delete://\">" + tr("Delete") + "</a>");
    deleteLabel->setContextMenuPolicy(Qt::NoContextMenu);
    deleteLabel->setAlignment(Qt::AlignTop | Qt::AlignRight);
    deleteLabel->setFont(commentsFont);
    deleteLabel->setToolTip(tr("Delete this comment"));
    connect(deleteLabel, SIGNAL(linkActivated(QString)),
            this, SLOT(deleteComment()));


    // The likes count
    likesCountLabel = new QLabel();
    likesCountLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
    commentsFont.setBold(false);
    likesCountLabel->setFont(commentsFont);

    this->setLikesCount(object->getLikesCount(),
                        object->getLastLikesList());

    // The comment itself
    this->commentOriginalText = object->getContent();


    contentLabel = new QLabel();
    contentLabel->setAlignment(Qt::AlignLeft);
    contentLabel->setFont(commentsFont);
    contentLabel->setTextInteractionFlags(Qt::TextBrowserInteraction);
    contentLabel->setWordWrap(true);
    contentLabel->setOpenExternalLinks(true);
    contentLabel->setTextFormat(Qt::RichText);
    contentLabel->setSizePolicy(sizePolicy);
    contentLabel->setMaximumHeight(4096);
    connect(contentLabel, SIGNAL(linkHovered(QString)),
            this, SLOT(showUrlInfo(QString)));



    pendingImagesList = MiscHelpers::htmlWithReplacedImages(commentOriginalText,
                                                            64); // Arbitrary width
    pendingImagesList.removeFirst(); // First one is the HTML with images replaced
    this->getPendingImages();


    // Layout
    leftLayout = new QVBoxLayout();
    leftLayout->setAlignment(Qt::AlignLeft);
    leftLayout->addWidget(avatarButton,    0, Qt::AlignLeft | Qt::AlignTop);
    leftLayout->addWidget(likesCountLabel, 1, Qt::AlignHCenter | Qt::AlignTop);

    rightTopLayout = new QHBoxLayout();
    rightTopLayout->addWidget(fullNameLabel,  0, Qt::AlignLeft);
    rightTopLayout->addWidget(timestampLabel, 0, Qt::AlignLeft);
    rightTopLayout->addWidget(likeLabel,      0, Qt::AlignLeft);
    rightTopLayout->addWidget(quoteLabel,     0, Qt::AlignLeft);
    rightTopLayout->addStretch(1);
    if (commentIsOwn)
    {
        rightTopLayout->addWidget(editLabel,   0, Qt::AlignRight);
        rightTopLayout->addWidget(deleteLabel, 0, Qt::AlignRight);
    }

    rightLayout = new QVBoxLayout();
    rightLayout->addLayout(rightTopLayout, 0);
    rightLayout->addSpacing(4); // 4px vertical space separation
    rightLayout->addWidget(contentLabel,   1);
    rightLayout->addSpacing(2); // and 2px more as margin


    mainLayout = new QHBoxLayout();
    mainLayout->addLayout(leftLayout);
    mainLayout->addLayout(rightLayout);
    this->setLayout(mainLayout);


    setCommentContents();


    qDebug() << "Comment created" << this->commentId;
}


Comment::~Comment()
{
    qDebug() << "Comment destroyed" << this->commentId;
}



void Comment::fixLikeLabelText()
{
    if (commentIsLiked)
    {
        this->likeLabel->setText("<a href=\"unlike://\">" + tr("Unlike") +"</a>");
    }
    else
    {
        this->likeLabel->setText("<a href=\"like://\">" + tr("Like") +"</a>");
    }
}


void Comment::setLikesCount(QString count, QVariantList namesVariantList)
{
    if (count != "0")
    {
        QString likesString;
        foreach (QVariant likesMap, namesVariantList)
        {
            likesString.append(likesMap.toMap().value("displayName").toString() + ", ");
        }
        likesString.remove(-2, 2); // remove last comma+space: ", "
        if (count != "1")
        {
            likesString = tr("%1 like this comment",
                             "Plural: %1=list of people like John, Jane, Smith").arg(likesString);
        }
        else // just 1 like
        {
            likesString = tr("%1 likes this comment",
                             "Singular: %1=name of just 1 person").arg(likesString);
        }


        likesCountLabel->setText(QString::fromUtf8("\342\231\245")  // heart symbol
                                 + QString(" %1").arg(count));
        // set tooltip as HTML, so it gets wordwrapped
        likesCountLabel->setToolTip(likesString + "<b></b>");
    }
    else
    {
        likesCountLabel->clear();
        likesCountLabel->setToolTip("");
    }
}


void Comment::setFuzzyTimestamps()
{
    QString timestamp = Timestamp::fuzzyTime(createdAt);
    if (createdAt != updatedAt) // Comment has been edited
    {
        timestamp.prepend("**"); // FIXME, somehow show edit time
    }

    this->timestampLabel->setText(timestamp);
}


/*
 * Set the contents of the comment, parsing images, etc.
 *
 */
void Comment::setCommentContents()
{
    int imageWidth = this->contentLabel->width() - 20; // Kinda TMP

    QStringList commentImageList = MiscHelpers::htmlWithReplacedImages(commentOriginalText,
                                                                       imageWidth);

    QString commentContents = commentImageList.takeAt(0); // Comment's HTML with images replaced
    this->contentLabel->setText(commentContents);
}



void Comment::getPendingImages()
{
    if (!pendingImagesList.isEmpty())
    {
        foreach (QString imageUrl, pendingImagesList)
        {
            pController->enqueueImageForDownload(imageUrl);
        }

        connect(pController, SIGNAL(imageStored(QString)),
                this, SLOT(redrawImages(QString)));
    }
}



/****************************************************************************/
/******************************** SLOTS *************************************/
/****************************************************************************/



void Comment::likeComment(QString clickedLink)
{
    if (clickedLink == "like://")
    {
        commentIsLiked = true;
    }
    else // unlike://
    {
        commentIsLiked = false;
    }

    this->pController->likePost(this->commentId,
                                this->objectType,
                                this->commentIsLiked);
    this->fixLikeLabelText();
}


/*
 * This will be called when hovering the "Quote" link
 *
 * Store the selected text, so it can be used when actually quoting
 *
 */
void Comment::saveCommentSelectedText()
{
    this->commentSelectedText = contentLabel->selectedText();
    // TMP / FIXME: issues here; the signal is not emitted every time
    qDebug() << "### comment SELECTED TEXT: " << contentLabel->selectedText();
}


/*
 * Take the contents of a comment and put them as a quote block
 * in the comment composer.
 *
 * If some text has been selected, saveCommentSelectedText()
 * will have it stored in a variable, used here.
 *
 */
void Comment::quoteComment()
{
    QString quotedComment;

    if (commentSelectedText.isEmpty())
    {
        // Quote full comment
        quotedComment = MiscHelpers::quotedText(this->fullNameLabel->text(),
                                                this->contentLabel->text());
    }
    else
    {
        // Quote the selection only
        quotedComment = MiscHelpers::quotedText(this->fullNameLabel->text(),
                                                this->commentSelectedText);
    }

    commentSelectedText.clear();

    emit commentQuoteRequested(quotedComment);
}



void Comment::editComment()
{
    emit commentEditRequested(this->commentId,
                              this->contentLabel->text());
}



void Comment::deleteComment()
{
    int confirmation = QMessageBox::question(this, tr("WARNING: Delete comment?"),
                                             tr("Are you sure you want to delete this comment?"),
                                             tr("&Yes, delete it"), tr("&No"), "", 1, 1);

    if (confirmation == 0)
    {
        qDebug() << "Deleting comment" << this->commentId;
        this->pController->deletePost(this->commentId, this->objectType);

        this->setDisabled(true); // disable... maybe hide?
    }
    else
    {
        qDebug() << "Confirmation canceled, not deleting the comment";
    }
}


/*
 * Show the URL of a link hovered in a comment
 *
 */
void Comment::showUrlInfo(QString url)
{
    if (!url.isEmpty())
    {
        this->pController->showTransientMessage(url);

        qDebug() << "Link hovered in comment:" << url;
    }
    else
    {
        this->pController->showTransientMessage("");
    }
}




/*
 * Redraw comment contents after receiving downloaded images
 *
 */
void Comment::redrawImages(QString imageUrl)
{
    if (pendingImagesList.contains(imageUrl))
    {
        this->pendingImagesList.removeAll(imageUrl);
        if (pendingImagesList.isEmpty()) // If there are no more, disconnect
        {
            disconnect(pController, SIGNAL(imageStored(QString)),
                       this, SLOT(redrawImages(QString)));

            setCommentContents();
        }
    }
}




/****************************************************************************/
/****************************** PROTECTED ***********************************/
/****************************************************************************/


/*
 * Ensure url info in statusbar is hidden when the mouse leaves the comment
 *
 */
void Comment::leaveEvent(QEvent *event)
{
    this->pController->showTransientMessage("");

    event->accept();
}


void Comment::resizeEvent(QResizeEvent *event)
{
    qDebug() << "Comment::resizeEvent()"
             << event->oldSize() << ">" << event->size();

    int height = contentLabel->heightForWidth(contentLabel->width());
    this->contentLabel->setMinimumHeight(height);
    this->contentLabel->setMaximumHeight(height);

    event->accept();
}

