/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "imageviewer.h"

ImageViewer::ImageViewer(QString fileURI,
                         QString title,
                         QWidget *parent) : QWidget(parent)
{
    this->setMinimumSize(420, 320);

    QString tooltipTitle;
    if (title.isEmpty())
    {
        title = "--";
    }
    else
    {
        tooltipTitle = "<b><u>" + title + "</u></b><br><br>";
    }

    fileURI.remove(0, 7); // remove "image:/" from filename URI
    this->originalFileURI = fileURI;

    QDesktopWidget desktopWidget;
    originalPixmap = QPixmap(originalFileURI);
    QString resolution = QString("%1x%2")
                         .arg(originalPixmap.width())
                         .arg(originalPixmap.height());

    QString imageDetails = resolution + ", "
                           + MiscHelpers::fileSizeString(originalFileURI);

    this->setWindowTitle(tr("Image") + ": " + title + " - Dianara");
    this->setWindowIcon(QIcon::fromTheme("folder-image"));

    imageLabel = new QLabel();
    imageLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    imageLabel->setAlignment(Qt::AlignCenter);
    imageLabel->setToolTip(tooltipTitle
                           + tr("Resolution and file size") + ":<br>"
                           + imageDetails
                           + "<br><br>"
                           + tr("ESC to close, secondary-click for options"));


    saveImageButton = new QPushButton(QIcon::fromTheme("document-save-as"),
                                      tr("&Save As..."));
    connect(saveImageButton, SIGNAL(clicked()),
            this, SLOT(saveImage()));

    infoLabel = new QLabel(imageDetails);
    infoLabel->setAlignment(Qt::AlignCenter);

    closeButton = new QPushButton(QIcon::fromTheme("window-close"),
                                  tr("&Close"));
    connect(closeButton, SIGNAL(clicked()),
            this, SLOT(close()));


    this->createContextMenu();


    // Layout
    buttonsLayout = new QHBoxLayout();
    buttonsLayout->addWidget(saveImageButton);
    buttonsLayout->addWidget(infoLabel, 1);
    buttonsLayout->addWidget(closeButton);

    mainLayout = new QVBoxLayout();
    mainLayout->addWidget(imageLabel);
    mainLayout->addLayout(buttonsLayout);
    this->setLayout(mainLayout);


    // Set initial window size according to image size and desktop (screen) size
    int windowHeight = originalPixmap.height() + 100;
    if (windowHeight > desktopWidget.height())
    {
        windowHeight = desktopWidget.height();
    }

    int windowWidth = originalPixmap.width() + 100;
    if (windowWidth > desktopWidget.width())
    {
        windowWidth = desktopWidget.width();
    }

    this->resize(windowWidth, windowHeight);

    qDebug() << "ImageViewer created";
}


ImageViewer::~ImageViewer()
{
    qDebug() << "ImageViewer destroyed";
}



void ImageViewer::createContextMenu()
{
    contextMenu = new QMenu("imageViewerMenu");


    contextMenu->addAction(QIcon::fromTheme("document-save-as"),
                           tr("Save Image..."),
                           this, SLOT(saveImage()));
    contextMenu->addAction(QIcon::fromTheme("window-close"),
                           tr("Close Viewer"),
                           this,
                           SLOT(close()));
}




/****************************************************************************/
/************************************ SLOTS *********************************/
/****************************************************************************/


void ImageViewer::saveImage()
{
    bool savedCorrectly;

    QString filename;
    filename = QFileDialog::getSaveFileName(this, tr("Save Image As..."),
                                            QDir::homePath(),
                                            tr("Image files") + "(*.jpg *.png);;"
                                            + tr("All files") + " (*)");

    if (!filename.isEmpty())
    {
        // Save pixmap from original file
        savedCorrectly = QPixmap(this->originalFileURI).save(filename);

        if (!savedCorrectly)
        {
            QMessageBox::warning(this,
                                 tr("Error saving image"),
                                 tr("There was a problem while saving %1.\n\n"
                                    "Filename should end in .jpg "
                                    "or .png extensions.").arg(filename));
        }
    }
}



/****************************************************************************/
/********************************** PROTECTED *******************************/
/****************************************************************************/


void ImageViewer::closeEvent(QCloseEvent *event)
{
    qDebug() << "ImageViewer::closeEvent()";
    event->accept();

    this->deleteLater();
}

void ImageViewer::resizeEvent(QResizeEvent *event)
{
    qDebug() << "ImageViewer::resizeEvent()";

    int width = imageLabel->width() - 16;   // 16px as margin
    if (width > originalPixmap.width())
    {
        width = originalPixmap.width();
    }

    int height = imageLabel->height() - 16;
    if (height > originalPixmap.height())
    {
        height = originalPixmap.height();
    }

    imageLabel->setPixmap(originalPixmap.scaled(width, height,
                                                Qt::KeepAspectRatio,
                                                Qt::SmoothTransformation));

    event->accept();
}


void ImageViewer::keyPressEvent(QKeyEvent *event)
{
    if (event->key() == Qt::Key_Escape)
    {
        this->close();
    }
    else
    {
        event->accept();
    }
}


void ImageViewer::contextMenuEvent(QContextMenuEvent *event)
{
    this->contextMenu->exec(event->globalPos());
}
