/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "minorfeeditem.h"


MinorFeedItem::MinorFeedItem(ASActivity *activity,
                             QStringList highlightColors,
                             bool highlightedByFilter,
                             PumpController *pumpController,
                             QWidget *parent) : QFrame(parent)
{
    this->pController = pumpController;

    this->itemIsNew = false;

    // This sizePolicy prevents cut messages, and the huge space at the end
    // of the feed, after clicking "More" several times
    this->setSizePolicy(QSizePolicy::Minimum,
                        QSizePolicy::MinimumExpanding);

    activity->setParent(this); // reparent the passed activity

    QFont mainFont;
    mainFont.setPointSize(mainFont.pointSize() - 2);

    QString authorId = activity->author()->getId();

    avatarButton = new AvatarButton(activity->author(),
                                    this->pController,
                                    QSize(32,32));

    QString generator = activity->getGenerator();
    if (!generator.isEmpty())
    {
        generator = "\n"
                  + tr("using %1",
                       "Application used to generate this activity").arg(generator);
    }

    createdAtTimestamp = activity->getCreatedAt();
    timestampLabel = new QLabel();
    mainFont.setBold(true);
    timestampLabel->setFont(mainFont);
    timestampLabel->setWordWrap(true);
    timestampLabel->setToolTip(Timestamp::localTimeDate(createdAtTimestamp) + generator);
    timestampLabel->setAlignment(Qt::AlignCenter);
    timestampLabel->setAutoFillBackground(true);
    timestampLabel->setForegroundRole(QPalette::Text);
    timestampLabel->setBackgroundRole(QPalette::Base);
    timestampLabel->setFrameStyle(QFrame::Panel | QFrame::Raised);
    this->setFuzzyTimeStamp();


    /////////////////////////// Tooltip contents
    QString activityTooltip;

    // Object's author name and ID
    QString activityObjectAuthorName = activity->object()->author()->getName();
    QString activityObjectAuthorId = activity->object()->author()->getId();
    if (activityObjectAuthorName.isEmpty())
    {
        activityObjectAuthorName = activityObjectAuthorId;
    }

    // If there's a name and/or an ID, show them
    if (!activityObjectAuthorName.isEmpty())
    {
        activityTooltip = "<b>" + activityObjectAuthorName + "</b><br>"
                          "<i>" + activityObjectAuthorId + "</i>"
                          "<hr><br>"; // horizontal rule as separator
    }


    // Title
    QString activityObjectTitle = activity->object()->getTitle();
    if (activityObjectTitle.isEmpty())
    {
        activityTooltip.append("<b></b>");
    }
    else
    {
        activityTooltip.append("<b><u>" + activityObjectTitle + "</u></b><br>");
    }

    // Content
    QString activityObjectContent = activity->object()->getContent();
    if (!activityObjectContent.isEmpty())
    {
        activityTooltip.append(activityObjectContent + "<br>");
    }

    // Object ID, interesting to show if it's a person
    if (activity->object()->getType() == "person")
    {
        activityTooltip.append("<br><i>"
                               + activity->object()->getId()
                               + "</i><br><br>");
    }


    // Target info
    QString activityTargetName = activity->target()->getTitle();
    QString activityTargetUrl = activity->target()->getUrl();
    if (activityTargetUrl.isEmpty())
    {
        activityTargetUrl = activity->target()->getId();
    }
    if (!activityTargetUrl.isEmpty())
    {
        activityTooltip.append("<hr>&gt;&gt; "); // Horizontal rule, and >>

        if (!activityTargetName.isEmpty())
        {
            activityTooltip.append(QString("<b>%1</b>"
                                           "<br><i>%2</i>")
                                   .arg(activityTargetName)
                                   .arg(activityTargetUrl));
        }
        else
        {
            activityTooltip.append("<i>" + activityTargetUrl + "</i>");
        }
    }


    // Remove last <br>, if needed
    if (activityTooltip.endsWith("<br>"))
    {
        activityTooltip.remove(-4, 4);
        // Check again, for double <br>'s
        if (activityTooltip.endsWith("<br>"))
        {
            activityTooltip.remove(-4, 4);
        }
    }

    // If the tooltip is only this, remove it / FIXME
    if (activityTooltip == "<b></b>")
    {
        activityTooltip.clear();
    }


    activityDescriptionLabel = new QLabel(activity->getContent());
    mainFont.setBold(false);
    activityDescriptionLabel->setFont(mainFont);
    activityDescriptionLabel->setWordWrap(true);
    activityDescriptionLabel->setOpenExternalLinks(true);
    activityDescriptionLabel->setToolTip(activityTooltip.trimmed());
    connect(activityDescriptionLabel, SIGNAL(linkHovered(QString)),
            this, SLOT(showUrlInfo(QString)));


    openButton = new QPushButton("+");
    openButton->setSizePolicy(QSizePolicy::Ignored,
                              QSizePolicy::Maximum);
    openButton->setToolTip(tr("Open referenced post"));

    originalObjectMap = activity->object()->getOriginalObject();

    inReplyToMap = activity->object()->getInReplyTo();
    QString inReplyToAuthor = inReplyToMap.value("author").toMap().value("id").toString();
    if (inReplyToAuthor.startsWith("acct:"))
    {
        inReplyToAuthor.remove(0,5);
    }



    // Highlight this item if it's about the user, with different colors
    itemHighlightType = -1; // False
    QString highlightItemColor;

    // We are in the recipient list of the activity
    if (activity->getRecipientsIdList().contains("acct:" + pController->currentUserId()))
    {
        itemHighlightType = 0;

        if (QColor::isValidColor(highlightColors.at(itemHighlightType)))
        {
            highlightItemColor = highlightColors.at(itemHighlightType);
        }
    }

    // Activity is in reply to something made by us
    if (inReplyToAuthor == pController->currentUserId())
    {
        itemHighlightType = 1;

        if (QColor::isValidColor(highlightColors.at(itemHighlightType)))
        {
            highlightItemColor = highlightColors.at(itemHighlightType);
        }
    }

    // We are the object; someone added us, etc.
    if (activity->object()->getId() == pController->currentUserId())
    {
        itemHighlightType = 2;

        if (QColor::isValidColor(highlightColors.at(itemHighlightType)))
        {
            highlightItemColor = highlightColors.at(itemHighlightType);
        }
    }

    // The object is ours; someone favorited our note, or something
    if (activity->object()->author()->getId() == pController->currentUserId())
    {
        itemHighlightType = 3;

        if (QColor::isValidColor(highlightColors.at(itemHighlightType)))
        {
            highlightItemColor = highlightColors.at(itemHighlightType);
        }
    }

    //// Special case: highlighting the item because there's a filter rule for it
    if (highlightedByFilter && itemHighlightType == -1) // Only if there's no other reason for HL
    {
        itemHighlightType = 4;

        if (QColor::isValidColor(highlightColors.at(itemHighlightType)))
        {
            highlightItemColor = highlightColors.at(itemHighlightType);
        }
    }


    if (itemHighlightType != -1)
    {
        // Unless the user ID is also empty!
        if (!pController->currentUserId().isEmpty())
        {
            if (QColor::isValidColor(highlightItemColor)) // Valid color
            {
                // CSS for horizontal gradient from configured color to transparent
                QString css = QString("MinorFeedItem "
                                      "{ background-color: "
                                      "qlineargradient(spread:pad, "
                                      "x1:0, y1:0, x2:1, y2:0, "
                                      "stop:0 %1, stop:1 rgba(0, 0, 0, 0)); "
                                      "}")
                              .arg(highlightItemColor);

                this->setStyleSheet(css);
            }
            else // If there's no valid color, highlight with a border
            {
                this->setFrameStyle(QFrame::Panel);
            }
        }
    }




    ////// Layout
    leftLayout = new QVBoxLayout();
    leftLayout->setAlignment(Qt::AlignTop);
    leftLayout->addWidget(avatarButton, 1, Qt::AlignTop | Qt::AlignLeft);

    // Original post available (inReplyTo) or object available (note, image...)
    if (!inReplyToMap.isEmpty()
      || ( (activity->object()->getType() == "note"
            || activity->object()->getType() == "image"
            || activity->object()->getType() == "audio"
            || activity->object()->getType() == "video"
            || activity->object()->getType() == "file")
          && activity->object()->getDeletedTime().isEmpty() ) )
    {
        connect(openButton, SIGNAL(clicked()),
                this, SLOT(openOriginalPost()));
        leftLayout->addWidget(openButton, 0, Qt::AlignHCenter);
    }


    rightLowerLayout = new QHBoxLayout();
    rightLowerLayout->setContentsMargins(0, 0, 0, 0);
    rightLowerLayout->addWidget(activityDescriptionLabel, 1);
    // This may also contain an AvatarButton for an "object person"

    rightLayout = new QVBoxLayout();
    rightLayout->setAlignment(Qt::AlignTop);
    rightLayout->addWidget(timestampLabel);
    rightLayout->addLayout(rightLowerLayout);

    // If the object is a person, such as someone following someone else, add an AvatarButton for them
    if (activity->object()->getType() == "person")
    {
        ASPerson *personObject = activity->personObject();
        if (personObject->getId() != authorId) // avoid cases like "JohnDoe updated JohnDoe"
        {
            AvatarButton *targetAvatar = new AvatarButton(personObject,
                                                          pController,
                                                          QSize(28,28));
            rightLowerLayout->addWidget(targetAvatar, 0, Qt::AlignBottom);
        }
    }


    mainLayout = new QHBoxLayout();
    if (authorId == pController->currentUserId())
    {
        mainLayout->addLayout(rightLayout, 10);
        mainLayout->addLayout(leftLayout,  1);
    }
    else // Normal item, not ours
    {
        mainLayout->addLayout(leftLayout,  1);
        mainLayout->addLayout(rightLayout, 10);
    }
    this->setLayout(mainLayout);

    qDebug() << "MinorFeedItem created";
}



MinorFeedItem::~MinorFeedItem()
{
    qDebug() << "MinorFeedItem destroyed";
}


/*
 * Pseudo-highlight for new items
 *
 */
void MinorFeedItem::setItemAsNew(bool isNew)
{
    itemIsNew = isNew;

    if (itemIsNew)
    {
        this->setAutoFillBackground(true);
        this->setBackgroundRole(QPalette::Mid);

        this->timestampLabel->setForegroundRole(QPalette::HighlightedText);
        this->timestampLabel->setBackgroundRole(QPalette::Highlight);
    }
    else
    {
        this->setAutoFillBackground(false);
        this->setBackgroundRole(QPalette::Window);

        this->timestampLabel->setForegroundRole(QPalette::Text);
        this->timestampLabel->setBackgroundRole(QPalette::Base);

        emit itemRead();
    }

    // Avoid possible flickering of the "new" effect later
    this->timestampLabel->update();
}


/*
 * Set/Update the fuzzy timestamp
 *
 * This will be called from time to time
 *
 */
void MinorFeedItem::setFuzzyTimeStamp()
{
    this->timestampLabel->setText(Timestamp::fuzzyTime(createdAtTimestamp));
}


int MinorFeedItem::getItemHighlightType()
{
    return this->itemHighlightType;
}




/****************************************************************************/
/******************************** SLOTS *************************************/
/****************************************************************************/


void MinorFeedItem::openOriginalPost()
{
    // Create a fake activity for the object
    QVariantMap originalPostMap;
    if (!inReplyToMap.isEmpty())
    {
        originalPostMap.insert("object", this->inReplyToMap);
        originalPostMap.insert("actor", this->inReplyToMap.value("author").toMap());
    }
    else
    {
        originalPostMap.insert("object", this->originalObjectMap);
        originalPostMap.insert("actor", this->originalObjectMap.value("author").toMap());
    }

    ASActivity *originalPostActivity = new ASActivity(originalPostMap, this);

    QStringList colorsList;
    colorsList << "" << "" << "" << "" << ""; // Empty color list, needs 5 colors to avoid crashes

    Post *referencedPost = new Post(originalPostActivity,
                                    colorsList,
                                    false, // Not highlighted
                                    true,  // Post is standalone
                                    pController,
                                    this->parentWidget()); // Pass parent widget (MinorFeed) instead
                                                           // of 'this', so it won't be killed by reloads
    referencedPost->setWindowFlags(Qt::Window);
    referencedPost->setWindowModality(Qt::WindowModal);
    referencedPost->show();
    connect(pController, SIGNAL(commentsReceived(QVariantList,QString)),
            referencedPost, SLOT(setAllComments(QVariantList,QString)));
    referencedPost->getAllComments();
}



void MinorFeedItem::showUrlInfo(QString url)
{
    if (!url.isEmpty())
    {
        this->pController->showTransientMessage(url);

        qDebug() << "Link hovered in Minor Feed:" << url;
    }
    else
    {
        this->pController->showTransientMessage("");
    }
}


/****************************************************************************/
/******************************* PROTECTED **********************************/
/****************************************************************************/


/*
 * On mouse click in any part of the item, set it as read
 *
 */
void MinorFeedItem::mousePressEvent(QMouseEvent *event)
{
    if (itemIsNew)
    {
        this->setItemAsNew(false);
    }

    event->accept();
}


/*
 * Ensure URL info in statusbar is hidden when the mouse leaves the item
 *
 */
void MinorFeedItem::leaveEvent(QEvent *event)
{
    this->pController->showTransientMessage("");

    event->accept();
}
