/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#ifndef PUMPCONTROLLER_H
#define PUMPCONTROLLER_H

#include <QObject>
#include <QStringList>
#include <QByteArray>
#include <QUrl>
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
#include <QUrlQuery>
#endif
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QNetworkAccessManager>
#include <QSslError> // TMP
#include <QTimer>
#include <QFile>

#include <QSettings>
#include <QDesktopServices>

#include <QDebug>


/// For JSON parsing
#include <QVariant>
#include <QVariantMap>
#include <qjson/parser.h>
#include <qjson/serializer.h>

// For OAuth authentication
#include <QtOAuth>

#include "mischelpers.h"
#include "asperson.h"


class PumpController : public QObject
{
    Q_OBJECT

public:
    explicit PumpController(QObject *parent = 0);
    ~PumpController();

    void setPostsPerPageMain(int ppp);
    void setPostsPerPageOther(int ppp);

    void setUpdatesToTimelineBlocked(bool blocked);


    void setNewUserId(QString userId);
    void setUserCredentials(QString userId);
    QString currentUserId();
    QString currentUsername();
    QString currentFollowersUrl();
    int currentFollowersCount();
    int currentFollowingCount();

    void getUserProfile(QString userId);
    void updateUserProfile(QString avatarUrl, QString fullName,
                           QString hometown, QString bio);

    void enqueueAvatarForDownload(QString url);
    void enqueueImageForDownload(QString url);
    void getAvatar(QString avatarUrl);
    void getImage(QString imageUrl);
    void notifyAvatarStored(QString avatarUrl, QString avatarFilename);
    void notifyImageStored(QString imageUrl);

    void getContactList(QString listType, int offset=0);
    bool userInFollowing(QString contactId);
    void updateInternalFollowingIdList(QStringList idList);
    void removeFromInternalFollowingList(QString id);

    void getListsList();
    void createPersonList(QString name, QString description);
    void deletePersonList(QString id);
    void getPersonList(QString url);
    void addPersonToList(QString listId, QString personId);
    void removePersonFromList(QString listId, QString personId);

    void getMainTimeline(int timelineOffset);
    void getDirectTimeline(int timelineOffset);
    void getActivityTimeline(int timelineOffset);
    void getFavoritesTimeline(int timelineOffset);

    void getTimeline(int timelineType, QString url);

    void getPostLikes(QString postLikesURL);
    void getPostComments(QString postCommentsURL);
    void getPostShares(QString postSharesURL);

    void getMinorFeed(int offset = 0);

    QNetworkRequest prepareRequest(QString url,
                                   QOAuth::HttpMethod method,
                                   int requestType,
                                   QOAuth::ParamMap paramMap = QOAuth::ParamMap(),
                                   QString contentTypeString="application/json");

    void uploadFile(QString filename, QString contentType,
                    int uploadType = UploadFileRequest);

    QList<QVariantList> processAudience(QMap<QString,QString> audienceMap);

    void showTransientMessage(QString message);

    enum requestTypes
    {
        NoRequest,

        ClientRegistrationRequest,
        TokenRequest,

        UserProfileRequest,
        UpdateProfileRequest,

        FollowingListRequest,
        FollowersListRequest,
        ListsListRequest,
        CreatePersonListRequest,
        DeletePersonListRequest,
        PersonListRequest,
        AddMemberToListRequest,
        RemoveMemberFromListRequest,

        MainTimelineRequest,
        DirectTimelineRequest,
        ActivityTimelineRequest,
        FavoritesTimelineRequest,

        PostLikesRequest,
        PostCommentsRequest,
        PostSharesRequest,

        MinorFeedRequest,

        PublishPostRequest,
        LikePostRequest,
        CommentPostRequest,
        SharePostRequest,
        UnsharePostRequest,
        DeletePostRequest,

        UpdatePostRequest,
        UpdateCommentRequest,

        FollowContactRequest,
        UnfollowContactRequest,


        AvatarRequest,
        ImageRequest,

        UploadFileRequest,
        UploadMediaForPostRequest,
        UploadAvatarRequest,
        PublishAvatarRequest
    };




signals:
    void openingAuthorizeURL(QUrl url);
    void authorizationStatusChanged(bool authorized);

    void profileReceived(QString avatarURL, QString fullName,
                         QString hometown, QString bio,
                         QString email);
    void contactListReceived(QString listType, QVariantList contactList,
                             int totalReceivedCount);
    void contactFollowed(ASPerson *contact);
    void contactUnfollowed(ASPerson *contact);

    void listsListReceived(QVariantList listsList);
    void personListReceived(QVariantList personList, QString listUrl);
    void personAddedToList(QString id, QString name, QString avatarUrl);
    void personRemovedFromList(QString id);


    void mainTimelineReceived(QVariantList postList, int postsPerPage,
                              QString previousLink, QString nextLink);
    void directTimelineReceived(QVariantList postList, int postsPerPage,
                                QString previousLink, QString nextLink);
    void activityTimelineReceived(QVariantList postList, int postsPerPage,
                                  QString previousLink, QString nextLink);
    void favoritesTimelineReceived(QVariantList postList, int postsPerPage,
                                   QString previousLink, QString nextLink);

    void likesReceived(QVariantList likesList, QString originatingPostURL);
    void commentsReceived(QVariantList commentsList, QString originatingPostURL);

    void minorFeedReceived(QVariantList activitiesList);

    void avatarPictureReceived(QByteArray pictureData, QUrl pictureURL);
    void imageReceived(QByteArray pictureData, QUrl pictureURL);
    void avatarStored(QString avatarUrl, QString avatarFilename);
    void imageStored(QString imageUrl);

    void postPublished();
    void postPublishingFailed();
    void likeSet();
    void commentPosted();
    void commentPostingFailed();

    void avatarUploaded(QString url);

    void showNotification(QString message);
    void currentJobChanged(QString message);
    void transientStatusBarMessage(QString message);
    void logMessage(QString message, QString url="");


public slots:
    void requestFinished(QNetworkReply *reply);

    void sslErrorsHandler(QNetworkReply *reply, QList<QSslError> errorList);

    void getToken();
    void authorizeApplication(QString verifierCode);

    void getInitialData();

    void postNote(QMap<QString,QString> audienceMap,
                  QString postText,
                  QString postTitle);
    void postMedia(QMap<QString,QString> audienceMap,
                   QString postText, QString postTitle,
                   QString mediaFilename, QString mediaType,
                   QString mimeContentType);
    void postMediaStepTwo(QString id);
    void postMediaStepThree(QString id);

    void postAvatarStepTwo(QString id);


    void updatePost(QString id, QString content, QString title);

    void likePost(QString postID, QString postType, bool like);

    void addComment(QString comment, QString postID, QString postType);
    void updateComment(QString id, QString content);

    void sharePost(QString postID, QString postType);
    void unsharePost(QString postId, QString postType);
    void deletePost(QString postID, QString postType);

    void followContact(QString address);
    void unfollowContact(QString address);

    void tmpfixmeNotifyQNAMdestroyed(); // FIXME TMP KILL IT!

private:
    QNetworkAccessManager nam;
    QByteArray userAgentString;

    // QOAuth-related
    QOAuth::Interface *qoauth;

    bool isApplicationAuthorized;

    QString clientID;
    QString clientSecret;

    QByteArray token;
    QByteArray tokenSecret;


    QString userId;  // Full webfinger address, user@host.tld
    QString userName;
    QString serverURL;

    QString userFollowersURL;
    int userFollowersCount;
    int userFollowingCount;
    QStringList followingIdList;
    int totalReceivedFollowers;
    int totalReceivedFollowing;

    QTimer *initialDataTimer;
    int initialDataStep;
    bool haveProfile;
    bool haveFollowing;
    bool haveFollowers;
    bool havePersonLists;


    int postsPerPageMain;
    int postsPerPageOther;
    bool updatesToTimelineBlocked;


    // For multi-step operations in posts
    QString currentPostTitle;
    QString currentPostDescription;
    QMap<QString, QString> currentAudienceMap;
    QString currentPostType;

    // Avatars / Images queue
    QStringList pendingAvatarsList;
    QStringList pendingImagesList;
};

#endif // PUMPCONTROLLER_H
