/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "mischelpers.h"

MiscHelpers::MiscHelpers(QObject *parent) : QObject(parent)
{
    // Creating object not required, all static functions
}




QString MiscHelpers::getCachedAvatarFilename(QString url)
{
    QString localFilename;

    if (!url.isEmpty())
    {
#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
        localFilename = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
#else
        localFilename = QStandardPaths::standardLocations(QStandardPaths::DataLocation).first();
#endif
        localFilename.append("/avatars/");
        localFilename.append(url.trimmed().toUtf8().toBase64());

        QString fileExtension = url;
        fileExtension.remove(QRegExp(".*\\.")); // remove all but the extension

        localFilename.append(".");
        localFilename.append(fileExtension);
    }

    return localFilename;
}







QString MiscHelpers::getCachedImageFilename(QString url)
{
    QString localFilename;

    if (!url.isEmpty())
    {
        url = url.trimmed();
        if (url.startsWith("http://"))
        {
            url.remove(0, 7); // remove http://
        }
        if (url.startsWith("https://"))
        {
            url.remove(0, 8); // remove https://
        }

        QByteArray base64url = url.toUtf8().toBase64();
        base64url.truncate(255); // Limit filename length! Just in case the URL is VERY long


#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
        localFilename = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
#else
        localFilename = QStandardPaths::standardLocations(QStandardPaths::DataLocation).first();
#endif
        localFilename.append("/images/");
        localFilename.append(base64url);
    }

    return localFilename;
}



/*
 * Return MIME content type, like image/png, audio/ogg, etc.
 *
 * using libmagic
 *
 */
QString MiscHelpers::getFileMimeType(QString fileUri)
{
    qDebug() << "getFileMimeType() file:" << fileUri;

    magic_t magicCookie = magic_open(MAGIC_MIME_TYPE);
    if (magicCookie == NULL)
    {
        qDebug() << "libmagic init error!";
        return QString();
    }

    if (magic_load(magicCookie, NULL) != 0)
    {
        qDebug() << "magic_load() error! No system-wide libmagic DB?";
        // Try loading from a subdirectory instead (in mswindows, maybe osx)
        if (magic_load(magicCookie, "plugins/magic") != 0)
        {
            qDebug() << "magic_load() error; Can't load plugins/magic.mgc";
            magic_close(magicCookie);
            return QString();
        }
    }

    const char *magicMimeString;
    magicMimeString = magic_file(magicCookie, fileUri.toLocal8Bit());

    QString mimeType = QString(magicMimeString);
    qDebug() << "File MIME type:" << mimeType;

    magic_close(magicCookie);

    return mimeType;
}



/*
 * Return width of an image
 *
 */
int MiscHelpers::getImageWidth(QString fileURI)
{
    QImageReader imageReader(fileURI);

    return imageReader.size().width();
}



QString MiscHelpers::fixLongName(QString name)
{
    // very TMP optimization of LOOONG names / FIXME
    if (name.length() > 16)
    {
        name.replace("@", "@ ");
        name.replace(".", ". ");
    }

    return name;
}



/*
 * Return a pretty string with the size of a file, like
 * "33 KiB", "512 bytes" or "3 MiB"
 *
 */
QString MiscHelpers::fileSizeString(QString fileURI)
{
    QFileInfo fileInfo(fileURI);
    int fileSize = fileInfo.size();

    QString sizeUnit = tr("bytes");
    if (fileSize > 1024) // if > 1024 bytes, transform to KiB
    {
        fileSize /= 1024;
        sizeUnit = "KiB";
    }
    if (fileSize > 1024) // if > 1024 KiB, transform to MiB
    {
        fileSize /= 1024;
        sizeUnit = "MiB";
    }


    return QString("%1 %2").arg(fileSize).arg(sizeUnit);
}




/*
 * Parse a string of HTML and replace the URL in each <img src=""> tag with
 * the corresponding locally cached filename.
 *
 * Return also the string list of the URL's to download
 *
 */
QStringList MiscHelpers::htmlWithReplacedImages(QString originalHtml, int postWidth)
{
    // if no <img tag is found, just return the original HTML
    if (!originalHtml.contains("<img", Qt::CaseInsensitive))
    {
        //qDebug() << "MiscHelpers::htmlWithReplacedImages(); HTML does NOT contain <img> tags...";
        return QStringList(originalHtml);
    }

    //qDebug() << "MiscHelpers::htmlWithReplacedImages(); HTML contains some <img> tags...";
    QString newHtml = originalHtml;

    QStringList imageList;
    QString imgSrc;

    QRegExp regExp("\\<img(.+)src=(\"|\\')([^\"\\']+)(\"|\\')(.*)\\>");
    regExp.setMinimal(true);

    int matchedLength = 0;
    int stringPos = 0;
    while (matchedLength != -1)
    {
        stringPos = regExp.indexIn(newHtml, stringPos);
        matchedLength = regExp.matchedLength();


        //qDebug() << "#######\n\nregExp match = " << regExp.cap(0);
        //qDebug() << "Groups:" << regExp.cap(1) << " // " << regExp.cap(2)
        //         << " // " << regExp.cap(3) << " // " << regExp.cap(4)
        //         << " // " << regExp.cap(5);
        //qDebug() << "Matched length is:" << matchedLength;

        imgSrc = regExp.cap(3);
        if (!imgSrc.isEmpty()) // if not an empty string, add to the list, and replace HTML
        {
            imageList.append(imgSrc);

            QString cachedImageFilename = getCachedImageFilename(imgSrc);

            int imageWidth = getImageWidth(cachedImageFilename);
            // if width is bigger than the post, make it smaller to fit
            if (imageWidth > postWidth - 32)
            {
                // Some margins, to account for a scrollbar or a tab space before the image
                imageWidth = postWidth - 32;
            }

            newHtml.replace(stringPos,
                            matchedLength,
                            "<img src=\"" + cachedImageFilename +  "\" "
                            "width=\"" + QString("%1").arg(imageWidth) + "\" />");
        }

        stringPos += matchedLength; // FIXME: error control
    }

    imageList.insert(0, newHtml); // The modified HTML goes before the image list
    //qDebug() << "Returned HTML and images:\n" << imageList << "\n#################";

    return imageList;
}


/*
 * Basic cleanup of HTML stuff
 *
 */
QString MiscHelpers::cleanupHtml(QString originalHtml)
{
    QString cleanHtml = originalHtml;

    cleanHtml.replace("\n", " ");  // Remove line breaks, as that results in server error 500


    QRegExp doctypeRE("<!DOCTYPE.*>");
    doctypeRE.setMinimal(true);
    cleanHtml.remove(doctypeRE);

    QRegExp headRE("<html><head>.*</head>");
    headRE.setMinimal(true);
    cleanHtml.remove(headRE);
    cleanHtml.remove("</body></html>");

    QRegExp bodyRE("<body style.*>");
    bodyRE.setMinimal(true);

    cleanHtml.remove(bodyRE);

    return cleanHtml.trimmed();
}



/*
 * Return some HTML with a blockquote, quote symbols, etc.
 *
 *
 */
QString MiscHelpers::quotedText(QString author, QString content)
{
    QTextDocument textDocument;
    textDocument.setHtml(content);

    content = textDocument.toPlainText().trimmed();
    content.replace("\n", "<br>");

    QString quoteHtml = "&gt;&gt; <b><u>"+ author + ":</u></b>"  // >>> + name
                        "<blockquote>"
                        "<small><i>&ldquo;" + content + "&rdquo;</i></small>"
                        "</blockquote> <br>";

    return quoteHtml;
}
