/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "mainwindow.h"


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent)
{
    this->setWindowTitle("Dianara");
    this->setWindowIcon(QIcon::fromTheme("dianara",
                                         QIcon(":/icon/64x64/dianara.png")));
    this->setMinimumSize(400, 400);

    QSettings settings;

    firstRun = true;
    prepareDataDirectory(); // This sets this->dataDirectory

    reallyQuitProgram = false;
    trayIconAvailable = false;
    trayCurrentNewCount = 0;
    trayCurrentHLCount = 0;


    QString currentIconset = QIcon::themeName();
    qDebug() << "System iconset:" << currentIconset;
    qDebug() << "Icon theme search paths:" << QIcon::themeSearchPaths();
    if (currentIconset.isEmpty() || currentIconset == "hicolor")
    {
        qDebug() << ">> No system iconset (or hicolor) configured; trying to use Oxygen";
        QIcon::setThemeName("oxygen"); // VERY TMP; FIXME
    }

    // Network control
    pumpController = new PumpController(this);

    // Global object to connect different classes directly
    globalObject = new GlobalObject(this);


    // Filter checker
    filterChecker = new FilterChecker(this);


    ////// GUI

    // User's profile editor, in its own window
    profileEditor = new ProfileEditor(pumpController, this);


    // Splitter to divide the window horizontally
    mainSplitter = new QSplitter(Qt::Horizontal, this);
    mainSplitter->setChildrenCollapsible(false);
    mainSplitter->setContentsMargins(0, 0, 0, 0);
    mainSplitter->setHandleWidth(4);



    // Left side
    avatarIconButton = new QPushButton();
    avatarIconButton->setIcon(QIcon(QPixmap(":/images/no-avatar.png")
                                    .scaled(64, 64,
                                            Qt::KeepAspectRatio,
                                            Qt::SmoothTransformation)));
    avatarIconButton->setSizePolicy(QSizePolicy::Maximum,
                                    QSizePolicy::Maximum);
    avatarIconButton->setFlat(true);
    avatarIconButton->setIconSize(QSize(64, 64));
    connect(avatarIconButton, SIGNAL(clicked()),
            profileEditor, SLOT(show()));


    fullNameLabel = new QLabel("[-------------]");
    fullNameLabel->setWordWrap(true);

    QFont userDetailsFont;
    userDetailsFont.setBold(true);
    userDetailsFont.setItalic(true);
    userDetailsFont.setPointSize(userDetailsFont.pointSize() - 2);

    userIdLabel = new QLabel();
    userIdLabel->setWordWrap(true);
    userIdLabel->setFont(userDetailsFont);
    userIdLabel->setSizePolicy(QSizePolicy::Ignored,
                               QSizePolicy::Minimum);

    userDetailsFont.setBold(false);
    userDetailsFont.setItalic(false);

    userHometownLabel = new QLabel();
    userHometownLabel->setWordWrap(true);
    userHometownLabel->setFont(userDetailsFont);


    userInfoLayout = new QVBoxLayout();
    userInfoLayout->addSpacing(2);
    userInfoLayout->addWidget(fullNameLabel);
    userInfoLayout->addWidget(userIdLabel);
    userInfoLayout->addWidget(userHometownLabel);

    leftTopLayout = new QHBoxLayout();
    leftTopLayout->addWidget(avatarIconButton, 0, Qt::AlignLeft);
    leftTopLayout->addLayout(userInfoLayout,   1);


    leftPanel = new QToolBox(); // For now, only one "section"...

    meanwhileFeed = new MinorFeed(pumpController,
                                  globalObject,
                                  filterChecker);
    connect(meanwhileFeed, SIGNAL(newItemsCountChanged(int,int)),
            this, SLOT(setMinorFeedTitle(int,int)));
    connect(meanwhileFeed, SIGNAL(newItemsReceived(int,int)),
            this, SLOT(notifyMinorFeedUpdate(int,int)));
    leftPanel->addItem(meanwhileFeed,
                       QIcon::fromTheme("clock"),
                       "*meanwhile*");
    this->setMinorFeedTitle(0, 0); // Set initial title (0 new, 0 HL)


    leftSideWidget = new QWidget();
    leftLayout = new QVBoxLayout();
    leftLayout->setContentsMargins(0, 0, 0, 0);
    leftLayout->addLayout(leftTopLayout);       // Avatar + user info
    leftLayout->addSpacing(2);
    leftLayout->addWidget(leftPanel);           // Meanwhile feed

////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////// TMP GROUP MANAGER STUFF
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
#if 0
    GroupsManager *TMPGROUPSMANAGER = new GroupsManager(this->pumpController,
                                                        this);

    QPushButton *TMPEDITGROUPSBUTTON = new QPushButton(QIcon::fromTheme("user-group-properties"),
                                                       "*MANAGE GROUPS*");
    connect(TMPEDITGROUPSBUTTON, SIGNAL(clicked()),
            TMPGROUPSMANAGER, SLOT(show()));
    this->leftLayout->addWidget(TMPEDITGROUPSBUTTON);
#endif
////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////// TMP GROUP MANAGER STUFF
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

    this->leftSideWidget->setLayout(leftLayout);


    // Right side
    rightSideWidget = new QWidget();
    rightLayout = new QVBoxLayout();
    rightLayout->setContentsMargins(0, 1, 1, 1);

    publisher = new Publisher(pumpController,
                              globalObject,
                              this);


    /// START SETTING UP TIMELINES


    // Main timeline //

    mainTimeline = new TimeLine(TimeLine::TimelineTypeMain,
                                pumpController,
                                globalObject,
                                filterChecker,
                                this);
    mainTimelineScrollArea = new QScrollArea();
    mainTimelineScrollArea->setContentsMargins(1, 1, 1, 1);
    mainTimelineScrollArea->setFrameStyle(QFrame::NoFrame);
    mainTimelineScrollArea->setWidget(mainTimeline);  // Make timeline scrollable
    mainTimelineScrollArea->setWidgetResizable(true);
    mainTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(mainTimeline, SIGNAL(scrollTo(QAbstractSlider::SliderAction)),
            this, SLOT(scrollMainTimelineTo(QAbstractSlider::SliderAction)));
    connect(mainTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));
    connect(mainTimeline, SIGNAL(unreadPostsCountChanged(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));

    connect(mainTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(notifyTimelineUpdate(int,int,int)));

    // To ensure comment composer is visible
    connect(mainTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollMainTimelineToWidget(QWidget*)));


    // Direct timeline //

    directTimeline = new TimeLine(TimeLine::TimelineTypeDirect,
                                  pumpController,
                                  globalObject,
                                  filterChecker,
                                  this);
    directTimelineScrollArea = new QScrollArea();
    directTimelineScrollArea->setContentsMargins(1, 1, 1, 1);
    directTimelineScrollArea->setFrameStyle(QFrame::NoFrame);
    directTimelineScrollArea->setWidget(directTimeline);
    directTimelineScrollArea->setWidgetResizable(true);
    directTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(directTimeline, SIGNAL(scrollTo(QAbstractSlider::SliderAction)),
            this, SLOT(scrollDirectTimelineTo(QAbstractSlider::SliderAction)));
    connect(directTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));
    connect(directTimeline, SIGNAL(unreadPostsCountChanged(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));

    connect(directTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(notifyTimelineUpdate(int,int,int)));

    // To ensure comment composer is visible
    connect(directTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollDirectTimelineToWidget(QWidget*)));


    // Activity timeline //

    activityTimeline = new TimeLine(TimeLine::TimelineTypeActivity,
                                    pumpController,
                                    globalObject,
                                    filterChecker,
                                    this);
    activityTimelineScrollArea = new QScrollArea();
    activityTimelineScrollArea->setContentsMargins(1, 1, 1, 1);
    activityTimelineScrollArea->setFrameStyle(QFrame::NoFrame);
    activityTimelineScrollArea->setWidget(activityTimeline);  // Make it scrollable
    activityTimelineScrollArea->setWidgetResizable(true);
    activityTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(activityTimeline, SIGNAL(scrollTo(QAbstractSlider::SliderAction)),
            this, SLOT(scrollActivityTimelineTo(QAbstractSlider::SliderAction)));
    connect(activityTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));
    connect(activityTimeline, SIGNAL(unreadPostsCountChanged(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));

    connect(activityTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(notifyTimelineUpdate(int,int,int)));

    // To ensure comment composer is visible
    connect(activityTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollActivityTimelineToWidget(QWidget*)));


    // Favorites timeline //

    favoritesTimeline = new TimeLine(TimeLine::TimelineTypeFavorites,
                                     pumpController,
                                     globalObject,
                                     filterChecker,
                                     this);
    favoritesTimelineScrollArea = new QScrollArea();
    favoritesTimelineScrollArea->setContentsMargins(1, 1, 1, 1);
    favoritesTimelineScrollArea->setFrameStyle(QFrame::NoFrame);
    favoritesTimelineScrollArea->setWidget(favoritesTimeline);
    favoritesTimelineScrollArea->setWidgetResizable(true);
    favoritesTimelineScrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    connect(favoritesTimeline, SIGNAL(scrollTo(QAbstractSlider::SliderAction)),
            this, SLOT(scrollFavoritesTimelineTo(QAbstractSlider::SliderAction)));
    connect(favoritesTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));
    connect(favoritesTimeline, SIGNAL(unreadPostsCountChanged(int,int,int,int)),
            this, SLOT(setTimelineTabTitle(int,int,int,int)));

    connect(favoritesTimeline, SIGNAL(timelineRendered(int,int,int,int)),
            this, SLOT(notifyTimelineUpdate(int,int,int)));

    // To ensure comment composer is visible
    connect(favoritesTimeline, SIGNAL(commentingOnPost(QWidget*)),
            this, SLOT(scrollFavoritesTimelineToWidget(QWidget*)));


    /// END SETTING UP TIMELINES


    // The contact list has its own tabs with its own scroll areas
    contactManager = new ContactManager(pumpController,
                                        globalObject,
                                        this);


    tabWidget = new QTabWidget();
    tabWidget->addTab(mainTimelineScrollArea,
                      QIcon::fromTheme("view-list-details"),
                      "MAIN TIMELINE TAB");
    this->setTimelineTabTitle(TimeLine::TimelineTypeMain, 0, 0, 0);
    tabWidget->addTab(directTimelineScrollArea,
                      QIcon::fromTheme("mail-message"),
                      "MESSAGES TAB");
    this->setTimelineTabTitle(TimeLine::TimelineTypeDirect, 0, 0, 0);
    tabWidget->addTab(activityTimelineScrollArea,
                      QIcon::fromTheme("user-home"),
                      "ACTIVITY TAB");
    this->setTimelineTabTitle(TimeLine::TimelineTypeActivity, 0, 0, 0);
    tabWidget->addTab(favoritesTimelineScrollArea,
                      QIcon::fromTheme("folder-favorites"),
                      "FAVORITES TAB");
    this->setTimelineTabTitle(TimeLine::TimelineTypeFavorites, 0, 0, 0);
    tabWidget->addTab(contactManager,
                      QIcon::fromTheme("system-users"),
                      tr("&Contacts"));
    tabWidget->setTabToolTip(4, tr("The people you follow, and the ones who follow you"));
    connect(tabWidget, SIGNAL(currentChanged(int)),
            this, SLOT(setTitleAndTrayInfo(int)));



    rightLayout->addWidget(publisher, 1); // stretch 1/10
    rightLayout->addWidget(tabWidget, 9); // stretch 9/10

    this->rightSideWidget->setLayout(rightLayout);


    mainSplitter->addWidget(leftSideWidget);
    mainSplitter->addWidget(rightSideWidget);
    this->setCentralWidget(mainSplitter);


    // FreeDesktop.org notifications handler
    fdNotifier = new FDNotifications();
    connect(fdNotifier, SIGNAL(showFallbackNotification(QString,QString)),
            this, SLOT(showTrayFallbackMessage(QString,QString)));


    // Timeline updates timer
    updateTimer = new QTimer(this); // Interval is set from loadSettings()
    connect(updateTimer, SIGNAL(timeout()),
            this, SLOT(updateMainDirectMinorTimelines()));
    updateTimer->start();

    // Timestamps refresh timer
    timestampsTimer = new QTimer(this);
    timestampsTimer->setInterval(60000); // 60 sec
    connect(timestampsTimer, SIGNAL(timeout()),
            this, SLOT(refreshAllTimestamps()));
    timestampsTimer->start();

    // Delayed timeline resize timer
    delayedResizeTimer = new QTimer(this);
    delayedResizeTimer->setSingleShot(true);
    connect(delayedResizeTimer, SIGNAL(timeout()),
            this, SLOT(adjustTimelineSizes()));


    ////////////////// Load configuration from disk
    loadSettings();



    //// External widgets which live in their own windows

    // Account wizard
    accountDialog = new AccountDialog(pumpController, this);
    connect(accountDialog, SIGNAL(userIDChanged(QString)),
            this, SLOT(updateUserID(QString)));

    // If this is the first run, show the account Dialog
    if (firstRun)
    {
        accountDialog->show();
    }

    // Configuration dialog
    configDialog = new ConfigDialog(this->globalObject,
                                    this->dataDirectory,
                                    this->updateInterval,
                                    this->postsPerPageMain,
                                    this->postsPerPageOther,
                                    this->tabsPosition,
                                    this->tabsMovable,
                                    this->publicPosts,
                                    this->fdNotifier,
                                    this);
    connect(configDialog, SIGNAL(configurationChanged()),
            this, SLOT(updateConfigSettings()));


    // Filter editor
    filterEditor = new FilterEditor(filterChecker, this);
    connect(configDialog, SIGNAL(filterEditorRequested()),
            filterEditor, SLOT(show()));


    // Log viewer
    logViewer = new LogViewer(this);
    connect(pumpController, SIGNAL(logMessage(QString,QString)),
            logViewer, SLOT(addToLog(QString,QString)));


    // Help widget; "Getting started", etc.
    helpWidget = new HelpWidget(this);


    ///
    //////////////////////////////// Connections for PumpController //////////
    ///

    connect(pumpController, SIGNAL(profileReceived(QString,QString,QString,QString,QString)),
            this, SLOT(updateProfileData(QString,QString,QString,QString,QString)));

    connect(pumpController, SIGNAL(avatarPictureReceived(QByteArray,QUrl)),
            this, SLOT(storeAvatar(QByteArray,QUrl)));
    connect(pumpController, SIGNAL(imageReceived(QByteArray,QUrl)),
            this, SLOT(storeImage(QByteArray,QUrl)));

    connect(pumpController, SIGNAL(authorizationStatusChanged(bool)),
            this, SLOT(toggleWidgetsByAuthorization(bool)));
    connect(pumpController, SIGNAL(authorizationFailed(QString,QString)),
            this, SLOT(showAuthError(QString,QString)));

    connect(pumpController, SIGNAL(initializationComplete()),
            this, SLOT(onInitializationComplete()));


    // After receiving timeline contents, update corresponding timeline
    connect(pumpController, SIGNAL(mainTimelineReceived(QVariantList,int,QString,QString,int)),
            mainTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString,int)));
    connect(pumpController, SIGNAL(directTimelineReceived(QVariantList,int,QString,QString,int)),
            directTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString,int)));
    connect(pumpController, SIGNAL(activityTimelineReceived(QVariantList,int,QString,QString,int)),
            activityTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString,int)));
    connect(pumpController, SIGNAL(favoritesTimelineReceived(QVariantList,int,QString,QString,int)),
            favoritesTimeline, SLOT(setTimeLineContents(QVariantList,int,QString,QString,int)));


    // After sucessful posting, request updated timeline, with your post included
    connect(pumpController, SIGNAL(postPublished()),
            this, SLOT(updateMainActivityMinorTimelines()));


    // After successful liking, update likes count
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            mainTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            directTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            activityTimeline, SLOT(setLikesInPost(QVariantList,QString)));
    // We don't update likes count in favorites timeline,
    // since it still doesn't know about this post

    // Instead, reload favorites timeline
    connect(pumpController, SIGNAL(likesReceived(QVariantList,QString)),
            favoritesTimeline, SLOT(goToFirstPage()));


    // After commenting successfully, refresh list of comments in that post
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            mainTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            directTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            activityTimeline, SLOT(setCommentsInPost(QVariantList,QString)));
    connect(pumpController, SIGNAL(commentsReceived(QVariantList,QString)),
            favoritesTimeline, SLOT(setCommentsInPost(QVariantList,QString)));


    // After successful sharing....

    // TODO***


    // After receiving the minor feed ("Meanwhile"), update it
    connect(pumpController, SIGNAL(minorFeedReceived(QVariantList,QString,QString)),
            meanwhileFeed, SLOT(setFeedContents(QVariantList,QString,QString)));


    // Show notifications for events sent from pumpController
    connect(pumpController, SIGNAL(showNotification(QString)),
            fdNotifier, SLOT(showMessage(QString)));

    // Update statusBar message from pumpController's infos
    connect(pumpController, SIGNAL(currentJobChanged(QString)),
            this, SLOT(setStatusBarMessage(QString)));
    connect(pumpController, SIGNAL(transientStatusBarMessage(QString)),
            this, SLOT(setTransientStatusMessage(QString)));


    // Add menus and toolbar
    createMenus();
    createToolbar();

    // StatusBar stuff
    this->createStatusbarWidgets();
    this->statusBar()->showMessage(tr("Initializing..."));

    // Add the system tray icon
    createTrayIcon();


    settings.beginGroup("MainWindow");

    // Now set the "view side panel" checkable menu to its saved state
    // This will also trigger the action to hide/show it
    viewSidePanel->setChecked(settings.value("viewSidePanel", true).toBool());

    // Check view > toolbar if needed; HIDDEN by default
    viewToolbar->setChecked(settings.value("viewToolbar", false).toBool());

    // Set the "view status bar" checkable menu to its saved state
    viewStatusBar->setChecked(settings.value("viewStatusBar", true).toBool());

    settings.endGroup();


    logViewer->addToLog(tr("Dianara started."));

    // If User ID is defined, set PumpController in motion
    if (!userID.isEmpty())
    {
        pumpController->setUserCredentials(this->userID);
        fdNotifier->setCurrentUserId(this->userID);
        // getUserProfile() will be called from setUserCredentials()
    }
    else // Otherwise, just say so in the statusbar
    {
        QString message = tr("Your account is not configured yet.");
        this->setStatusBarMessage(message);
        logViewer->addToLog(message);
    }

    // Post-init timer
    postInitTimer = new QTimer(this);
    postInitTimer->setSingleShot(true);
    connect(postInitTimer, SIGNAL(timeout()),
            this, SLOT(postInit()));
    postInitTimer->start(1000);


    qDebug() << "MainWindow created";
}



MainWindow::~MainWindow()
{
    qDebug() << "MainWindow destroyed";
}




/*
 * Prepare the data directory. Create if necessary
 *
 */
void MainWindow::prepareDataDirectory()
{
    #if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
    dataDirectory = QDesktopServices::storageLocation(QDesktopServices::DataLocation);
    #else
    dataDirectory = QStandardPaths::standardLocations(QStandardPaths::DataLocation).first();
    #endif
    qDebug() << "Data directory:" << this->dataDirectory;

    QDir dataDir;

    // Base directory
    if (!dataDir.exists(dataDirectory))
    {
        qDebug() << "Creating data directory";
        if (dataDir.mkpath(dataDirectory))
        {
            qDebug() << "Data directory created";
        }
        else
        {
            qDebug() << "Error creating data directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/images"))
    {
        qDebug() << "Creating images directory";
        if (dataDir.mkpath(dataDirectory + "/images"))
        {
            qDebug() << "Images directory created";
        }
        else
        {
            qDebug() << "Error creating images directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/audios"))
    {
        qDebug() << "Creating audios directory";
        if (dataDir.mkpath(dataDirectory + "/audios"))
        {
            qDebug() << "Audios directory created";
        }
        else
        {
            qDebug() << "Error creating audios directory!";
        }
    }

    if (!dataDir.exists(dataDirectory + "/videos"))
    {
        qDebug() << "Creating videos directory";
        if (dataDir.mkpath(dataDirectory + "/videos"))
        {
            qDebug() << "Videos directory created";
        }
        else
        {
            qDebug() << "Error creating videos directory!";
        }
    }

    // Avatars directory
    if (!dataDir.exists(dataDirectory + "/avatars"))
    {
        qDebug() << "Creating avatars directory";
        if (dataDir.mkpath(dataDirectory + "/avatars"))
        {
            qDebug() << "Avatars directory created";
        }
        else
        {
            qDebug() << "Error creating avatars directory!";
        }
    }

}



/*
 * Populate the menus
 *
 */
void MainWindow::createMenus()
{
    sessionMenu = new QMenu(tr("&Session"));

    sessionUpdateMainTimeline = new QAction(QIcon::fromTheme("view-refresh",
                                                             QIcon(":/images/menu-refresh.png")),
                                            tr("&Update Timeline"),
                                            this);
    sessionUpdateMainTimeline->setShortcut(QKeySequence(Qt::Key_F5));
    sessionUpdateMainTimeline->setDisabled(true); // Disabled until authorization checked
    connect(sessionUpdateMainTimeline, SIGNAL(triggered()),
            mainTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateMainTimeline);

    sessionUpdateDirectTimeline = new QAction(QIcon::fromTheme("view-refresh",
                                                               QIcon(":/images/menu-refresh.png")),
                                              tr("Update &Messages"),
                                              this);
    sessionUpdateDirectTimeline->setShortcut(QKeySequence(Qt::Key_F6));
    sessionUpdateDirectTimeline->setDisabled(true);
    connect(sessionUpdateDirectTimeline, SIGNAL(triggered()),
            directTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateDirectTimeline);

    sessionUpdateActivityTimeline = new QAction(QIcon::fromTheme("view-refresh",
                                                                 QIcon(":/images/menu-refresh.png")),
                                                tr("Update &Activity"),
                                                this);
    sessionUpdateActivityTimeline->setShortcut(QKeySequence(Qt::Key_F7));
    sessionUpdateActivityTimeline->setDisabled(true);
    connect(sessionUpdateActivityTimeline, SIGNAL(triggered()),
            activityTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateActivityTimeline);


    sessionUpdateFavoritesTimeline = new QAction(QIcon::fromTheme("view-refresh",
                                                                  QIcon(":/images/menu-refresh.png")),
                                                 tr("Update Fa&vorites"),
                                                 this);
    sessionUpdateFavoritesTimeline->setShortcut(QKeySequence(Qt::Key_F8));
    sessionUpdateFavoritesTimeline->setDisabled(true);
    connect(sessionUpdateFavoritesTimeline, SIGNAL(triggered()),
            favoritesTimeline, SLOT(goToFirstPage()));
    sessionMenu->addAction(sessionUpdateFavoritesTimeline);


    sessionUpdateMinorFeed = new QAction(QIcon::fromTheme("view-refresh",
                                                          QIcon(":/images/menu-refresh.png")),
                                         tr("Update Minor &Feed"),
                                         this);
    sessionUpdateMinorFeed->setShortcut(QKeySequence(Qt::Key_F10));
    sessionUpdateMinorFeed->setDisabled(true);
    connect(sessionUpdateMinorFeed, SIGNAL(triggered()),
            meanwhileFeed, SLOT(updateFeed()));
    sessionMenu->addAction(sessionUpdateMinorFeed);


    sessionUpdateAllTimelines = new QAction(QIcon::fromTheme("view-refresh",
                                                             QIcon(":/images/menu-refresh.png")),
                                         tr("Update All Timelines"),
                                         this);
    sessionUpdateAllTimelines->setDisabled(true);
    connect(sessionUpdateAllTimelines, SIGNAL(triggered()),
            this, SLOT(updateAllTimelines()));
    sessionMenu->addAction(sessionUpdateAllTimelines);

    sessionAutoUpdates = new QAction(QIcon::fromTheme("clock"),
                                     tr("Auto-update &Timelines"),
                                     this);
    sessionAutoUpdates->setCheckable(true);
    sessionAutoUpdates->setChecked(true);
    sessionAutoUpdates->setDisabled(true); // until initialization is complete
    connect(sessionAutoUpdates, SIGNAL(toggled(bool)),
            this, SLOT(toggleAutoUpdates(bool)));
    sessionMenu->addAction(sessionAutoUpdates);

    sessionMenu->addSeparator(); // ------

    sessionMarkAllAsRead = new QAction(QIcon::fromTheme("mail-mark-read"),
                                       tr("Mark All as Read"),
                                       this);
    sessionMarkAllAsRead->setShortcut(QKeySequence("Ctrl+R"));
    connect(sessionMarkAllAsRead, SIGNAL(triggered()),
            this, SLOT(markAllAsRead()));
    sessionMenu->addAction(sessionMarkAllAsRead);    

    sessionMenu->addSeparator();

    sessionPostNote = new QAction(QIcon::fromTheme("document-edit",
                                                   QIcon(":/images/button-edit.png")),
                                          tr("&Post a Note"),
                                          this);
    sessionPostNote->setShortcut(QKeySequence("Ctrl+N"));
    connect(sessionPostNote, SIGNAL(triggered()),
            publisher, SLOT(setFullMode()));
    connect(sessionPostNote, SIGNAL(triggered()),
            this, SLOT(show()));  // show window, in case it's hidden and
                                  // Post a Note is used from tray menu
    sessionMenu->addAction(sessionPostNote);

    sessionMenu->addSeparator();

    sessionQuit = new QAction(QIcon::fromTheme("application-exit"),
                              tr("&Quit"), this);
    sessionQuit->setShortcut(QKeySequence("Ctrl+Q"));
    connect(sessionQuit, SIGNAL(triggered()),
            this, SLOT(quitProgram()));
    sessionMenu->addAction(sessionQuit);

    this->menuBar()->addMenu(sessionMenu);


    viewMenu = new QMenu(tr("&View"));

    viewSidePanel = new QAction(QIcon::fromTheme("view-sidetree"),
                                tr("Side &Panel"), this);
    connect(viewSidePanel, SIGNAL(toggled(bool)),
            this, SLOT(toggleSidePanel(bool)));
    viewSidePanel->setCheckable(true);
    viewSidePanel->setChecked(true);
    viewSidePanel->setShortcut(Qt::Key_F9);

    viewMenu->addAction(viewSidePanel);


    viewToolbar = new QAction(QIcon::fromTheme("configure-toolbars"),
                                tr("&Toolbar"), this);
    connect(viewToolbar, SIGNAL(toggled(bool)),
            this, SLOT(toggleToolbar(bool)));
    viewToolbar->setCheckable(true);

    viewMenu->addAction(viewToolbar);


    viewStatusBar = new QAction(QIcon::fromTheme("configure-toolbars"),
                                tr("Status &Bar"), this);
    connect(viewStatusBar, SIGNAL(toggled(bool)),
            this, SLOT(toggleStatusBar(bool)));
    viewStatusBar->setCheckable(true);
    viewStatusBar->setChecked(true);

    viewMenu->addAction(viewStatusBar);

    viewMenu->addSeparator();


    viewFullscreenAction = new QAction(QIcon::fromTheme("view-fullscreen"),
                                       tr("Full &Screen"), this);
    connect(viewFullscreenAction, SIGNAL(toggled(bool)),
            this, SLOT(toggleFullscreen(bool)));
    viewFullscreenAction->setCheckable(true);
    viewFullscreenAction->setChecked(false);
    viewFullscreenAction->setShortcut(Qt::Key_F11);

    viewMenu->addAction(viewFullscreenAction);

    viewLogAction = new QAction(QIcon::fromTheme("text-x-log",
                                                 QIcon(":/images/log.png")),
                                tr("&Log"), this);
    connect(viewLogAction, SIGNAL(triggered()),
            logViewer, SLOT(show()));
    viewLogAction->setShortcut(Qt::Key_F12);

    viewMenu->addAction(viewLogAction);


    this->menuBar()->addMenu(viewMenu);



    settingsMenu = new QMenu(tr("S&ettings"));

    settingsEditProfile = new QAction(QIcon::fromTheme("user-properties",
                                                       QIcon(":/images/no-avatar.png")),
                                      tr("Edit &Profile"), this);
    settingsEditProfile->setShortcut(QKeySequence("Ctrl+Shift+P"));
    connect(settingsEditProfile, SIGNAL(triggered()),
            profileEditor, SLOT(show()));
    settingsMenu->addAction(settingsEditProfile);

    settingsAccount = new QAction(QIcon::fromTheme("dialog-password"),
                                  tr("&Account"), this);
    connect(settingsAccount, SIGNAL(triggered()),
            accountDialog, SLOT(show()));
    settingsMenu->addAction(settingsAccount);

    settingsMenu->addSeparator();

    settingsFilters = new QAction(QIcon::fromTheme("view-filter",
                                                   QIcon(":/images/button-filter.png")),
                                  tr("&Filters and Highlighting"), this);
    settingsFilters->setShortcut(QKeySequence("Ctrl+Shift+F"));
    connect(settingsFilters, SIGNAL(triggered()),
            filterEditor, SLOT(show()));
    settingsMenu->addAction(settingsFilters);

    settingsConfigure = new QAction(QIcon::fromTheme("configure",
                                                     QIcon(":/images/button-configure.png")),
                                    tr("&Configure Dianara"), this);
    settingsConfigure->setShortcut(QKeySequence("Ctrl+Shift+S"));
    connect(settingsConfigure, SIGNAL(triggered()),
            configDialog, SLOT(show()));
    settingsMenu->addAction(settingsConfigure);
    this->menuBar()->addMenu(settingsMenu);
    this->menuBar()->addSeparator();



    helpMenu = new QMenu(tr("&Help"));

    helpBasicHelp = new QAction(QIcon::fromTheme("help-browser"),
                                tr("Basic &Help"), this);
    helpBasicHelp->setShortcut(Qt::Key_F1);
    connect(helpBasicHelp, SIGNAL(triggered()),
            helpWidget, SLOT(show()));
    helpMenu->addAction(helpBasicHelp);

    helpMenu->addSeparator();

    helpVisitWebsite = new QAction(QIcon::fromTheme("internet-web-browser"),
                            tr("Visit &Website"), this);
    connect(helpVisitWebsite, SIGNAL(triggered()),
            this, SLOT(visitWebSite()));
    helpMenu->addAction(helpVisitWebsite);

    helpVisitPumpFAQ = new QAction(QIcon::fromTheme("internet-web-browser"),
                                  tr("&Frequently Asked Questions about Pump.io"), this);
    connect(helpVisitPumpFAQ, SIGNAL(triggered()),
            this, SLOT(visitFAQ()));
    helpMenu->addAction(helpVisitPumpFAQ);

    helpVisitPumpTips = new QAction(QIcon::fromTheme("internet-web-browser"),
                                    tr("Some Pump.io &Tips"), this);
    connect(helpVisitPumpTips, SIGNAL(triggered()),
            this, SLOT(visitTips()));
    helpMenu->addAction(helpVisitPumpTips);

    helpVisitPumpUserList = new QAction(QIcon::fromTheme("internet-web-browser"),
                                        tr("List of Some Pump.io &Users"), this);
    connect(helpVisitPumpUserList, SIGNAL(triggered()),
            this, SLOT(visitUserList()));
    helpMenu->addAction(helpVisitPumpUserList);


    helpMenu->addSeparator();

    helpAbout = new QAction(QIcon(":/icon/64x64/dianara.png"),
                            tr("About &Dianara"), this);
    connect(helpAbout, SIGNAL(triggered()),
            this, SLOT(aboutDianara()));
    helpMenu->addAction(helpAbout);
    this->menuBar()->addMenu(helpMenu);


    this->menuBar()->setContextMenuPolicy(Qt::PreventContextMenu);


    ///// Context menu for the tray icon
    trayTitleSeparatorAction = new QAction("Dianara", this);
    trayTitleSeparatorAction->setSeparator(true);

    trayShowWindowAction = new QAction(QIcon(":/icon/64x64/dianara.png"),
                                       "*show-window*", this);
    connect(trayShowWindowAction, SIGNAL(triggered()),
            this, SLOT(toggleMainWindow()));

    trayContextMenu = new QMenu("Tray Context Menu");
    trayContextMenu->setSeparatorsCollapsible(false);
    trayContextMenu->addAction(trayTitleSeparatorAction); // Acts as title
    trayContextMenu->addAction(trayShowWindowAction);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionUpdateMainTimeline);
    trayContextMenu->addAction(sessionUpdateMinorFeed);
    trayContextMenu->addAction(sessionAutoUpdates);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionMarkAllAsRead);
    trayContextMenu->addAction(sessionPostNote);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(settingsEditProfile);
    trayContextMenu->addAction(settingsConfigure);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(helpBasicHelp);
    trayContextMenu->addAction(helpAbout);
    trayContextMenu->addSeparator();
    trayContextMenu->addAction(sessionQuit);

    // FIXME: if mainwindow is hidden, program quits
    // after closing Configure or About window (now partially fixed)


    qDebug() << "Menus created";
}


void MainWindow::createToolbar()
{
    this->mainToolBar = addToolBar(tr("Toolbar"));
    mainToolBar->setToolButtonStyle(Qt::ToolButtonFollowStyle);
    mainToolBar->setContextMenuPolicy(Qt::PreventContextMenu);
    mainToolBar->setMovable(false);

    mainToolBar->addAction(this->sessionUpdateMainTimeline);
    mainToolBar->addAction(this->sessionUpdateMinorFeed);
    mainToolBar->addAction(this->sessionMarkAllAsRead);
    mainToolBar->addSeparator();
    this->settingsFilters->setPriority(QAction::LowPriority); // Don't show text in besides-icon mode
    mainToolBar->addAction(this->settingsFilters);
    this->settingsConfigure->setPriority(QAction::LowPriority);
    mainToolBar->addAction(this->settingsConfigure);


    mainToolBar->hide();
}


void MainWindow::createStatusbarWidgets()
{
    this->statusStateButton = new QToolButton();
    this->setStateIcon(MainWindow::Initializing);
    connect(statusStateButton, SIGNAL(clicked()),
            sessionAutoUpdates, SLOT(toggle()));

    this->statusLogButton = new QToolButton();
    statusLogButton->setIcon(QIcon::fromTheme("text-x-log",
                                              QIcon(":/images/log.png")));
    statusLogButton->setToolTip(tr("Open the log viewer"));
    connect(statusLogButton, SIGNAL(clicked()),
            viewLogAction, SLOT(trigger()));

    this->statusBar()->addPermanentWidget(statusLogButton);
    this->statusBar()->addPermanentWidget(statusStateButton);
    this->statusBar()->setSizeGripEnabled(false);
}


void MainWindow::setStateIcon(MainWindow::StatusType statusType)
{
    if (statusType == Initializing)
    {
        statusStateButton->setDisabled(true); // Until initialization is done
        statusStateButton->setToolTip(tr("Initializing..."));
        statusStateButton->setIcon(QIcon::fromTheme("user-offline",
                                                    QIcon(":/images/button-offline.png")));
    }
    else if (statusType == Autoupdating)
    {
        statusStateButton->setEnabled(true);
        statusStateButton->setToolTip("<b></b>" + tr("Auto-updating enabled"));
        statusStateButton->setIcon(QIcon::fromTheme("user-online",
                                                    QIcon(":/images/button-online.png")));
    }
    else // MainWindow::Stopped
    {
        statusStateButton->setEnabled(true);
        statusStateButton->setToolTip("<b></b>" + tr("Auto-updating disabled"));
        statusStateButton->setIcon(QIcon::fromTheme("user-busy",
                                                    QIcon(":/images/button-busy.png")));
    }
}


/*
 * Create an icon in the system tray, define its contextual menu, etc.
 *
 */
void MainWindow::createTrayIcon()
{
    trayIcon = new QSystemTrayIcon(this);

    if (trayIcon->isSystemTrayAvailable())
    {
        trayIconAvailable = true;

        this->setTrayIconPixmap(); // Set icon for "no unread messages" initially
        this->setTitleAndTrayInfo(this->tabWidget->currentIndex());


        // Catch clicks on icon
        connect(trayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
                this, SLOT(trayControl(QSystemTrayIcon::ActivationReason)));

        // clicking in a popup notification (balloon-type) will show the window
        connect(trayIcon, SIGNAL(messageClicked()),
                this, SLOT(show()));

        // Set contextual menu for the icon
        trayIcon->setContextMenu(this->trayContextMenu);


        trayIcon->show();

        qDebug() << "Tray icon created";
    }
    else
    {
        trayIconAvailable = false;

        qDebug() << "System tray not available";
    }
}


/*
 * Set the tray icon's pixmap, with number of unread messages, or nothing
 *
 */
void MainWindow::setTrayIconPixmap(int count, int hightlightedCount)
{
    this->trayCurrentNewCount = count;
    this->trayCurrentHLCount = hightlightedCount;

    QPixmap iconPixmap;

    switch (this->trayIconType)
    {
    case 0: // Default
        iconPixmap = QIcon(":/icon/32x32/dianara.png")
                     .pixmap(32, 32)
                     .scaled(32, 32);
        break;

    case 1: // system iconset, if available
        iconPixmap = QIcon::fromTheme("dianara",
                                       QIcon(":/icon/32x32/dianara.png"))
                                       .pixmap(32, 32)
                                       .scaled(32, 32);
        break;

    case 2: // Use your own avatar
        iconPixmap = this->avatarIconButton->icon()
                     .pixmap(32, 32)
                     .scaled(32, 32);
        break;

    case 3: // Custom icon
        iconPixmap = this->trayCustomPixmap;

        break;

    }


    // Paint the number of unread messages on top of the pixmap, if != 0
    if (trayCurrentNewCount > 0)
    {
        // Draw a pseudo-shadow on the side first
        QPainter painter(&iconPixmap);
        if (trayCurrentHLCount > 0)
        {
            // Paint a shadow that covers the higher part too
            painter.drawPixmap(0, 0,
                               32, 32,
                               QPixmap(":/images/tray-bg-high.png"));
        }
        else
        {
            // Shadow for the lower part only
            painter.drawPixmap(0, 0,
                               32, 32,
                               QPixmap(":/images/tray-bg-low.png"));
        }

        QFont font;
        font.setPixelSize(16);
        font.setWeight(QFont::Black);


        // The number
        QString messagesCountString = QString("%1").arg(trayCurrentNewCount);

        QPen pen;
        pen.setBrush(Qt::white);
        painter.setFont(font);
        painter.setPen(pen);
        // Draw the number of new messages
        painter.drawText(0, 0,
                         32, 34,  // End painting outside, at Y=34, to skip margins
                         Qt::AlignRight | Qt::AlignBottom,
                         messagesCountString);

        if (trayCurrentHLCount > 0)
        {
            // The other number
            messagesCountString = QString("%1").arg(trayCurrentHLCount);

            pen.setBrush(Qt::cyan);
            painter.setPen(pen);

            // Draw the number of highlighted messages
            painter.drawText(0, -2,  // Start painting outside, at Y=-2, to avoid some margins
                             32, 32,
                             Qt::AlignRight | Qt::AlignTop,
                             messagesCountString);
        }
    }

    this->trayIcon->setIcon(iconPixmap);
}



/*
 * Load general program settings and state: size, position...
 *
 */
void MainWindow::loadSettings()
{
    QSettings settings;

    firstRun = settings.value("firstRun", true).toBool();
    if (firstRun)
    {
        qDebug() << "This is the first run";
    }


    userID = settings.value("userID", "").toString();
    this->setTitleAndTrayInfo(tabWidget->currentIndex());



    // Main window state
    settings.beginGroup("MainWindow");

    this->resize(settings.value("windowSize", QSize(740, 580)).toSize());
    if (!firstRun) // So we should have a proper position saved
    {
        this->move(settings.value("windowPosition").toPoint());
    }
    this->mainSplitter->restoreState(settings.value("mainSplitterState",
                                                    mainSplitter->saveState()).toByteArray());
    // Set childrenCollapsible to false AFTER loading state, to make sure state does not mess with it
    mainSplitter->setChildrenCollapsible(false);

    settings.endGroup();


    // General program configuration
    this->updateConfigSettings();


    qDebug() << "Settings loaded";
}

/*
 * Save general program settings and state: size, position...
 *
 */
void MainWindow::saveSettings()
{
    QSettings settings;

    settings.setValue("firstRun", false);

    // General main window status
    settings.beginGroup("MainWindow");

    settings.setValue("windowSize",        this->size());
    settings.setValue("windowPosition",    this->pos());
    settings.setValue("mainSplitterState", this->mainSplitter->saveState());

    settings.setValue("viewSidePanel",     this->viewSidePanel->isChecked());
    settings.setValue("viewToolbar",       this->viewToolbar->isChecked());
    settings.setValue("viewStatusBar",     this->viewStatusBar->isChecked());

    settings.endGroup();


    qDebug() << "MainWindow settings saved";
}



//////////////////////////////////////////////////////////////////////////////
/////////////////////////////////// SLOTS ////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



/*
 * Update UserID string from signal emitted in AccountDialog
 *
 */
void MainWindow::updateUserID(QString newUserID)
{
    this->userID = newUserID;

    // update window title and tray icon tooltip
    this->setTitleAndTrayInfo(tabWidget->currentIndex());

    this->pumpController->setUserCredentials(userID);
    this->fdNotifier->setCurrentUserId(userID);

    // Remove current user's name, id and avatar
    this->fullNameLabel->setText("--");
    this->userIdLabel->setText("_@_");
    this->userHometownLabel->setText("--");

    avatarIconButton->setIcon(QIcon(QPixmap(":/images/no-avatar.png")
                                    .scaled(64, 64,
                                            Qt::KeepAspectRatio,
                                            Qt::SmoothTransformation)));


    qDebug() << "UserID updated from AccountDialog:" << userID;
}



/*
 * Update settings changed from ConfigDialog()
 *
 */
void MainWindow::updateConfigSettings()
{
    QSettings settings;
    settings.beginGroup("Configuration");

    this->updateInterval = settings.value("updateInterval", 5).toInt();
    this->updateTimer->setInterval(updateInterval * 1000 * 60);  // min > msec

    this->postsPerPageMain = settings.value("postsPerPageMain", 20).toInt();
    this->pumpController->setPostsPerPageMain(this->postsPerPageMain);

    this->postsPerPageOther = settings.value("postsPerPageOther", 5).toInt();
    this->pumpController->setPostsPerPageOther(this->postsPerPageOther);

    this->tabsPosition = settings.value("tabsPosition", QTabWidget::North).toInt();
    tabWidget->setTabPosition((QTabWidget::TabPosition)tabsPosition);

    this->tabsMovable = settings.value("tabsMovable", true).toBool();
    tabWidget->setMovable(tabsMovable);


    this->publicPosts = settings.value("publicPosts", false).toBool();
    this->publisher->setDefaultPublicPosting(this->publicPosts);

    int selectedProxyType = settings.value("proxyType", 0).toInt(); // 0 means "no proxy"
    QNetworkProxy::ProxyType proxyType = QNetworkProxy::NoProxy;
    if (selectedProxyType == 1)
    {
        proxyType = QNetworkProxy::Socks5Proxy;
    }
    else if (selectedProxyType == 2)
    {
        proxyType = QNetworkProxy::HttpProxy;
    }

    QString proxyHostname = settings.value("proxyHostname").toString();
    int proxyPort = settings.value("proxyPort", 0).toInt();
    bool proxyUseAuth = settings.value("proxyUseAuth", false).toBool();
    QString proxyUser = settings.value("proxyUser").toString();
    QByteArray proxyPassword = QByteArray::fromBase64(settings.value("proxyPassword").toByteArray());

    this->pumpController->setProxyConfig(proxyType,
                                         proxyHostname, proxyPort,
                                         proxyUseAuth,
                                         proxyUser,
                                         QString::fromLocal8Bit(proxyPassword));



    // System tray icon type and custom icon filename, if any
    this->trayIconType = settings.value("systrayIconType", 0).toInt();
    QString trayIconFilename = settings.value("systrayCustomIconFN").toString();
    this->trayCustomPixmap = QPixmap(trayIconFilename).scaled(32, 32);
    if (trayCustomPixmap.isNull()) // Custom icon image is gone or something
    {
        trayCustomPixmap = QPixmap(":/icon/32x32/dianara.png");
    }
    if (trayIconAvailable)
    {
        this->setTrayIconPixmap(trayCurrentNewCount,
                                trayCurrentHLCount); // Sync icon
    }

    settings.endGroup();


    qDebug() << "updateInterval updated:" << updateInterval << updateInterval*60000;
    qDebug() << "postsPerPage Main/Other:" << postsPerPageMain << postsPerPageOther;
    qDebug() << "tabsPosition updated:" << tabsPosition << tabWidget->tabPosition();
    qDebug() << "tabsMovable updated:" << tabsMovable;
    qDebug() << "public posts updated:" << publicPosts;
    qDebug() << "tray icon type updated:" << trayIconType;
}


/*
 * Enable or disable some widgets, depending on whether the applicacion
 * is authorized or not
 *
 */
void MainWindow::toggleWidgetsByAuthorization(bool authorized)
{
    this->sessionUpdateMainTimeline->setEnabled(authorized);
    this->sessionUpdateDirectTimeline->setEnabled(authorized);
    this->sessionUpdateActivityTimeline->setEnabled(authorized);
    this->sessionUpdateFavoritesTimeline->setEnabled(authorized);
    this->sessionUpdateMinorFeed->setEnabled(authorized);
    this->sessionUpdateAllTimelines->setEnabled(authorized);

    if (authorized)
    {
        // TODO FIXME
    }
    else
    {
        // TODO
    }
}


void MainWindow::onInitializationComplete()
{
    this->sessionAutoUpdates->setEnabled(true);

    if (this->sessionAutoUpdates->isChecked())
    {
        this->setStateIcon(MainWindow::Autoupdating);
    }
    else
    {
        this->setStateIcon(MainWindow::Stopped);
    }

}


/*
 * Stuff executed 1 second after MainWindow is created
 *
 */
void MainWindow::postInit()
{
    qDebug() << "postInit();";

    this->adjustTimelineSizes();

    // Ask for proxy password if needed
    if (this->pumpController->needsProxyPassword())
    {
        QString proxyPassword = QInputDialog::getText(this, tr("Proxy password required"),
                                                      tr("You have configured a "
                                                         "proxy server with "
                                                         "authentication, but the "
                                                         "password is not set.")
                                                      + "\n\n"
                                                      + tr("Enter the password "
                                                           "for your proxy server:"),
                                                      QLineEdit::Password);
        // FIXME: if this is canceled, nothing will be loaded until program restart
        this->pumpController->setProxyPassword(proxyPassword);
    }
}




/*
 * Control interaction with the system tray icon
 *
 */
void MainWindow::trayControl(QSystemTrayIcon::ActivationReason reason)
{
    qDebug() << "Tray icon activation reason:" << reason;

    if (reason != QSystemTrayIcon::Context) // Simple "main button" click in icon
    {
        /*
        qDebug() << "trayControl()";
        qDebug() << "isHidden?"    << this->isHidden();
        qDebug() << "isVisible?"   << this->isVisible();
        qDebug() << "isMinimized?" << this->isMinimized();
        qDebug() << "hasFocus?"    << this->hasFocus();
        */

        // Hide or show the main window
        if (this->isMinimized())
        {
            // hide and show, because raise() wouldn't work
            this->hide();
            this->showNormal();
            qDebug() << "RAISING from minimized state";
        }
        else
        {
            this->toggleMainWindow();
        }
    }
}



/*
 * If FreeDesktop.org notifications are not available,
 * fall back to Qt's balloon ones
 *
 */
void MainWindow::showTrayFallbackMessage(QString title, QString message)
{
    this->trayIcon->showMessage(title,
                                message,
                                QSystemTrayIcon::Information,
                                4000); // 4 secs
}



void MainWindow::updateProfileData(QString avatarUrl, QString fullName,
                                   QString hometown, QString bio,
                                   QString eMail)
{
    QString bioTooltip = bio;
    if (!bio.isEmpty())
    {
        bioTooltip.prepend("<b></b>"); // make it rich text, so it gets wordwrap
        bioTooltip.replace("\n", "<br>"); // HTML newlines
    }
    else
    {
        bioTooltip = "<i>" + tr("Your biography is empty") + "</i>";
    }

    this->fullNameLabel->setText(fullName);
    this->fullNameLabel->setToolTip(bioTooltip);
    this->userIdLabel->setText(this->userID);
    this->userIdLabel->setToolTip(bioTooltip);
    this->userHometownLabel->setText(hometown);
    this->userHometownLabel->setToolTip(bioTooltip);
    qDebug() << "Updated profile data from server:" << fullName << " @" << hometown;

    this->avatarURL = avatarUrl;
    qDebug() << "Own avatar URL:" << avatarURL;


    // Get local file name, which is stored in base64 hash form
    QString avatarFilename = MiscHelpers::getCachedAvatarFilename(avatarURL);

    if (QFile::exists(avatarFilename))
    {
        // Load avatar if already cached
        this->avatarIconButton->setIcon(QIcon(QPixmap(avatarFilename)
                                              .scaled(64, 64,
                                                      Qt::KeepAspectRatio,
                                                      Qt::SmoothTransformation)));

        qDebug() << "Using cached avatar for user";
    }
    else
    {
        pumpController->getAvatar(avatarURL);
    }
    this->avatarIconButton->setToolTip(bioTooltip
                                       + "<br><hr>"
                                         "<b><i>"
                                       + tr("Click to edit your profile")
                                       + "</i></b>");


    // Fill/update this info in the profile editor too
    this->profileEditor->setProfileData(avatarURL, fullName,
                                        hometown, bio,
                                        eMail);
}

/*
 * Enable or disable the timer that auto-updates the timelines
 *
 */
void MainWindow::toggleAutoUpdates(bool checked)
{
    QString message;

    if (checked)
    {
        this->setStateIcon(MainWindow::Autoupdating);
        this->updateTimer->start();
        message = tr("Starting automatic update of timelines, "
                     "once every %1 minutes.").arg(this->updateInterval);
    }
    else
    {
        this->setStateIcon(MainWindow::Stopped);
        this->updateTimer->stop();
        message = tr("Stopping automatic update of timelines.");
    }

    this->setStatusBarMessage(message);
    this->logViewer->addToLog(message);
}


/*
 * Update all timelines
 *
 */
void MainWindow::updateAllTimelines()
{
    mainTimeline->goToFirstPage(); // received timeline will come in a SIGNAL()
    directTimeline->goToFirstPage();
    activityTimeline->goToFirstPage();
    favoritesTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated all timelines by menu";
}

/*
 * Update some of the timelines:
 *
 * Main, Direct messages, and Minor feed
 *
 */
void MainWindow::updateMainDirectMinorTimelines()
{
    mainTimeline->goToFirstPage();
    directTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated some timelines by menu or after:" << this->updateInterval << "min";
}


void MainWindow::updateMainActivityMinorTimelines()
{
    mainTimeline->goToFirstPage();
    activityTimeline->goToFirstPage();

    meanwhileFeed->updateFeed();

    qDebug() << "Updated some timelines by menu or after posting";
}



void MainWindow::scrollMainTimelineTo(QAbstractSlider::SliderAction sliderAction)
{
    this->mainTimelineScrollArea->verticalScrollBar()->triggerAction(sliderAction);

//    this->adjustTimelineSizes();
}

void MainWindow::scrollDirectTimelineTo(QAbstractSlider::SliderAction sliderAction)
{
    this->directTimelineScrollArea->verticalScrollBar()->triggerAction(sliderAction);
}

void MainWindow::scrollActivityTimelineTo(QAbstractSlider::SliderAction sliderAction)
{
    this->activityTimelineScrollArea->verticalScrollBar()->triggerAction(sliderAction);
}

void MainWindow::scrollFavoritesTimelineTo(QAbstractSlider::SliderAction sliderAction)
{
    this->favoritesTimelineScrollArea->verticalScrollBar()->triggerAction(sliderAction);
}

/*
 * Scroll timelines to make sure the commenter block of the post
 * currently being commented is shown
 *
 */
void MainWindow::scrollMainTimelineToWidget(QWidget *widget)
{
    this->mainTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}

void MainWindow::scrollDirectTimelineToWidget(QWidget *widget)
{
    this->directTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}

void MainWindow::scrollActivityTimelineToWidget(QWidget *widget)
{
    this->activityTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}

void MainWindow::scrollFavoritesTimelineToWidget(QWidget *widget)
{
    this->favoritesTimelineScrollArea->ensureWidgetVisible(widget, 1, 1);
}



void MainWindow::notifyTimelineUpdate(int timelineType,
                                      int newPostCount, int highlightCount)
{
    QString statusBarMessage = tr("Timeline updated.");

    if (newPostCount > 0)
    {
        QString timelineUpdatedAt = tr("Timeline updated at %1.")
                                    .arg(QTime::currentTime().toString());
        QString newPostsString;
        if (newPostCount == 1)
        {
            newPostsString = tr("There is 1 new post.");
        }
        else
        {
            newPostsString = tr("There are %1 new posts.").arg(newPostCount);
        }

        statusBarMessage.append(" " + newPostsString);

        QString hlPostsString;
        if (highlightCount > 0)
        {
            if (highlightCount == 1)
            {
                hlPostsString = tr("1 highlighted.",
                                   "singular, refers to a post");
            }
            else
            {
                hlPostsString = tr("%1 highlighted.",
                                   "plural, refers to posts").arg(highlightCount);
            }

            statusBarMessage.append(" " + hlPostsString);
        }

        this->setStatusBarMessage(statusBarMessage);

        // Only for the main timeline
        if (timelineType == TimeLine::TimelineTypeMain)
        {
            if (fdNotifier->getNotifyHLTimeline() && highlightCount > 0)
            {
                fdNotifier->showMessage(timelineUpdatedAt + "\n\n"
                                        + newPostsString + "\n"
                                        + hlPostsString);
            }
            else if (fdNotifier->getNotifyNewTimeline())
            {
                fdNotifier->showMessage(timelineUpdatedAt + "\n\n"
                                        + newPostsString);
            }

            this->logViewer->addToLog(statusBarMessage);
        }
    }
    else
    {
        statusBarMessage.append(" " + tr("No new posts."));
        this->setStatusBarMessage(statusBarMessage);
    }


    // In either case, restart update timer, so every manual update of timeline
    // will postpone the auto-update but ONLY IF autoupdates are enabled
    if (this->sessionAutoUpdates->isChecked())
    {
        this->updateTimer->stop();
        this->updateTimer->start();
    }

    // And adjust timelines sizes
    this->adjustTimelineSizes();
}


/*
 * Update timelines titles with number of new posts
 *
 */
void MainWindow::setTimelineTabTitle(int timelineType,
                                     int newPostCount, int highlightCount,
                                     int totalItemsCount)
{
    int updatedTab = 0;

    QString messageCountString;
    if (newPostCount > 0)
    {
        messageCountString = QString(" (%1)").arg(newPostCount);
        if (highlightCount > 0)
        {
            messageCountString.append(QString(" [%1]").arg(highlightCount));
        }
    }


    QString totalItemsString = "\n\n" + tr("Total posts: %1").arg(totalItemsCount);

    switch (timelineType)
    {
    case TimeLine::TimelineTypeMain:
        this->tabWidget->setTabText(0, tr("&Timeline") + messageCountString);
        this->tabWidget->setTabToolTip(0, tr("The main timeline")
                                          + totalItemsString);
        updatedTab = 0;
        break;

    case TimeLine::TimelineTypeDirect:
        this->tabWidget->setTabText(1, tr("&Messages") + messageCountString);
        this->tabWidget->setTabToolTip(1, tr("Messages sent explicitly to you")
                                          + totalItemsString);
        updatedTab = 1;
        break;

    case TimeLine::TimelineTypeActivity:
        this->tabWidget->setTabText(2, tr("&Activity") + messageCountString);
        this->tabWidget->setTabToolTip(2, tr("Your own posts")
                                          + totalItemsString);
        updatedTab = 2;
        break;

    case TimeLine::TimelineTypeFavorites:
        this->tabWidget->setTabText(3, tr("Favor&ites") + messageCountString);
        this->tabWidget->setTabToolTip(3, tr("Your favorited posts")
                                          + totalItemsString);
        updatedTab = 3;
        break;
    }

    // If the updated tab is the current one, set also window title, etc.
    if (updatedTab == tabWidget->currentIndex())
    {
        this->setTitleAndTrayInfo(updatedTab);
    }


    // If it's the main timeline, set the tray icon pixmap with the newmsg count
    if (updatedTab == 0)
    {
        if (trayIconAvailable)
        {
            this->setTrayIconPixmap(newPostCount,
                                    highlightCount);
        }
    }

}


/*
 * Set mainWindow's title based on current tab, and user ID
 *
 * Use that same title as tray icon's tooltip
 *
 */
void MainWindow::setTitleAndTrayInfo(int currentTab)
{
    QString currentTabTitle;
    currentTabTitle = this->tabWidget->tabText(currentTab);
    currentTabTitle.remove("&"); // Remove accelators

    QString title = currentTabTitle;
    if (!this->userID.isEmpty())
    {
        title.append(" - " + userID);
    }
    title.append(" - Dianara");

    /*  Not sure if I like it

    if (currentTabTitle.endsWith(")")) // kinda TMP
    {
        title.prepend("* ");
    }
    */

    this->setWindowTitle(title);
    if (trayIconAvailable)
    {
        QString idToShow = this->userID.isEmpty()
                           ? tr("Your Pump.io account is not configured")
                             : userID;
#ifdef Q_OS_UNIX
        title = "<b>Dianara</b>: "
              + currentTabTitle
              + "<br><br>["
              + idToShow
              + "]";
#else
        // Some OSes don't render HTML in the tray tooltip
        title = "Dianara: "
              + currentTabTitle
              + "\n\n["
              + idToShow
              + "]";
#endif


        this->trayIcon->setToolTip(title);

        this->trayTitleSeparatorAction->setText("Dianara - " + this->userID);
    }
}


/*
 * Set the title for the minor feed (Meanwhile), with new item count
 *
 */
void MainWindow::setMinorFeedTitle(int newItemsCount,
                                   int newHighlightedItemsCount)
{
    QString title = tr("Meanwhile...");
    if (newItemsCount > 0)
    {
        title.append(QString(" (%1)").arg(newItemsCount));
        if (newHighlightedItemsCount > 0)
        {
            title.append(QString(" [%1]").arg(newHighlightedItemsCount));
        }
    }

    this->leftPanel->setItemText(0, title);
}




void MainWindow::notifyMinorFeedUpdate(int newItemsCount, int highlightedCount)
{
    QString statusBarMessage = tr("Minor feed updated.");

    if (newItemsCount > 0)
    {
        QString mwUpdatedAt = tr("Minor feed updated at %1.")
                              .arg(QTime::currentTime().toString());

        QString newItemsString;
        if (newItemsCount == 1)
        {
            newItemsString = tr("There is 1 new activity.");
        }
        else
        {
            newItemsString = tr("There are %1 new activities.")
                             .arg(newItemsCount);
        }

        statusBarMessage.append(" " + newItemsString);

        if (highlightedCount > 0)
        {
            QString hlItemsString;
            if (highlightedCount == 1)
            {
                hlItemsString = tr("1 highlighted.",
                                   "singular, refers to an activity");
            }
            else
            {
                hlItemsString = tr("%1 highlighted.",
                                   "plural, refers to activities")
                                .arg(highlightedCount);
            }

            statusBarMessage.append(" " + hlItemsString);


            if (fdNotifier->getNotifyHLMeanwhile())
            {
                fdNotifier->showMessage(mwUpdatedAt + "\n\n"
                                        + newItemsString + "\n"
                                        + hlItemsString);
            }
        }
        else
        {
            if (fdNotifier->getNotifyNewMeanwhile())
            {
                fdNotifier->showMessage(mwUpdatedAt + "\n\n"
                                        + newItemsString);
            }

        }


        this->setStatusBarMessage(statusBarMessage);
        this->logViewer->addToLog(statusBarMessage);
    }
    else
    {
        statusBarMessage.append(" " + tr("No new activities."));
        this->setStatusBarMessage(statusBarMessage);
    }
}




/*
 * Store avatars on disk
 *
 */
void MainWindow::storeAvatar(QByteArray avatarData, QUrl avatarUrl)
{
    QString fileName = MiscHelpers::getCachedAvatarFilename(avatarUrl.toString());
    qDebug() << "Saving avatar to disk: " << fileName;

    QFile avatarFile(fileName);
    avatarFile.open(QFile::WriteOnly);
    avatarFile.write(avatarData);
    avatarFile.close();

    this->pumpController->notifyAvatarStored(avatarUrl.toString(),
                                             avatarFile.fileName());

    if (avatarUrl == this->avatarURL)
    {
        this->avatarIconButton->setIcon(QIcon(QPixmap(avatarFile.fileName())
                                              .scaled(64, 64,
                                                      Qt::KeepAspectRatio,
                                                      Qt::SmoothTransformation)));
    }

    qDebug() << "avatarData size:" << avatarData.size();
}





/*
 * Store images on disk
 *
 */
void MainWindow::storeImage(QByteArray imageData, QUrl imageUrl)
{
    QString fileName = MiscHelpers::getCachedImageFilename(imageUrl.toString());
    qDebug() << "Saving image to disk: " << fileName;

    QFile imageFile(fileName);
    imageFile.open(QFile::WriteOnly);
    imageFile.write(imageData);
    imageFile.close();

    this->pumpController->notifyImageStored(imageUrl.toString());

    qDebug() << "imageData size:" << imageData.size();
}



/*
 * Update status bar message, from pumpController's signals
 *
 */
void MainWindow::setStatusBarMessage(QString message)
{
    this->oldStatusBarMessage = "[" + QTime::currentTime().toString() + "] " + message;
    this->statusBar()->showMessage(oldStatusBarMessage, 0);
}


/*
 * Show a temporary message in the statusBar
 *
 * If message is empty, restore the old one
 *
 */
void MainWindow::setTransientStatusMessage(QString message)
{
    if (!message.isEmpty())
    {
        if (message.startsWith("http"))
        {
            message = tr("Link to: %1").arg(message);
        }
        this->statusBar()->showMessage(message, 0);
    }
    else
    {
        // Old status message was saved in setStatusBarMessage()
        this->statusBar()->showMessage(oldStatusBarMessage, 0);
    }
}


/*
 * Mark all posts in all timelines as read
 * and clear their counters
 *
 */
void MainWindow::markAllAsRead()
{
    mainTimeline->markPostsAsRead();
    directTimeline->markPostsAsRead();
    activityTimeline->markPostsAsRead();
    favoritesTimeline->markPostsAsRead();

    meanwhileFeed->markAllAsRead();
    this->setMinorFeedTitle(0, 0);
}


/*
 * Refresh all timestamps in the minor feed, in timelines posts,
 * and in comments
 *
 */
void MainWindow::refreshAllTimestamps()
{
    this->meanwhileFeed->updateFuzzyTimestamps();
    this->mainTimeline->updateFuzzyTimestamps();
    this->directTimeline->updateFuzzyTimestamps();
    this->activityTimeline->updateFuzzyTimestamps();
    this->favoritesTimeline->updateFuzzyTimestamps();
}



/*
 * Adjust timelines maximum widths according to their scrollareas sizes,
 * which in turn depend on the window size
 *
 * Called from the resizeEvent(), among other places
 *
 */
void MainWindow::adjustTimelineSizes()
{
    int timelineWidth;
    int timelineHeight;

    // Get the right timelinewidth based on currently active tab's timeline width
    switch (this->tabWidget->currentIndex())
    {
    case 0: // main
        timelineWidth = mainTimelineScrollArea->viewport()->width();
        timelineHeight = mainTimelineScrollArea->viewport()->height();
        mainTimeline->resizePosts();
        break;
    case 1: // direct
        timelineWidth = directTimelineScrollArea->viewport()->width();
        timelineHeight = directTimelineScrollArea->viewport()->height();
        directTimeline->resizePosts();
        break;
    case 2: // activity
        timelineWidth = activityTimelineScrollArea->viewport()->width();
        timelineHeight = activityTimelineScrollArea->viewport()->height();
        activityTimeline->resizePosts();
        break;
    case 3: // favorites
        timelineWidth = favoritesTimelineScrollArea->viewport()->width();
        timelineHeight = favoritesTimelineScrollArea->viewport()->height();
        favoritesTimeline->resizePosts();
        break;

    default: // Contacts tab, ATM
        timelineWidth = contactManager->visibleRegion().boundingRect().width();
                //->width() - scrollbarWidth;
        timelineHeight = contactManager->height();
    }

    // Then set the maximum width for all of them based on that
    mainTimeline->setMaximumWidth(timelineWidth);
    directTimeline->setMaximumWidth(timelineWidth);
    activityTimeline->setMaximumWidth(timelineWidth);
    favoritesTimeline->setMaximumWidth(timelineWidth);


    // Some basic 1:2 proportions TMP FIXME
    if (timelineHeight > (timelineWidth * 2))
    {
        timelineHeight = timelineWidth * 2;
    }
    mainTimeline->setMinMaxHeightForPosts(timelineHeight);
    directTimeline->setMinMaxHeightForPosts(timelineHeight);
    activityTimeline->setMinMaxHeightForPosts(timelineHeight);
    favoritesTimeline->setMinMaxHeightForPosts(timelineHeight);
}




/*
 * Hide or show the side panel
 *
 */
void MainWindow::toggleSidePanel(bool shown)
{
    // Unless the publisher (really, the composer) has focus already,
    // give focus to timeline before, to avoid giving focus to the publisher
    if (!this->publisher->hasFocus())
    {
        // FIXME: check if publisher's composer has focus!
        this->mainTimeline->setFocus();
    }

    qDebug() << "Showing side panel:" << shown;
    this->leftSideWidget->setVisible(shown);


    if (!shown)
    {
        qApp->processEvents();
        this->adjustTimelineSizes();
    }
}


void MainWindow::toggleToolbar(bool shown)
{
    qDebug() << "Showing toolbar:" << shown;
    this->mainToolBar->setVisible(shown);
}


/*
 * Hide or show the status bar
 *
 */
void MainWindow::toggleStatusBar(bool shown)
{
    qDebug() << "Showing status bar:" << shown;
    this->statusBar()->setVisible(shown);
}


/*
 * Toggle between fullscreen mode and normal window mode
 *
 */
void MainWindow::toggleFullscreen(bool enabled)
{
    if (enabled)
    {
        this->showFullScreen();
    }
    else
    {
        this->showNormal();
    }
}





/*
 * Open website in browser
 *
 */
void MainWindow::visitWebSite()
{
    qDebug() << "Opening website in browser";
    QDesktopServices::openUrl(QUrl("http://jancoding.wordpress.com/dianara"));
}

/*
 * Open pump.io's FAQ in web browser
 * (currently at GitHub)
 *
 */
void MainWindow::visitFAQ()
{
    qDebug() << "Opening Pump.io FAQ in browser";
    QDesktopServices::openUrl(QUrl("https://github.com/e14n/pump.io/wiki/FAQ"));
}

void MainWindow::visitTips()
{
    qDebug() << "Opening Pump.io tips in browser";
    QDesktopServices::openUrl(QUrl("http://communicationfreedom.wordpress.com/2014/03/17/pump-io-tips/"));
}

void MainWindow::visitUserList()
{
    qDebug() << "Opening JPope.org user list in browser";
    QDesktopServices::openUrl(QUrl("https://static.jpope.org/users.html"));
}




/*
 * About... message
 *
 */
void MainWindow::aboutDianara()
{
    QMessageBox::about(this, tr("About Dianara"),
                       "<b>Dianara v1.2.4</b><br>"
                       "Copyright 2012-2014  JanKusanagi<br><br>"
                       "<a href=\"http://jancoding.wordpress.com/dianara\">"
                       "http://jancoding.wordpress.com/dianara</a><br><br>"

                       + tr("Dianara is a pump.io social networking client.")
                       + "<br><br>"

                       + tr("With Dianara you can see your timelines, "
                            "create new posts, upload pictures and "
                            "other media, interact with posts, manage "
                            "your contacts and follow new people.")
                       + "<br><br>"

                       + tr("Thanks to all the testers, translators and packagers, "
                            "who help make Dianara better!")
                       + "<br><br>"

                       + tr("English translation by JanKusanagi.",
                            "TRANSLATORS: Change this with your language and name. "
                            "If there was another translator before you, add your "
                            "name after theirs ;)")
                       + "<br><br>"

                       + tr("Dianara is licensed under the GNU GPL license, and "
                            "uses some Oxygen icons: http://www.oxygen-icons.org/ "
                            "(LGPL licensed)")
                       + "<br><br>"
                         "<a href=\"http://www.gnu.org/licenses/gpl-2.0.html\">"
                         "GNU GPL v2</a> - "
                         "<a href=\"http://www.gnu.org/licenses/lgpl-2.1.html\">"
                         "GNU LGPL v2.1</a>");


    if (this->isHidden()) // FIXME: ugly workaround to avoid closing the program
    {                     // after closing About dialog, if mainWindow hidden
        this->show();
        this->hide();     // This hack might be causing problems under LXDE
        qDebug() << "MainWindow was hidden, showing and hiding again";
    }
}



void MainWindow::toggleMainWindow()
{
    if (this->isHidden())
    {
        this->trayShowWindowAction->setText(tr("&Hide Window"));
        this->show();
        qDebug() << "SHOWING";
    }
    else
    {
        this->trayShowWindowAction->setText(tr("&Show Window"));
        this->hide();
        qDebug() << "HIDING";
    }
}

void MainWindow::showAuthError(QString title, QString message)
{
    QMessageBox::warning(this, title, message);
}



/*
 * Close the program. Needed to quit correctly from context menu
 *
 */
void MainWindow::quitProgram()
{
    // Add more needed shutdown stuff here

    if (this->isHidden())
    {
        this->show();
    }

    saveSettings();


    reallyQuitProgram = false;
    if (this->publisher->isFullMode())  // If composer is active, ask for confirmation
    {                                   // FIXME: composing comments should also block here
        int confirmation = QMessageBox::question(this,
                              tr("Quit?"),
                              tr("You are composing a note or a comment.")
                              + "\n\n"
                              + tr("Do you really want to close Dianara?"),
                              tr("&Yes, close the program"), tr("&No"),
                              "", 1, 1);
        if (confirmation == 0)
        {
            qDebug() << "Exit confirmed; quitting, bye!";
            reallyQuitProgram = true;
        }
        else
        {
            qDebug() << "Confirmation canceled, not exiting";
        }

    }
    else
    {
        reallyQuitProgram = true;
    }


    if (reallyQuitProgram)
    {
        std::cout << "\nShutting down Dianara ("
                     + this->userID.toStdString()
                     + ")...\n";
        std::cout.flush();

        // Manually remove some stuff; trying to minimize shutdown time
        this->mainTimeline->clearTimeLineContents();
        this->directTimeline->clearTimeLineContents();
        this->activityTimeline->clearTimeLineContents();
        this->favoritesTimeline->clearTimeLineContents();
        this->meanwhileFeed->clearContents();

        this->publisher->deleteLater();
        this->contactManager->deleteLater();

        qApp->processEvents(QEventLoop::ExcludeUserInputEvents
                          | QEventLoop::ExcludeSocketNotifiers);

        qApp->closeAllWindows();
        //qApp->quit();

        //qApp->processEvents();
        std::cout << "All windows closed, bye! o/\n";
        std::cout.flush();
    }
}




//////////////////////////////////////////////////////////////////////////////
///////////////////////////////// PROTECTED //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void MainWindow::closeEvent(QCloseEvent *event)
{
    qDebug() << "MainWindow::closeEvent()";

    // FIXME: handle cases where tray icon isn't available

    if (reallyQuitProgram)
    {
        event->accept(); // really close, if called from Quit menu
        qDebug() << "Quit called from menu, shutting down program";
    }
    else
    {
        this->hide();     // Hide window, app accessible via tray icon
        qDebug() << "Tried to close main window, so hiding to tray";
        event->ignore();  // ignore the closeEvent
    }
}


void MainWindow::resizeEvent(QResizeEvent *event)
{
    qDebug() << "MainWindow::resizeEvent()" << event->size();

    delayedResizeTimer->stop();
    delayedResizeTimer->start(500); // call adjustTimelineSizes() a little later

    event->accept();
}
