/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "pumpcontroller.h"

PumpController::PumpController(QObject *parent) :  QObject(parent)
{
    this->userAgentString = "Dianara/1.2.4";

    this->postsPerPageMain = 20;
    this->postsPerPageOther = 10;
    this->updatesToTimelineBlocked = false;

    this->proxyUsesAuth = false;

    qoauth = new QOAuth::Interface();
    qoauth->setRequestTimeout(10000); // 10 sec timeout

    QSettings settings;
    this->clientID = settings.value("clientID").toString();
    this->clientSecret = settings.value("clientSecret").toString();
    qoauth->setConsumerKey(clientID.toLocal8Bit());
    qoauth->setConsumerSecret(clientSecret.toLocal8Bit());


    this->isApplicationAuthorized = settings.value("isApplicationAuthorized", false).toBool();

    if (isApplicationAuthorized)
    {
        qDebug() << "Dianara is already authorized for user ID:" << settings.value("userID").toString();

        this->token = settings.value("token").toString().toLocal8Bit();
        this->tokenSecret = settings.value("tokenSecret").toString().toLocal8Bit();

        qDebug() << "Using token" << token;
        qDebug() << "And token secret" << tokenSecret.left(5) + "********** (hidden)";
    }



    connect(&nam, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(requestFinished(QNetworkReply*)));

    // FIXME: setting this up for now, to at least have debug messages just in case
    connect(&nam, SIGNAL(sslErrors(QNetworkReply*,QList<QSslError>)),
            this, SLOT(sslErrorsHandler(QNetworkReply*,QList<QSslError>)));


    this->initialDataStep = 0;
    this->initialDataAttempts = 0;

    initialDataTimer = new QTimer(this);
    initialDataTimer->setSingleShot(false); // Triggered constantly until stopped
    connect(initialDataTimer, SIGNAL(timeout()),
            this, SLOT(getInitialData()));



    qDebug() << "PumpController created";
}


PumpController::~PumpController()
{
    qDebug() << "PumpController destroyed";
}


void PumpController::setProxyConfig(QNetworkProxy::ProxyType proxyType,
                                    QString hostname, int port,
                                    bool useAuth,
                                    QString user, QString password)
{
    QNetworkProxy proxy;
    proxy.setType(proxyType);
    proxy.setHostName(hostname);
    proxy.setPort(port);
    if (useAuth)
    {
        proxy.setUser(user);
        proxy.setPassword(password);

        // If using auth, and proxy type is set
        if (proxyType != QNetworkProxy::NoProxy)
        {
            this->proxyUsesAuth = true;
        }
    }
    nam.setProxy(proxy);
    qoauth->networkAccessManager()->setProxy(proxy);

    qDebug() << "Proxy config applied:" << hostname << port << user;
}

bool PumpController::needsProxyPassword()
{
    if (this->proxyUsesAuth && this->nam.proxy().password().isEmpty())
    {
        return true;
    }

    return false;
}

void PumpController::setProxyPassword(QString password)
{
    QNetworkProxy proxy = this->nam.proxy();
    proxy.setPassword(password);
    this->nam.setProxy(proxy);
    this->qoauth->networkAccessManager()->setProxy(proxy);
}



void PumpController::setPostsPerPageMain(int ppp)
{
    this->postsPerPageMain = ppp;
    qDebug() << "PumpController: setting postsPerPage (main) to" << this->postsPerPageMain;
}

void PumpController::setPostsPerPageOther(int ppp)
{
    this->postsPerPageOther = ppp;
    qDebug() << "PumpController: setting postsPerPage (other) to" << this->postsPerPageOther;

}

/*
 * Block timeline updates. Used while commenting.
 *
 */
void PumpController::setUpdatesToTimelineBlocked(bool blocked)
{
    this->updatesToTimelineBlocked = blocked;
    qDebug() << "PumpController: Updates to timeline blocked? " << this->updatesToTimelineBlocked;
}



/*
 * Set new user ID (user@domain.tld) and clear OAuth-related tokens/secrets
 *
 *
 */
void PumpController::setNewUserId(QString userID)
{
    this->userId = userID;
    QStringList splittedUserID = this->userId.split("@");
    this->userName = splittedUserID.at(0); // get username, before @
    this->serverURL = splittedUserID.at(1); // get URL, after @

    qDebug() << "Server URL to connect:" << serverURL << "; username:" << userName;

    this->clientID.clear();
    this->clientSecret.clear();
    this->token.clear();
    this->tokenSecret.clear();

    this->isApplicationAuthorized = false;
    emit this->authorizationStatusChanged(isApplicationAuthorized);
}




/*
 * Get "pumpserver.org" and "user" from "user@pumpserver.org", set OAuth token from Account dlg
 *
 */
void PumpController::setUserCredentials(QString userID)
{
    this->initialDataTimer->stop(); // Just in case it was running before


    this->userId = userID;
    QStringList splittedUserID = this->userId.split("@");
    this->userName = splittedUserID.at(0);
    this->serverURL = splittedUserID.at(1);
    qDebug() << "New userID is:" << this->userId;


    emit this->authorizationStatusChanged(isApplicationAuthorized);

    this->haveProfile = false;
    this->haveFollowing = false;
    this->haveFollowers = false;
    this->havePersonLists = false;
    this->haveMainTL = false;
    this->haveDirectTL = false;
    this->haveActivityTL = false;
    this->haveFavoritesTL = false;
    this->haveMinorFeed = false;

    this->initialDataStep = 0;
    this->initialDataAttempts = 0;

    // This will call getUserProfile(), getContactList(), getMainTimeline()...
    if (isApplicationAuthorized)
    {
        this->initialDataTimer->start(2000);  // start 2 seconds after setting the ID
                                              // (mainly on program startup)
        emit logMessage(tr("Authorized to use account %1. Getting initial data.")
                        .arg(this->userId));
    }
    else
    {
        emit logMessage(tr("There is no authorized account."));
    }
}



QString PumpController::currentUserId()
{
    return this->userId;
}

QString PumpController::currentUsername()
{
    return this->userName;
}

QString PumpController::currentFollowersUrl()
{
    return this->userFollowersURL;
}

int PumpController::currentFollowersCount()
{
    return this->userFollowersCount;
}

int PumpController::currentFollowingCount()
{
    return this->userFollowingCount;
}




/*
 * Get any user's profile (not only our own)
 *
 * GET https://pumpserver.example/api/user/username
 *
 */
void PumpController::getUserProfile(QString userID)
{
    QStringList splittedUserID = userID.split("@");

    QString url = "https://" + splittedUserID.at(1) + "/api/user/" + splittedUserID.at(0);

    QNetworkRequest userProfileRequest = this->prepareRequest(url, QOAuth::GET,
                                                              UserProfileRequest);
    nam.get(userProfileRequest);

    qDebug() << "Requested user profile:" << userProfileRequest.url().toString();
}



/*
 * Update user's profile
 *
 */
void PumpController::updateUserProfile(QString avatarUrl, QString fullName,
                                       QString hometown, QString bio)
{
    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/profile";

    QNetworkRequest updateProfileRequest = this->prepareRequest(url, QOAuth::PUT,
                                                                UpdateProfileRequest);


    QVariantMap jsonVariantImage;
    jsonVariantImage.insert("url", avatarUrl);
    jsonVariantImage.insert("width",  90);   // FIXME: don't hardcode this
    jsonVariantImage.insert("height", 90);   // get values from actual pixmap

    QVariantMap jsonVariantLocation;
    jsonVariantLocation.insert("objectType", "place");
    jsonVariantLocation.insert("displayName", hometown);

    QVariantMap jsonVariant;
    jsonVariant.insert("objectType", "person");

    if (!avatarUrl.isEmpty()) // Only add image object if a new image was uploaded
    {
        jsonVariant.insert("image", jsonVariantImage);
    }
    jsonVariant.insert("displayName", fullName);
    jsonVariant.insert("location", jsonVariantLocation);
    jsonVariant.insert("summary", bio);


    QByteArray data = this->prepareJSON(jsonVariant);
    nam.put(updateProfileRequest, data);
    qDebug() << "Updating user profile" << fullName << hometown;
}


/*
 * Add an avatar URL to the queue of pending
 *
 */
void PumpController::enqueueAvatarForDownload(QString url)
{
    if (QFile::exists(MiscHelpers::getCachedAvatarFilename(url))
     || pendingAvatarsList.contains(url))
    {
        qDebug() << "PumpController() Using cached avatar, or it is pending download...";
    }
    else
    {
        this->pendingAvatarsList.append(url);
        this->getAvatar(url);
        qDebug() << "PumpController() Avatar not cached, downloading" << url;
    }
}



/*
 * Add the URL of an image to the queue of pending-download
 *
 * connect() signal/slot if necessary to refresh when done
 *
 */
void PumpController::enqueueImageForDownload(QString url)
{
    if (QFile::exists(MiscHelpers::getCachedImageFilename(url))
     || pendingImagesList.contains(url))
    {
        qDebug() << "PumpController::enqueueImageForDownload(), "
                    "Using cached image, or requested image is pending download...";
    }
    else
    {
        this->pendingImagesList.append(url);
        this->getImage(url);

        qDebug() << "PumpController::enqueueImageForDownload(), "
                    "image not cached, downloading" << url;
    }
}




void PumpController::getAvatar(QString avatarUrl)
{
    if (avatarUrl.isEmpty())
    {
        return;
    }

    qDebug() << "Getting avatar" << avatarUrl;

    QNetworkRequest avatarRequest(QUrl((const QString)avatarUrl));
    avatarRequest.setRawHeader("User-Agent", userAgentString);
    avatarRequest.setAttribute(QNetworkRequest::User,
                               QVariant(AvatarRequest));

    nam.get(avatarRequest);
}



void PumpController::getImage(QString imageUrl)
{
    if (imageUrl.isEmpty())
    {
        return;
    }

    QNetworkRequest imageRequest = this->prepareRequest(imageUrl,
                                                        QOAuth::GET,
                                                        ImageRequest);
    nam.get(imageRequest);
    qDebug() << "getImage() imageRequest sent";
}


QNetworkReply *PumpController::getMedia(QString mediaUrl)
{
    QNetworkRequest mediaRequest = this->prepareRequest(mediaUrl,
                                                        QOAuth::GET,
                                                        MediaRequest);

    qDebug() << "getMedia() sending mediaRequest for:" << mediaUrl;
    return nam.get(mediaRequest);
}



void PumpController::notifyAvatarStored(QString avatarUrl, QString avatarFilename)
{
    pendingAvatarsList.removeAll(avatarUrl);

    emit avatarStored(avatarUrl, avatarFilename);
}

void PumpController::notifyImageStored(QString imageUrl)
{
    pendingImagesList.removeAll(imageUrl);

    emit imageStored(imageUrl);
}




/*
 * GET https://pumpserver.example/api/user/username/following or /followers
 *
 */
void PumpController::getContactList(QString listType, int offset)
{
    qDebug() << "Getting contact list, type" << listType << "; offset:" << offset;

    QString url = "https://" + this->serverURL + "/api/user/"
                  + this->userName + "/" + listType;

    QOAuth::ParamMap paramMap;
    paramMap.insert("count",  "200"); // 200 each time
    paramMap.insert("offset", QString("%1").arg(offset).toLocal8Bit());


    QNetworkRequest contactListRequest;
    if (listType == "following")
    {
        contactListRequest = this->prepareRequest(url, QOAuth::GET,
                                                  FollowingListRequest,
                                                  paramMap);
        if (offset == 0)
        {
            totalReceivedFollowing = 0;
            followingIdList.clear();
            this->showStatusMessageAndLogIt(tr("Getting list of 'Following'..."));
        }
    }
    else
    {
        contactListRequest = this->prepareRequest(url, QOAuth::GET,
                                                  FollowersListRequest,
                                                  paramMap);
        if (offset == 0)
        {
            totalReceivedFollowers = 0;
        }
        this->showStatusMessageAndLogIt(tr("Getting list of 'Followers'..."));
    }

    nam.get(contactListRequest);
}


/*
 * Check if a userID is in the "following" list
 *
 */
bool PumpController::userInFollowing(QString contactId)
{
    if (followingIdList.contains(contactId))
    {
        return true;
    }
    else
    {
        return false;
    }
}


void PumpController::updateInternalFollowingIdList(QStringList idList)
{
    this->followingIdList.append(idList);
}

void PumpController::removeFromInternalFollowingList(QString id)
{
    this->followingIdList.removeAll(id);
}



/*
 * GET https://pumpserver.example/api/user/username/lists/person
 *
 */
void PumpController::getListsList()
{
    qDebug() << "Getting list of lists";

    QString url = "https://" + this->serverURL + "/api/user/" +this->userName + "/lists/person";

    QNetworkRequest listsListRequest = this->prepareRequest(url, QOAuth::GET,
                                                            ListsListRequest);

    emit currentJobChanged(tr("Getting list of person lists..."));

    nam.get(listsListRequest);
}


/*
 * Create a person list
 *
 */
void PumpController::createPersonList(QString name, QString description)
{
    qDebug() << "PumpController() creating person list:" << name;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       CreatePersonListRequest);


    QVariantList jsonVariantObjectTypes;
    jsonVariantObjectTypes << "person";

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "collection");
    jsonVariantObject.insert("objectTypes", jsonVariantObjectTypes);
    jsonVariantObject.insert("displayName", name);
    jsonVariantObject.insert("content", description);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "create");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;


    emit currentJobChanged(tr("Creating person list..."));
    nam.post(postRequest, data);
}


void PumpController::deletePersonList(QString id)
{
    qDebug() << "PumpController::deletePersonList() deleting list" << id;

    QNetworkRequest deleteRequest = this->prepareRequest(id, QOAuth::DELETE,
                                                         DeletePersonListRequest);

    emit currentJobChanged(tr("Deleting person list..."));
    nam.deleteResource(deleteRequest);
}


void PumpController::getPersonList(QString url)
{
    qDebug() << "Getting a person list:" << url;

    QOAuth::ParamMap paramMap;
    paramMap.insert("count",  "200"); // Get 200 members

    QNetworkRequest personListRequest = this->prepareRequest(url, QOAuth::GET,
                                                             PersonListRequest,
                                                             paramMap);
    emit currentJobChanged(tr("Getting a person list..."));

    nam.get(personListRequest);
}



/*
 * Add a new member to a list
 *
 */
void PumpController::addPersonToList(QString listId, QString personId)
{
    qDebug() << "PumpController() adding person to list:" << personId << listId;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       AddMemberToListRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", personId);

    QVariantMap jsonVariantTarget;
    jsonVariantTarget.insert("objectType", "collection");
    jsonVariantTarget.insert("id", listId);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "add");
    jsonVariant.insert("object", jsonVariantObject);
    jsonVariant.insert("target", jsonVariantTarget);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;


    emit currentJobChanged(tr("Adding person to list..."));
    nam.post(postRequest, data);
}



/*
 * Remove member from a list
 *
 */
void PumpController::removePersonFromList(QString listId, QString personId)
{
    qDebug() << "PumpController() removing person from list:" << personId << listId;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       RemoveMemberFromListRequest);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", personId);

    QVariantMap jsonVariantTarget;
    jsonVariantTarget.insert("objectType", "collection");
    jsonVariantTarget.insert("id", listId);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "remove");
    jsonVariant.insert("object", jsonVariantObject);
    jsonVariant.insert("target", jsonVariantTarget);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;


    emit currentJobChanged(tr("Removing person from list..."));
    nam.post(postRequest, data);
}



/*
 * Create a group where users can join and post messages for the other
 * members, similar to the StatusNet groups.
 *
 */
void PumpController::createGroup(QString name,
                                 QString summary,
                                 QString description)
{
    qDebug() << "PumpController() creating group:" << name;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       CreateGroupRequest);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "group");
    if (!name.isEmpty())
    {
        jsonVariantObject.insert("displayName", name);
    }
    jsonVariantObject.insert("summary", summary);
    jsonVariantObject.insert("content", description);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "create");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;

    emit currentJobChanged(tr("Creating group..."));
    nam.post(postRequest, data);
}


/*
 * Join a group to be able to send messages to it.
 * Joining based on ID.
 *
 */
void PumpController::joinGroup(QString id)
{
    qDebug() << "PumpController() joining group:" << id;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       JoinGroupRequest);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "group");
    jsonVariantObject.insert("id", id);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "join");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;

    emit currentJobChanged(tr("Joining group..."));
    nam.post(postRequest, data);
}


void PumpController::leaveGroup(QString id)
{
    qDebug() << "PumpController() leaving group:" << id;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       LeaveGroupRequest);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "group");
    jsonVariantObject.insert("id", id);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "leave");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;

    emit currentJobChanged(tr("Leaving group..."));
    nam.post(postRequest, data);
}




/*
 * Get main timeline
 *
 * GET https://pumpserver.example/api/username/inbox/major
 *
 */
void PumpController::getMainTimeline(int timelineOffset)
{
    if (this->updatesToTimelineBlocked)
    {
        emit currentJobChanged(tr("Main timeline update requested, "
                                  "but updates are blocked."));
        qDebug() << "Updating timelines requested, but updates are blocked; ignoring";
        return;
    }

    emit currentJobChanged(tr("Getting main timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/major";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count",  QString("%1").arg(this->postsPerPageMain).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           MainTimelineRequest,
                                                           paramMap);

/*
    qDebug() << "===================================";
    qDebug() << "Getting timeline" << url;
    qDebug() << "with params:" << paramMap;
    qDebug() << "\nAuthorization Header:" << timelineRequest.rawHeader("Authorization");
    qDebug() << "\n\nFinal URL to retrieve:" << timelineRequest.url().toString();
    qDebug() << "===================================";
*/
    nam.get(timelineRequest);
}



/*
 * Get direct timeline, posts with the user's address in the "To:" field,
 * that is, sent explicitly to the user
 *
 * GET https://pumpserver.example/api/username/inbox/direct
 *
 */
void PumpController::getDirectTimeline(int timelineOffset)
{
    if (this->updatesToTimelineBlocked)
    {
        emit currentJobChanged(tr("Direct timeline update requested, "
                                  "but updates are blocked."));
        return;
    }

    emit currentJobChanged(tr("Getting direct messages timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/direct/major";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPageOther).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           DirectTimelineRequest,
                                                           paramMap);

    nam.get(timelineRequest);
}



/*
 * Get activity timeline, user's own posts
 *
 * GET https://pumpserver.example/api/username/feed/major
 *
 */
void PumpController::getActivityTimeline(int timelineOffset)
{
    if (this->updatesToTimelineBlocked)
    {
        emit currentJobChanged(tr("Activity timeline update requested, "
                                  "but updates are blocked."));
        return;
    }

    emit currentJobChanged(tr("Getting activity timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed/major";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPageOther).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           ActivityTimelineRequest,
                                                           paramMap);

    nam.get(timelineRequest);
}



/*
 * Get favorites timeline, posts where user clicked "like"
 *
 * GET https://pumpserver.example/api/username/favorites
 *
 */
void PumpController::getFavoritesTimeline(int timelineOffset)
{
    if (this->updatesToTimelineBlocked)
    {
        emit currentJobChanged(tr("Favorites timeline update requested, "
                                  "but updates are blocked."));
        return;
    }

    emit currentJobChanged(tr("Getting favorites timeline..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/favorites";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", QString("%1").arg(this->postsPerPageOther).toLocal8Bit());
    paramMap.insert("offset", QString("%1").arg(timelineOffset).toLocal8Bit());

    QNetworkRequest timelineRequest = this->prepareRequest(url, QOAuth::GET,
                                                           FavoritesTimelineRequest,
                                                           paramMap);

    nam.get(timelineRequest);
}



void PumpController::getTimeline(int timelineType, QString url)
{
    if (this->updatesToTimelineBlocked)
    {
        emit currentJobChanged(tr("Timeline update requested, "
                                  "but updates are blocked."));
        return;
    }

    emit currentJobChanged(tr("Getting timeline..."));

    QStringList urlParts = url.split("?");
    QString baseUrl = urlParts.at(0);
    QStringList pageParameter;
    if (urlParts.length() > 1)
    {
        pageParameter = urlParts.at(1).split("="); // since/before=http...
    }

    // 'since' or 'before'
    QByteArray sinceBeforeWord;
    sinceBeforeWord = pageParameter.at(0).toLocal8Bit();

    // The url parameter
    QByteArray sinceBeforeUrl;
    sinceBeforeUrl = pageParameter.at(1).toLocal8Bit();
                     //QByteArray::fromPercentEncoding(pageParameter.at(1).toLocal8Bit());


    QOAuth::ParamMap paramMap;
    paramMap.insert(sinceBeforeWord, sinceBeforeUrl); // since=http...
    paramMap.insert("count", QString("%1").arg(this->postsPerPageMain).toLocal8Bit());


    QNetworkRequest timelineRequest = this->prepareRequest(baseUrl, QOAuth::GET,
                                                           MainTimelineRequest,
                                                           paramMap);

    qDebug() << "\n\n==== getTimeLine() =========================================";
    qDebug() << "===================================";
    qDebug() << "Getting timeline" << url;
    qDebug() << "Base URL:" << baseUrl;
    qDebug() << "with params:" << paramMap;
    qDebug() << "\n\nAuthorization Header:" << timelineRequest.rawHeader("Authorization");
    qDebug() << "\n\nFinal URL to retrieve:" << timelineRequest.url().toString();
    qDebug() << "===================================";

    nam.get(timelineRequest);
}





/*
 * Get list of people who liked a specific post
 *
 */
void PumpController::getPostLikes(QString postLikesUrl)
{
    qDebug() << "Getting likes for post" << postLikesUrl;

    emit currentJobChanged(tr("Getting likes..."));

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "100"); // TMP, up to 100 likes

    QNetworkRequest likesRequest = this->prepareRequest(postLikesUrl,
                                                        QOAuth::GET,
                                                        PostLikesRequest,
                                                        paramMap);
    nam.get(likesRequest);
}



/*
 * Get comments for one specific post
 *
 * GET https://pumpserver.example/api/note/#id#/replies
 * or proxyed URL. URL is given by the post itself anyway
 *
 */
void PumpController::getPostComments(QString postCommentsUrl)
{
    qDebug() << "Getting comments for post" << postCommentsUrl;

    emit currentJobChanged(tr("Getting comments..."));


    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "200"); // TMP, up to 200 comments / FIXME?

    QNetworkRequest commentsRequest = this->prepareRequest(postCommentsUrl,
                                                           QOAuth::GET,
                                                           PostCommentsRequest,
                                                           paramMap);

    nam.get(commentsRequest);
}


/*
 * Get list of people who shared a specific post
 *
 */
void PumpController::getPostShares(QString postSharesUrl)
{
    // TODO
}



/*
 * Get the minor feed, used in the "Meanwhile" column
 *
 * GET https://pumpserver.example/api/username/inbox/minor
 *
 */
void PumpController::getMinorFeed(int offset)
{
    emit currentJobChanged(tr("Getting minor feed..."));

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/minor";

    QOAuth::ParamMap paramMap;
    paramMap.insert("count", "50");
    paramMap.insert("offset", QString("%1").arg(offset).toLocal8Bit());

    QNetworkRequest feedRequest = this->prepareRequest(url, QOAuth::GET,
                                                       MinorFeedRequest,
                                                       paramMap);
    nam.get(feedRequest);
}


void PumpController::getMinorFeedProper(QString url)
{
    emit currentJobChanged(tr("Getting minor feed...") + " [PROPER/testing]"); // FIXME

    QOAuth::ParamMap paramMap;
    //paramMap.insert("count", "50");

    if (url.isEmpty())
    {
        url = "https://" + this->serverURL + "/api/user/" + this->userName + "/inbox/minor";
    }
    else
    {
        QStringList splitUrl = url.split("?");
        url = splitUrl.at(0);
        QStringList splitParams = splitUrl.at(1).split("=");

        paramMap.insert(splitParams.at(0).toUtf8(),
                        //QString("https://blah").toUtf8());
                        //QByteArray::fromPercentEncoding(splitParams.at(1).toLocal8Bit()));
                        splitParams.at(1).toUtf8());
    }

    QNetworkRequest feedRequest = this->prepareRequest(url, QOAuth::GET,
                                                       MinorFeedRequest,
                                                       paramMap);

    qDebug() << "****\nfeedRequest:\n";
    qDebug() << feedRequest.rawHeader("Authorization");
    qDebug() << "\n*\n\nURL: " << feedRequest.url().toString() << "\n\n*******";

    nam.get(feedRequest);
}



/*
 * Prepare a QNetworkRequest with OAuth header, content type and user agent.
 *
 */
QNetworkRequest PumpController::prepareRequest(QString url, QOAuth::HttpMethod method,
                                               int requestType, QOAuth::ParamMap paramMap,
                                               QString contentTypeString)
{
    QByteArray authorizationHeader = qoauth->createParametersString(url,
                                                                 method,
                                                                 this->token,
                                                                 this->tokenSecret,
                                                                 QOAuth::HMAC_SHA1,
                                                                 paramMap,
                                                                 QOAuth::ParseForHeaderArguments);
    //qDebug() << "QOAuth::error()" << qoauth->error() << " (200=OK)";

    QNetworkRequest request;

    // Don't append inline parameters if they're empty, that can mess up things
    if (!paramMap.isEmpty())
    {
        url.append(qoauth->inlineParameters(paramMap,
                                            QOAuth::ParseForInlineQuery));
    }

    request.setUrl(QUrl(url));

    // Only add Authorization header if we're requesting something in our server
    if (request.url().host() == this->serverURL)
    {
        request.setRawHeader("Authorization", authorizationHeader);
    }
    request.setHeader(QNetworkRequest::ContentTypeHeader, contentTypeString);
    request.setRawHeader("User-Agent", userAgentString);

    request.setAttribute(QNetworkRequest::User, QVariant(requestType));

    return request;
}


/*
 * Generate JSON plaintext from a VarianMap.
 * Uses QJSON at this point, but will use the integrated Qt5 method
 * when building with Qt5.
 *
 */
QByteArray PumpController::prepareJSON(QVariantMap jsonVariantMap)
{
    QJson::Serializer serializer;
    // bool ok;
    // QByteArray data = serializer.serialize(jsonVariantMap, &ok);

    // Without using the bool, to make it work with QJSON 0.7.x
    return serializer.serialize(jsonVariantMap);
}


/*
 * Upload a file to the /uploads feed for the user
 *
 * Used to upload pictures, audio, video and misc files
 *
 */
QNetworkReply *PumpController::uploadFile(QString filename,
                                          QString contentType,
                                          int uploadType)
{
    qDebug() << "PumpController::uploadFile()" << filename << contentType;
    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/uploads";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       uploadType,
                                                       QOAuth::ParamMap(),
                                                       contentType);

    QFile file(filename);
    file.open(QIODevice::ReadOnly);
    QByteArray data = file.readAll();  // FIXME: This can use a lot of RAM with big files
    file.close();

    this->showStatusMessageAndLogIt(tr("Uploading %1",
                                       "1=filename").arg(filename)
                                    + QString(" (%1, %2)")
                                      .arg(contentType)
                                      .arg(MiscHelpers::fileSizeString(filename)));

    return nam.post(postRequest, data);
}



QList<QVariantList> PumpController::processAudience(QMap<QString, QString> audienceMap)
{
    QVariantList jsonVariantTo;
    QVariantList jsonVariantCC;

    QVariantMap jsonVariantAudienceItem;
    while (!audienceMap.isEmpty())
    {
        jsonVariantAudienceItem.clear();

        if (audienceMap.keys().contains("to|collection")) // To:
        {
            QString collectionID = audienceMap.take("to|collection");
            jsonVariantAudienceItem.insert("objectType", "collection");
            jsonVariantAudienceItem.insert("id", collectionID);

            jsonVariantTo.append(jsonVariantAudienceItem);
        }
        else if (audienceMap.keys().contains("to|person"))
        {
            QString personID = audienceMap.take("to|person");
            jsonVariantAudienceItem.insert("objectType", "person");
            jsonVariantAudienceItem.insert("id", personID);

            jsonVariantTo.append(jsonVariantAudienceItem);
        }
        else if (audienceMap.keys().contains("cc|collection")) // CC:
        {
            QString collectionID = audienceMap.take("cc|collection");
            jsonVariantAudienceItem.insert("objectType", "collection");
            jsonVariantAudienceItem.insert("id", collectionID);

            jsonVariantCC.append(jsonVariantAudienceItem);
        }
        else if (audienceMap.keys().contains("cc|person"))
        {
            QString personID = audienceMap.take("cc|person");
            jsonVariantAudienceItem.insert("objectType", "person");
            jsonVariantAudienceItem.insert("id", personID);

            jsonVariantCC.append(jsonVariantAudienceItem);
        }
    }


    QList<QVariantList> jsonVariantToAndCCList;
    jsonVariantToAndCCList.append(jsonVariantTo);
    jsonVariantToAndCCList.append(jsonVariantCC);


    return jsonVariantToAndCCList;
}


void PumpController::showTransientMessage(QString message)
{
    emit transientStatusBarMessage(message);
}


void PumpController::showStatusMessageAndLogIt(QString message)
{
    emit currentJobChanged(message);
    emit logMessage(message);
}



/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/*********************************** SLOTS *********************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/




void PumpController::requestFinished(QNetworkReply *reply)
{
    int httpCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();
    int requestType = reply->request().attribute(QNetworkRequest::User).toInt();
    bool finished = reply->isFinished();
    QString replyUrl = reply->url().toString();
    QString replyHost = reply->url().host();
    QString replyErrorString = reply->errorString();
    QByteArray replyFirstLine = reply->readLine();
    QByteArray replyData = replyFirstLine + reply->readAll();
    QString prettyLogMessage;

    qDebug() << "Request finished. HTTP code:" << httpCode;
    qDebug() << "Size:" << replyData.length() << "bytes; URL:" << replyUrl;
    qDebug() << "isFinished()?" << finished << "; Request type:" << requestType;

    // We got all necessary data, clean up
    reply->deleteLater();


    // Special control after sending a post or a comment
    if (httpCode != 200) // if not OK
    {
        if (requestType == PublishPostRequest)
        {
            emit postPublishingFailed();
        }
        else if (requestType == UpdatePostRequest)
        {
            emit postPublishingFailed();  // kinda TMP
        }
        else if (requestType == CommentPostRequest)
        {
            emit commentPostingFailed();
        }
        else if (requestType == UpdateCommentRequest)
        {
            emit commentPostingFailed(); // also, kinda TMP
        }
        else if (requestType == UploadMediaForPostRequest)
        {
            emit postPublishingFailed(); // FIXME?
        }
        else if (requestType == MediaRequest)
        {
            emit downloadFailed(replyUrl);
        }
    }


    const QString httpErrorString = tr("HTTP error",
                                       "For the following HTTP error codes"
                                       "you can check "
                                       "http://en.wikipedia.org/wiki/List_of_HTTP_status_codes "
                                       "in your language") + ": ";
    QString errorTypeString;

    switch (httpCode)
    {
    //////////////////////////////////////////////// First, handle error codes
    case 504:
        errorTypeString = httpErrorString
                        + tr("Gateway Timeout", "HTTP 504 error string")
                        + " (504)";
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "HTTP 504: Gateway Timeout.";
        qDebug() << "Data:  " << replyData;
        return;

    case 503:
        errorTypeString = httpErrorString
                        + tr("Service Unavailable", "HTTP 503 error string")
                        +  " (503) " + replyFirstLine;

        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);

        // if not just an image, it's important, so popup notification too
        if (requestType != ImageRequest && requestType != AvatarRequest)
        {
            emit showNotification(errorTypeString + "\n" + replyUrl);
        }

        qDebug() << "HTTP 503: Service Unavailable.";
        qDebug() << "Data:  " << replyData;
        return;

    case 500:
        errorTypeString = httpErrorString
                        + tr("Internal Server Error", "HTTP 500 error string")
                        + " (500) " + replyFirstLine;

        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);

        // if not just an image, it's important, so popup notification too
        if (requestType != ImageRequest && requestType != AvatarRequest)
        {
            emit showNotification(errorTypeString + "\n"
                                  + replyUrl);
        }

        qDebug() << "HTTP 500: Internal Server Error.";
        qDebug() << "Data:  " << replyData;
        return;


    case 410:
        errorTypeString = httpErrorString
                        + tr("Gone", "HTTP 410 error string")
                        + " (410)";
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "HTTP 410: Gone.";
        qDebug() << "Data:  " << replyData;
        return;

    case 404:
        errorTypeString = httpErrorString
                        + tr("Not Found", "HTTP 404 error string")
                        + " (404)";
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "HTTP 404: Not Found.";
        qDebug() << "Data:  " << replyData;
        return;

    case 403:
        errorTypeString = httpErrorString
                        + tr("Forbidden", "HTTP 403 error string")
                        + " (403) " + replyFirstLine;
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "HTTP 403: Forbidden.";
        qDebug() << "Data:  " << replyData;
        return;

    case 401:
        errorTypeString = httpErrorString
                        + tr("Unauthorized", "HTTP 401 error string")
                        + " (401) " + replyFirstLine;
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "HTTP 401: Unauthorized.";
        qDebug() << "Data:  " << replyData;
        return;


    case 400:
        errorTypeString = httpErrorString
                        + tr("Bad Request", "HTTP 400 error string")
                        + " (400) " + replyFirstLine;

        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);

        // if not just an image, it's important, so popup notification too
        if (requestType != ImageRequest && requestType != AvatarRequest)
        {
            emit showNotification(errorTypeString + "\n"
                                  + replyUrl);
        }

        qDebug() << "HTTP 400: Bad Request.";
        qDebug() << "Data:  " << replyData;
        return;

    case 302:
        errorTypeString = httpErrorString
                        + tr("Moved Temporarily", "HTTP 302 error string")
                        + " (302)";
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "HTTP 302: Moved Temporarily.";
        qDebug() << "Data:  " << replyData;
        return;

    case 301:
        errorTypeString = httpErrorString
                        + tr("Moved Permanently", "HTTP 301 error string")
                        + " (301)";
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "HTTP 301: Moved Permanently.";
        qDebug() << "Data:  " << replyData;
        return;

    case 0: // Other kinds of network errors
        errorTypeString = tr("Error connecting to %1").arg(replyHost);
        emit currentJobChanged(errorTypeString + ": " + replyErrorString);
        emit logMessage(errorTypeString + ": " + replyErrorString, replyUrl);

        qDebug() << "Error connecting to" << replyHost;
        return;

    // Other HTTP codes
    default:
        errorTypeString = tr("Unhandled HTTP error code %1").arg(httpCode);
        emit currentJobChanged(errorTypeString + ": " + replyUrl);
        emit logMessage(errorTypeString, replyUrl);
        qDebug() << "Unhandled HTTP error " << httpCode;
        qDebug() << "Data:  " << replyData;
        return;


    //////////////////////////////////////// The good one!
    case 200:
        qDebug() << "HTTP 200: OK!";
    }


    // At this point, httpCode should be 200 = OK



    // Prepare the JSON parser
    QJson::Parser jsonParser;
    bool jsonParsedOK = false;
    QVariantMap jsonData;
    QVariantList jsonDataList;


    // Unless it was an AvatarRequest, ImageRequest or MediaRequest,
    // it should be JSON, so parse it
    if (requestType != AvatarRequest
     && requestType != ImageRequest
     && requestType != MediaRequest)
    {
        jsonData = jsonParser.parse(replyData, &jsonParsedOK).toMap();
        qDebug() << "JSON data size (items):" << jsonData.size();
        qDebug() << "Keys:" << jsonData.keys();
    }


    //////////////////////////////////////////////////////////////////

    switch (requestType)
    {

    case ClientRegistrationRequest:
        qDebug() << "Client Registration was requested";

        qDebug() << "Raw JSON:" << jsonData;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            this->clientID = jsonData["client_id"].toString();
            this->clientSecret = jsonData["client_secret"].toString();

            // FIXME: error control, etc.
            // check if jsonData.keys().contains("client_id") !!

            QSettings settings;
            settings.setValue("clientID",     this->clientID);
            settings.setValue("clientSecret", this->clientSecret);

            this->getToken();
        }

        break;



    case UserProfileRequest:
        qDebug() << "A user profile was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            this->showStatusMessageAndLogIt(tr("Profile received."));

            QVariantMap profileMap = jsonData["profile"].toMap();
            if (profileMap.value("id").toString() == "acct:" + this->userId)
            {
                qDebug() << "Received OWN profile; keys:" << profileMap.keys();
                qDebug() << "Links from profile:" << profileMap.value("links").toMap().keys();
                qDebug() << "Lists from profile:" << profileMap.value("lists").toMap()
                                                               .value("totalItems").toInt();
                qDebug() << "Lists URL:" << profileMap.value("lists").toMap()
                                                      .value("url").toString();
                this->haveProfile = true;

                QString profImageUrl = profileMap.value("image").toMap().value("url").toString();
                QString profDisplayName = profileMap.value("displayName").toString();
                QString profLocation = profileMap.value("location").toMap().value("displayName").toString();
                QString profSummary = profileMap.value("summary").toString();

                QString userEmail = jsonData["email"].toString();
                qDebug() << "E-mail configured for the account:" << userEmail;

                emit profileReceived(profImageUrl, profDisplayName,
                                     profLocation, profSummary,
                                     userEmail);

                // Store also the user's followers URL, for posting to Followers
                this->userFollowersURL = profileMap.value("followers").toMap().value("url").toString();

                this->userFollowersCount = profileMap.value("followers").toMap().value("totalItems").toInt();
                this->userFollowingCount = profileMap.value("following").toMap().value("totalItems").toInt();

                qDebug() << "Followers count:" << userFollowersCount;
                qDebug() << "Following count:" << userFollowingCount;
            }
        }

        break;


    case UpdateProfileRequest:
        this->showStatusMessageAndLogIt(tr("Profile updated."));
        this->getUserProfile(this->userId);

        break;




    ////////////////////////////////////////////////// If a timeline was requested
    case MainTimelineRequest:
        // just jump to the next
    case DirectTimelineRequest:
        // just jump to next
    case ActivityTimelineRequest:
        // just... yeah, jump
    case FavoritesTimelineRequest:
        qDebug() << "A timeline was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            prettyLogMessage = tr("Timeline received. Updating post list...");
            emit currentJobChanged(prettyLogMessage);

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in timeline batch:" << jsonDataList.size();

            QString previousLink = jsonData["links"].toMap().value("prev").toMap().value("href").toString();
            QString nextLink = jsonData["links"].toMap().value("next").toMap().value("href").toString();

            int totalItems = jsonData["totalItems"].toInt();

            if (requestType == MainTimelineRequest)
            {
                qDebug() << "It was the main timeline";
                this->haveMainTL = true;
                emit mainTimelineReceived(jsonDataList, this->postsPerPageMain,
                                          previousLink, nextLink,
                                          totalItems);
            }
            else if (requestType == DirectTimelineRequest)
            {
                qDebug() << "It was the direct messages timeline";
                this->haveDirectTL = true;
                emit directTimelineReceived(jsonDataList, this->postsPerPageOther,
                                            previousLink, nextLink,
                                            totalItems);
            }
            else if (requestType == ActivityTimelineRequest)
            {
                qDebug() << "It was the own activity timeline";
                this->haveActivityTL = true;
                emit activityTimelineReceived(jsonDataList, this->postsPerPageOther,
                                              previousLink, nextLink,
                                              totalItems);
            }
            else if (requestType == FavoritesTimelineRequest)
            {
                qDebug() << "It was the favorites timeline";
                this->haveFavoritesTL = true;
                emit favoritesTimelineReceived(jsonDataList, this->postsPerPageOther,
                                               previousLink, nextLink,
                                               totalItems);
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }
        break;



    ////////////////////////////////////
    case PublishPostRequest:
        this->showStatusMessageAndLogIt(tr("Post published successfully."));
        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QString objectId = jsonData["object"].toMap().value("id").toString();
            qDebug() << "Post ID:" << objectId;

            QString objectType = jsonData["object"].toMap().value("objectType").toString();
            if (objectType == "image"
             || objectType == "audio"
             || objectType == "video"
             || objectType == "file")
            {
                // Update the object with title and description
                this->postMediaStepThree(objectId);
            }
            else
            {
                // Not a a media post, notify "posted OK"
                emit postPublished();
                qDebug() << "Non-media post published correctly";
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }

        break;



    case PublishAvatarRequest:
        this->showStatusMessageAndLogIt(tr("Avatar published successfully."));
        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QString imageUrl = jsonData["object"].toMap().value("image").toMap().value("url").toString();
            qDebug() << "Avatar post ID:" << imageUrl;
            // FIXME: get "pump_io: fullImage: url" too

            if (jsonData["object"].toMap().value("objectType").toString() == "image")
            {
                emit avatarUploaded(imageUrl);
            }
            else
            {
                qDebug() << "Avatar uploaded, but type is not IMAGE!";
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }

        break;


    ////////////////////////////////////
    case UpdatePostRequest:
        this->showStatusMessageAndLogIt(tr("Post updated successfully."));

        emit postPublished();
        break;

    case UpdateCommentRequest:
        this->showStatusMessageAndLogIt(tr("Comment updated successfully."));

        emit commentPosted();
        break;




    ///////////////////////////////////// If liking a post was requested
    case LikePostRequest:
        this->showStatusMessageAndLogIt(tr("Message liked or unliked successfully."));

        emit likeSet();
        break;


    ///////////////////////////////////// If the likes for a post were requested
    case PostLikesRequest:
        qDebug() << "Likes for a post were requested" << replyUrl;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            emit currentJobChanged(tr("Likes received."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in comments list:" << jsonDataList.size();

            emit likesReceived(jsonDataList, replyUrl);
        }
        else
        {
            qDebug() << "Error parsing received comment JSON data!";
            qDebug() << "Raw data:" << replyData;
        }
        break;



    ///////////////////////////////////// If commenting on a post was requested
    case CommentPostRequest:
        this->showStatusMessageAndLogIt(tr("Comment posted successfully."));

        emit commentPosted(); // This will be caught by Commenter()

        break;



    ///////////////////////////////////// If the comments for a post were requested
    case PostCommentsRequest:
        qDebug() << "Comments for a post were requested" << replyUrl;

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            jsonDataList = jsonData["items"].toList();

            int commentCount = jsonDataList.size();
            qDebug() << "Number of items in comments list:" << commentCount;

            if (commentCount == 1)
            {
                emit currentJobChanged(tr("1 comment received.")
                                       .arg(commentCount));
            }
            else
            {
                emit currentJobChanged(tr("%1 comments received.")
                                       .arg(commentCount));
            }

            emit commentsReceived(jsonDataList, replyUrl);
        }
        else
        {
            qDebug() << "Error parsing received comment JSON data!";
            qDebug() << "Raw data:" << replyData;
        }
        break;



    case SharePostRequest:
        qDebug() << "Post shared OK";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            // FIXME: this should be done using ASActivity+ASObject
            QVariantMap authorMap = jsonData.value("object").toMap().value("author").toMap();
            QString authorName = authorMap.value("displayName").toString();
            if (authorName.isEmpty())
            {
                authorName = ASPerson::cleanupId(authorMap.value("id").toString());
            }
            this->showStatusMessageAndLogIt(tr("Post by %1 shared successfully.",
                                               "1=author of the post we are sharing")
                                            .arg(authorName));
        }

        break;

    case PostSharesRequest:
        // TODO
        break;



    case MinorFeedRequest:
        qDebug() << "The minor feed was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";
            this->haveMinorFeed = true;
            emit currentJobChanged(tr("Minor feed received."));

            jsonDataList = jsonData["items"].toList();
            qDebug() << "Number of items in minor feed:" << jsonDataList.size();

            QString previousLink = jsonData["links"].toMap().value("prev").toMap().value("href").toString();
            QString nextLink = jsonData["links"].toMap().value("next").toMap().value("href").toString();

            emit minorFeedReceived(jsonDataList, previousLink, nextLink);
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }
        break;



    case DeletePostRequest:
        this->showStatusMessageAndLogIt(tr("Message deleted successfully."));

        break;


    case FollowContactRequest:
        // Just jump to next
    case UnfollowContactRequest:
        if (jsonParsedOK && jsonData.size() > 0)
        {
            ASPerson *contact = new ASPerson(jsonData.value("object").toMap(),
                                             this);

            if (requestType == FollowContactRequest)
            {
                prettyLogMessage = tr("Following %1 (%2) successfully.",
                                      "%1 is a person's name, %2 is the ID")
                                   .arg(contact->getName())
                                   .arg(contact->getId());

                emit contactFollowed(contact);
            }
            else
            {
                prettyLogMessage = tr("Stopped following %1 (%2) successfully.",
                                      "%1 is a person's name, %2 is the ID")
                                   .arg(contact->getName())
                                   .arg(contact->getId());

                emit contactUnfollowed(contact);
            }

            this->showStatusMessageAndLogIt(prettyLogMessage);
        }

        break;


    case FollowingListRequest:
        // just go to the next
    case FollowersListRequest:
        qDebug() << "A contact list was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant contactsVariant = jsonData.value("items");
            if (contactsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing contacts...";

                if (requestType == FollowingListRequest)
                {
                    totalReceivedFollowing += 200;

                    emit contactListReceived("following",
                                             contactsVariant.toList(),
                                             this->totalReceivedFollowing);

                    if (totalReceivedFollowing >= this->userFollowingCount)
                    {
                        this->haveFollowing = true;
                        this->showStatusMessageAndLogIt(tr("List of 'following' "
                                                           "completely received."));
                    }
                    else
                    {
                        emit currentJobChanged(tr("Partial list of 'following' "
                                                  "received."));
                    }
                }
                else // == FollowersListRequest
                {
                    totalReceivedFollowers += 200;

                    emit contactListReceived("followers",
                                             contactsVariant.toList(),
                                             this->totalReceivedFollowers);

                    if (totalReceivedFollowers >= this->userFollowersCount)
                    {
                        this->haveFollowers = true;
                        this->showStatusMessageAndLogIt(tr("List of 'followers' "
                                                           "completely received."));
                    }
                    else
                    {
                        emit currentJobChanged(tr("Partial list of 'followers' "
                                                  "received."));
                    }
                }
            }
            else
            {
                qDebug() << "Expected a list of contacts, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }

        break;



    case ListsListRequest:
        qDebug() << "The list of person lists was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant listsVariant = jsonData.value("items");
            if (listsVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing lists...";

                this->havePersonLists = true;
                emit currentJobChanged(tr("List of 'lists' received."));
                emit listsListReceived(listsVariant.toList());
            }
            else
            {
                qDebug() << "Expected a list of lists, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }

        break;

    ///////////////////////////////////////////////////////////////////// Lists

    // Person list created OK
    case CreatePersonListRequest:
        this->showStatusMessageAndLogIt(tr("Person list '%1' created successfully.")
                                        .arg(jsonData.value("object").toMap()
                                                     .value("displayName").toString()));

        this->getListsList(); // And reload the person lists (FIXME!)
        break;


    // Person list deleted OK
    case DeletePersonListRequest:
        this->showStatusMessageAndLogIt(tr("Person list deleted successfully."));

        this->getListsList(); // And reload the person lists (FIXME!)
        break;


    case PersonListRequest:
        qDebug() << "A person list was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QVariant personListVariant = jsonData.value("items");
            if (personListVariant.type() == QVariant::List)
            {
                qDebug() << "Parsed a List, listing people in list..."
                         << jsonData.value("displayName");

                emit currentJobChanged(tr("Person list received."));
                emit personListReceived(personListVariant.toList(),
                                        replyUrl);
            }
            else
            {
                qDebug() << "Expected a list of people, received something else:";
                qDebug() << jsonData;
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }

        break;


    // Person added to list OK
    case AddMemberToListRequest:
        if (jsonParsedOK && jsonData.size() > 0)
        {
            QString personId = ASPerson::cleanupId(jsonData.value("object").toMap()
                                                           .value("id").toString());
            QString personName = jsonData.value("object").toMap()
                                         .value("displayName").toString();
            QString personAvatar = jsonData.value("object").toMap()
                                           .value("image").toMap()
                                           .value("url").toString();

            emit personAddedToList(personId, personName, personAvatar);

            this->showStatusMessageAndLogIt(tr("%1 (%2) added to list successfully.",
                                               "1=contact name, 2=contact ID")
                                            .arg(personName)
                                            .arg(personId));
        }
        break;


    // Person removed from list OK
    case RemoveMemberFromListRequest:
        if (jsonParsedOK && jsonData.size() > 0)
        {
            QString personId = ASPerson::cleanupId(jsonData.value("object").toMap()
                                                            .value("id").toString());
            QString personName = jsonData.value("object").toMap()
                                         .value("displayName").toString();

            emit personRemovedFromList(personId);

            this->showStatusMessageAndLogIt(tr("%1 (%2) removed from list successfully.",
                                               "1=contact name, 2=contact ID")
                                            .arg(personName)
                                            .arg(personId));
        }
        break;


    //////////////////////////////////////////////////////////////////// Groups
    case CreateGroupRequest:
        this->showStatusMessageAndLogIt(tr("Group %1 created successfully.")
                                        .arg(jsonData.value("object").toMap()
                                                     .value("displayName").toString()));

        qDebug() << "Group created; ID:" << jsonData.value("object").toMap()
                                                    .value("id").toString();
        break;


    case JoinGroupRequest:
        this->showStatusMessageAndLogIt(tr("Group %1 joined successfully.")
                                        .arg(jsonData.value("object").toMap()
                                                     .value("displayName").toString()));

        qDebug() << "Group joined; ==========================";
        qDebug() << "Keys: " << jsonData.keys();
        qDebug() << jsonData.value("object").toString();

        break;


    case LeaveGroupRequest:
        this->showStatusMessageAndLogIt(tr("Left the %1 group successfully.")
                                        .arg(jsonData.value("object").toMap()
                                                     .value("displayName").toString()));

        break;



    case AvatarRequest:
        qDebug() << "Received AVATAR data, from " << replyUrl;
        if (finished)
        {
            qDebug() << "Avatar received 100%";
            emit avatarPictureReceived(replyData, replyUrl);
        }
        else
        {
            qDebug() << "Avatar not complete yet";
        }
        break;


    case ImageRequest:
        qDebug() << "Received IMAGE data, from " << replyUrl;
        if (finished)
        {
            qDebug() << "Image received 100%";
            emit imageReceived(replyData, replyUrl);
        }
        else
        {
            qDebug() << "Image not complete yet";
        }
        break;


    case MediaRequest:
        qDebug() << "Received MEDIA data, from " << replyUrl;
        if (finished)
        {
            this->showStatusMessageAndLogIt(tr("File downloaded successfully."));

            emit downloadCompleted(replyUrl);

            qDebug() << "Media received 100%" << replyData.length() << "bytes";
        }
        else
        {
            qDebug() << "Media not complete yet";
        }
        break;


    //////////////////////////////////////// If uploading a file was requested
    case UploadAvatarRequest:
        // just jump to next
    case UploadMediaForPostRequest:
        // just jump
    case UploadFileRequest:
        qDebug() << "Uploading a file was requested";

        if (jsonParsedOK && jsonData.size() > 0)
        {
            qDebug() << "JSON parsed OK";

            QString uploadedFileId = jsonData["id"].toString();
            qDebug() << "Uploaded file ID:" << uploadedFileId;


            QString objectType = jsonData["objectType"].toString();
            if (objectType == "image"
             || objectType == "audio"
             || objectType == "video"
             || objectType == "file")
            {
                if (requestType == UploadMediaForPostRequest)
                {
                    prettyLogMessage = tr("File uploaded successfully. "
                                          "Posting message...");
                    emit currentJobChanged(prettyLogMessage);
                    emit logMessage(prettyLogMessage, replyUrl);

                    this->postMediaStepTwo(uploadedFileId);
                }
                else if (requestType == UploadAvatarRequest)
                {
                    this->showStatusMessageAndLogIt(tr("Avatar uploaded."));
                    this->postAvatarStepTwo(uploadedFileId);
                }
            }
        }
        else
        {
            qDebug() << "Error parsing received JSON data!";
            qDebug() << "Raw data:" << replyData; // JSON directly
        }

        break;


    }
    // end switch (requestType)

    qDebug() << "requestFinished() ended; " << replyUrl;
}







void PumpController::sslErrorsHandler(QNetworkReply *reply, QList<QSslError> errorList)
{
    qDebug() << "\n==== SSL errors!! ====";
    qDebug() << "At:" << reply->url().toString();
    qDebug() << "Error list:" << errorList << "\n\n";

    // ignore completely for now
    qDebug() << "Ignoring these errors...";
    reply->ignoreSslErrors(errorList);

    /*
    // Check list of SSL errors
    foreach (QSslError sslError, errorList)
    {
        // Ignore the "certificate is self-signed" SSL error, it's expected
        if (sslError.error() == QSslError::SelfSignedCertificate)
        {
            qDebug() << "Ignoring self-signed certificate error";
            reply->ignoreSslErrors(); // FIXME, maybe ask the user
            return;
        }
    }
    */

    reply->deleteLater();
}




void PumpController::getToken()
{
    // If we do not have client_id or client_secret, do dynamic client registration
    if (this->clientID.isEmpty() || this->clientSecret.isEmpty())
    {
        qDebug() << "PumpController::getToken()";
        qDebug() << "We do not have client_id/client_secret yet; doing Dynamic Client Registration";

        this->showStatusMessageAndLogIt(tr("The application is not registered with "
                                           "your server yet. Registering..."));

        // POST to https://yourserver.example/api/client/register
        QNetworkRequest postRequest(QUrl("https://" + this->serverURL + "/api/client/register"));
        postRequest.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");
        postRequest.setRawHeader("User-Agent", userAgentString);
        postRequest.setAttribute(QNetworkRequest::User,
                                 QVariant(ClientRegistrationRequest));

        QByteArray data("{"
                        " \"type\": \"client_associate\",  "
                        " \"application_type\": \"native\", "
                        " \"application_name\": \"Dianara\" "
                        "}");

        qDebug() << "About to POST:" << data;


        // upon receiving data (id+secret), will execute getToken() again
        nam.post(postRequest, data);
    }
    else
    {
        qDebug() << "Using saved client_id and client_secret:" << this->clientID << this->clientSecret;

        // OAuth stuff.....
        // 1. obtaining an unauthorized Request Token from the Service Provider,
        // 2. asking the User to authorize the Request Token,
        // 3. exchanging the Request Token for the Access Token


        this->showStatusMessageAndLogIt(tr("Getting OAuth token..."));

        qDebug() << "Doing OAuth token stuff...";
        qDebug() << "NOTE: if you see a crash here, you need QCA and its openSSL plugin:";
        qDebug() << ">> qca2-plugin-openssl, libqca2-plugin-ossl, or similar";
        qDebug() << "If you compiled Dianara from source, check the INSTALL file carefully";


        QStringList QCAsupportedFeatures = QCA::supportedFeatures();
        qDebug() << "QCA Supported Features:" << QCAsupportedFeatures;

        if (QCAsupportedFeatures.contains("hmac(sha1)"))
        {
            qDebug() << "HMAC-SHA1 support is OK";
        }
        else
        {
            qDebug() << "Warning, HMAC-SHA1 doesn't seem to be supported!";

            // Notify the user about missing plugin
            emit authorizationFailed(tr("OAuth support error"),
                                     tr("Your installation of QOAuth, a library "
                                        "used by Dianara, doesn't seem to have "
                                        "HMAC-SHA1 support.")
                                     + "\n"
                                     + tr("You probably need to install the OpenSSL "
                                          "plugin for QCA: %1, %2 or similar.")
                                       .arg("qca2-plugin-openssl")
                                       .arg("libqca2-plugin-ossl"));
        }

        qoauth->setConsumerKey(this->clientID.toLocal8Bit());
        qoauth->setConsumerSecret(this->clientSecret.toLocal8Bit());


        QString requestTokenURL = "https://" + this->serverURL + "/oauth/request_token";
        qDebug() << "GET: " << requestTokenURL
                 << "with" << qoauth->consumerKey() << qoauth->consumerSecret();

        QOAuth::ParamMap oAuthParams;
        oAuthParams.insert("oauth_callback", "oob");

        QOAuth::ParamMap reply = qoauth->requestToken(requestTokenURL,
                                                      QOAuth::GET,
                                                      QOAuth::HMAC_SHA1,
                                                      oAuthParams);

        if (qoauth->error() == QOAuth::NoError)
        {
            qDebug() << "requestToken OK:" << reply.keys();

            token = reply.value(QOAuth::tokenParameterName());
            tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

            qDebug() << "Token:" << token;
            qDebug() << "Token Secret:" << tokenSecret.left(5) + "********** (hidden)";

            QUrl oAuthAuthorizeURL("https://" + this->serverURL + "/oauth/authorize");
#if QT_VERSION < QT_VERSION_CHECK(5, 0, 0)
            oAuthAuthorizeURL.addQueryItem("oauth_token", token);
#else
            QUrlQuery query;
            query.addQueryItem("oauth_token", token);
            oAuthAuthorizeURL.setQuery(query);
#endif
            QDesktopServices::openUrl(oAuthAuthorizeURL);

            // Send also a signal, so AccountDialog can show the URL in
            // a label, in case the browser didn't launch
            emit openingAuthorizeURL(oAuthAuthorizeURL);

            // Now, user should enter VERIFIER in AccountDialog to authorize the program
        }
        else
        {
            qDebug() << "QOAuth error" << qoauth->error() << "!";
            qDebug() << reply.keys();

            emit authorizationFailed(tr("Authorization error"),
                                     tr("There was an OAuth error while trying "
                                        "to get the authorization token.")
                                     + "\n\n"
                                     + tr("QOAuth error %1").arg(qoauth->error()));
        }
    }
}



void PumpController::authorizeApplication(QString verifierCode)
{
    qDebug() << "Verifier code entered by user:" << verifierCode;

    QOAuth::ParamMap moreParams;
    moreParams.insert("oauth_verifier", verifierCode.toUtf8()); // verifier as QByteArray

    QString requestAuthorizationURL = "https://" + this->serverURL + "/oauth/access_token";

    QOAuth::ParamMap reply = qoauth->accessToken(requestAuthorizationURL,
                                                 QOAuth::GET,
                                                 token,
                                                 tokenSecret,
                                                 QOAuth::HMAC_SHA1,
                                                 moreParams);

    if (qoauth->error() == QOAuth::NoError) // Woooohooo!!
    {
        qDebug() << "Got authorized token; Dianara is authorized to access the account";
        token = reply.value(QOAuth::tokenParameterName());
        tokenSecret = reply.value(QOAuth::tokenSecretParameterName());

        this->isApplicationAuthorized = true;

        this->showStatusMessageAndLogIt(tr("Application authorized successfully."));

        QSettings settings;
        settings.setValue("isApplicationAuthorized", this->isApplicationAuthorized);
        settings.setValue("token",       this->token);
        settings.setValue("tokenSecret", this->tokenSecret);

        qDebug() << "Token:" << token;
        qDebug() << "TokenSecret:" << tokenSecret;

        emit this->authorizationStatusChanged(isApplicationAuthorized);
    }
    else
    {
        this->showStatusMessageAndLogIt(tr("OAuth error while authorizing application."));
        qDebug() << "OAuth error while authorizing application" << qoauth->error();
    }

}




/*
 * Called by a QTimer;
 * Get initial data (profile, contacts, timelines), one step at a time
 *
 */
void PumpController::getInitialData()
{
    qDebug() << "PumpController::getInitialData() step" << initialDataStep;

    initialDataTimer->setInterval(3000);  // Every 3 sec

    // If we're still waiting for a proxy password, don't do anything
    if (this->needsProxyPassword())
    {
        emit currentJobChanged(tr("Waiting for proxy password..."));
        return;
    }

    /*
     * FIXME: this needs to be way more elaborate.
     *
     * Ensure certain stuff has been received before continuing.
     *
     * For instance, getting the "following" list is needed before
     * being able to post, and to know the state of following/not Following
     * the author of a post in a timeline
     *
     */

    /* // TMP, tests for getMinorFeedProper()
    this->getMinorFeed();
    initialDataTimer->stop();
    return;
    */

    switch (this->initialDataStep)
    {
    case 0:
        if (!haveProfile)
        {
            this->getUserProfile(this->userId);
        }
        break;

    case 1:
        // Ensure we have the profile already, before getting contact lists...
        if (!haveProfile)
        {
            initialDataStep = -1; // Restart initialization!
            this->showStatusMessageAndLogIt(tr("Still waiting for profile. "
                                               "Trying again..."));
        }
        break;

    case 2:
        if (!haveFollowing)
        {
            if (haveProfile) // Require profile first, since it provides followingCount
            {
                this->getContactList("following");
            }
            else
            {
                --initialDataStep; // go back 1 to retry
            }
        }
        break;

    case 3:
        if (!haveFollowers)
        {
            if (haveProfile) // Required for followersCount
            {
                this->getContactList("followers");
            }
            else
            {
                --initialDataStep; // retry
            }
        }
        break;

    case 4:
        // Just some extra time to get 'following' before timelines
        break;

    case 5:
        if (!haveMainTL)
        {
            this->getMainTimeline(0);
        }
        break;

    case 6:
        if (!haveMinorFeed)
        {
            this->getMinorFeed();
        }
        break;

    case 7:
        if (!havePersonLists)
        {
            this->getListsList();
        }
        break;

    case 8:
        if (!haveDirectTL)
        {
            this->getDirectTimeline(0);
        }
        break;

    case 9:
        if (!haveActivityTL)
        {
            this->getActivityTimeline(0);
        }
        break;

    case 10:
        if (!haveFavoritesTL)
        {
            this->getFavoritesTimeline(0);
        }
        break;

    case 11:
        // If some data is still missing, go back to the beginning of the cycle
        if (!haveProfile || !haveFollowing || !haveFollowers
         || !havePersonLists || !haveMainTL || !haveMinorFeed
         || !haveDirectTL || !haveActivityTL || !haveFavoritesTL)
        {
            // Unless we've already tried several times, like people with broken
            // contact lists, for instance, which will never be received
            if (initialDataAttempts < 5)
            {
                initialDataStep = -1;
                ++initialDataAttempts;

                this->showStatusMessageAndLogIt(tr("Some initial data was not "
                                                   "received. Restarting "
                                                   "initialization.")
                                                + QString(" (%1)")
                                                  .arg(initialDataAttempts));
            }
            else
            {
                this->showStatusMessageAndLogIt(tr("Some initial data was not "
                                                   "received after several "
                                                   "attempts. Something might "
                                                   "be wrong with your server. "
                                                   "You might still be able to "
                                                   "use the service normally."));
            }
        }
        else
        {
            this->showStatusMessageAndLogIt(tr("All initial data received. "
                                               "Initialization complete."));
            // If there are no problems, this will just wait one interval,
            // so it takes longer for the final (default) step to arrive
        }
        break;


    default:
        initialDataTimer->stop();

        this->showStatusMessageAndLogIt(tr("Ready."));
        emit initializationComplete();

        qDebug() << "--------------------------------------";
        qDebug() << "-- All initial data loaded -----------";
        qDebug() << "--------------------------------------";
    }

    ++initialDataStep;
}





/*
 * Send a NOTE to the server
 *
 */
void PumpController::postNote(QMap<QString, QString> audienceMap,
                              QString postText, QString postTitle)
{
    qDebug() << "PumpController::postNote()";

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url,
                                                       QOAuth::POST,
                                                       PublishPostRequest);

    qDebug() << "Should be posting to:" << audienceMap;



    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "note");
    if (!postTitle.isEmpty())
    {
        jsonVariantObject.insert("displayName", postTitle);
    }
    //jsonVariantObject.insert("summary", "summary test");
    jsonVariantObject.insert("content", postText);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);

    QList<QVariantList> audience = processAudience(audienceMap);
    jsonVariant.insert("to", audience.at(0));
    jsonVariant.insert("cc", audience.at(1));


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;

    nam.post(postRequest, data);
}


/*
 * Post media-type of object (image, audio/ video)
 *
 * First, upload the file.
 * Then we get it's ID in a signal, and create the post itself
 *
 */
QNetworkReply *PumpController::postMedia(QMap<QString, QString> audienceMap,
                                         QString postText, QString postTitle,
                                         QString mediaFilename, QString mediaType,
                                         QString mimeContentType)
{
    qDebug() << "PumpController::postMedia()";
    qDebug() << "Uploading" << mediaFilename << "with title:" << postTitle;

    // Store postTitle, postText, and audienceMap, then upload
    this->currentPostTitle = postTitle;
    this->currentPostDescription = postText;
    this->currentAudienceMap = audienceMap;
    this->currentPostType = mediaType;

    QNetworkReply *networkReply;
    networkReply =  this->uploadFile(mediaFilename,
                                     mimeContentType,
                                     UploadMediaForPostRequest);
    return networkReply;
}


/*
 * Post Media, step 2: after getting the ID in the file upload request,
 * create the post itself
 *
 */
void PumpController::postMediaStepTwo(QString id)
{
    qDebug() << "PumpController::postMediaStepTwo()"
             <<  currentPostType << "ID:" << id;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url,
                                                       QOAuth::POST,
                                                       PublishPostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", this->currentPostType);
    jsonVariantObject.insert("id", id);

    QList<QVariantList> audience = processAudience(this->currentAudienceMap);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);
    jsonVariant.insert("to", audience.at(0));
    jsonVariant.insert("cc", audience.at(1));


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;
    nam.post(postRequest, data);
}


/*
 * Workaround for the non-title-non-description issue
 *
 * Update the image/audio/video post with the title and the description
 *
 */
void PumpController::postMediaStepThree(QString id)
{
    qDebug() << "PumpController::postMediaStepThree() post ID:" << id;

    // Using the ID directly as URL
    QNetworkRequest postRequest = this->prepareRequest(id,
                                                       QOAuth::PUT,
                                                       UpdatePostRequest);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "update"); // "post" would suffice
    if (!this->currentPostTitle.isEmpty())
    {
        jsonVariant.insert("displayName", this->currentPostTitle);
    }
    jsonVariant.insert("content", this->currentPostDescription);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to PUT:" << data;
    nam.put(postRequest, data);
}



/*
 * Second step for avatar upload.
 *
 * Post the image to Public
 *
 */
void PumpController::postAvatarStepTwo(QString id)
{
    qDebug() << "PumpController::postAvatarStepTwo() image ID:" << id;

    QString url = "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest postRequest = this->prepareRequest(url, QOAuth::POST,
                                                       PublishAvatarRequest);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "image");
    jsonVariantObject.insert("id", id);

    // audience, CC: Public
    QVariantMap jsonVariantPublic;
    jsonVariantPublic.insert("objectType", "collection");
    jsonVariantPublic.insert("id", "http://activityschema.org/collection/public");

    QVariantList jsonVariantAudience;
    jsonVariantAudience.append(jsonVariantPublic);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);
    jsonVariant.insert("cc", jsonVariantAudience); // CC: Public


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to POST:" << data;
    nam.post(postRequest, data);
}



/*
 * Update post contents (the object)
 *
 */
void PumpController::updatePost(QString id,
                                QString content,
                                QString title)
{
    qDebug() << "PumpController::updatePost(), post ID:" << id;

    // Using the ID directly as URL
    QNetworkRequest postRequest = this->prepareRequest(id,
                                                       QOAuth::PUT,
                                                       UpdatePostRequest);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "update");     // "post" would suffice
    jsonVariant.insert("displayName", title); // "title" can be empty, as a way to remove titles
    jsonVariant.insert("content", content);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to PUT:" << data;
    nam.put(postRequest, data);
}



/*
 * Like (favorite) a post, by its ID (URL)
 *
 */
void PumpController::likePost(QString postID, QString postType, bool like)
{
    qDebug() << "PumpController::likePost() liking post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest likeRequest = this->prepareRequest(url,
                                                       QOAuth::POST,
                                                       LikePostRequest);


    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", like ? "favorite":"unfavorite"); // like or unlike
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "about to POST:" << data;
    nam.post(likeRequest, data);
}



void PumpController::addComment(QString comment, QString postID, QString postType)
{
    qDebug() << "PumpController::addComment() sending comment to this post:" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest commentRequest = this->prepareRequest(url,
                                                          QOAuth::POST,
                                                          CommentPostRequest);

    QVariantMap jsonVariantInReplyTo;
    jsonVariantInReplyTo.insert("id", postID);
    jsonVariantInReplyTo.insert("objectType", postType);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "comment");
    jsonVariantObject.insert("content", comment);
    jsonVariantObject.insert("inReplyTo", jsonVariantInReplyTo);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "post");
    jsonVariant.insert("object", jsonVariantObject);

    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "about to POST:" << data;
    nam.post(commentRequest, data);
}


/*
 * Update comment contents (object)
 *
 */
void PumpController::updateComment(QString id, QString content)
{
    qDebug() << "PumpController::updateComment(), comment ID:" << id;

    // Using the ID directly as URL
    QNetworkRequest postRequest = this->prepareRequest(id,
                                                       QOAuth::PUT,
                                                       UpdateCommentRequest);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "update");
    jsonVariant.insert("content", content);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "About to PUT:" << data;
    nam.put(postRequest, data);
}




void PumpController::sharePost(QString postID, QString postType)
{
    qDebug() << "PumpController::sharePost() sharing post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest shareRequest = this->prepareRequest(url,
                                                        QOAuth::POST,
                                                        SharePostRequest);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "share");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "about to POST:" << data;
    nam.post(shareRequest, data);
}


void PumpController::unsharePost(QString postId, QString postType)
{
    qDebug() << "PumpController::unsharePost() unsharing post" << postId;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest unshareRequest = this->prepareRequest(url,
                                                          QOAuth::POST,
                                                          UnsharePostRequest);

    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postId);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "unshare");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "about to POST:" << data;
    nam.post(unshareRequest, data);
}



void PumpController::deletePost(QString postID, QString postType)
{
    qDebug() << "PumpController::deletePost() deleting post" << postID;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest deleteRequest = this->prepareRequest(url,
                                                         QOAuth::POST,
                                                         DeletePostRequest);
    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", postType);
    jsonVariantObject.insert("id", postID);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "delete");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "about to POST:" << data;
    nam.post(deleteRequest, data);
}




/*
 * Add a contact to the /following list with their webfinger address
 *
 * This will trigger a re-request of the contact list, upon receiving the HTTP 200
 * confirming the addition of the contact to /following
 *
 */
void PumpController::followContact(QString address)
{
    qDebug() << "PumpController::followContact()" << address;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest followRequest = this->prepareRequest(url,
                                                         QOAuth::POST,
                                                         FollowContactRequest);
    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", "acct:" + address);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "follow");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "about to POST:" << data;
    nam.post(followRequest, data);
}



/*
 * Remove a contact from the /following list with their webfinger address
 *
 * This will trigger a re-request of the contact list, upon receiving the HTTP 200
 * confirming the removal of the contact from /following
 *
 */
void PumpController::unfollowContact(QString address)
{
    qDebug() << "PumpController::unfollowContact()" << address;

    QString url =  "https://" + this->serverURL + "/api/user/" + this->userName + "/feed";

    QNetworkRequest unfollowRequest = this->prepareRequest(url,
                                                           QOAuth::POST,
                                                           UnfollowContactRequest);
    QVariantMap jsonVariantObject;
    jsonVariantObject.insert("objectType", "person");
    jsonVariantObject.insert("id", "acct:" + address);

    QVariantMap jsonVariant;
    jsonVariant.insert("verb", "stop-following");
    jsonVariant.insert("object", jsonVariantObject);


    QByteArray data = this->prepareJSON(jsonVariant);
    qDebug() << "about to POST:" << data;
    nam.post(unfollowRequest, data);
}
