/*****************************************************************************************/
/* Copyright 2008,2009,2010,2011,2012,2013 Elias Potapov. */
/* Copyright 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007
   2008, 2009, 2010, 2011 The GSL Team. */

/*****************************************************************************************/
/* This file is part of DINAMICA. */

/* DINAMICA is free software: you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation, either version 3 of the License, or */
/* (at your option) any later version. */

/* DINAMICA is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with DINAMICA.  If not, see <http://www.gnu.org/licenses/>. */
/****************************************************************************************/
/****************************************************************************************/
/* Original author is Elias Potapov <elias.potapov@gmail.com>
   Lomonosov Moscow State University, Biophysics Dep..
   Tampere University of Technology, Dep. of Signal Processing.
   Moscow, Russia / Tampere, Finland*/
/****************************************************************************************/
/* It is the core integration algorithm of DINAMICA. It uses four
   methods: my own Runge-Kutta 4-th order without any modifications,
   Runge-Kutta-Fehlberg (4,5), Runge-Kutta Cash-Karp (4,5) and
   Runge-Kutta Prince-Dormand (8,9). For each integrating method there
   is comparising test (if strcmp...) for string in the mynum.method
   struct member. The only function run() uses three parameters: total
   time of integration, write flag and transient flag. 
   Total time is value of upper limit of independent variable till
   what we integrate.
   Write flag(wr) is flag for whether we should write output
   trajectory file.
   Transient flag(tr) is flag for whether our integration is
   transient, e.g. is used for getting to attractor in slow
   systems. Transient integration is always free of writing output
   trajectory file.
   Common integration may be either with writing to disk or without
   it.*/  

#include "init.h"
#include "errors.h"
#include "random.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_odeiv2.h>
//#include <plot.h>

int run_lin()
{
  int i,j,k;
  double t1 = mynum.total_time;
  double h = mynum.step;
  double x_lin[DIM];
  int npr;
  t = 0.0;
  for(i=0;i<DIM;i++)
    x_lin[i]=x[i];
  npr = 1;
  while(t <= t1)
    {
      if(t<h){
	printf("Initials:\n");
	for(i=0;i<DIM;i++)
	  printf("%G ",x[i]);
	printf("\n");
	for(i=0;i<DIM;i++)
	  printf("%G ",x_lin[i]);
	printf("\n");
	printf("-----------\n");
      }
      if(t>(t1/10)*npr){
	for(i=0;i<DIM;i++)
	  printf("%G ",x[i]);
	printf("\n");
	for(i=0;i<DIM;i++)
	  printf("%G ",x_lin[i]);
	printf("\n");
	printf("-----------\n");
	npr++;
      }
      solver_lin(x,x_lin,f,h,ode_lin);
      solver(x,h,func_odeiv);
      
      t = t+h;
    }

  return 0;
}

int run (const char *method, const double time, const int short wr,
	 const int short tr, const int nruns)
{
  int i,j,k,n_wr_step,outPrint = 1;
  /* Static flag showing what run we are in. 0 -- not complex OR before the first run
     and after the second one of complex, 1 -- complex first run and 2 --
     complex second run.*/
  int short static complex_run_flag = 0;
  /*** Data file ***/
  FILE *data;
  if(strlen(data_name) == 0) strcpy(data_name,"dat");
  if(wr && !tr){/* We need to open file */
    /* if(strcmp(method,"complex") != 0){ */
    /*   /\* simple method: do not open file for complex one. *\/ */
    /*   if(complex_run_flag != 2) */
    /* 	data = fopen(data_name,"w"); */
    /*   else */
    /* 	data = fopen(data_name,"a"); */
    /* } */
    if(!complex_run_flag)
      data = fopen(data_name,"w");
    else
      data = fopen(data_name,"a");
  }
  /*** Technical details of the timeseries to the file ***/
  if(wr && !tr){
    if(!complex_run_flag){
      if(strcmp(method,"complex")==0){
	fprintf(data,"#1,%d,0,%d\n",nruns,lang_flag);
	fclose(data);
      }
      else{
	fprintf(data,"#0,%d,",nruns);
	if(strcmp(method,"discrete")==0 || lang_flag)
	  fprintf(data,"0,%d\n",lang_flag);/* not deterministic */
	else
	  fprintf(data,"1,%d\n",lang_flag);/* Pure deterministic flag */
      }
    }
  }
  /*** Set initial time, time step etc. ***/
  t = 0.0;
  double t1 = time;
  double h = mynum.step;
  double tau;/*Next reaction time(Gill)*/
  int mur;/*Next reaction(Gill)*/
  double A_sum;/*Sum of all reactions' propensities(Gill)*/
  double y_dbl[DIM];/*Double version of y(Gill)*/
  double y_in[DIM];/*Initial value for each run(swap variable)*/
  n_steps = 0;
  write_count = 0;
  double tmp,tmp1;
  /*** Random number generation initialization. ***/
  if(strcmp(method,"discrete") == 0 || lang_flag){
    /* Do not initialize for complex method */
    if(seed_flag == 2)
      generate_seed();
    init_rand(sd);
  }
  /*** Warning about multiple runs for the deterministic method ***/
  if((strcmp(method,"complex")!=0) && (nruns>1) &&	\
     (strcmp(method,"discrete") != 0) && (!lang_flag)){
    fprintf(stderr,"Method must be <stochastic>\n");
    fprintf(stderr,"If you want to run deterministic simulations multiple");
    fprintf(stderr," times,\nuse transient runs instead (see `rt').\n");
    return 45;
  }
  /*** COMPLEX METHOD: recursive calls to run(...) ***/
  /* Complex method: one deterministic simulation plus nruns of */
  /*  stochastic; requires to specify two methods respectively */
  if((strcmp(method,"complex") == 0)){
    printf("Run deterministic method: %s...\n",mynum.method2);
    complex_run_flag = 1;
    /* First run deterministic simulation */
    if(lang_flag == 0)
      run(mynum.method2,time,wr,tr,1);
    else{
      lang_flag = 0;
      run(mynum.method2,time,wr,tr,1);
      lang_flag = 1;
    }
    complex_run_flag = 2;
    /* Then, run stochastic method nruns times */
    if(lang_flag == 0){/* we have discrete method, this was checked */
      printf("Run stochastic method: discrete...\n");
      run("discrete",time,wr,tr,nruns);
    }
    else{
      printf("Run stochastic method: langevin...\n");
      run(mynum.method2,time,wr,tr,nruns);
    }
    /* Finished complex run */
    complex_run_flag = 0;
  }
  /*** Own routine for Gillespie algorithm ***/
  else if((strcmp(method,"discrete")==0)){/*Gillespie algorithm*/
    for(i=0;i<DIM;i++)/* Initial value stored in the swap */
      y_in[i] = y[i];
    for(k=0;k<nruns;k++){
      /*Init gillespie*/
      t = 0.0;
      n_wr_step = 0;
      for(i=0;i<DIM;i++){/*Doublify and initialize*/
	y[i] = y_in[i];
	y_dbl[i] = (double)y[i];
      }
      gill_init(y,pr,&A_sum);
      while (t < t1){
	/*Compute auxillary...*/
	auxillary(t,mu.P,y_dbl,a);
	/*Generate two random numbers*/
	tmp1 = gsl_rng_uniform_pos(r);/*R1*/
	tau = (1/A_sum)*log(1/tmp1);/* Time of the next reaction */
	tmp = gsl_rng_uniform_pos(r);/*R2*/
	tmp1 = A_sum;
	for(i=NREAC;i>0;i--){/* Choosing the next reaction */
	  tmp1 = tmp1-pr[i-1];
	  if((tmp*A_sum) > tmp1){
	    mur = i;
	    break;
	  }
	}
	if(!tr){/* write the output to file */
	  if(wr){
	    while((t+tau) >= (n_wr_step*mynum.smp_frq)){
	      fprintf(data,"%G ",(double)n_wr_step*mynum.smp_frq);
	      for(j=0;j<DIM;j++)
		fprintf(data,"%ld ",y[j]);
	      fprintf(data,"\n");
	      n_wr_step++;
	    }
	  }
	}
	t = t + tau;/*Forward in time*/
	update(y,mur);/*Update system according to the reaction*/
	/*Recompute sum of the propensities*/
	A_sum = 0.0;
	for(i=0;i<DIM;i++)/*Doublify*/
	  y_dbl[i]=(double)y[i];
	propensity(y_dbl,mu.P,pr);/* Recompute propensities, all of them */
	for(i=0;i<NREAC;i++)
	  A_sum = A_sum + pr[i];
      }
      if(!tr)
	fprintf(data,"\n\n");/*Separating data in the file*/
      if(nruns >= 5){
	i = (int)(outPrint*nruns/5);
	if(k == i){
	  outPrint++;
	  printf("Completed: %d/%d\n",k+1,nruns);
	}
      }
    }
  }
  else if((strcmp(method,"run-kut4")==0) || \
	  (strcmp(method,"eu")==0)) {
    for(i=0;i<DIM;i++)/* Initial value stored in the swap */
      y_in[i] = x[i];
    for(k=0;k<nruns;k++){
      t = 0.0;
      n_steps = 0;
      write_count = 0;
      for(i=0;i<DIM;i++)/*initialize: take from the swap*/
	x[i] = y_in[i];
      while(t < t1) {
	/*Compute auxillary...*/
	auxillary(t,mu.P,x,a);
	if(!tr){
	  if(n_steps == write_count*mynum.write_step) {
	    /* if(write_count < BUFFER)//mynum.global_buffer */
	    /*   write_data_array(); */
	    /* if(buffer_overfull_check()) break; */
	    if(wr)
	      write_data_file(data);
	    write_count++;
	  }
	}
	solver(x,h,func_odeiv);//Solver for run-kut4 and Euler, own.
	t = t+h;
	n_steps++;
      }
      if(!tr)
	fprintf(data,"\n\n");/*Separating data in the file*/
      if(nruns >= 5){
	i = (int)(outPrint*nruns/5);
	if(k == i){
	  outPrint++;
	  printf("Completed: %d/%d\n",k+1,nruns);
	}
      }
    }
  }
  else{
    gsl_odeiv2_step *s;
    if(strcmp(method,"rkf45") == 0)
      s = gsl_odeiv2_step_alloc(gsl_odeiv2_step_rkf45,DIM);
    if(strcmp(method,"rk8pd") == 0)
      s = gsl_odeiv2_step_alloc(gsl_odeiv2_step_rk8pd,DIM);
    if(strcmp(method,"rkck") == 0)
      s = gsl_odeiv2_step_alloc(gsl_odeiv2_step_rkck,DIM);
    if(strcmp(method,"bsimp") == 0)
      s = gsl_odeiv2_step_alloc(gsl_odeiv2_step_bsimp,DIM);
    gsl_odeiv2_control *c = gsl_odeiv2_control_standard_new (eps_abs_int,eps_rel_int,a_y,a_dydt);
    gsl_odeiv2_evolve *e = gsl_odeiv2_evolve_alloc (DIM);
    gsl_odeiv2_system sys = {func_odeiv,jac,DIM,&mu};
    
    for(i=0;i<DIM;i++)/* Initial value stored in the swap */
      y_in[i] = x[i];
    for(k=0;k<nruns;k++){
      /* This multiple runs for the deterministic methods might be used only for
      Langevin simulation for now. In the future, the method for Langevin approach
      should be fixed. */
      if(k>0)
	gsl_odeiv2_step_reset(s);/* Resetting the stepping function */
      t = 0.0;
      n_steps = 0;
      write_count = 0;
      for(i=0;i<DIM;i++)/*initialize: take from the swap*/
	x[i] = y_in[i];
      while(t < t1) {
	/*Compute auxillary...*/
	auxillary(t,mu.P,x,a);
	if(!tr){
	  if(n_steps == write_count*mynum.write_step) {
	    /* if(write_count < BUFFER)//mynum.global_buffer */
	    /*   write_data_array(); */
    	    /* if(buffer_overfull_check()) break; */
	    if(wr)
	      write_data_file(data);
	    write_count++;
	  }
	}
	int status = gsl_odeiv2_evolve_apply (e, c, s,
					     &sys, &t, t1, &h, x);
	if(status != GSL_SUCCESS)
	  break;
	n_steps++;
      }//while (t<t1)
      if(!tr)
	fprintf(data,"\n\n");/*Separating data in the file*/
      if(nruns >= 5){
	i = (int)(outPrint*nruns/5);
	if(k == i){
	  outPrint++;
	  printf("Completed: %d/%d\n",k+1,nruns);
	}
      }
    }//k over nruns
    //mynum.global_buffer = BUFFER;
    gsl_odeiv2_evolve_free(e);
    gsl_odeiv2_control_free(c);
    gsl_odeiv2_step_free(s);
  }
  /***Closing data file stream.***/
  if(wr && !tr && (strcmp(method,"complex")!=0))
    fclose(data);
  /***Free random generator***/
  if(strcmp(method,"discrete") == 0 || lang_flag)
    free_rand();
  /***Plotting results***/
  if(wr && !tr && graph_flag && (strcmp(method,"complex")!=0)){
    gplot_results(data_name,complex_run_flag,method);
  }
  /*** Analysis of the trajectory ***/
  if(!complex_run_flag){
    //if complex method or other method not included inside complex run
    analyze_traj(data_name);
  }
  /*** Final printing to the user ***/
  if(strcmp(method,"complex")!=0){
    if((strcmp(method,"discrete")==0) || (lang_flag))
      fprintf(stdout,"Time: %G. Number of runs: %d\n",time,nruns);
    else{
      if(lang_flag)
	fprintf(stdout,"Time: %G. Number of runs: %d\n",time,nruns);
      else
	fprintf(stdout,"Time: %G. Number of steps: %d\n",t,n_steps);
    }
  }
  return 0;
}

int analyze_traj(const char *fname)
{/* The function is intended to compute the common entities out from the data file */
  int complex_true,nRuns,pure_det,lf,*info;
  FILE *tmp = fopen(fname,"r");
  if((info=get_info_data(info,tmp)) == NULL){
    fprintf(stderr,"Error occurred. Exit.\n");
    return 1;
  }
  fclose(tmp);
  printf("complex=%d,nRuns=%d,pure_det=%d,lf=%d\n",info[0],info[1],info[2],info[3]);
  complex_true = info[0];
  nRuns = info[1];
  pure_det = info[2];
  lf = info[3];
  if(complex_true){/* Complex run */
    perDet = compute_period(perDet,&nPerDet,
			    per_method,fname,1,0);
    perStoch = compute_period(perStoch,&nPerStoch,
			      per_method,fname,nRuns,1);
  }
  else{/* Not complex run */
    if(!pure_det){
      /* Stochastic method: langevin or discrete*/
      nPerDet = 0;
      perStoch = compute_period(perStoch,&nPerStoch,
				per_method,fname,nRuns,0);
    }
    else{
      /* Only deterministic method */
      nPerStoch = 0;
      perDet = compute_period(perDet,&nPerDet,
			      per_method,fname,nRuns,0);
    }
  }
  
  return 0;
}

int buffer_overfull_check(){
  int i;
  if(write_count == BUFFER){
    fprintf(stdout,"Buffer(%d) is overfull(t=%lf), continue?[Y/n]\n",n_steps,t);
    i=getchar();
    while(i!='n' && i!='N' && i!='\n' && i!='y' && i!='Y'){
      fprintf(stdout,"Please `Y' or 'N':\n");
      i=getchar();
    }
    if(i=='n' || i=='N') return 1;
    if(i=='\n' || i=='y' || i=='Y'){
      BUFFER += BUF_INCR;
      ts = (double *)realloc(ts,BUFFER*sizeof(double));
      if(ts == NULL)
	printf("Error: couldn't allocate memory for time storage\n");
      xs = (double **)realloc(xs,BUFFER*sizeof(double *));
      if(xs == NULL)
	printf("Error: couldn't allocate memory for variable storage\n");
      for(i=0;i<BUFFER;i++){
	xs[i] = (double *)realloc(xs[i],DIM*sizeof(double));
	if(xs[i] == NULL)
	  printf("Error: couldn't allocate memory for variable points storage\n");
      }
      fprintf(stdout,"Allocated successfully %d points.\n",BUFFER);
    }
  }
  return 0;
}

void write_data_file(FILE *data){
  int i;
  fprintf(data,"%G ",t);
  for(i=0;i<DIM;i++)
    fprintf(data,"%G ",x[i]);
  for(i=0;i<AUXNUM;i++)
    fprintf(data,"%G ",a[i]);
  fprintf(data,"\n");
}

void write_data_array(){
  int i;
  ts[write_count] = t;
  for(i=0;i<DIM;i++)
    xs[write_count][i] = x[i];
}

char *app_ext(char const * basename, char const * ext, char * out)
{/* This function appends extension specified to the basename of file
    names*/
  int i,j;
  //char *out;
  out=(char *)calloc((strlen(basename)+strlen(ext)+2),sizeof(char));
  for(i=0;i<strlen(basename);i++)
    out[i]=basename[i];
  out[i]='.';i++;
  for(j=0;j<strlen(ext);j++)
    out[i+j]=ext[j];
  out[i+j]='\0';

  return out;

}
