/****************************************************************************
   GLASHCtl - a simple tray applet for controlling lashd
   
   Copyright (C) 2006  Lars Luthman <lars.luthman@gmail.com>
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
****************************************************************************/

#include <algorithm>
#include <iostream>
#include <ctime>
#include <map>
#include <string>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/errno.h>


#include <gtkmm/stock.h>
#include <gtkmm/main.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/entry.h>

#include <sigc++/sigc++.h>

#include "trayicon.hpp"


using namespace Gdk;
using namespace Glib;
using namespace Gtk;
using namespace sigc;
using namespace std;


TrayIcon::TrayIcon() 
  : m_pixbuf(Pixbuf::create_from_file(DATA_DIR "/lash_96px.png")),
    m_iconset(m_pixbuf),
    m_image(m_iconset, ICON_SIZE_LARGE_TOOLBAR),
    m_startitem("Start session handler"),
    m_stopitem("Stop session handler"),
    m_restoreitem("Restore session"),
    m_saveitem("Save session"),
    m_closeitem("Close session"),
    m_setsessiondiritem("Move session directory"),
    m_setsessionnameitem("Rename session"),
    m_session_dialog("Select session", FILE_CHOOSER_ACTION_SELECT_FOLDER),
    m_session_save_dialog("Enter directory name", FILE_CHOOSER_ACTION_SAVE),
    m_lashd_running(false) {
  
  m_session_dialog.add_button(Stock::OK, RESPONSE_OK);
  m_session_dialog.add_button(Stock::CANCEL, RESPONSE_CANCEL);
  m_session_dialog.set_default_response(RESPONSE_OK);
  
  m_session_save_dialog.add_button(Stock::OK, RESPONSE_OK);
  m_session_save_dialog.add_button(Stock::CANCEL, RESPONSE_CANCEL);
  m_session_save_dialog.set_default_response(RESPONSE_OK);

  m_image.set_sensitive(false);
  add(m_image);
  show_all();
  
  set_events(BUTTON_PRESS_MASK);
  
  m_menu.append(m_restoreitem);
  m_restoreitem.set_sensitive(false);
  m_restoreitem.signal_activate().connect(mem_fun(*this, 
                                                  &TrayIcon::do_restore));

m_menu.append(m_setsessionnameitem);
  m_setsessionnameitem.set_sensitive(false);
  m_setsessionnameitem.signal_activate().
    connect(mem_fun(*this, &TrayIcon::do_set_session_name));
  
  m_menu.append(m_saveitem);
  m_saveitem.set_sensitive(false);
  m_saveitem.signal_activate().connect(signal_save_session);

  m_menu.append(m_setsessiondiritem);
  m_setsessiondiritem.set_sensitive(false);
  m_setsessiondiritem.signal_activate().
    connect(mem_fun(*this, &TrayIcon::do_set_session_dir));

  m_menu.append(m_closeitem);
  m_closeitem.set_sensitive(false);
  m_closeitem.signal_activate().connect(signal_close_session);
  m_menu.append(*manage(new SeparatorMenuItem));

  m_startitem.signal_activate().connect(signal_start_lashd);
  m_menu.append(m_startitem);
  m_stopitem.signal_activate().connect(signal_stop_lashd);
  m_stopitem.set_sensitive(false);
  m_menu.append(m_stopitem);
  m_menu.append(*manage(new SeparatorMenuItem));
  
  MenuItem* quititem = manage(new MenuItem("Quit"));
  quititem->signal_activate().connect(mem_fun(*this, &TrayIcon::do_quit));
  m_menu.append(*quititem);
  
  m_menu.show_all();
}


bool TrayIcon::on_button_press_event(GdkEventButton* event) {
  if (event->button == 1)
    signal_toggle_window();
  if (event->button == 3)
    m_menu.popup(event->button, event->time);
  return true;
}


void TrayIcon::lashd_started() {
  m_image.set_sensitive(true);
  m_startitem.set_sensitive(false);
  m_stopitem.set_sensitive(true);
  m_restoreitem.set_sensitive(true);
  m_lashd_running = true;
}


void TrayIcon::lashd_stopped() {
  m_image.set_sensitive(false);
  m_stopitem.set_sensitive(false);
  m_startitem.set_sensitive(true);
  m_restoreitem.set_sensitive(false);
  m_saveitem.set_sensitive(false);
  m_setsessiondiritem.set_sensitive(false);
  m_setsessionnameitem.set_sensitive(false);
  m_closeitem.set_sensitive(false);
  dynamic_cast<Label*>(m_saveitem.get_child())->set_text("Save session");
  dynamic_cast<Label*>(m_setsessiondiritem.get_child())->
    set_text("Move session directory");
  dynamic_cast<Label*>(m_closeitem.get_child())->set_text("Close session");
  dynamic_cast<Label*>(m_setsessionnameitem.get_child())->
    set_text("Rename session");
  m_lashd_running = false;
  internal_signal_lashd_stopped();
}


void TrayIcon::session_changed(const string& name) {
  m_session_name = name;
  if (name == "") {
    m_saveitem.set_sensitive(false);
    m_setsessiondiritem.set_sensitive(false);
    m_setsessionnameitem.set_sensitive(false);
    m_closeitem.set_sensitive(false);
    m_stopitem.set_sensitive(true);
    m_restoreitem.set_sensitive(true);
    dynamic_cast<Label*>(m_saveitem.get_child())->set_text("Save session");
    dynamic_cast<Label*>(m_closeitem.get_child())->set_text("Close session");
    dynamic_cast<Label*>(m_setsessiondiritem.get_child())->
      set_text("Move session directory");
    dynamic_cast<Label*>(m_setsessionnameitem.get_child())->
      set_text("Rename session");
}
  else {
    dynamic_cast<Label*>(m_saveitem.get_child())->
      set_text(string("Save '") + name + "'");
    dynamic_cast<Label*>(m_closeitem.get_child())->
      set_text(string("Close '") + name + "'");
    dynamic_cast<Label*>(m_setsessiondiritem.get_child())->
      set_text(string("Move '") + name + "' directory");
    dynamic_cast<Label*>(m_setsessionnameitem.get_child())->
      set_text(string("Rename '") + name + "'");
    m_saveitem.set_sensitive(true);
    m_setsessiondiritem.set_sensitive(true);
    m_setsessionnameitem.set_sensitive(true);
    m_closeitem.set_sensitive(true);
    m_stopitem.set_sensitive(false);
    m_restoreitem.set_sensitive(false);
  }
}


void TrayIcon::do_set_session_dir() {
  struct stat stats;
  if (m_session_save_dialog.run() == RESPONSE_OK) {
    if ((stat(m_session_save_dialog.get_filename().c_str(), &stats) == -1) 
        && (errno == ENOENT)) {
      signal_set_session_dir(m_session_save_dialog.get_filename());
    }
    else {
      // TODO: this case will never happen as the filechooser won't let us 
      // choose existing directories. it rather enters them upon pressing
      // return.
      MessageDialog dialog("Couldn't move session directory. "
                           "Directory exists."); 
      dialog.run();
    }
  }
  m_session_save_dialog.hide();
}


void TrayIcon::do_set_session_name() {
  Gtk::Dialog dialog("Enter new session name");

  dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);
  dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  dialog.set_default_response(Gtk::RESPONSE_OK);
  Gtk::Entry entry;
  dialog.get_vbox()->pack_end(entry);
  entry.show();
  entry.set_text(m_session_name);
  entry.signal_activate().
    connect(bind<int>(mem_fun(dialog, &Gtk::Dialog::response), 
                      Gtk::RESPONSE_OK));
  if ((dialog.run() == Gtk::RESPONSE_OK) && (entry.get_text() != ""))
    signal_set_session_name(entry.get_text());
}


void TrayIcon::do_restore() {
  if (m_session_dialog.run() == RESPONSE_OK)
    signal_restore_session(m_session_dialog.get_filename());
  m_session_dialog.hide();
}


void TrayIcon::do_quit() {
  if (m_lashd_running) {
    MessageDialog dlg("Do you want to shut down the session handler?",
                      false, MESSAGE_QUESTION, BUTTONS_YES_NO);
    if (dlg.run() == RESPONSE_YES) {
      internal_signal_lashd_stopped.connect(&Main::quit);
      signal_stop_lashd();
    }
    else
      Main::quit();
  }
  else
    Main::quit();
}
