# Class SourceFile definition

# Copyright (C) 2011 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Purpose: encapsulates data about a source file, taken from the file itself
#   or from the database as appropriate

class SourceFile
  attr_reader :in_db, :inode, :md5, :mime_type, :mtime, :path, :sha1

  # Create a new SourceFile
  # ~~~~~~~~~~~~~~~~~~~~~~~
  def initialize( path )
    $log.write( Log::DEBUG, "SourceFile: initializing #{ path }" )
    @path = "#{ path }"
    @inode, @mtime = GetInodeAndMtime( @path )
    #$log.write( Log::DEBUG, "SourceFile.initialize: after GetInodeAndMtime: @inode is #{ @inode }" )
    if @inode == nil; raise end

    # Lookup this file in the database
    @in_db, rec_inode, rec_md5, rec_mime_type, rec_mtime, rec_sha1 = \
      LookupSourceFileByPath( @path )
    #$log.write( Log::DEBUG, "SourceFile.initialize: after LookupSourceFileByPath" )

    # Delete from database if file has changed
    if @in_db && ( @inode != rec_inode || @mtime != rec_mtime )
      #$log.write( Log::DEBUG, "SourceFile.initialize: deleting DB record" )
      delete_from_db( )
    end

    # Populate remaining file properties
    $log.write( Log::DEBUG, "SourceFile.initialize: populating remaining file properties" )
    if @in_db 
      @md5 = rec_md5
      @mime_type = rec_mime_type
      @sha1 = rec_sha1
    else
      @md5, @sha1 = GetChecksums( @path )
      #$log.write( Log::DEBUG, "SourceFile.initialize: after GetChecksums: @md5: #{ @md5 }" )
      if @md5 == nil; raise end
      @mime_type = GetMIMEtype( @path )
      #$log.write( Log::DEBUG, "SourceFile.initialize: after GetMIMEtype: @mime_type: #{ @mime_type }" )
      if @mime_type == nil; raise end
      #$log.write( Log::DEBUG, "SourceFile.initialize: inserting DB record" )
      insert_into_db( )
    end
  end

  # Insert this SourceFile into the database
  # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  def insert_into_db
    InsertSourceFileIntoDB( @inode, @md5, @mime_type, @mtime, @path, @sha1 )
    @in_db = true
  end

  # Delete this SourceFile from the database
  # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  def delete_from_db
    DeleteSourceFileFromDB( @path )
    @in_db = false
  end

  # Update this SourceFile in the database
  # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  def update_db
    UpdateSourceFileInDB( @inode, @md5, @mime_type, @mtime, @path, @sha1 )
    @in_db = true
  end

end
