# Copyright (C) 2012 Charles Atkinson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

# Purpose: common code library for bash scripts

# Usage: 
#	* Source this file via the . or source command
#   * Global variables referenced by functions in this library:
#     - $debugging: set to $true or $false. Set by library user.
#     - $false: set by this library
#     - $global_error_flag: set by this library's msg function when called with msgclass E
#     - $global_warning_flag: set by this library's msg function when called with msgclass W
#     - $have_tty: set to $true or $false. Set by this library.
#     - $my_nam: caller's short name.  Set by this library.
#     - $redirect_afn: name of file to receive any redirection. Set by this library.
#     - $true: set by this library

#   * Functions called by functions in this library:
#     - fct: function call trace, called by some functions in this library. In this library.
#     - finalise: clean up and exit. Provided by library user.
	
# Environment:
#	Developed and tested on Debian 6.0 Squeeze 64 bit with bash 4.1.5

# Configure shell
# ~~~~~~~~~~~~~~~
export PATH=/usr/sbin:/sbin:/usr/bin:/bin
set -o nounset
shopt -s extglob nullglob
umask 0022

# Utility variables
# ~~~~~~~~~~~~~~~~~
readonly false=
readonly my_nam=${0##*/}
readonly true=true

# Reporting control variables
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~
global_error_flag=$false
global_warning_flag=$false

# Determine whether there is a terminal available for output
# ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
# The "standard" test is to check $PS1 but this test is more reliable
have_tty=$false
case "$( /bin/ps -p $$ -o tty 2>&1 )" in
    *TT*-* | *TT*\?* ) 
        ;;  
    *TT* )
        have_tty=$true
esac

# Set traps
# ~~~~~~~~~
trap 'finalise 129' 'HUP'
trap 'finalise 130' 'INT'
trap 'finalise 131' 'QUIT'
trap 'finalise 143' 'TERM'

# Function definitions
# ~~~~~~~~~~~~~~~~~~~~
# In alphabetical order:
# * ck_cmd
# * ck_file
# * ck_integer
# * fct
# * msg

#--------------------------
# Name: ck_cmd
# Purpose: checks that the command is executable
# Usage: ck_cmd <command>
# Output: writes any errors to stdout
# Return: non-zero on error
#--------------------------
function ck_cmd {

    local buf command command_canonical

	command=$1

    if [[ ! $command =~ ^/ ]]; then
		case $( type -t "$command" ) in
            alias )
				echo "$command is an alias" >&2
                return 1
				;;
            keyword )
				echo "$command is a keyword" >&2
                return 1
				;;
    	    function | builtin | file )
                return 0
				;;
            '' )
				echo "$command not found" >&2
                return 1
        esac
    fi

    command_canonical=$( readlink --canonicalize-existing "$command" )
    [[ $command_canonical = '' ]] && { echo "$command: no such file" >&2; return 1; }
    if [[ $command_canonical = $command ]]; then
        ck_file "$command" f:rx:
        return $?
    else
        buf=$( ck_file "$command_canonical" f:rx: 2>&1 )
        [[ $? -eq 0 ]] && return 0
        echo "$command is $command_canonical after resolving synlinks"$'\n'"$buf"
        return 1
    fi

}  #  end of function ck_cmd

#--------------------------
# Name: ck_file
# Purpose: for each file listed in the argument list: checks that it is 
#   reachable, exists and that the user has the listed permissions
# Usage: ck_file [ file_name <file_type>:<permission>...:[a] ]
#   where 
#       file_name is a file name
#       file_type is b (block special file), f (file) or d (directoy)
#       permission is r, w and x
#       a requires that file_name must be absolute (begin with /)
#   Example: ck_file foo d:rwx:
# Output: writes any errors to stdout
# Return: non-zero on error
#--------------------------
function ck_file {

    local absolute_flag buf file_name file_type perm perms retval

    # For each file ...
    # ~~~~~~~~~~~~~~~~~
    retval=0
    while [[ $# -gt 0 ]]
    do  
        file_name=$1
        file_type=${2%%:*}
        buf=${2#$file_type:}
        perms=${buf%%:*}
        absolute=${buf#$perms:}
		case $absolute in 
			'' )
            	absolute_flag=$false
				;;
			a )
            	absolute_flag=$true
				;;
			* )
                echo "$my_nam: ck_file: invalid absoluteness flag in '$2' specified for file '$file_name'" >&2
                let retval=retval+1
            	absolute_flag=$false
		esac
        shift 2

        # Is the file reachable and does it exist?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        case $file_type in
            b ) 
                if [[ ! -b $file_name ]]; then
                    echo "file '$file_name' is unreachable, does not exist or is not a block special file" >&2
                    let retval=retval+1
                    continue
                fi  
                ;;  
            f ) 
                if [[ ! -f $file_name ]]; then
                    echo "file '$file_name' is unreachable, does not exist or is not an ordinary file" >&2
                    let retval=retval+1
                    continue
                fi  
                ;;  
            d ) 
                if [[ ! -d $file_name ]]; then
                    echo "directory '$file_name' is unreachable, does not exist or is not a directory" >&2
                    let retval=retval+1
                    continue
                fi
                ;;
            * )
                echo "$my_nam: ck_file: invalid file type '$file_type' specified for file '$file_name'" >&2
                return 1
        esac

        # Does the file have the requested permissions?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        buf="$perms"
        while [[ $buf ]]
        do
            perm="${buf:0:1}"
            buf="${buf:1}"
            case $perm in
                r )
                    if [[ ! -r $file_name ]]; then
                        echo "file '$file_name' does not have requested read permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                w )
                    if [[ ! -w $file_name ]]; then
                        echo "file '$file_name' does not have requested write permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                x )
                    if [[ ! -x $file_name ]]; then
                        echo "file '$file_name' does not have requested execute permission" >&2
                        let retval=retval+1
                        continue
                    fi
                    ;;
                * )
                    echo "$my_nam: ck_file: unexpected permisssion '$perm' requested for file '$file_name'" >&2
                    return 1
            esac
        done

        # Does the file have the requested absoluteness?
        # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        if [[ $absolute_flag && ${file_name:0:1} != / ]]; then
          	echo "file '$file_name' does not begin with /" >&2
            let retval++
		fi

    done

return $retval

}  #  end of function ck_file

#--------------------------
# Name: ck_integer
# Purpose: checks whether a string is an integer between upper and lower bounds
# Usage:
#   $1 - string to check
#   $2 - lower bound (may be negative)
#   $3 - upper bound (may be negative)
#--------------------------
function ck_integer {

	if [[ ! $1 =~ '^-?[0-9]+$' ]]; then
		echo "'$1' is not an integer" >&2
	else
		[[ $1 -lt $2 ]] && echo "'$1' is less than $2" >&2
		[[ $1 -gt $3 ]] && echo "'$1' is greater than $3" >&2
	fi

}  # end of function ck_integer

#--------------------------
# Name: fct
# Purpose: function call trace (for debugging)
# $1 - name of calling function 
# $2 - message.  If it starts with "started" or "returning" then the output is prettily indented
#--------------------------
function fct {

	[[ ! $debugging ]] && return 0

	fct_ident="${fct_indent:=}"

	case $2 in
		'started'* )
			fct_indent="$fct_indent  "
			msg -l "$log_afn" I "FCT: $fct_indent$1: $2"
			;;
		'returning'* )
			msg -l "$log_afn" I "FCT: $fct_indent$1: $2"
			fct_indent="${fct_indent#  }"
			;;
		* )
			msg -l "$log_afn" I "FCT: $fct_indent$1: $2"
	esac

}  # end of function fct

#--------------------------
# Name: msg
# Purpose: generalised messaging interface
# Usage: msg class msg_text
#    class: must be one of D, E, I or W indicating Debug, Error, Information or Warning
#    msg_text: is the text of the message
# Output: information messages to stdout; the rest to stderr
# Return code: always zero
#--------------------------
function msg {

    local class message_text prefix

    class="${1:-}"
    case "$class" in 
    	D ) 
			prefix='DEBUG: '
	    	;;
    	E ) 
			global_error_flag=$true
			prefix='ERROR: '
	    	;;
    	I ) 
			prefix=
	    	;;
    	W ) 
			global_warning_flag=$true
			prefix='WARN: '
	    	;;
	    * )
	    	msg E "$my_nam: msg: invalid class '$class': '$*'"
			class=I
    esac
    message_text="${2:-}"

	if [[ $class = I ]]; then
		echo "$( date '+%H:%M:%S ' )$prefix$message_text"
	else
		echo "$( date '+%H:%M:%S ' )$prefix$message_text" >&2
	fi

    return 0

}  #  end of function msg

#--------------------------
# Name: my_readlink
# Purpose: generates message component if argument path contains symlinks
# Usage: my_readlink path
# Output: if pathcontains symlinks "(<readlink output>)" else nothing
# Return code: always zero
#--------------------------
function my_readlink {

    local buf path

    path=${1%/}
    buf=$( readlink --canonicalize --no-newline "$path" 2>&1 )
    [[ $buf != $path ]] && echo -n -E "($buf)"
    return 0

}  #  end of function my_readlink

#--------------------------
# Name: parse_omindex_sh_cfg
# Purpose: parses the configuration file
# $1 - pathname of file to parse
#--------------------------
function parse_omindex_sh_cfg {

    local buf buf2 cfg_fn data emsg keyword

    # Does the file exist?
    # ~~~~~~~~~~~~~~~~~~~~
    cfg_fn="$1"
    buf="$( ck_file $cfg_fn f:r: 2>&1 )"
    if [[ $buf != '' ]]
    then
        msg 'E' "Terminating on configuration file problem: $buf"
        finalise 1
    fi

    # Initialisation
    # ~~~~~~~~~~~~~~
    java_opts=
    emsg=''
    exec 3< $cfg_fn                                # set up the config file for reading on file descriptor 3

    # For each line in the config file ...
    # ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    filters=
    while read -u 3 buf                             # for each line of the config file
    do
        buf="${buf%%#*}"                            # strip any comment
        buf="${buf%%*(  )}"                         # strip any trailing spaces and tabs
        buf="${buf##*(  )}"                         # strip any leading spaces and tabs
        if [[ $buf = '' ]]; then
            continue                                # empty line
        fi
        keyword="${buf%%=*}"
        keyword="${keyword%%*([[:space:]])}"        # strip any trailing spaces and tabs
        data="${buf#*=}"
        data="${data##*([[:space:]])}"              # strip any leading spaces and tabs
        case "$keyword" in
            'JAVA_OPTS' )
                java_opts=$data
                ;;
            'omindex log cleaning' )
                omindex_log_cleaning=$data
                ;;
            'filter' )
                filters="$filters"$'\n'"$data"
                ;;
            'omega index database directory' )
                index_db_dir=${data%%*(/)}/         # silently ensure a single trailing slash
                ;;
            * )
                emsg="$emsg"$'\n'"  Unrecognised keyword. '$keyword'"
                ;;
        esac

    done
    exec 3<&- # free file descriptor 3

    [[ $filters != '' ]] && filters=${filters#$'\n'}

    return 0

}  # end of function parse_omindex_sh_cfg

