/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "eyebrows.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include <stdlib.h>
#include "util.h"
#include "object.h"

DoodleHashEyeBrowPair* doodlehash_eyebrows_new()
{
  DoodleHashEyeBrowPair *e = g_malloc (sizeof (DoodleHashEyeBrowPair));
  if (!e)
    return NULL;
  memset (e, 0, sizeof (DoodleHashEyeBrowPair));
  e->left = g_malloc (sizeof (DoodleHashEyeBrow));
  if (!e->left)
    {
      g_free (e);
      return NULL;
    }
  memset (e->left, 0, sizeof (DoodleHashEyeBrow));
  e->left->base = doodlehash_object_new();
  e->right = g_malloc (sizeof (DoodleHashEyeBrow));
  if (!e->right)
    {
      g_free (e->left);
      g_free (e);
      return NULL;
    }
  memset (e->right, 0, sizeof (DoodleHashEyeBrow));
  e->right->base = doodlehash_object_new();
  return e;
}
void doodlehash_eyebrow_copy(const DoodleHashEyeBrow*orig, DoodleHashEyeBrow *copy)
{
  g_free (copy->base);
  copy->base = doodlehash_object_copy(orig->base);
  copy->connect_to_eye_x = orig->connect_to_eye_x;
  copy->connect_to_eye_y = orig->connect_to_eye_y;
}

DoodleHashEyeBrowPair* doodlehash_eyebrows_copy(const DoodleHashEyeBrowPair *orig)
{
  DoodleHashEyeBrowPair *eyebrows = doodlehash_eyebrows_new();
  if (orig->left)
    doodlehash_eyebrow_copy(orig->left, eyebrows->left);
  if (orig->right)
    doodlehash_eyebrow_copy(orig->right, eyebrows->right);
  return eyebrows;
}

int doodlehash_eyebrow_load_from_svg(DoodleHashEyeBrow *e, const char *file)
{
  if (doodlehash_object_load_from_svg_file(e->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)e->base->data, 
                                                 e->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-eye", 
                         &e->connect_to_eye_x, &e->connect_to_eye_y) != 0)
    return -2;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashEyeBrowPair* doodlehash_eyebrows_load_from_svg_file(const gchar *left, const gchar *right)
{
  DoodleHashEyeBrowPair *eyebrow = doodlehash_eyebrows_new();
  if (doodlehash_eyebrow_load_from_svg(eyebrow->left, left) == 0 &&
      doodlehash_eyebrow_load_from_svg(eyebrow->right, right) == 0)
    return eyebrow;
  else
    {
      doodlehash_eyebrows_free(eyebrow);
      return NULL;
    }
}
void doodlehash_eyebrows_get_width_and_height(DoodleHashEyeBrowPair* eyebrow, guint side, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(side == DOODLEHASH_SIDE_LEFT ? eyebrow->left->base : eyebrow->right->base, width, height);
}

void doodlehash_eyebrows_set_width(DoodleHashEyeBrowPair* eb, guint side, guint32 width)
{
  DoodleHashEyeBrow *e = (side == DOODLEHASH_SIDE_LEFT) ? eb->left : eb->right;
  double scale = (double) width / (double)e->base->width;
  e->connect_to_eye_x *= scale;
  doodlehash_object_set_width (e->base, e->base->width * scale);
}

guint32 doodlehash_eyebrows_get_width(DoodleHashEyeBrowPair* eb, guint side)
{
  return doodlehash_object_get_width(side == DOODLEHASH_SIDE_LEFT ? eb->left->base : eb->right->base);
}

guint32 doodlehash_eyebrows_get_height(DoodleHashEyeBrowPair* eyebrow, guint side)
{
  return doodlehash_object_get_height(side == DOODLEHASH_SIDE_LEFT ? eyebrow->left->base : eyebrow->right->base);
}

void doodlehash_eyebrows_set_height(DoodleHashEyeBrowPair* eb, guint side, guint32 height)
{
  DoodleHashEyeBrow *e = (side == DOODLEHASH_SIDE_LEFT) ? eb->left : eb->right;
  double scale = (double) height / (double)e->base->height;
  e->connect_to_eye_x *= scale;
  doodlehash_object_set_height (e->base, e->base->height * scale);
}

void doodlehash_eyebrows_get_eye_connection_point(DoodleHashEyeBrowPair* eyebrow, guint side, guint32* x, guint32* y)
{
  DoodleHashEyeBrow *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = eyebrow->left;
  else
    e = eyebrow->right;
  *x = e->connect_to_eye_x;
  *y = e->connect_to_eye_y;
}

void doodlehash_eyebrows_set_eye_connection_point(DoodleHashEyeBrowPair* eyebrow, guint side, guint32 x, guint32 y)
{
  DoodleHashEyeBrow *e;
  if (side == DOODLEHASH_SIDE_LEFT)
    e = eyebrow->left;
  else
    e = eyebrow->right;
  e->connect_to_eye_x = x;
  e->connect_to_eye_y = y;
}

guint32 doodlehash_eyebrows_get_eye_connection_point_x(DoodleHashEyeBrowPair* eyebrow, guint side)
{
  return side == DOODLEHASH_SIDE_LEFT ? eyebrow->left->connect_to_eye_x : eyebrow->right->connect_to_eye_x;
}

guint32 doodlehash_eyebrows_get_eye_connection_point_y(DoodleHashEyeBrowPair* eyebrow, guint side)
{
  return side == DOODLEHASH_SIDE_LEFT ? eyebrow->left->connect_to_eye_y : eyebrow->right->connect_to_eye_y;
}

void doodlehash_eyebrows_set_random_colours(DoodleHashEyeBrowPair* eyebrow, guint code, int matching)
{
  doodlehash_object_set_random_matching_colours(eyebrow->left->base, 
                                                eyebrow->right->base, 
                                                code, matching);
}
void doodlehash_eyebrows_set_colours(DoodleHashEyeBrowPair* eyebrow, guint side, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(side == DOODLEHASH_SIDE_LEFT ? eyebrow->left->base : eyebrow->right->base, code, s, f);
}

gchar *doodlehash_eyebrows_get_fill_colour(DoodleHashEyeBrowPair* eyebrow, guint side, guint code)
{
  return doodlehash_object_get_fill_colour(side == DOODLEHASH_SIDE_LEFT ? eyebrow->left->base : eyebrow->right->base, code);
}

gchar *doodlehash_eyebrows_get_stroke_colour(DoodleHashEyeBrowPair* eyebrow, guint side, guint code)
{
  return doodlehash_object_get_stroke_colour(side == DOODLEHASH_SIDE_LEFT ? eyebrow->left->base : eyebrow->right->base, code);
}

void doodlehash_eyebrows_scale(DoodleHashEyeBrowPair *eb, guint side, double scale)
{
  DoodleHashEyeBrow *e = (side == DOODLEHASH_SIDE_LEFT) ? eb->left : eb->right;
  e->connect_to_eye_x *= scale;
  e->connect_to_eye_y *= scale;
  doodlehash_object_scale(e->base, scale);
}

void doodlehash_eyebrows_free(DoodleHashEyeBrowPair* e)
{
  if (e && e->left && e->left->base)
    doodlehash_object_free (e->left->base);
  if (e && e->right && e->right->base)
    doodlehash_object_free (e->right->base);
  if (e && e->right)
    g_free (e->right);
  if (e && e->left)
    g_free (e->left);
  g_free (e);
}

