/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "nose.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include "object.h"

DoodleHashNose* doodlehash_nose_new()
{
  DoodleHashNose *n = g_malloc (sizeof (DoodleHashNose));
  if (!n)
    return NULL;
  memset (n, 0, sizeof (DoodleHashNose));
  n->base = doodlehash_object_new();
  return n;
}

DoodleHashNose* doodlehash_nose_copy(const DoodleHashNose*orig)
{
  DoodleHashNose *nose = doodlehash_nose_new();
  g_free (nose->base);
  nose->base = doodlehash_object_copy(orig->base);
  nose->connect_to_head_x = orig->connect_to_head_x;
  nose->connect_to_head_y = orig->connect_to_head_y;
  return nose;
}


int doodlehash_nose_load_from_svg(DoodleHashNose *n, const char *file)
{
  if (doodlehash_object_load_from_svg_file(n->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)n->base->data, 
                                                 n->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-head", 
                         &n->connect_to_head_x, &n->connect_to_head_y) != 0)
    return -2;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashNose* doodlehash_nose_load_from_svg_file(const char *file)
{
  DoodleHashNose *nose = doodlehash_nose_new();
  if (doodlehash_nose_load_from_svg(nose, file) == 0)
    return nose;
  else
    {
      doodlehash_nose_free(nose);
      return NULL;
    }
}

void doodlehash_nose_get_width_and_height(DoodleHashNose* nose, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(nose->base, width, height);
}

void doodlehash_nose_set_width(DoodleHashNose* nose, guint32 width)
{
  double scale = (double) width / (double)nose->base->width;
  nose->connect_to_head_x *= scale;
  doodlehash_object_set_width(nose->base, nose->base->width * scale);
}

void doodlehash_nose_set_height(DoodleHashNose* nose, guint32 height)
{
  double scale = (double)nose->base->height / (double)height;
  nose->connect_to_head_y *= scale;
  return doodlehash_object_set_height(nose->base, nose->base->height * scale);
}

void doodlehash_nose_get_head_connection_point(DoodleHashNose* nose, guint32* x, guint32* y)
{
  *x = nose->connect_to_head_x;
  *y = nose->connect_to_head_y;
}

void doodlehash_nose_set_head_connection_point(DoodleHashNose* nose, guint32 x, guint32 y)
{
  nose->connect_to_head_x = x;
  nose->connect_to_head_y = y;
}

guint32 doodlehash_nose_get_width(DoodleHashNose* nose)
{
  return doodlehash_object_get_width(nose->base);
}

guint32 doodlehash_nose_get_height(DoodleHashNose* nose)
{
  return doodlehash_object_get_height(nose->base);
}

guint32 doodlehash_nose_get_head_connection_point_x(DoodleHashNose* nose)
{
  return nose->connect_to_head_x;
}

guint32 doodlehash_nose_get_head_connection_point_y(DoodleHashNose* nose)
{
  return nose->connect_to_head_y;
}

void doodlehash_nose_set_random_colours(DoodleHashNose* nose, guint code)
{
  doodlehash_object_set_random_colours(nose->base, code);
}

void doodlehash_nose_set_colours(DoodleHashNose* nose, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(nose->base, code, s, f);
}

gchar *doodlehash_nose_get_fill_colour(DoodleHashNose* nose, guint code)
{
  return doodlehash_object_get_fill_colour(nose->base, code);
}

gchar *doodlehash_nose_get_stroke_colour(DoodleHashNose* nose, guint code)
{
  return doodlehash_object_get_stroke_colour(nose->base, code);
}

void doodlehash_nose_scale(DoodleHashNose *n, double scale)
{
  n->connect_to_head_x *= scale;
  n->connect_to_head_y *= scale;
  doodlehash_object_scale(n->base, scale);
}

void doodlehash_nose_free(DoodleHashNose* n)
{
  if (n && n->base)
    doodlehash_object_free (n->base);
  g_free (n);
}
