/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope teyes it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "eyes.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include "object.h"
DoodleHashEyes* doodlehash_eyes_new()
{
  DoodleHashEyes *m = g_malloc (sizeof (DoodleHashEyes));
  if (!m)
    return NULL;
  memset (m, 0, sizeof (DoodleHashEyes));
  m->base = doodlehash_object_new();
  return m;
}
DoodleHashEyes* doodlehash_eyes_copy(const DoodleHashEyes*orig)
{
  DoodleHashEyes *eyes = doodlehash_eyes_new();
  g_free (eyes->base);
  eyes->base = doodlehash_object_copy(orig->base);
  eyes->connect_to_head_x = orig->connect_to_head_x;
  eyes->connect_to_head_y = orig->connect_to_head_y;
  eyes->connect_to_left_brow_x = orig->connect_to_left_brow_x;
  eyes->connect_to_left_brow_y = orig->connect_to_left_brow_y;
  eyes->connect_to_right_brow_x = orig->connect_to_right_brow_x;
  eyes->connect_to_right_brow_y = orig->connect_to_right_brow_y;
  return eyes;
}

int doodlehash_eyes_load_from_svg(DoodleHashEyes *m, const char *file)
{
  if (doodlehash_object_load_from_svg_file(m->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)m->base->data, 
                                                 m->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-left-brow", 
                         &m->connect_to_left_brow_x, 
                         &m->connect_to_left_brow_y) != 0)
    return -2;
  if (find_connect_point(handle, "#connect-to-right-brow", 
                         &m->connect_to_right_brow_x, 
                         &m->connect_to_right_brow_y) != 0)
    return -3;
  if (find_connect_point(handle, "#connect-to-head", 
                         &m->connect_to_head_x, &m->connect_to_head_y) != 0)
    return -4;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashEyes* doodlehash_eyes_load_from_svg_file(const char *file)
{
  DoodleHashEyes *eyes = doodlehash_eyes_new();
  if (doodlehash_eyes_load_from_svg(eyes, file) == 0)
    return eyes;
  else
    {
      doodlehash_eyes_free(eyes);
      return NULL;
    }
}
void doodlehash_eyes_get_width_and_height(DoodleHashEyes* eyes, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(eyes->base, width, height);
}

void doodlehash_eyes_set_width(DoodleHashEyes* eyes, guint32 width)
{
  double scale = (double) width / (double)eyes->base->width;
  eyes->connect_to_head_x *= scale;
  doodlehash_object_set_width(eyes->base, eyes->base->width * scale);
}

void doodlehash_eyes_set_height(DoodleHashEyes* eyes, guint32 height)
{
  double scale = (double)eyes->base->height / (double)height;
  eyes->connect_to_head_y *= scale;
  doodlehash_object_set_height(eyes->base, eyes->base->height * scale);
}

void doodlehash_eyes_get_head_connection_point(DoodleHashEyes* eyes, guint32* x, guint32* y)
{
  *x = eyes->connect_to_head_x;
  *y = eyes->connect_to_head_y;
}

guint32 doodlehash_eyes_get_head_connection_point_x(DoodleHashEyes* eyes)
{
  return eyes->connect_to_head_x;
}

guint32 doodlehash_eyes_get_head_connection_point_y(DoodleHashEyes* eyes)
{
  return eyes->connect_to_head_y;
}

void doodlehash_eyes_set_head_connection_point(DoodleHashEyes* eyes, guint32 x, guint32 y)
{
  eyes->connect_to_head_x = x;
  eyes->connect_to_head_y = y;
}

guint32 doodlehash_eyes_get_width(DoodleHashEyes* eyes)
{
  return doodlehash_object_get_width(eyes->base);
}

guint32 doodlehash_eyes_get_height(DoodleHashEyes* eyes)
{
  return doodlehash_object_get_height(eyes->base);
}

void doodlehash_eyes_get_left_brow_connection_point(DoodleHashEyes* eyes, guint32* x, guint32* y)
{
  *x = eyes->connect_to_left_brow_x;
  *y = eyes->connect_to_left_brow_y;
}

void doodlehash_eyes_set_left_brow_connection_point(DoodleHashEyes* eyes, guint32 x, guint32 y)
{
  eyes->connect_to_left_brow_x = x;
  eyes->connect_to_left_brow_y = y;
}

guint32 doodlehash_eyes_get_left_brow_connection_point_x(DoodleHashEyes* eyes)
{
  return eyes->connect_to_left_brow_x;
}

guint32 doodlehash_eyes_get_left_brow_connection_point_y(DoodleHashEyes* eyes)
{
  return eyes->connect_to_left_brow_y;
}

void doodlehash_eyes_get_right_brow_connection_point(DoodleHashEyes* eyes, guint32* x, guint32* y)
{
  *x = eyes->connect_to_right_brow_x;
  *y = eyes->connect_to_right_brow_y;
}

void doodlehash_eyes_set_right_brow_connection_point(DoodleHashEyes* eyes, guint32 x, guint32 y)
{
  eyes->connect_to_right_brow_x = x;
  eyes->connect_to_right_brow_y = y;
}

guint32 doodlehash_eyes_get_right_brow_connection_point_x(DoodleHashEyes* eyes)
{
  return eyes->connect_to_right_brow_x;
}

guint32 doodlehash_eyes_get_right_brow_connection_point_y(DoodleHashEyes* eyes)
{
  return eyes->connect_to_right_brow_y;
}

void doodlehash_eyes_set_random_colours(DoodleHashEyes* eyes, guint code)
{
  doodlehash_object_set_random_colours(eyes->base, code);
}

void doodlehash_eyes_set_colours(DoodleHashEyes* eyes, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(eyes->base, code, s, f);
}

gchar *doodlehash_eyes_get_fill_colour(DoodleHashEyes* eyes, guint code)
{
  return doodlehash_object_get_fill_colour(eyes->base, code);
}

gchar *doodlehash_eyes_get_stroke_colour(DoodleHashEyes* eyes, guint code)
{
  return doodlehash_object_get_stroke_colour(eyes->base, code);
}

void doodlehash_eyes_scale(DoodleHashEyes *e, double scale)
{
  e->connect_to_head_x *= scale;
  e->connect_to_head_y *= scale;
  doodlehash_object_scale(e->base, scale);
}

void doodlehash_eyes_free(DoodleHashEyes* e)
{
  if (e && e->base)
    doodlehash_object_free (e->base);
  g_free (e);
}
