/*
 * dvdspanky - a video to DVD MPEG conversion frontend for transcode
 * Copyright (C) 2007  Jeffrey Grembecki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <pcre.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <stdarg.h>
#include <dirent.h>
#include <fnmatch.h>
#include "dvdspanky.h"
#include "config.h"

static char gp_title[] = "dvdspanky 0.10.1 - (C) 2007 Jeffrey Grembecki\n\n";
static char gp_licence[] =
"This program is free software; you can redistribute it and/or modify\n"
"it under the terms of the GNU General Public License as published by\n"
"the Free Software Foundation; either version 2 of the License, or\n"
"(at your option) any later version.\n"
"\n"
"This program is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"GNU General Public License for more details.\n"
"\n"
"You should have received a copy of the GNU General Public License\n"
"along with this program; if not, write to the Free Software\n"
"Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,\n"
"MA 02110-1301, USA.\n";

static char gp_help[] =
"%s -i file [ -o file ] [-12bBFhILnOmNpTv] [-s size] [-r rate] [-R rate] [-c channels] [-V pixels] [-H pixels] [-a mode] [-f filters]\n"
"\n"
"following; [ ] = default, { } = requirements, * = experimental\n"
"\n"
"-1, --firstpass        1st pass only [off]\n"
"-2, --secondpass       2nd pass only including multiplex [off]\n"
"-a, --aspect=MODE      aspect ratio 2=4:3, 3=16:9 [auto] {2,3}\n"
"-b, --border           automatic black border [off]\n"
"-B, --clip             clips edges to fit aspect [off]\n"
"-c, --channels=CHANS   number of audio channels [auto] {1 - 5}\n"
"-f, --filters=FILTERS  specify aditional transcode filters [none]\n"
"-F, --frameadjust      interlace video for framerate conversion [off] *\n"
"-h, --help             this help text\n"
"-H, --hcrop=PIXELS     as above [off]\n"
"-i, --input=FILE       input file\n"
"-I, --nonice           do not run at nice priority 19 [off]\n"
"-L, --licence          show the program licence\n"
"-m, --multi            2 pass mode [off]\n"
"-M, --mplex            multiplex only [off]\n"
"-n, --ntsc             set ntsc [auto]\n"
"-N, --ntscfix          fix variable framerate ntsc video [off] *\n"
"-o, --output=FILE      output filename prefix (w/o extension)\n"
"-O, --normalise        normalise audio volume [off]\n"
"-r, --vrate=RATE       video bitrate [auto] {600 - 7500}\n"
"-p, --pal              set pal [auto]\n"
"-P, --preview          previews 5 frames using feh then exits [off]\n"
"-R, --arate=RATE       audio bitrate [auto] {64 - 448, 192 is good}\n"
"-s, --size=KB          final mpg size in kilobytes\n"
"-T, --postprocess      enable post processing filter [off]\n"
"-v, --verbose          be verbose [off]\n"
"-V, --vcrop=PIXELS     pos for crop, neg for black border [off]\n"
"\nsee man 1 dvdspanky for more information\n";

static struct option gp_longopts[] =
{
	{"firstpass", 0, 0, '1'},
	{"secondpass", 0, 0, '2'},
	{"aspect", 1, 0, 'a'},
	{"border", 0, 0, 'b'},
	{"clip", 0, 0, 'B'},
	{"channels", 1, 0, 'c'},
	{"filters", 1, 0, 'f'},
	{"frameadjust", 0, 0, 'F'},
	{"hcrop", 1, 0, 'H'},
	{"help", 0, 0, 'h'},
	{"input", 1, 0, 'i'},
	{"nonice", 0, 0, 'I'},
	{"licence", 0, 0, 'L'},
	{"mplex", 0, 0, 'M'},
	{"multi", 0, 0, 'm'},
	{"multipass", 0, 0, 'm'},
	{"ntsc", 0, 0, 'n' },
	{"ntscfix", 0, 0, 'N'},
	{"output", 1, 0, 'o'},
	{"normalise", 0, 0, 'O'},
	{"normalize", 0, 0, 'O'},
	{"pal", 0, 0, 'p' },
	{"preview", 0, 0, 'P'},
	{"vrate", 1, 0, 'r'},
	{"arate", 1, 0, 'R'},
	{"size", 1, 0, 's'},
	{"postprocess", 0, 0, 'T'},
	{"vcrop", 1, 0, 'V'},
	{"verbose", 0, 0, 'v'},
};

static char gp_shortopts[] = "12a:bBc:f:FhH:i:ILmnNo:OpPr:R:s:TvV:";

pcre *gp_idregex = NULL;
pcre *gp_transregex = NULL;
pcre *gp_tcvideoregex = NULL;
pcre *gp_tcaudioregex = NULL;

static void printerror(const char *p_format, ...)
{
	va_list argp;

	fprintf(stderr, "[dvdspanky] error: ");
	va_start(argp, p_format);
	vfprintf(stderr, p_format, argp);
	va_end(argp);
	fprintf(stderr, "\n");
}

static void printwarning(const char *p_format, ...)
{
	va_list argp;

	fprintf(stderr, "[dvdspanky] warning: ");
	va_start(argp, p_format);
	vfprintf(stderr, p_format, argp);
	va_end(argp);
	fprintf(stderr, "\n");
}

static void printverbose(const char *p_format, ...)
{
	va_list argp;

	fprintf(stderr, "[dvdspanky] verbose: ");
	va_start(argp, p_format);
	vfprintf(stderr, p_format, argp);
	va_end(argp);
	fprintf(stderr, "\n");
}

/* termination cleanup */
void cleanup()
{
	if(gp_idregex)
		pcre_free(gp_idregex);
	if(gp_transregex)
		pcre_free(gp_transregex);
	if(gp_tcvideoregex)
		pcre_free(gp_tcvideoregex);
	gp_idregex = gp_transregex = gp_tcvideoregex = NULL;
}

/* prepare encoding */
int encode(video_t *p_in, video_t *p_out, int options, const char *p_filters)
{
	char p_cmdline[4096];
	char p_print[1000];
	char p_num[20];
	int asr;

	p_cmdline[0] = 0;

	/* handy safe cat macros */
#define safecat(_s) strncat(p_cmdline, _s, sizeof(p_cmdline) - strlen(p_cmdline) - 1)
#define safecati(_n) { snprintf(p_num, sizeof(p_num), "%d", _n); \
	strncat(p_cmdline, p_num, sizeof(p_cmdline) - strlen(p_cmdline) - 1); }
#define safecatf(_f) { snprintf(p_num, sizeof(p_num), "%f", _f); \
	strncat(p_cmdline, p_num, sizeof(p_cmdline) - strlen(p_cmdline) - 1); }

	/* prepare command line */
	p_cmdline[0] = p_print[0] = p_num[0] = 0;
	safecat("transcode");

	/* nice */
	if(!(options & SPANK_NONICE))
		safecat(" --nice 19");

	/* infile */
	safecat(" -i ");
	safecat(p_in->p_filename);

	/* audio */
	if(p_out->achans)
	{
		safecat(" -E 48000,");
		safecati(p_out->abits);
		safecat(",");
		safecati(p_out->achans);
	}
	else
	{
		safecat(" -E 48000");
	}
	safecat(" -N 0x2000 -b ");
	safecati(p_out->arate);

	/* filters */
	if(options & SPANK_DOUBLEFPS)
		safecat(" -J doublefps");
	if(options & SPANK_PPROCESS)
		safecat(" -J pp=de");
	if(options & SPANK_NORMALISE)
		safecat(" -J normalize");
	if(p_filters[0])
	{
		safecat(" -J ");
		safecat(p_filters);
	}

	/* video */
	if(p_out->aspect < 1.5555)
		asr = 2;
	else
		asr = 3;
	snprintf(p_print, sizeof(p_print) - 1,
		" -w %d --export_fps %.3f --export_asr %d --import_asr %d",
		p_out->vrate, p_out->fps, asr, asr);
	safecat(p_print);
	if(p_out->p_vcodec[0] == 'P')
		safecat(" --encode_fields t");
	else
		safecat(" --encode_fields b");

	/* alter fps */
	if(options & SPANK_NEWFPS)
	{
		safecat(" -M 0 -J ivtc,32detect=force_mode=3,decimate -f ");
		safecatf(p_in->fps);
	}

	/* ntsc fix */
	if(options & SPANK_FIXNTSC)
	{
		safecat(" -M 2 -J ivtc,32detect=force_mode=5,decimate --hard_fps");
	}

	/* croping / bordering */
	if(p_out->vcrop || p_out->hcrop)
	{
		snprintf(p_print, sizeof(p_print), " -j %d,%d,%d,%d",
			p_out->vcrop, p_out->hcrop, p_out->vcrop, p_out->hcrop);
		safecat(p_print);
	}

	/* mplayer input if transcode doesn't understand the input */
	if(options & SPANK_MPLAYER)
	{
		unlink("stream.yuv");
		snprintf(p_print, sizeof(p_print),
			" -H 0 -x mplayer,mplayer -f %.3f -g %dx%d -n 0x1 -e %d,%d,%d",
			p_in->fps, p_in->width, p_in->height, p_in->hz,
			(p_in->abits) ? p_in->abits : 16, p_in->achans);
		safecat(p_print);
	}

	/* preview settings */
	if(options & SPANK_PREVIEW)
	{
		struct dirent **pp_dirent;
		int count;

		p_out->options = options & SPANK_VERBOSE;
		printf("generating previews..\n");
		if(forkitf(NULL, NULL, TRANSCODE_BIN,
			"%s -y jpg,null -F 100 -Z %dx%d -c 300-301,600-601,900-901,1200-1201,1500-1501 -o pre",
			p_cmdline, (int)((double)p_out->height * p_out->aspect), p_out->height,
			p_out->p_filename))
			return 1;

		count = scandir(".", &pp_dirent, 0, 0);
		if(count >= 0)
		{
			printf("displaying previews..\n");
			char p_cmd[512];
			int cmdlen, found;

			found = count;

			/* construct feh command line */
			strcpy(p_cmd, "feh");
			cmdlen = strlen(p_cmd);
			while(count--)
			{
				if(!fnmatch("pre*.jpg", pp_dirent[count]->d_name, 0))
				{
					strncat(p_cmd, " ", sizeof(p_cmd) - cmdlen - 1);
					cmdlen++;
					strncat(p_cmd, pp_dirent[count]->d_name, sizeof(p_cmd) - cmdlen - 1);
					cmdlen += strlen(pp_dirent[count]->d_name);
				}
			}

			/* start feh */
			forkit(NULL, NULL, FEH_BIN, p_cmd);

			/* remove preview images */
			count = found;
			while(count--)
			{
				if(!fnmatch("pre*.jpg", pp_dirent[count]->d_name, 0))
					unlink(pp_dirent[count]->d_name);
			}

			free(pp_dirent);
		}
		return 0;
	}

	/* create ffmpeg.cfg */
	system("echo \"[mpeg2video]\nvrc_minrate=0\nvrc_maxrate=7500\nvrc_buf_size=1792\n\" > ffmpeg.cfg");

	/* execute 1st pass */
	if(options & SPANK_FIRST && !(options & SPANK_MULTIPLEX))
	{
		printf("starting 1st VBR pass..\n");
		p_out->options = (options & SPANK_VERBOSE) | SPANK_FIRST;
		if(forkitf(cb_transcode, (void*)p_out, TRANSCODE_BIN,
			"%s -Z %dx%d -y ffmpeg,null -F mpeg2video -R 1,%s.pass1",
			p_cmdline, p_out->width, p_out->height, p_out->p_filename))
			return 1;
		printf("\n");
	}

	/* execute 2nd pass */
	if(options & SPANK_SECOND && !(options & SPANK_MULTIPLEX))
	{
		printf("starting 2nd VBR pass..\n");
		p_out->options = (options & SPANK_VERBOSE) | SPANK_SECOND;
		if(forkitf(cb_transcode, (void*)p_out, TRANSCODE_BIN,
			"%s -Z %dx%d -y ffmpeg -F mpeg2video -R 2,%s.pass1 -o %s -m %s.ac3",
			p_cmdline, p_out->width, p_out->height, p_out->p_filename,
			p_out->p_filename, p_out->p_filename))
			return 1;
		printf("\n");
	}

	/* CBR encoding */
	if(!(options & (SPANK_FIRST | SPANK_SECOND)) && !(options & SPANK_MULTIPLEX))
	{
		printf("starting CBR encode..\n");
		p_out->options = options & SPANK_VERBOSE;
		if(forkitf(cb_transcode, (void*)p_out, TRANSCODE_BIN,
			"%s -Z %dx%d -y ffmpeg -F mpeg2video -o %s -m %s.ac3",
			p_cmdline, p_out->width, p_out->height, p_out->p_filename, p_out->p_filename))
			return 2;
		printf("\n");
	}

	/* multiplex */
	if((options & SPANK_SECOND) || !(options & (SPANK_FIRST | SPANK_SECOND)) ||
		(options & SPANK_MULTIPLEX))
	{
		printf("multiplexing..\n");
		if((options & SPANK_SECOND))
		{
			if(forkitf(NULL, NULL, MPLEX_BIN, "mplex -V -f 8 -o %s.mpg %s.m2v %s.ac3",
				p_out->p_filename, p_out->p_filename, p_out->p_filename))
				return 3;
		}
		else
		{
			if(forkitf(NULL, NULL, MPLEX_BIN, "mplex -f 8 -o %s.mpg %s.m2v %s.ac3",
				p_out->p_filename, p_out->p_filename, p_out->p_filename))
				return 3;
		}

		snprintf(p_print, sizeof(p_print), "%s.m2v", p_out->p_filename);
		unlink(p_print);
		snprintf(p_print, sizeof(p_print), "%s.ac3", p_out->p_filename);
		unlink(p_print);
	}

#undef safecat
#undef safecati
#undef safecatf

	return 0;
}

/* tcprobe identify callback */
int cb_tcprobe(const char *p_string, void *p_options)
{
	char *p_buffer;
	video_t *p_video;
	int numstrings, length, p_vector[20];
	int i;

	/* option is a video_t */
	p_video = (video_t*)p_options;

	/* initialisation command */
	if(!p_string)
	{
		/* pre set feedback as nothing found */
		p_video->feedback = 0;
		return 0;
	}

	/* video regex */
	length = strlen(p_string);
	numstrings = pcre_exec(gp_tcvideoregex, NULL, p_string, length, 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));

	if(numstrings > 0)
	{
		p_video->feedback = 1;

		/* allocate, copy string and terminate vectors */
		length = strlen(p_string);
		p_buffer = malloc(length + 1);
		memcpy(p_buffer, p_string, length + 1);
		for(i = 1; i < 6; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;

		/* video settings found, update any missing */
		if(!p_video->fps)
			p_video->fps = atof(&p_buffer[p_vector[2]]);

		if(!p_video->p_vcodec[0])
			strncpy(p_video->p_vcodec, &p_buffer[p_vector[4]], sizeof(p_video->p_vcodec));

		if(!p_video->frames)
			p_video->frames = atoi(&p_buffer[p_vector[6]]);

		if(!p_video->width)
			p_video->width = atoi(&p_buffer[p_vector[8]]);

		if(!p_video->height)
			p_video->width = atoi(&p_buffer[p_vector[10]]);

		if(p_video->width && p_video->height)
			p_video->aspect = (double)p_video->width / (double)p_video->height;

		free(p_buffer);

		return 0;
	}

	/* audio regex */
	length = strlen(p_string);
	numstrings = pcre_exec(gp_tcaudioregex, NULL, p_string, length, 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));

	if(numstrings > 0)
	{
		p_video->feedback = 1;

		/* allocate, copy string and terminate vectors */
		length = strlen(p_string);
		p_buffer = malloc(length + 1);
		memcpy(p_buffer, p_string, length + 1);
		for(i = 1; i < 5; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;

		/* audio settings found, update any missing */
		if(!p_video->hz)
			p_video->hz = atoi(&p_buffer[p_vector[2]]);

		if(!p_video->abits)
			p_video->abits = atoi(&p_buffer[p_vector[4]]);

		if(!p_video->achans)
			p_video->achans = atoi(&p_buffer[p_vector[6]]);

		if(!p_video->arate)
			p_video->arate = atoi(&p_buffer[p_vector[8]]);

		free(p_buffer);

		return 0;
	}

	return 0;
}

/* transcode progress callback */
int cb_transcode(const char *p_string, void *p_options)
{
	char p_buffer[1024];
	int numstrings, length, p_vector[20];
	double videopos;
	video_t *p_video;
	int i;

	/* initialisation command */
	if(!p_string)
		return 0;

	/* option is a video_t */
	p_video = (video_t*)p_options;

	/* regex string */
	length = strlen(p_string);
	numstrings = pcre_exec(gp_transregex, NULL, p_string, length, 0, 0,
		p_vector, sizeof(p_vector) / sizeof(int));

	/* make copy of string and terminate vectors with 0 */
	if(numstrings > 0)
	{
		if(length >= sizeof(p_buffer))
			return 1;
		memcpy(p_buffer, p_string, length);
		for(i = 1; i < 5; i++)
			p_buffer[p_vector[i * 2 + 1]] = 0;
		videopos = atof(&p_buffer[p_vector[4]]) * 3500.0 +
			atof(&p_buffer[p_vector[6]]) * 60.0 +
			atof(&p_buffer[p_vector[8]]);
		fprintf(stderr,"  %7.2f fps %7.2f%% complete  \r", atof(&p_buffer[p_vector[2]]),
			videopos * 100.0 / (double)p_video->length);
	}
	else if(p_video->options & SPANK_VERBOSE)
	{
		printverbose("%s\n", p_buffer);
	}

	return 0;
}

/* mplayer identify callback */
int cb_mplayer(const char *p_string, void *p_options)
{
	const char *p_id;
	const char *p_value;
	video_t *p_video;
	int p_vector[20];
	int numstrings;

	/* initialisation command */
	if(!p_string)
		return 0;

	/* option is output to a video_t */
	p_video = (video_t*)p_options;

	/* examine the string, return if nothing */
	numstrings = pcre_exec(gp_idregex, NULL, p_string, strlen(p_string), 0, 0, p_vector, 20);
	if(numstrings < 0)
		return 0;

	/* create temp strings and feed data into video_t, speed is not needed here */
	pcre_get_substring(p_string, p_vector, numstrings, 1, &p_id);
	pcre_get_substring(p_string, p_vector, numstrings, 2, &p_value);

	if(!strcmp(p_id, "VIDEO_FORMAT"))
	{
		strncpy(p_video->p_vcodec, p_value, sizeof(p_video->p_vcodec) - 1);
	}
	if(!strcmp(p_id, "VIDEO_BITRATE"))
	{
		p_video->vrate = atoi(p_value) / 1000;
	}
	if(!strcmp(p_id, "VIDEO_WIDTH"))
	{
		p_video->width = atoi(p_value);
	}
	if(!strcmp(p_id, "VIDEO_HEIGHT"))
	{
		p_video->height = atoi(p_value);
		p_video->aspect = (float)p_video->width / (float)p_video->height;
	}
	if(!strcmp(p_id, "VIDEO_FPS"))
	{
		p_video->fps = atof(p_value);
	}
	if(!strcmp(p_id, "AUDIO_CODEC"))
	{
		strncpy(p_video->p_acodec, p_value, sizeof(p_video->p_acodec) - 1);
	}
	if(!strcmp(p_id, "AUDIO_BITRATE"))
	{
		p_video->arate = atoi(p_value) / 1000;
	}
	if(!strcmp(p_id, "AUDIO_RATE"))
	{
		p_video->hz = atoi(p_value);
	}
	if(!strcmp(p_id, "AUDIO_NCH"))
	{
		p_video->achans = atoi(p_value);
	}
	if(!strcmp(p_id, "LENGTH"))
	{
		p_video->length = atoi(p_value);
		p_video->hours   = p_video->length / 3600;
		p_video->minutes = p_video->length % 3600 / 60;
		p_video->seconds = p_video->length % 60;
	}

	pcre_free_substring(p_id);
	pcre_free_substring(p_value);

	return 0;
}

/* fork another program */
int forkit(forkitcallback callback, void *p_options, const char *p_path, const char *p_cmd)
{
	char *pp_cmdopts[100], *p_cmdline, p_buffer[4096];
	int cmdlength;
	int p_pipe[2], pid;
	int status = 0;
	int retval = 0;
	int count;
	int i, pos, start, copy;

	cmdlength = strlen(p_cmd);
	p_cmdline = malloc(cmdlength + 1);

	/* create parameter index */
	strncpy(p_cmdline, p_cmd, cmdlength + 1);
	pos = start = 0;
	/* fprintf(stderr, "[debug] fork %s, %s\n", p_path,p_cmd); */
	for(i = 0; i < sizeof(pp_cmdopts) / sizeof(char**); i++)
	{
		while(p_cmdline[pos] != ' ' && p_cmdline[pos] != 0)
			pos++;

		pp_cmdopts[i] = &p_cmdline[start];
		if(p_cmdline[pos] == 0)
		{
			/* fprintf(stderr, "  [debug-end] %s\n", &p_cmdline[start]); */
			pp_cmdopts[i + 1] = 0;
			break;
		}
		p_cmdline[pos] = 0;
		/* fprintf(stderr, "  [debug-add] %s\n", &p_cmdline[start]); */
		pos++;
		start = pos;
	}

	/* open comminication pipe */
	if(pipe(p_pipe) == -1)
		return 1;

	/* fork */
	pid = fork();
	switch(pid)
	{
		case -1:
			/* error */
			close(p_pipe[0]);
			close(p_pipe[1]);
			retval = -2;
			break;

		case 0:
			/* child, run program */
			close(p_pipe[0]);
			dup2(p_pipe[1], fileno(stdout));
			dup2(p_pipe[1], fileno(stderr));
			execv(p_path, pp_cmdopts);
			fprintf(stderr, "[dvdspanky] fork %s,%s failed\n", p_path,p_cmd);
			free(p_cmdline);
			exit(1);

		default:
			/* parent, send output to callback */
			close(p_pipe[1]);

			/* reset the callback */
			if(callback)
				callback(NULL, p_options);

			/* read input while waiting for child to term */
			start = 0;
			while(waitpid(pid, &status, WNOHANG) != pid)
			{
				count = read(p_pipe[0], &p_buffer[start], sizeof(p_buffer) - 1 - start);
				if(count)
				{
					/* seperate lines and send each to callback */
					p_buffer[start + count] = 0;
					for(start = 0, pos = 0; pos < sizeof(p_buffer) - 1; pos++)
					{
						if(p_buffer[pos] == '\n' || p_buffer[pos] == '\r')
						{
							/* send line to callback */
							p_buffer[pos] = 0;
							if(callback && callback(&p_buffer[start], p_options))
								retval = -3;
							start = pos + 1;
						}
						else if(p_buffer[pos] == 0)
						{
							/* copy unread buffer to beginning */
							for(copy = 0; copy < pos - start; copy++)
								p_buffer[copy] = p_buffer[start + copy];

							/* set position ready for continued reading */
							start = pos - start;

							break;
						}
					}
				}
			}
			/* return value same as exit status or -1 if terminated by signal */
			if(!retval)
			{
				if(WIFEXITED(status))
					retval = WEXITSTATUS(status);
				else
					retval = -1;
			}

			close(p_pipe[0]);
			break;
	}

	free(p_cmdline);
	return retval;
}

/* formatted forking of another program */
int forkitf(forkitcallback callback, void *p_options, const char *p_path, const char *p_cmd, ...)
{
	va_list argp;
	char p_buffer[4096];

	va_start(argp, p_cmd);
	vsnprintf(p_buffer, sizeof(p_buffer) - 1, p_cmd, argp);
	va_end(argp);

	return forkit(callback, p_options, p_path, p_buffer);
}

/* entry point */
int main(int argc, char **argv)
{
	const char *p_error;
	int erroffset;
	int opt;
	char p_filters[512];
	int vrate = 0, arate = 0, aspect = 0, vcrop = 0, hcrop = 0, channels = 0;
	int calcsize = 0, calcborder = 0, calcclip = 0;
	char vidmode = 'a';
	int options = 0;
	video_t invid, outvid;

	/* clear video data */
	memset(&invid, '\0', sizeof(video_t));
	memset(&outvid, '\0', sizeof(video_t));

	atexit(cleanup);

	/* show title */
	printf(gp_title);

	/* compile regex */
	if(!(gp_transregex = pcre_compile(
		"^encoding\\sframes\\s.*?,\\s*(\\d+[.]\\d+)\\sfps.*?EMT:\\s*(\\d+):(\\d+):(\\d+),.*",
		0, &p_error, &erroffset, NULL)))
	{
		fprintf(stderr, "error: gp_transregex compile failed - %s\n", p_error);
		exit(1);
	}
	if(!(gp_idregex = pcre_compile("^ID_([^=]+)=(.*)$", 0, &p_error, &erroffset, NULL)))
	{
		fprintf(stderr, "error: gp_idregex compile failed - %s\n", p_error);
		exit(1);
	}
	if(!(gp_tcvideoregex = pcre_compile("^\\[.*?\\]\\sV:\\s([^ ]*)\\sfps,\\scodec=([^,]*),"
		"\\sframes=([^,]*),\\swidth=([^,]*),\\sheight=(.*)$", 0, &p_error, &erroffset, NULL)))
	{
		fprintf(stderr, "error: gp_tcvideoregex compile failed - %s\n", p_error);
		exit(1);
	}
	if(!(gp_tcaudioregex = pcre_compile("^\\[.*?\\]\\sA:\\s([^ ]*)\\sHz,\\sformat=[^,]*,"
		"\\sbits=([^,]*),\\schannels=([^,]*),\\sbitrate=(.*),$", 0, &p_error, &erroffset, NULL)))
	{
		fprintf(stderr, "error: gp_tcaudioregex compile failed - %s\n", p_error);
		exit(1);
	}

	/* parse options */
	while( (opt = getopt_long(argc, argv, gp_shortopts, gp_longopts, NULL)) != -1 )
	{
		switch(opt)
		{
			case '1':
				options |= SPANK_FIRST;
				break;
			case '2':
				options |= SPANK_SECOND;
				break;
			case 'a':
				aspect = atoi(optarg);
				if(aspect != 2 && aspect != 3)
				{
					printerror("aspect not 2 or 3");
					exit(1);
				}
				break;
			case 'b':
				calcborder = 1;
				break;
			case 'B':
				calcclip = 1;
				break;
			case 'c':
				channels = atoi(optarg);
				if(!channels || channels > 6)
				{
					printerror("channels not 1-6");
					exit(1);
				}
				break;
			case 'f':
				strncpy(p_filters, optarg, sizeof(p_filters) - 1);
				break;
			case 'F':
				options |= SPANK_NEWFPS;
				break;
			case 'h':
				printf(gp_help, argv[0]);
				exit(0);
				break;
			case 'H':
				hcrop = atoi(optarg);
				break;
			case 'i':
				strncpy(invid.p_filename, optarg, sizeof(invid.p_filename) - 1);
				break;
			case 'I':
				options |= SPANK_NONICE;
				break;
			case 'L':
				printf(gp_licence);
				exit(0);
				break;
			case 'm':
				options |= SPANK_FIRST | SPANK_SECOND;
				break;
			case 'M':
				options |= SPANK_MULTIPLEX;
				break;
			case 'n':
				vidmode = 'n';
				break;
			case 'N':
				options |= SPANK_FIXNTSC;
				break;
			case 'p':
				vidmode = 'p';
				break;
			case 'P':
				options |= SPANK_PREVIEW;
				break;
			case 'o':
				strncpy(outvid.p_filename, optarg, sizeof(outvid.p_filename) - 1);
				break;
			case 'O':
				options |= SPANK_NORMALISE;
				break;
			case 'r':
				vrate = atoi(optarg);
				if(vrate < 600 || vrate > 7500)
				{
					printerror("vrate not 600 - 7500");
					exit(1);
				}
				break;
			case 'R':
				arate = atoi(optarg);
				if(arate < 64 || arate > 448)
				{
					printerror("arate not 64 - 488");
					exit(1);
				}
				break;
			case 's':
				calcsize = atoi(optarg);
				break;
			case 'T':
				options |= SPANK_PPROCESS;
				break;
			case 'v':
				options |= SPANK_VERBOSE;
				break;
			case 'V':
				vcrop = atoi(optarg);
				break;
			case '?':
				exit(1);
				break;
			default:
				printerror("unhandled option %c", opt);
				exit(1);
				break;
		}
	}

	/* test input filename */
	if(!invid.p_filename[0])
	{
		/* TODO: check for existance of file */
		printerror("no input file");
		exit(1);
	}

	/* create output filename based on input filename if none given */
	if(!outvid.p_filename[0])
	{
		pcre *p_regex;
		int p_vector[10];
		int found;

		p_regex = pcre_compile("^.*?([^/.]+)(?:\\.|)[^/.]*$", 0, &p_error, &erroffset, NULL);
		if(!p_regex)
		{
			printerror("failed to compile filename regex");
			exit(1);
		}
		found = pcre_exec(p_regex, NULL, invid.p_filename, strlen(invid.p_filename), 0, 0,
			p_vector, sizeof(p_vector) / sizeof(int));
		if(found > 0)
		{
			strncpy(outvid.p_filename, &invid.p_filename[p_vector[2]], p_vector[3] - p_vector[2]);
			outvid.p_filename[p_vector[3] - p_vector[2]] = 0;
		}
		else
		{
			printerror("unable to fathom output filename from input");
			pcre_free(p_regex);
			exit(1);
		}
		pcre_free(p_regex);
	}

	/* get source video details */
	forkitf(cb_mplayer, &invid, MPLAYER_BIN,
		"mplayer %s -identify -frames 0 -ao null -vo null", invid.p_filename);
	forkitf(cb_tcprobe, &invid, TCPROBE_BIN, "tcprobe -i %s", invid.p_filename);
	if(!invid.feedback) /* if tcprobe unable to detect, flag use of mplayer for input */
		options |= SPANK_MPLAYER;

	/* set audio bitrate / codec and scale */
	strcpy(outvid.p_acodec, "AC3");
	if(!arate)
		arate = invid.arate;
	for(outvid.arate = 64; outvid.arate < arate; outvid.arate += 32)
		;
	arate = outvid.arate;

	/* set audio channels */
	if(channels)
		outvid.achans = channels;
	else
		outvid.achans = invid.achans;

	/* set audio junk */
	outvid.abits = 16;
	outvid.hz = 48000;

	/* calculate video bitrate for final size allowing 5% overhead */
	if(calcsize)
	{
		vrate = (calcsize - ((arate + arate / 20) / 8 * invid.length)) * 8
			/ invid.length;
		vrate -= vrate / 20;

		if(vrate < 600 || vrate > 7500)
		{
			printerror("calculated video rate %d not 600 - 7500 for size %d KB", vrate, calcsize);
			printerror("suggested minimum size setting of %d KB",
				((631 + arate + arate / 20) / 8 * invid.length) / 1000 * 1000);
			printerror("suggested maximum size setting of %d KB",
				((7875 + arate + arate / 20) / 8 * invid.length) / 1000 * 1000);
			exit(1);
		}
	}

	/* adjust minimum automatic bitrate */
	if(!vrate && invid.vrate < 600)
		vrate = 600; /* TODO: set rate based on input filesize if rate == 0 */

	/* set video rate */
	if(vrate)
	{
		outvid.vrate = vrate;
	}
	else
	{
		outvid.vrate = invid.vrate;
		vrate = outvid.vrate;
	}

	/* set aspect ratio */
	if(aspect == 2)
	{
		outvid.aspect = 1.333333;
	}
	else if(aspect == 3)
	{
		outvid.aspect = 1.777777;
	}
	else
	{
		if(invid.aspect < 1.555555)
			outvid.aspect = 1.333333;
		else
			outvid.aspect = 1.777777;
	}

	/* set doublefps if needed */
	if(invid.fps <= 15)
		options |= SPANK_DOUBLEFPS;

	/* set pal/ntsc */
	if(vidmode == 'a')
	{
		if(invid.fps == 25)
			vidmode = 'p';
		else
			vidmode = 'n';
	}
	outvid.width = 720;
	if(vidmode == 'p')
	{
		outvid.height = 576;
		strcpy(outvid.p_vcodec, "PAL");
	}
	else
	{
		outvid.height = 480;
		strcpy(outvid.p_vcodec, "NTSC");
	}

	/* set length */
	outvid.length  = invid.length;
	outvid.hours   = invid.hours;
	outvid.minutes = invid.minutes;
	outvid.seconds = invid.seconds;
	outvid.frames  = invid.frames;

	/* set fps */
	if(vidmode == 'p' && invid.fps == 25)
	{
		options &= 0xffff ^ SPANK_NEWFPS;
		outvid.fps = 25;
	}
	else if(vidmode =='n' && (invid.fps == 23.976 || invid.fps == 29.97))
	{
		options &= 0xffff ^ SPANK_NEWFPS;
		outvid.fps = invid.fps;
	}
	else if(vidmode == 'p')
	{
		outvid.fps = 25;
	}
	else
	{
		outvid.fps = 23.976;
	}

	/* set border/crop */
	if(calcborder)
	{
		hcrop = 0;
		if(outvid.aspect < 1.5) /* 4:3 */
		{
			vcrop = -(invid.width * 10 / 4 * 3 - invid.height * 10) / 40 * 2;
			if(vcrop > 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 3 * 4) / 40 * 2;
				vcrop = 0;
			}
		}
		else /* 16:9 */
		{
			vcrop = -(invid.width * 10 / 16 * 9 - invid.height * 10) / 40 * 2;
			if(vcrop > 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 9 * 16) / 40 * 2;
				vcrop = 0;
			}
		}
	}
	if(calcclip)
	{
		hcrop = 0;
		if(outvid.aspect < 1.5) /* 4:3 */
		{
			vcrop = -(invid.width * 10 / 4 * 3 - invid.height * 10) / 40 * 2;
			if(vcrop < 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 3 * 4) / 40 * 2;
				vcrop = 0;
			}
		}
		else /* 16:9 */
		{
			vcrop = -(invid.width * 10 / 16 * 9 - invid.height * 10) / 40 * 2;
			if(vcrop < 0)
			{
				hcrop = (invid.width * 10 - invid.height * 10 / 9 * 16) / 40 * 2;
				vcrop = 0;
			}
		}
	}
	outvid.vcrop = vcrop;
	outvid.hcrop = hcrop;

	/* print source and dest video encoding info */
	printf(
		" source | %s\n"
		"  video | %s, %dx%d, %d kbps, %.3f fps\n"
		"        | %.4f aspect, %d frames, %d:%02d:%02d\n"
		"  audio | %s, %d kbps, %d Hz, %d bit, %d channel\n",
		invid.p_filename,
		invid.p_vcodec, invid.width, invid.height, invid.vrate, invid.fps,
		invid.aspect, invid.frames,	invid.hours, invid.minutes, invid.seconds,
		invid.p_acodec, invid.arate, invid.hz, invid.abits, invid.achans
		);
	if(options & SPANK_MPLAYER)
		printf(
		"   info | using mplayer decoder\n"
		);
	printf(
		"\n"
		);


	printf(
		"   dest | %s.mpg\n"
		"  video | %s, %dx%d, %d kbps, %.3f fps\n"
		"        | %.4f aspect, %d frames, %d:%02d:%02d\n"
		"        | %d:%d crop, %s\n"
		"  audio | %s, %d kbps, %d Hz, %d bit, %d channel\n"
		"   file | ~%d KB\n\n",
		outvid.p_filename,
		outvid.p_vcodec, outvid.width, outvid.height, outvid.vrate, outvid.fps,
		outvid.aspect, outvid.frames, outvid.hours, outvid.minutes, outvid.seconds,
		outvid.vcrop, outvid.hcrop, (options & SPANK_MULTI) ? "VBR" : "CBR",
		outvid.p_acodec, outvid.arate, outvid.hz, outvid.abits, outvid.achans,
		(vrate + vrate / 20 + arate + arate / 20) / 8 * invid.length
		);

	exit(encode(&invid, &outvid, options, p_filters));

	return 0;
}
