/*
 * dvdspanky - a video to DVD MPEG conversion frontend for transcode
 * Copyright (C) 2007  Jeffrey Grembecki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <pcre.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include "config.h"

#define SPANK_FIRST 	0x1
#define SPANK_SECOND	0x2
#define SPANK_NONICE	0x4
#define SPANK_FIXNTSC	0x8
#define SPANK_VERBOSE	0x10
#define SPANK_NEWFPS	0x20
#define SPANK_DOUBLEFPS 0x40

static char gp_title[] = "dvdspanky 0.9.1 - (C) 2007 Jeffrey Grembecki\n\n";
static char gp_licence[] =
"This program is free software; you can redistribute it and/or modify\n"
"it under the terms of the GNU General Public License as published by\n"
"the Free Software Foundation; either version 2 of the License, or\n"
"(at your option) any later version.\n"
"\n"
"This program is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"GNU General Public License for more details.\n"
"\n"
"You should have received a copy of the GNU General Public License\n"
"along with this program; if not, write to the Free Software\n"
"Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,\n"
"MA 02110-1301, USA.\n";

static char gp_help[] =
"%s -i file -o file [-bBpnNm12FILhv] [-s size] [-r rate] [-R rate] [-c channels] [-V pixels] [-H pixels] [-a mode] [-f filters]\n"
"\n"
"Following; [ ] = default, { } = requirements, * = experimental\n"
"\n"
"-i, --input=FILE       input file\n"
"-o, --output=FILE      output filename prefix (w/o extension)\n"
"-s, --size=KB          final mpg size in kilobytes\n"
"-r, --vrate=RATE       video bitrate [auto] {600 - 7500}\n"
"-R, --arate=RATE       audio bitrate [auto] {64 - 448, 192 is good}\n"
"-c, --channels=CHANS   number of audio channels [auto] {1 - 5}\n"
"-V, --vcrop=PIXELS     pos for crop, neg for black border [off]\n"
"-H, --hcrop=PIXELS     as above [off]\n"
"-b, --border           automatic black border [off]\n"
"-B, --clip             clips edges to fit aspect [off]\n"
"-p, --pal              set pal [auto]\n"
"-n, --ntsc             set ntsc [auto]\n"
"-N, --ntscfix          fix variable framerate ntsc video [off] *\n"
"-a, --aspect=MODE      aspect ratio 2=4:3, 3=16:9 [auto] {2,3}\n"
"-m, --multi            2 pass mode [off]\n"
"-1, --firstpass        1st pass only [off]\n"
"-2, --secondpass       2nd pass only including multiplex [off]\n"
"-F, --frameadjust      interlace video for framerate conversion [off] *\n"
"-f, --filters=FILTERS  eg. -f pp=de,hqdn3d [off]\n"
"-I, --nonice           do not run at nice priority 19 [off]\n"
"-v, --verbose          be verbose [off]\n"
"-L, --licence          show the program licence\n"
"-h, --help             this help text\n"
"\nSee man 1 dvdspanky for more information\n";

static struct option gp_longopts[] =
{
	{"input", 1, 0, 'i'},
	{"output", 1, 0, 'o'},
	{"vrate", 1, 0, 'r'},
	{"arate", 1, 0, 'R'},
	{"size", 1, 0, 's'},
	{"channels", 1, 0, 'c'},
	{"vcrop", 1, 0, 'V'},
	{"hcrop", 1, 0, 'H'},
	{"border", 0, 0, 'b'},
	{"clip", 0, 0, 'B'},
	{"pal", 0, 0, 'p' },
	{"ntsc", 0, 0, 'n' },
	{"ntscfix", 0, 0, 'N'},
	{"aspect", 1, 0, 'a'},
	{"multi", 0, 0, 'm'},
	{"firstpass", 0, 0, '1'},
	{"secondpass", 0, 0, '2'},
	{"frameadjust", 0, 0, 'F'},
	{"filters", 1, 0, 'f'},
	{"nonice", 0, 0, 'I'},
	{"licence", 0, 0, 'L'},
	{"help", 0, 0, 'h'},
	{"verbose", 0, 0, 'v'},
};
static char gp_shortopts[] = "vi:o:r:R:s:c:V:H:bBpnNa:mf:ILhF12";

static int gp_inpipe[2];

pcre *gp_idregex = NULL;
pcre *gp_transregex = NULL;
/* pcre *gp_mplexregex = NULL;
pcre_extra *gp_mplexstudy = NULL; */

typedef struct {
	int width, height, vrate, arate, achans;
	int length, hours, minutes, seconds;
	int vcrop, hcrop;
	char p_vcodec[20], p_acodec[10];
	float aspect, fps;
} video_t;

video_t g_sourcevideo, g_destvideo;

void clearregex()
{
	if(gp_idregex)
		pcre_free(gp_idregex);
	if(gp_transregex)
		pcre_free(gp_transregex);
	/*
	if(gp_mplexregex)
		pcre_free(gp_mplexregex);
	if(gp_mplexstudy)
		pcre_free(gp_mplexstudy);
	*/
	gp_idregex = gp_transregex = NULL;
	/* gp_mplexregex = NULL; gp_mplexstudy = NULL; */
}

int transcode(const char *p_cmd, int options)
{
	char p_cmdline[4096];
	char p_buffer[1024];
	char *pp_cmdopts[100];
	int i, pos, start;
	const char *p_error;
	int p_vector[20];
	int erroffset;
	int numstrings;
	double length;

	pid_t pid;
	int status;
	int count;

	if(!gp_transregex)
	{
		gp_transregex = pcre_compile("^encoding\\sframes\\s.*?,\\s*(\\d+[.]\\d+)\\sfps.*?EMT:\\s*(\\d+):(\\d+):(\\d+),.*", 0, &p_error, &erroffset, NULL);
		if(!gp_transregex)
		{
			fprintf(stderr, "error: gp_transregex compile failed - %s\n", p_error);
			return 1;
		}
	}

	/* create parameter index */
	strncpy(p_cmdline, p_cmd, sizeof(p_cmdline) - 1);
	pos = start = 0;
	for(i = 0; i < sizeof(pp_cmdopts) - 1; i++)
	{
		while(p_cmdline[pos] != ' ' && p_cmdline[pos] != 0)
			pos++;

		pp_cmdopts[i] = &p_cmdline[start];
		if(p_cmdline[pos] == 0)
		{
			/*printf("opt: %s\n", pp_cmdopts[i]);*/
			break;
		}
		p_cmdline[pos] = 0;
		/*printf("opt: %s\n", pp_cmdopts[i]);*/
		pos++;
		start = pos;
	}
	pp_cmdopts[i + 1] = 0;

	/* setup pipe */
	if(pipe(gp_inpipe) == -1)
		return 1;

	/* fork transcode */
	pid = fork();
	switch(pid)
	{
		case -1: /* error */
			close(gp_inpipe[0]);
			close(gp_inpipe[1]);
			return 1;
		case 0: /* child */
			close(gp_inpipe[0]);
			dup2(gp_inpipe[1], fileno(stdout));
			dup2(gp_inpipe[1], fileno(stderr));
			execv(TRANSCODE_BIN, pp_cmdopts);
			exit(1);
			break;
		default: /* parent */
			fcntl(gp_inpipe[0], F_SETFL, O_NONBLOCK);
			close(gp_inpipe[1]);
			pos = 0;
			/* translate output into progress line */
			do
			{
				count = read(gp_inpipe[0], &p_buffer[pos], 1);
				if(count > 0)
				{
					if(p_buffer[pos] == '\n' || p_buffer[pos] == '\r')
					{
						p_buffer[pos] = 0;

						numstrings = pcre_exec(gp_transregex, NULL, p_buffer, strlen(p_buffer), 0, 0, p_vector, 20);
						if(numstrings > 0)
						{
							for(i = 1; i < 5; i++)
								p_buffer[p_vector[i * 2 + 1]] = 0;
							length = atof(&p_buffer[p_vector[4]]) * 3500.0 +
								atof(&p_buffer[p_vector[6]]) * 60.0 +
								atof(&p_buffer[p_vector[8]]);
							if(options & SPANK_FIRST)
								fprintf(stderr, "pass 1: ");
							else if(options & SPANK_SECOND)
								fprintf(stderr, "pass 2: ");
							else
								fprintf(stderr, "cbr encode: ");
							fprintf(stderr,"%7.2f fps %7.2f%%   \r", atof(&p_buffer[p_vector[2]]),
								length * 100.0 / (double)g_destvideo.length);
						}
						else if(options & SPANK_VERBOSE)
						{
							fprintf(stderr, "%s\n", p_buffer);
						}
						/* todo: some logging here might be nice */
						pos = 0;
					}
					else
					{
						pos++;
					}
				}
				else
				{
					usleep(10000);
				}
			} while(!waitpid(pid, &status, WNOHANG));

			fprintf(stderr, "\n");
			close(gp_inpipe[0]);
			break;
	}

	return 0;
}

int multiplex(const char *p_cmd, int options)
{
	char p_cmdline[4096];
	char p_buffer[1024];
	char *pp_cmdopts[100];
	int i, pos, start;
	/* const char *p_error;
	int p_vector[20];
	int erroffset;
	int numstrings;
	double frames, curframe; */

	pid_t pid;
	int status;
	int count;

	/*
	if(!gp_mplexregex)
	{
		gp_mplexregex = pcre_compile("^\\-\\-DEBUG:\\s\\[mplex\\]\\sScanning\\s.*?frame\\s(\\d+).*", 0, &p_error, &erroffset, NULL);
		if(!gp_mplexregex)
		{
			fprintf(stderr, "error: gp_mpexregex compile failed - %s\n", p_error);
			return 1;
		}
		gp_mplexstudy = pcre_study(gp_mplexregex, 0, &p_error);
	}
	*/

	/* create parameter index */
	strncpy(p_cmdline, p_cmd, sizeof(p_cmdline) - 1);
	pos = start = 0;
	for(i = 0; i < sizeof(pp_cmdopts) - 1; i++)
	{
		while(p_cmdline[pos] != ' ' && p_cmdline[pos] != 0)
			pos++;

		pp_cmdopts[i] = &p_cmdline[start];
		if(p_cmdline[pos] == 0)
		{
			/*printf("opt: %s\n", pp_cmdopts[i]);*/
			break;
		}
		p_cmdline[pos] = 0;
		/*printf("opt: %s\n", pp_cmdopts[i]);*/
		pos++;
		start = pos;
	}
	pp_cmdopts[i + 1] = 0;

	/* setup pipe */
	if(pipe(gp_inpipe) == -1)
		return 1;

	/* fork mplex */
	pid = fork();
	switch(pid)
	{
		case -1: /* error */
			close(gp_inpipe[0]);
			close(gp_inpipe[1]);
			return 1;
		case 0: /* child */
			close(gp_inpipe[0]);
			dup2(gp_inpipe[1], fileno(stdout));
			dup2(gp_inpipe[1], fileno(stderr));
			execv(MPLEX_BIN, pp_cmdopts);
			exit(1);
			break;
		default: /* parent */
			fcntl(gp_inpipe[0], F_SETFL, O_NONBLOCK);
			close(gp_inpipe[1]);

			/* translate output into progress line */
			pos = 0;
			/* frames = g_destvideo.fps * (double)g_destvideo.length; */
			do
			{
				count = read(gp_inpipe[0], &p_buffer[pos], 1);
				if(count > 0)
				{
					if(p_buffer[pos] == '\n' || p_buffer[pos] == '\r')
					{
						/*
						p_buffer[pos] = 0;

						numstrings = pcre_exec(gp_mplexregex, gp_mplexstudy, p_buffer, strlen(p_buffer), 0, 0, p_vector, 20);
						if(numstrings > 0)
						{
							for(i = 1; i < 2; i++)
								p_buffer[p_vector[i * 2 + 1]] = 0;
							curframe = atof(&p_buffer[p_vector[4]]);
							fprintf(stderr,"multiplex: %7.2f%%   \r", 100.0 * curframe / frames);
						}
						else if(options & SPANK_VERBOSE)
						{
							fprintf(stderr, "%s  \n", p_buffer);
						}
						*/
						/* todo: some logging here might be nice */
						pos = 0;
					}
					else
					{
						pos++;
					}
				}
				else
				{
					usleep(10000);
				}
			} while(!waitpid(pid, &status, WNOHANG));

			fprintf(stderr, "\n");
			close(gp_inpipe[0]);
			break;
	}

	return 0;
}

int encode(const char *p_infile, const char *p_outfile, int options, const char *p_filters)
{
	char p_cmdline[4096];
	char p_cmdbackup[4096];
	char p_print[1000];
	char p_num[20];

	p_cmdline[0] = 0;

	/* handy safe cat macros */
#define safecat(_s) strncat(p_cmdline, _s, sizeof(p_cmdline) - strlen(p_cmdline) - 1)
#define safecati(_n) { snprintf(p_num, sizeof(p_num), "%d", _n); \
	strncat(p_cmdline, p_num, sizeof(p_cmdline) - strlen(p_cmdline) - 1); }
#define safecatf(_f) { snprintf(p_num, sizeof(p_num), "%f", _f); \
	strncat(p_cmdline, p_num, sizeof(p_cmdline) - strlen(p_cmdline) - 1); }

	/* prepare command line */
	p_cmdline[0] = p_cmdbackup[0] = p_print[0] = p_num[0] = 0;
	safecat("transcode");

	/* nice */
	if(!(options & SPANK_NONICE))
		safecat(" --nice 19");

	/* infile */
	safecat(" -i ");
	safecat(p_infile);

	/* audio */
	if(g_destvideo.achans)
	{
		safecat(" -E 48000,0,");
		safecati(g_destvideo.achans);
	}
	else
	{
		safecat(" -E 48000");
	}
	safecat(" -N 0x2000 -b ");
	safecati(g_destvideo.arate);

	/* filters */
	if(options & SPANK_DOUBLEFPS)
		safecat(" -J doublefps");
	if(p_filters[0])
	{
		safecat(" -J ");
		safecat(p_filters);
	}

	/* video */
	snprintf(p_print, sizeof(p_print) - 1, " -Z %dx%d -w %d --export_fps %f",
		g_destvideo.width, g_destvideo.height, g_destvideo.vrate, g_destvideo.fps);
	safecat(p_print);
	if(g_destvideo.p_vcodec[0] == 'P')
		safecat(" --encode_fields t");
	else
		safecat(" --encode_fields b");

	/* alter fps */
	if(options & SPANK_NEWFPS)
	{
		safecat(" -M 0 -J ivtc,32detect=force_mode=3,decimate -f ");
		safecatf(g_sourcevideo.fps);
	}

	/* ntsc fix */
	if(options & SPANK_FIXNTSC)
	{
		safecat(" -M 2 -J ivtc,32detect=force_mode=5,decimate --hard_fps");
	}

	/* croping */
	if(g_destvideo.vcrop || g_destvideo.hcrop)
	{
		snprintf(p_print, sizeof(p_print), " -j %d,%d,%d,%d",
			g_destvideo.vcrop, g_destvideo.hcrop, g_destvideo.vcrop, g_destvideo.hcrop);
		safecat(p_print);
	}

	/* create backup of base command line */
	strncpy(p_cmdbackup, p_cmdline, sizeof(p_cmdbackup) - 1);

	/* create ffmpeg.cfg */
	system("echo \"[mpeg2video]\nvrc_minrate=0\nvrc_maxrate=7500\nvrc_buf_size=1792\n\" > ffmpeg.cfg");

	/* execute 1st pass */
	if(options & SPANK_FIRST)
	{
		safecat(" -y ffmpeg,null -F mpeg2video -R 1,");
		safecat(p_outfile);
		safecat(".pass1");
		transcode(p_cmdline, SPANK_FIRST | (options & SPANK_VERBOSE));
		strncpy(p_cmdline, p_cmdbackup, sizeof(p_cmdline) - 1);
	}

	/* execute 2nd pass */
	if(options & SPANK_SECOND)
	{
		snprintf(p_print, sizeof(p_print) - 1, " -y ffmpeg -F mpeg2video -R 2,%s.pass1 -o %s -m %s.ac3",
			p_outfile, p_outfile, p_outfile);
		safecat(p_print);
		if(transcode(p_cmdline, SPANK_SECOND | (options & SPANK_VERBOSE)))
			return 1;
		strncpy(p_cmdline, p_cmdbackup, sizeof(p_cmdline) - 1);
	}

	/* CBR encoding */
	if(!(options & (SPANK_FIRST | SPANK_SECOND)))
	{
		snprintf(p_print, sizeof(p_print) - 1, " -y ffmpeg -F mpeg2video -o %s -m %s.ac3",
			p_outfile, p_outfile);
		safecat(p_print);
		if(transcode(p_cmdline, options & SPANK_VERBOSE))
			return 2;
		strncpy(p_cmdline, p_cmdbackup, sizeof(p_cmdline) - 1);
	}

	/* multiplex */
	if((options & SPANK_SECOND) || !(options & (SPANK_FIRST | SPANK_SECOND)))
	{
		printf("multiplexing..\n");
		if((options & SPANK_SECOND))
			snprintf(p_cmdline, sizeof(p_cmdline) - 1, "mplex -V -f 8 -o %s.mpg %s.m2v %s.ac3",
				p_outfile, p_outfile, p_outfile);
		else
			snprintf(p_cmdline, sizeof(p_cmdline) - 1, "mplex -f 8 -o %s.mpg %s.m2v %s.ac3",
				p_outfile, p_outfile, p_outfile);
		if(multiplex(p_cmdline, options & SPANK_VERBOSE))
			return 3;
	}

#undef safecat
#undef safecati
#undef safecatf

	return 0;
}

int parsemplayer(const char *p_string)
{
	const char *p_error;
	const char *p_id;
	const char *p_value;
	int p_vector[20];
	int erroffset;
	int numstrings;

	if(!gp_idregex)
	{
		gp_idregex = pcre_compile("^ID_([^=]+)=(.*)$", 0, &p_error, &erroffset, NULL);
		if(!gp_idregex)
		{
			fprintf(stderr, "error: gp_idregex compile failed - %s\n", p_error);
			return 1;
		}
	}

	numstrings = pcre_exec(gp_idregex, NULL, p_string, strlen(p_string), 0, 0, p_vector, 20);
	if(numstrings < 0)
		return 0;

	pcre_get_substring(p_string, p_vector, numstrings, 1, &p_id);
	pcre_get_substring(p_string, p_vector, numstrings, 2, &p_value);

	if(!strcmp(p_id, "VIDEO_FORMAT"))
	{
		strncpy(g_sourcevideo.p_vcodec, p_value, sizeof(g_sourcevideo.p_vcodec) - 1);
	}
	if(!strcmp(p_id, "VIDEO_BITRATE"))
	{
		g_sourcevideo.vrate = atoi(p_value) / 1000;
	}
	if(!strcmp(p_id, "VIDEO_WIDTH"))
	{
		g_sourcevideo.width = atoi(p_value);
	}
	if(!strcmp(p_id, "VIDEO_HEIGHT"))
	{
		g_sourcevideo.height = atoi(p_value);
		g_sourcevideo.aspect = (float)g_sourcevideo.width / (float)g_sourcevideo.height;
	}
	if(!strcmp(p_id, "VIDEO_FPS"))
	{
		g_sourcevideo.fps = atof(p_value);
	}
	if(!strcmp(p_id, "AUDIO_CODEC"))
	{
		strncpy(g_sourcevideo.p_acodec, p_value, sizeof(g_sourcevideo.p_acodec) - 1);
	}
	if(!strcmp(p_id, "AUDIO_BITRATE"))
	{
		g_sourcevideo.arate = atoi(p_value) / 1000;
	}
	if(!strcmp(p_id, "AUDIO_NCH"))
	{
		g_sourcevideo.achans = atoi(p_value);
	}
	if(!strcmp(p_id, "LENGTH"))
	{
		g_sourcevideo.length = atoi(p_value);
		g_sourcevideo.hours   = g_sourcevideo.length / 3600;
		g_sourcevideo.minutes = g_sourcevideo.length % 3600 / 60;
		g_sourcevideo.seconds = g_sourcevideo.length % 60;
	}

	pcre_free_substring(p_id);
	pcre_free_substring(p_value);

	return 0;
}

int getvideoinfo(const char *p_file)
{
	char p_buffer[1024];

	pid_t pid;
	int status;
	int count;
	int start;
	int pos;
	int copy;

	/* open comminication pipe */
	if(pipe(gp_inpipe) == -1)
		return 1;

	/* fork up mplayer */
	pid = fork();
	switch(pid)
	{
		case -1:
			close(gp_inpipe[0]);
			close(gp_inpipe[1]);
			return 1;
		case 0:
			close(gp_inpipe[0]);
			dup2(gp_inpipe[1], fileno(stdout));
			execl(MPLAYER_BIN, "mplayer", p_file, "-identify", "-frames", "0", "-ao", "null", "-vo", "null", NULL);
			exit(1);
			break;
		default:
			close(gp_inpipe[1]);
			while(waitpid(pid, &status, WNOHANG))
				;
			start = 0;
			while( (count = read(gp_inpipe[0], &p_buffer[start], sizeof(p_buffer) - 1 - start)) )
			{
				p_buffer[start + count] = 0;
				for(start = 0, pos = 0; pos < sizeof(p_buffer) - 1; pos++)
				{
					if(p_buffer[pos] == '\n' || p_buffer[pos] == '\r')
					{
						p_buffer[pos] = 0;
						parsemplayer(&p_buffer[start]);
						start = pos + 1;
					}
					else if(p_buffer[pos] == 0)
					{
						if(pos > start)
						{
							for(copy = 0; copy < pos - start; copy++)
								p_buffer[copy] = p_buffer[start + copy];
							start = pos - start;
						}
						else
						{
							start = 0;
						}
						break;
					}
				}
			}
			close(gp_inpipe[0]);
			break;
	}

	return 0;
}

int main(int argc, char **argv)
{
	int opt;
	char p_infile[512], p_outfile[512], p_filters[512];
	int vrate = 0, arate = 0, aspect = 0, vcrop = 0, hcrop = 0, channels = 0;
	int calcsize = 0, calcborder = 0, calcclip = 0;
	char vidmode = 'a';
	int options = 0;

	p_infile[0] = p_outfile[0] = p_filters[0] = 0;

	atexit(clearregex);

	/* show title */
	printf(gp_title);

	/* parse options */
	while( (opt = getopt_long(argc, argv, gp_shortopts, gp_longopts, NULL)) != -1 )
	{
		switch(opt)
		{
			case 'i':
				strncpy(p_infile, optarg, sizeof(p_infile) - 1);
				break;
			case 'o':
				strncpy(p_outfile, optarg, sizeof(p_outfile) - 1);
				break;
			case 'r':
				vrate = atoi(optarg);
				if(vrate < 600 || vrate > 7500)
				{
					fprintf(stderr, "error: vrate not 600 - 7500\n");
					exit(1);
				}
				break;
			case 'R':
				arate = atoi(optarg);
				if(arate < 64 || arate > 448)
				{
					fprintf(stderr, "error: arate not 64 - 448\n");
					exit(1);
				}
				break;
			case 's':
				calcsize = atoi(optarg);
				break;
			case 'c':
				channels = atoi(optarg);
				if(!channels || channels > 6)
				{
					fprintf(stderr, "error: channels not 1 - 6\n");
					exit(1);
				}
				break;
			case 'V':
				vcrop = atoi(optarg);
				break;
			case 'H':
				hcrop = atoi(optarg);
				break;
			case 'b':
				calcborder = 1;
				break;
			case 'B':
				calcclip = 1;
				break;
			case 'p':
				vidmode = 'p';
				break;
			case 'n':
				vidmode = 'n';
				break;
			case 'N':
				options |= SPANK_FIXNTSC;
				break;
			case 'F':
				options |= SPANK_NEWFPS;
				break;
			case 'a':
				aspect = atoi(optarg);
				if(aspect != 2 && aspect != 3)
				{
					fprintf(stderr, "error: aspect not 2 or 3\n");
					exit(1);
				}
				break;
			case 'm':
				options |= SPANK_FIRST | SPANK_SECOND;
				break;
			case '1':
				options |= SPANK_FIRST;
				break;
			case '2':
				options |= SPANK_SECOND;
				break;
			case 'f':
				strncpy(p_filters, optarg, sizeof(p_filters) - 1);
				break;
			case 'I':
				options |= SPANK_NONICE;
				break;
			case 'L':
				printf(gp_licence);
				exit(0);
				break;
			case 'h':
				printf(gp_help, argv[0]);
				exit(0);
				break;
			case 'v':
				options |= SPANK_VERBOSE;
				break;
			case '?':
				exit(1);
				break;
			default:
				fprintf(stderr, "error: unhandled option %c\n", opt);
				exit(1);
				break;
		}
	}

	/* get source video details */
	if(!p_infile[0])
	{
		fprintf(stderr, "error: no input file\n");
		exit(1);
	}
	getvideoinfo(p_infile);

	/* set audio bitrate / codec and scale */
	strcpy(g_destvideo.p_acodec, "AC3");
	if(!arate)
		arate = g_sourcevideo.arate;
	for(g_destvideo.arate = 64; g_destvideo.arate < arate; g_destvideo.arate += 32)
		;
	arate = g_destvideo.arate;

	/* set audio channels */
	if(channels)
		g_destvideo.achans = channels;
	else
		g_destvideo.achans = g_sourcevideo.achans;

	/* calculate video bitrate for final size allowing 5% overhead */
	if(calcsize)
	{
		vrate = (calcsize - ((arate + arate / 20) / 8 * g_sourcevideo.length)) * 8
			/ g_sourcevideo.length;
		vrate -= vrate / 20;

		if(vrate < 600 || vrate > 7500)
		{
			fprintf(stderr, "error: calculated video rate %d not 600 - 7500 for size %d KB\n",
				vrate, calcsize);
			calcsize = ((631 + arate + arate / 20) / 8 * g_sourcevideo.length) / 1000 * 1000;
			fprintf(stderr, "error: suggested minimum size setting of %d KB\n", calcsize);
			calcsize = ((7875 + arate + arate / 20) / 8 * g_sourcevideo.length) / 1000 * 1000;
			fprintf(stderr, "error: suggested maximum size setting of %d KB\n", calcsize);
			exit(1);
		}
	}

	/* adjust minimum automatic bitrate */
	if(!vrate && g_sourcevideo.vrate < 600)
		vrate = 600;

	/* set video rate */
	if(vrate)
	{
		g_destvideo.vrate = vrate;
	}
	else
	{
		g_destvideo.vrate = g_sourcevideo.vrate;
		vrate = g_destvideo.vrate;
	}

	/* set aspect ratio */
	if(aspect == 2)
	{
		g_destvideo.aspect = 1.333333;
	}
	else if(aspect == 3)
	{
		g_destvideo.aspect = 1.777777;
	}
	else
	{
		if(g_sourcevideo.aspect < 1.555555)
			g_destvideo.aspect = 1.333333;
		else
			g_destvideo.aspect = 1.777777;
	}

	/* set doublefps if needed */
	if(g_sourcevideo.fps <= 15)
		options |= SPANK_DOUBLEFPS;

	/* set pal/ntsc */
	if(vidmode == 'a')
	{
		if(g_sourcevideo.fps == 25)
			vidmode = 'p';
		else
			vidmode = 'n';
	}
	g_destvideo.width = 720;
	if(vidmode == 'p')
	{
		g_destvideo.height = 576;
		strcpy(g_destvideo.p_vcodec, "PAL");
	}
	else
	{
		g_destvideo.height = 480;
		strcpy(g_destvideo.p_vcodec, "NTSC");
	}

	/* set length */
	g_destvideo.length  = g_sourcevideo.length;
	g_destvideo.hours   = g_sourcevideo.hours;
	g_destvideo.minutes = g_sourcevideo.minutes;
	g_destvideo.seconds = g_sourcevideo.seconds;

	/* set fps */
	if(vidmode == 'p' && g_sourcevideo.fps == 25)
	{
		options &= 0xffff ^ SPANK_NEWFPS;
		g_destvideo.fps = 25;
	}
	else if(vidmode =='n' && (g_sourcevideo.fps == 23.976 || g_sourcevideo.fps == 29.97))
	{
		options &= 0xffff ^ SPANK_NEWFPS;
		g_destvideo.fps = g_sourcevideo.fps;
	}
	else if(vidmode == 'p')
	{
		g_destvideo.fps = 25;
	}
	else
	{
		g_destvideo.fps = 23.976;
	}

	/* set border/crop */
	if(calcborder)
	{
		hcrop = 0;
		if(g_destvideo.aspect < 1.5) /* 4:3 */
		{
			vcrop = -(g_sourcevideo.width * 10 / 4 * 3 - g_sourcevideo.height * 10) / 40 * 2;
			if(vcrop > 0)
			{
				hcrop = -vcrop;
				vcrop = 0;
			}
		}
		else /* 16:9 */
		{
			vcrop = -(g_sourcevideo.width * 10 / 16 * 9 - g_sourcevideo.height * 10) / 40 * 2;
			if(vcrop > 0)
			{
				hcrop = -vcrop;
				vcrop = 0;
			}
		}
	}
	if(calcclip)
	{
		hcrop = 0;
		if(g_destvideo.aspect < 1.5) /* 4:3 */
		{
			vcrop = -(g_sourcevideo.width * 10 / 4 * 3 - g_sourcevideo.height * 10) / 40 * 2;
			if(vcrop < 0)
			{
				hcrop = -vcrop;
				vcrop = 0;
			}
		}
		else /* 16:9 */
		{
			vcrop = -(g_sourcevideo.width * 10 / 16 * 9 - g_sourcevideo.height * 10) / 40 * 2;
			if(vcrop < 0)
			{
				hcrop = -vcrop;
				vcrop = 0;
			}
		}
	}
	if(vcrop)
		g_destvideo.vcrop = vcrop;
	else
		g_destvideo.vcrop = 0;
	if(hcrop)
		g_destvideo.hcrop = hcrop;
	else
		g_destvideo.hcrop = 0;

	/* print source and dest video encoding info */
	printf("source video: %s\n  video codec: %s\n  size: %dx%d\n  length: %d:%02d:%02d\n"
		"  video rate: %d kbps\n  fps: %.3f\n  aspect: %.06f\n  audio codec: %s\n"
		"  audio rate: %d kbps\n", p_infile,
		g_sourcevideo.p_vcodec, g_sourcevideo.width, g_sourcevideo.height, g_sourcevideo.hours,
		g_sourcevideo.minutes, g_sourcevideo.seconds, g_sourcevideo.vrate, g_sourcevideo.fps,
		g_sourcevideo.aspect, g_sourcevideo.p_acodec, g_sourcevideo.arate);

	printf("output video: %s.mpg %s.m2v %s.ac3\n  video format: %s\n  size: %dx%d\n"
		"  video rate: %d kbps\n  fps: %.3f\n  aspect: %.06f\n  crop: %d, %d\n  audio codec: %s\n"
		"  audio rate: %d kbps\n", p_outfile, p_outfile, p_outfile, g_destvideo.p_vcodec,
		g_destvideo.width, g_destvideo.height, g_destvideo.vrate, g_destvideo.fps,
		g_destvideo.aspect, g_destvideo.hcrop, g_destvideo.vcrop, g_destvideo.p_acodec,
		g_destvideo.arate);
	if(g_destvideo.achans)
		printf("  audio channels: %d\n", g_destvideo.achans);
	else
		printf("  audio channels: auto\n");
	printf("  file size: ~%d KB\n",
		(vrate + vrate / 20 + arate + arate / 20) / 8 * g_sourcevideo.length);

	encode(p_infile, p_outfile, options, p_filters);

	return 0;
}
