/*
 * vstr - a variable length string for C
 * Copyright (C) 2007  Jeffrey Grembecki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "vstr.h"

static int vs_realloc(vstr *p_vs, int length)
{
	char *p_s;
	if(length >= p_vs->_private_buffersize)
	{
		int bs;

		bs = p_vs->_private_buffersize;

		do
		{
			if(bs <= 32786)
				bs *= 2;
			else
				bs += 65536;
		} while(length >= bs);

		p_s = realloc(p_vs->s, bs);
		if(!p_s)
			return 1;

		p_vs->_private_buffersize = bs;
		p_vs->s = p_s;
	}
	else if(length < p_vs->_private_buffersize - 131072 && length >= 131072)
	{
		p_s = malloc(p_vs->_private_buffersize / 2);
		if(p_s)
		{
			free(p_vs->s);
			p_vs->_private_buffersize /= 2;
			p_vs->s = p_s;
		}
	}

	return 0;
}

vstr* vs_new()
{
	vstr *p_vs;

	p_vs = malloc(sizeof(vstr));
	if(!p_vs)
		return NULL;

	p_vs->s = malloc(256);
	if(!p_vs->s)
	{
		free(p_vs);
		return NULL;
	}

	p_vs->s[0] = 0;
	p_vs->len = 0;
	p_vs->_private_buffersize = 256;

	return p_vs;
}

void vs_free(vstr *p_vs)
{
	free(p_vs->s);
	free(p_vs);
}

void vs_freev(void *p_vs)
{
	free(((vstr*)p_vs)->s);
	free(p_vs);
}

int vs_print(vstr *p_vs, const char *p_format, ...)
{
	int length;
	va_list alist;

	va_start(alist, p_format);
	length = vsnprintf(NULL, 0, p_format, alist);
	va_end(alist);

	if(vs_realloc(p_vs, length))
		return -1;

	va_start(alist, p_format);
	vsprintf(p_vs->s, p_format, alist);
	va_end(alist);

	p_vs->len = length;

	return length;
}

int vs_cat(vstr *p_vs, const char *p_format, ...)
{
	int length;
	va_list alist;

	va_start(alist, p_format);
	length = vsnprintf(NULL, 0, p_format, alist) + p_vs->len;
	va_end(alist);

	if(vs_realloc(p_vs, length))
		return -1;

	va_start(alist, p_format);
	vsprintf(&p_vs->s[p_vs->len], p_format, alist);
	va_end(alist);

	p_vs->len = length;

	return length;
}

int vs_catn(vstr *p_vs, int n, const char *p_str)
{
	char *s;

	if(vs_realloc(p_vs, p_vs->len + n))
		return -1;

	s = &p_vs->s[p_vs->len];
	p_vs->len += n;

	while(n > 0)
	{
		*s = *p_str;
		s++;
		p_str++;
		n--;
	}
	*s = 0;

	return p_vs->len;
}

void vs_clear(vstr *p_vs)
{
	if(vs_realloc(p_vs, 0))
		return;

	p_vs->s[0] = 0;
	p_vs->len = 0;
}

int vs_copy(vstr *p_dest, vstr *p_source)
{
	char *p_s;

	if(!(p_s = malloc(p_source->_private_buffersize)))
		return 0;

	free(p_dest->s);
	p_dest->s = p_s;
	p_dest->len = p_source->len;
	p_dest->_private_buffersize = p_source->_private_buffersize;

	memcpy(p_dest->s, p_source->s, p_source->len + 1);

	return p_source->len;
}

int vs_size(vstr *p_vstr)
{
	return p_vstr->len;
}
