#!/usr/bin/perl

package Debian::DictionariesCommon;

use base qw(Exporter);

# List all exported symbols here.
our @EXPORT_OK = qw(parseinfo updatedb loaddb emacsen_support jed_support
		    getlibdir getsysdefault setsysdefault
		    getuserdefault setuserdefault
		    build_emacsen_support build_jed_support
                    build_pspell_support);
# Import :all to get everything.
our %EXPORT_TAGS = (all => [@EXPORT_OK]);

my $infodir = "/var/lib/dictionaries-common";
my $cachedir = "/var/cache/dictionaries-common";
my $ispelldefault = "ispell-default";
my $sysdefault = "/etc/dictionaries-common/$ispelldefault";
my $userdefault = "$ENV{HOME}/.$ispelldefault";
my $emacsensupport = "emacsen-ispell-dicts.el";
my $jedsupport = "jed-ispell-dicts.sl";

sub getlibdir {
  my $class = shift;
  return "$infodir/$class";
}

sub mydie {
  my $routine = shift;
  my $errmsg = shift;
  die __PACKAGE__, "($routine):E: $errmsg";
}

sub parseinfo {
  my $file = shift;
  open (DICT, "< $file");
  my $old_irs=$/; # Save current value for input record separator
  $/ = "";
  my %dictionaries =
    map {
      s/^([^:]+):/lc ($1) . ":"/meg;  # Lower case field names
      my %hash = /^([^:]+):\s*((?<!\n)[^\n]+)\s*$/mg;
      map { delete $hash{$_} if ($hash{$_} =~ /^\s+$/) } keys %hash;
      mydie ('parseinfo',
	     qq{Record in file $file does not have a "Language" entry})
	if not exists $hash{language};
      mydie ('parseinfo',
	     qq{Record in file $file does not have a "Hash-Name" entry})
	if not exists $hash{"hash-name"};
      my $lang = delete $hash{language};
      ($lang, \%hash);
    } <DICT>;
  $/ = $old_irs; # Reset value of input record separator
  return \%dictionaries;
}

# ------------------------------------------------------------------
sub dc_dumpdb {
# ------------------------------------------------------------------
# Save %dictionaries in Data::Dumper like format. This function
# should be enough for the limited needs of dictionaries-common
# ------------------------------------------------------------------
  my $class        = shift;
  my $dictionaries = shift;
  my @fullarray    = ();
  my @dictarray    = ();
  my $output       = "$cachedir/$class.db";
  my $dictentries  = '';
  my $thevalue     = '';
  
  foreach $thedict ( sort keys %{$dictionaries}){
    $dictentries = $dictionaries->{$thedict};
    @dictarray   = ();
    foreach $thekey ( sort keys %{$dictentries}){
      $thevalue = $dictentries->{$thekey};
      # Make sure \ and ' are escaped in keyvals
      $thevalue =~ s/(\\|\')/\\$1/g;
      push (@dictarray,"     \'$thekey\' => \'$thevalue\'");
    }
    # Make sure \ and ' are escaped in dict names
    $thedict =~ s/(\\|\')/\\$1/g;
    push (@fullarray,
	  "  \'$thedict\' => \{\n" . join(",\n",@dictarray) . "\n  \}");
  }
  
  mkdir $cachedir unless (-d $cachedir);
  
  open (DB,"> $output");
  print DB generate_comment("### ") . "\n";
  print DB "%dictionaries = (\n";
  print DB join (",\n",@fullarray);
  print DB "\n);\n\n1;\n";
  close DB;
}

sub updatedb {
  my $class = shift;
  opendir (DIR, "$infodir/$class");
  my @infofiles = grep {/^[^\.]/} readdir DIR;
  closedir DIR;
  my %dictionaries = ();
  foreach my $f (@infofiles) {
    next if $f =~ m/.*~$/;                         # Ignore ~ backup files
    my $dicts = parseinfo ("$infodir/$class/$f");
    %dictionaries = (%dictionaries, %$dicts);
  }
  &dc_dumpdb($class,\%dictionaries);
}

sub loaddb {
  my $class = shift;
  my $dbfile = "$cachedir/$class.db";
  if (-e $dbfile) {
    do $dbfile;
  }
  return \%dictionaries;
}

sub getdefault {
  $file = shift;
  if (-f $file) {
    my $lang = `cat $file`;
    chomp $lang;
    return $lang;
  }
  else {
    return undef;
  }
}

sub getuserdefault {
  getdefault ($userdefault);
}

sub getsysdefault {
  getdefault ($sysdefault);
}

sub setsysdefault {
  $value = shift;
  open (DEFAULT, "> $sysdefault");
  print DEFAULT $value;
  close DEFAULT;
}

sub setuserdefault {

  my $default = getuserdefault ();

  my $dictionaries = loaddb ("ispell");

  my @choices = sort keys %$dictionaries;

  if (scalar @choices == 0) {
    warn "Sorry, no ispell dictionary is installed in your system.\n";
    return;
  }

  my $initial = -1;
  if (defined $default) {
    for (my $i = 0; $i < scalar @choices; $i++) {
      if ($default eq $choices[$i]) {
	$initial = $i;
	last;
      }
    }
  }

  open (TTY, "/dev/tty");
  while (1) {
    $| = 1;
    print
      "\nSelect your personal ispell dictionary for use with ispell-wrapper\n\n";
    for ($i = 0; $i < scalar @choices; $i++) {
      print "  " . ($i == $initial ? "*" : " ")
	     . " [" . ($i+1) . "] $choices[$i]\n";
    }
    print qq(\nSelect number or "q" for quit)
      . ($initial != -1 ? " (* is the current default): " : ": ");
    my $sel = <TTY>;
    chomp $sel;
    last if $sel eq "q";
    if ($sel < 1 or $sel > scalar @choices) {
      print qq{\nInvalid choice "$sel".\n\n};
      next;
    }
    else {
      $sel--;
      open (DEFAULT, "> $userdefault");
      print DEFAULT $choices[$sel];
      close DEFAULT;
      last;
    }
  }
  close TTY;
}

sub generate_comment {
  my $commstr = shift;
  my $comment = "This file is part of the dictionaries-common package.
It has been automatically generated.
DO NOT EDIT!";
  $comment =~ s{^}{$commstr}mg;
  return "$comment\n";
}

sub build_emacsen_support {

  my $elisp = '';
  my $availability = '';
  my @classes=("aspell","ispell");
  my %entries = ();
  my %aspell_locales = ();
  my %emacsen_ispell = ();
  my %emacsen_aspell = ();
  
  foreach $class ( @classes ){
    my $dictionaries = loaddb ($class);
    
    foreach $k (keys %$dictionaries) {
      
      my $lang = $dictionaries->{$k};
      next if (exists $lang->{'emacs-display'} 
	       && $lang->{'emacs-display'} eq "no");
      
      my $hashname = $lang->{"hash-name"};
      my $casechars = exists $lang->{casechars} ?
	  $lang->{casechars} : "[a-zA-Z]";
      my $notcasechars = exists $lang->{"not-casechars"} ?
	  $lang->{"not-casechars"} : "[^a-zA-Z]";
      my $otherchars = exists $lang->{otherchars} ?
	  $lang->{otherchars} : "[']";
      my $manyothercharsp = exists $lang->{"many-otherchars"} ?
	  ($lang->{"many-otherchars"} eq "yes" ? "t" : "nil") : "nil";
      my $ispellargs = exists $lang->{"ispell-args"} ?
	  ('("' . join ('" "', split (/\s+/, $lang->{"ispell-args"}))
	   . '")') : (qq/("-d" "/ . $lang->{"hash-name"} . qq/")/) ;
      my $extendedcharactermode = exists $lang->{"extended-character-mode"} ?
	  ('"' . $lang->{"extended-character-mode"} . '"') : "nil";
      my $codingsystem = exists $lang->{"coding-system"} ?
	  $lang->{"coding-system"} : "nil";
      my $emacsenname = exists $lang->{"emacsen-name"} ?
	  $lang->{"emacsen-name"} : $hashname;
      
      if ( $class eq "ispell" ){
	$emacsen_ispell{$emacsenname}++;
      } elsif ( $class eq "aspell" ){
	$emacsen_aspell{$emacsenname}++;	
	if ( exists $lang->{"aspell-locales"} ){
	  foreach ( split(/\s*,\s*/,$lang->{"aspell-locales"}) ){
	    $aspell_locales{$_}=$emacsenname;
	  }
	}    
      }
      
      if ( exists $emacsen_ispell{$emacsenname} and $emacsen_aspell{$emacsenname} ){
	$availability = "all";
      } elsif ( exists $emacsen_ispell{$emacsenname} ){
	$availability = "ispell";
      } elsif ( exists $emacsen_aspell{$emacsenname} ){
	$availability = "aspell";
      } else {
	$availability = "none"; # This should not happen
      }
      
      $entries{$emacsenname} =  qq{
(debian-ispell-add-dictionary-entry
  \'("$emacsenname"
    "$casechars"
    "$notcasechars"
    "$otherchars"
    $manyothercharsp
    $ispellargs
    $extendedcharactermode
    $codingsystem)
  "$availability")};
    }
  }

  open (ELISP, "> $cachedir/$emacsensupport")
    or die "Cannot open emacsen cache file";

  print ELISP generate_comment (";;; ");
  $elisp .= join ("\n", map {"$entries{$_}"} reverse sort keys %entries);

  if ( scalar %aspell_locales ){
    $elisp .= "\n\n;; An assoc list that will try to map locales to emacsen names";
    $elisp .= "\n\n(setq debian-aspell-equivs-alist \'(\n";
    foreach ( sort keys %aspell_locales ){
      $elisp .= "     (\"$_\" \"$aspell_locales{$_}\")\n";
    }
    $elisp .= "))\n";
    # Obtain here debian-aspell-dictionary, after debian-aspell-equivs-alist
    # is loaded
    $elisp .="
;; Get default value for debian-aspell-dictionary. Will be used if
;; spellchecker is aspell and ispell-local-dictionary is not set.
;; We need to get it here, after debian-aspell-equivs-alist is loaded

(setq debian-aspell-dictionary (debian-get-aspell-default))\n\n";
  } else {
      $elisp .= "\n\n;; No emacsen-aspell-equivs entries were found\n";
  }

  print ELISP $elisp;
  close ELISP;
}

sub build_jed_support {

  my $dictionaries = loaddb ("ispell");
  my $slang = generate_comment ("%%% ");

  foreach $k (keys %$dictionaries) {

    my $lang = $dictionaries->{$k};
    next if (exists $lang->{'jed-display'} 
	     && $lang->{'jed-display'} eq "no");

    my $hashname = $lang->{"hash-name"};
    my $additionalchars = exists $lang->{additionalchars} ?
      $lang->{additionalchars} : "";
    my $otherchars = exists $lang->{otherchars} ?
      $lang->{otherchars} : "'";
    my $emacsenname = exists $lang->{"emacsen-name"} ?
      $lang->{"emacsen-name"} : $hashname;
    my $extendedcharmode = exists $lang->{"extended-character-mode"} ?
      $lang->{"extended-character-mode"} : "";
    my $ispellargs = exists $lang->{"ispell-args"} ?
      $lang->{"ispell-args"} : "";

    $slang .= qq{
ispell_add_dictionary (
  "$emacsenname",
  "$hashname",
  "$additionalchars",
  "$otherchars",
  "$extendedcharmode",
  "$ispellargs");
};
  }

  open (SLANG, "> $cachedir/$jedsupport")
    or die "Cannot open jed cache file";
  print SLANG $slang;
  close SLANG;
}

# Ensure we evaluate to true.
1;

__END__

#Local Variables:
#perl-indent-level: 2
#End: 

=head1 NAME

Debian::DictionariesCommon.pm - dictionaries-common library

=head1 SYNOPSIS

    use Debian::DictionariesCommon q(:all)
    $dictionaries = parseinfo ('/var/lib/dictionaries-common/ispell/iwolof');
    loaddb ('ispell')
    updatedb ('wordlist')

=head1 DESCRIPTION

(To be written)

=head1 SEE ALSO

(To be written)

=head1 AUTHORS

Rafael Laboissiere

=cut
