/*
  ecp is designed to copy files, and is named extended cp.
  Copyright (C) 2006  Andre Reffhaug <areffhaug@gmail.com>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  ecp homepage   : https://savannah.nongnu.org/projects/ecp/
  Contact author : areffhaug@gmail.com
*/




/*
  FIXME:

  Own realpath function would also be good, but thats in the 
  works.

  include functionality for mounting, ie copying right into
  devices? holy crap, batman, that'll be strenous

  have a look at size of CHUNK

  if recursively copying into a directory directly beneath the one
  we are copying, how to prevent it from forever recursing into 
  itself? one thought is to check if it does, and recursively copy
  it first, then add its inode to the checking mechanism. seems a 
  bit dense, though.

  what about human readable outputs? halfdecent hack thrown together

  hidden files and shell globbing: * doesnt include hidden files, which
  makes that whole shebang a little more crazy

  what about some sort of mode to output more information on each
  file in a sort of readable format? that'd be nice.

*/

#include <syscall.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <dirent.h>
#include <utime.h>
#include <errno.h>
#include <sys/mman.h>
#include "copyfun.h"
#include "dir.h"
#include "ecp_info.h"
#include "ftp_conn.h"

extern int path_max;


void
init_progress (struct progress *prog, off_t *bsize, char *dstname) 
{
    prog->fname = malloc (path_max * sizeof (*prog->fname));

    if (prog->fname == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory ==> aborting...\n", NAME);
        exit (EXIT_FAILURE);
    }
  
    prog->percent = 0;
    prog->in_total.bytes = *bsize;
    prog->in_total.kbytes = prog->in_total.bytes / 1000;
    prog->in_total.mbytes = prog->in_total.kbytes / 1000;
    prog->total.bytes = 0;

    strcpy (prog->fname, dstname);
  
}

void update_progress (struct progress *prog, struct opt *options)
{
    prog->percent = ((float) prog->total.bytes * 100) 
        / (float) prog->in_total.bytes;

    if ((options->ftp_transfer == true) && prog->in_total.bytes > 10000000)
        prog->chunk += (prog->in_total.bytes / (CHUNK*8));
    else
        prog->chunk += (prog->in_total.bytes / (CHUNK));

}

void do_progress (struct progress *prog, struct opt *options)
{

    if (options->hreadable == true) 
    {
        if ((prog->in_total.bytes < 100000))
            printf ("\r%6.2f%%  %3lu / %3lu kB ==> %s", prog->percent, 
                    prog->total.bytes / 1000, prog->in_total.kbytes, prog->fname);
        if (prog->in_total.bytes > 100000)
            printf ("\r%6.2f%%  %3lu / %3lu mB ==> %s", prog->percent, 
                    prog->total.bytes / 1000000, prog->in_total.mbytes, prog->fname);
    }
    else
        if ((prog->in_total.bytes < 100000) && options->hreadable != true)
            printf("\r%6.2f%%  %6lu / %5lu  B ==> %s", prog->percent, 
                   prog->total.bytes, prog->in_total.bytes, prog->fname);
        else printf("\r%6.2f%%  %6lu / %5lu kB ==> %s", prog->percent, 
                    prog->total.bytes / 1000, prog->in_total.kbytes, prog->fname);

    fflush (stdout);
}


bool do_copy (char *srcname, 
              char *outfile,
              struct opt *options, 
              bool todir)
{
    ssize_t readbytes, bwritten, buffersize;
    char * __restrict__ buf;
    char temp[path_max]; /* for readlink */
  
    int srcdesc;
    struct stat srcstat;

    char dstname[path_max];
    int dstdesc;
    struct stat dststat;

    struct progress *prog;
    struct utimbuf utimdef;

    prog = malloc (sizeof (*prog));

    if (prog == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory for struct prog ==> aborting...\n",
                 NAME);
        exit (EXIT_FAILURE);
    }
  
    /* set up our local copies of the filenames */
    strcpy (dstname, outfile);

    if (options->no_hidden_files == true && srcname[0] == '.') 
        return 1;

    if ((srcdesc = open (srcname, O_RDONLY, 0)) == -1) 
    {
        fprintf (stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
                 NAME, srcname, strerror (errno));
        return 0;
    }
    /* get hold of file information */
    lstat (srcname, &srcstat);


    if (S_ISLNK (srcstat.st_mode) && options->symlinks == true) 
    {
        readlink (srcname, srcname, path_max);
        lstat (srcname, &srcstat);
    } 

    if (options->no_zero_bytes == true)
        if (srcstat.st_size == 0) 
        {
            close (srcdesc);
            return 1;
        }



    if (todir == true) 
    {
        apptodir (dstname, base(srcname));
        if (S_ISLNK (srcstat.st_mode) && options->symlinks != true) 
        {
            memset (temp, '\0', sizeof (temp));
            readlink (srcname, temp, path_max);
            symlink (temp, dstname);
            close (srcdesc);
            if (options->kill == true) {
                if (remove (srcname) == -1)
                    fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                             NAME, srcname, strerror (errno));
            }
            return 1;
        } 
    }

    /* atleast it responds, but it really does need to be improved upon */
    if (options->update == true) {
        stat (dstname, &dststat);
        if (srcstat.st_mtime < dststat.st_mtime)
        {
            printf ("%s: target is newer or the same as '%s'\n", NAME, dstname);
            return 1;
        }
    }



    if ((dstdesc = open (dstname, O_CREAT | O_TRUNC | O_WRONLY, 
                         srcstat.st_mode)) == -1) 
    { 
        fprintf (stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
                 NAME, dstname, strerror (errno));
        return 0;
    }


    /* initializing the progress struct for our current file */
    init_progress (prog, &srcstat.st_size, dstname);

    if ((srcstat.st_size == 0) && options->quiet != true) 
    {
        prog->percent = 100;
        do_progress (prog, options);
        close (srcdesc);
        close (dstdesc);
        chmod (dstname, srcstat.st_mode&
               (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));
        if (options->kill == true) 
        {
            if ((chmod (srcname, 0777)) == -1)
                fprintf (stderr, "%s: Chmod for file '%s' failed\n", NAME, srcname);
          
            if (remove (srcname) == -1)
                fprintf (stderr, "%s: Couldn't remove source file '%s' (%s)\n", 
                         NAME, srcname, strerror (errno));
        }
        printf("\n");

        return 1;
    } 
    else if ((srcstat.st_size == 0) && options->quiet == true) 
    {
        close (srcdesc);
        close (dstdesc);
        if (options->kill == true) 
        {
            if ((chmod (srcname, 0777)) == -1)
            {
                fprintf (stderr, "%s: Chmod failed for file '%s'\n", NAME, srcname);
            }
            if (remove (srcname) == -1)
                fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                         NAME, srcname, strerror (errno));
        }

        return 0;
    }

    


    /* set up buffersize, according to disks blocksize */
    buffersize = srcstat.st_blksize;  
    /* if the file is actually smaller than the chunk we would like to mmap, we
       map the whole thing to memory, and write it back in one big swoop */

    buf = malloc (buffersize * sizeof (*buf));

    if (buf == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory for buffer in do_copy\n", NAME);
        exit (EXIT_FAILURE);
    }
    do
    {
        readbytes = read (srcdesc, buf, buffersize);

        if (readbytes == -1)
        {
            fprintf (stderr, "%s: An error occured while reading file '%s'\n", NAME, srcname);
            printf ("%s\n", strerror (errno));
            return 1;
        }
        bwritten = write (dstdesc, buf, readbytes);

        if (bwritten != readbytes)
        {
            fprintf (stderr, "%s: Error while writing to file '%s'\n", NAME, dstname);
            return 1;
        }
        prog->total.bytes += bwritten;
      
        if (options->quiet != true) 
        {
            if ((prog->total.bytes >= prog->chunk*2) || (bwritten < buffersize)) 
            {
                update_progress (prog, options);
                do_progress (prog, options);
            }
        }

    } while (readbytes != 0);


    

    if (options->quiet != true) printf("\n");

    if (options->preserve == true) 
    {
        utimdef.actime = srcstat.st_atime;
        utimdef.modtime = srcstat.st_mtime;
        if (!utime (dstname, &utimdef) == 0)
            fprintf (stderr, "%s: error setting utime of file '%s' (%s)\n", 
                     NAME, dstname, strerror (errno));
    }

    /* FIXME: still not working right, even though it now retains
       file permissions. Feels like some sort of cheap hack. what about
       S_ISUID (set-used-ID) and S_ISVTX (sticky bit) and S_ISGID? 
    */

    chmod (dstname, srcstat.st_mode&
           (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));

    /* clean up */
    if (options->kill == true) 
    {
        if ((chmod (srcname, 0777)) == -1)
            fprintf (stderr, "%s: Chmod for file '%s' failed.\n", NAME, srcname);
      
        if (remove (srcname) == -1)
            fprintf (stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                     NAME, srcname, strerror (errno));
    }

    close (srcdesc);
    close (dstdesc);

    return 1;
}

void copy_recursive(char *real_src_dir, 
                    char *real_target_dir,
                    int depth, ino_t orig_inode, 
                    struct opt *options) 
{
    char *copy_realsrc, *copy_targetdir, *new_dir;
    char *mkd;

    DIR *directory;
    struct dirent *current;
    struct stat statfile;

    copy_realsrc = malloc (path_max * sizeof (*copy_realsrc));
    copy_targetdir = malloc (path_max * sizeof (*copy_targetdir));

    realpath (real_src_dir, copy_realsrc);
    realpath (real_target_dir, copy_targetdir);


    lstat (real_src_dir, &statfile);
    if (statfile.st_ino == orig_inode) 
    {
        printf ("%s: skipping target directory '%s'\n", NAME, real_src_dir);
        return;
    }
    remove_trailing_slashes (copy_targetdir);

    if ((directory = opendir (copy_realsrc)) == NULL) 
    {
        fprintf (stderr, "%s: error opening directory '%s' (%s)\n", 
                 NAME, copy_realsrc, strerror (errno));
        return;
    }

    chdir (copy_realsrc);

    while ((current = readdir (directory)) != NULL) 
    {
        lstat (current->d_name, &statfile);
        if (S_ISDIR (statfile.st_mode)) 
        {
            new_dir = malloc (path_max * sizeof *new_dir);
            mkd = malloc (path_max * sizeof *mkd);

            /* if we are seeing ., .., continue */
            if ((strcmp (".", current->d_name) == 0) || 
                (strcmp ("..", current->d_name) == 0)) continue;

            /* check for hidden files */ 
            if (options->no_hidden_files == true)
                if (current->d_name[0] == '.')
                    continue;
    
            realpath (current->d_name, new_dir);

            strcpy (mkd, copy_targetdir);
            apptodir (mkd, current->d_name);
            mkdir (mkd, 0777);
            chmod (mkd, statfile.st_mode);
            copy_recursive (new_dir, mkd, depth+4, orig_inode,
                            options);
            if (options->kill == true) 
            { 
                chmod (new_dir, 0777); /* precaution against readonly dirs */
                if (remove (new_dir) != 0)
                    fprintf (stderr, "%s: removing '%s' failed (%s)\n", 
                             NAME, new_dir, strerror (errno));
            }

        }
        else 
        {
            do_copy (current->d_name, copy_targetdir, options, true);
        }

    }
    chdir ("..");
    closedir (directory);
}



