/*
  ecp is designed to copy files, and is named extended cp.
  Copyright (C) 2006  Andre Reffhaug <areffhaug@gmail.com>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  ecp homepage   : https://savannah.nongnu.org/projects/ecp/
  Contact author : areffhaug@gmail.com
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include "ftp_conn.h"
#include "copyfun.h"
#include "dir.h"
#include "ecp_info.h"

#define _GNU_SOURCE
#include <getopt.h>


#ifdef PATH_MAX
long path_max = PATH_MAX;

#else
long path_max = 0;

#endif

#define PATH_APPROX 4096

long palloc (void) 
{
    long paloc;

    if (path_max == 0) 
    {
        errno = 0;

        if ((paloc = pathconf ("/", _PC_PATH_MAX)) < 0) 
        {
            if (errno == 0)
            {
                paloc = PATH_APPROX;
            }
      
        }
    } else paloc = path_max;

    return (paloc);
}

void
init_opts (struct opt *opts)
{
    opts->information = false; /* implemented */
    opts->recursive = false; /* implemented */
    opts->preserve = false; /* implemented */
    opts->no_zero_bytes = false; /* implemented */
    opts->no_hidden_files = false; /* implemented */
    opts->create = false; /* implemented, depth 0 */
    opts->symlinks = false; /* implemented */
    opts->kill = false; /* implemented */
    opts->quiet = false; /* implemented */
    opts->update = false; /* half implemented */
    opts->hreadable = false; /* half implemented */
    opts->noclobber = false; /* not implemented */
    opts->verbose = false; /* half implemented */
    opts->ftp_transfer = false; /* implemented */
    opts->ftp_resume = false;
    opts->total_count = false;
}
    
int
main (int argc, char **argv)
{

    int opt = 0;
    struct opt options;
    struct stat destination;
    struct stat target;
    char *dest = NULL;

    /* ftp options */
    char localpath[1024];
    ftp_t *ftp;
    ftp_t *dataftp;

    char *messagebuf;
  
    int conres;

    /* for resolving the address */
    int port = 21;
  
    /* hacked in to avoid infinite recursion */
    struct stat origtarget;

    struct option longopts[] = {
        {"version", 0, NULL, 0}, /* output some versioning information */
        {"help", 0, NULL, 1}, /* output help messages */
        {"information", 0, NULL, 'i'}, /* general information */
        {"recursive", 0, NULL, 'R'}, /* for cp compatibility */
        {"preserve", 0, NULL, 'p'}, /* preserves timestamps */
        {"no-zero-bytes", 0, NULL, 2}, /* doesnt copy zero byte files */
        {"no-hidden-files", 0, NULL, 'n'}, /* omits hidden files */
        {"create", 0, NULL, 'c'}, /* creates destination directory if it does not exist */
        {"d-syms", 0, NULL, 's'}, /* dereference symlinks. change name */
        {"kill", 0, NULL, 'k'}, /* removes source directories/files */
        {"quiet", 0, NULL, 'q'}, /* stay quiet, no output at all */
        {"update", 0, NULL, 'u'}, /* copy only newer or modified files */
        {"human-readable", 0, NULL, 'h'}, /* try to output human readable numbers */
        {"no-clobber", 0, NULL, 3}, /* dont clobber already existing files */
        {"verbose", 0, NULL, 'v'},
        {"port", 1, NULL, 4},
        {"resume", 0, NULL, 5},
        {"total", 0, NULL, 't'},
        {0, 0, 0, 0}
    };

    opterr = 0;
    init_opts (&options);
//    memset (&options, 0, sizeof (options));
    path_max = palloc ();

    while ((opt = getopt_long (argc, argv, "iRpncskquhvt", longopts, NULL)) != -1)
    {
        switch (opt) 
        {
        case 0:
            version();
            author();
            printf ("\n");
            conditions ();
            exit (EXIT_SUCCESS);
            break;
        case 1:
            usage();
            printf ("\nBug reports should go to areffhaug@gmail.com\n");
            exit (EXIT_SUCCESS);
            break;
        case 2:
            options.no_zero_bytes = true;
            break;
        case 3:
            options.noclobber = true;
            break;
        case 4:
            port = atoi (optarg);
            break;
        case 5:
            options.ftp_resume = true;
            break;
        case 'n':
            options.no_hidden_files = true;
            break;
        case 'i':
            information();
            exit (EXIT_SUCCESS);
            break;
        case 'R':
            options.recursive = true;
            break;
        case 'p':
            /* this could have an option, like what to preserve */
            options.preserve = true;
            break;
        case 'c':
            options.create = true;
            break;
        case 'k':
            options.kill = true;
            break;
        case 's':
            options.symlinks = true;
            break;
        case 'q':
            options.quiet = true;
            break;
        case 'u':
            options.update = true;
            break;
        case 'h':
            options.hreadable = true;
            break;
        case 'v':
            options.verbose = true;
            break;
        case 't':
            options.total_count = true;
            break;
        case '?':
            fprintf (stderr, "%s: Unrecognized command line options\n", 
                     NAME);
            help_hint ();
            exit (EXIT_FAILURE);
            break;
        default:
            break;
        }
    }
  
    if (argc <= 1) 
    {
        version();
        printf ("%s: Missing arguments\n", NAME);
        exit (EXIT_SUCCESS);
    }

    /* if we are doing an ftp copy, just call the proper routines and exit */
    if ((strncmp (argv[optind], "ftp://", 6)) == 0) 
    {
        options.ftp_transfer = true;
      
        ftp = alloc_ftp ();
        ftp_parsecmd (ftp, argv[optind]);
        ftp_connect (ftp, port);
        ftp_login (ftp, port, &options, messagebuf);

        dataftp = ftp_mode (ftp, messagebuf);
        dataftp->socket = socket (AF_INET, SOCK_STREAM, 0);

        dataftp->len = sizeof (dataftp->address);

        conres = connect (dataftp->socket, (struct sockaddr *) &dataftp->address,
                          dataftp->len);

        if (conres == -1) 
        {
            fprintf (stderr, "%s: Could not connect to data socket (%s) ==> "
                     "aborting...\n", NAME, strerror (errno));
            exit (EXIT_FAILURE);
        }

        FD_ZERO (&dataftp->readfds);
        FD_SET (dataftp->socket, &dataftp->readfds);

        /* if user supplied a filename, we use that */
        if ((argc - optind) == 2)
        {
            lstat (argv[argc-1], &destination);
            if (S_ISDIR (destination.st_mode))
                apptodir (argv[argc-1], base (ftp->path));
            strcpy (localpath, argv[argc - 1]);
        }
        else
            /* otherwise we use the remote filename */
            strcpy (localpath, base (ftp->path));
        ftp_retr (ftp, dataftp, ftp->path, localpath, messagebuf, &options);

        close (ftp->socket);
        close (dataftp->socket);

        free (dataftp);
        free (ftp);
        if (options.quiet != true) printf ("\n");

        return 0;
    } /* END FTP DOWNLOAD BLOCK */

    if ((strncmp (argv[argc-1], "ftp://", 6)) == 0) /* we are copying TO an FTP */
    {
        ftp = alloc_ftp ();
        ftp_parsecmd (ftp, argv[argc-1]);
        ftp_connect (ftp, port);
        ftp_login (ftp, port, &options, messagebuf);
      
  
        ftp_sendcmd (ftp, "CWD", ftp->path);
        wait_server (ftp);
        messagebuf = ftp_reader (ftp);

        if (options.verbose == true) printf ("%s\n", messagebuf);
        ftp_getresp (ftp, messagebuf);

        if (ftp->response != FTP_CWDOK)
        {
            fprintf (stderr, "%s: Could not change to directory %s on remote server "
                     "==> aborting...\n", NAME, ftp->path);
            exit (EXIT_FAILURE);
        }

        dataftp = ftp_mode (ftp, messagebuf);
        dataftp->socket = socket (AF_INET, SOCK_STREAM, 0);

        dataftp->len = sizeof (dataftp->address);

        conres = connect (dataftp->socket, (struct sockaddr *) &dataftp->address,
                          dataftp->len);

        if (conres == -1) 
        {
            fprintf (stderr, "%s: Could not connect to data socket (%s) ==> "
                     "aborting...\n", NAME, strerror (errno));
            exit (EXIT_FAILURE);
        }
    
        FD_ZERO (&dataftp->readfds);
        FD_SET (dataftp->socket, &dataftp->readfds);
        stat (argv[optind], &destination);
        
        ftp_store (ftp, dataftp, argv[optind], messagebuf, &options, &destination.st_size);

/*         for (; optind < (argc-1); ++optind) */
/*         { */
/*             stat (argv[optind], &destination); */
/*             ftp_store (ftp, dataftp, argv[optind], messagebuf, &options, &destination.st_size); */
/*             ftp_sendcmd (ftp, "PASV", ""); */
/*             do */
/*             { */
/*                 messagebuf = ftp_reader (ftp); */
/*                 ftp_getresp (ftp, messagebuf); */
/*                 printf ("%s\n", messagebuf); */
/*             } while (ftp->multiline == true); */

/*             if (options.quiet != true) printf ("\n"); */
/*         } */
        if (options.quiet != true) printf ("\n");

        return 0;
      
    } /* END FTP UPLOAD BLOCK */

    options.ftp_transfer = false;
    
    /* START NORMAL FILE COPY BLOCK*/
    if ((dest = malloc (path_max * sizeof (*dest))) == NULL) 
    {
        fprintf (stderr, "%s: Could not malloc\n", NAME);
        exit (EXIT_FAILURE);
    }

    remove_trailing_slashes (argv[argc-1]);
    realpath (argv[argc-1], dest);
  
    lstat (dest, &origtarget);
    if (((argc-optind) > 3) && (!S_ISDIR (origtarget.st_mode)) 
        && options.create != true) 
    {
        printf ("%s: Target directory '%s' doesn't seem to exist ==> "
                "aborting\n", NAME, dest);
        exit (EXIT_FAILURE);
    }

    if (options.create == true) 
    {
        if (mkdir (dest, 0777) != 0) 
        {
            printf ("%s: Can't create directory: %s\n", NAME, strerror(errno));
            exit(EXIT_SUCCESS);
        }
    }
    
    /* if invoked with --total or -t, we need to count a little differently than
       usual */
    if (options.total_count == true)
    {
        
        unsigned long long fs = 0;
        int fc = 0;
        unsigned long long total_progress = 0;
        int files_copied = 1;
        unsigned long long bwritten;
        int copy_optind = optind;

        printf ("Traversing tree to compute total copy size...");
        fflush (stdout);
        /* walk the arguments, counting sizes and file count */
        for(; copy_optind < (argc-1); ++copy_optind) 
        {
            stat (argv[copy_optind], &destination);
            if (S_ISDIR (destination.st_mode))
            {
                count_total (argv[copy_optind], &fs, &fc, &options);
            }
            else
            {
                fs += destination.st_size;
                fc += 1;
            }
        }
        printf ("\rTraversing tree to compute total copy size... Done.\n");

        for(; optind < (argc-1); ++optind) 
        {
            stat (argv[optind], &destination);
            if (options.recursive == true) 
            {
                remove_trailing_slashes (argv[optind]);
                if (S_ISDIR (destination.st_mode)) 
                {
                    apptodir (dest, base(argv[optind]));
                    if (stat (dest, &target) == 0) 
                    {
                        if (S_ISREG (target.st_mode)) 
                        {
                            printf ("%s: Target '%s' is a regular file ==> aborting\n", 
                                    NAME, dest);
                            exit (EXIT_FAILURE);
                        }
                    }
                    mkdir (dest, 0777);
                    copy_recursive_total (argv[optind], dest, 0, origtarget.st_ino, &options,
                                          &total_progress, &fs, &files_copied, &fc);
                    chmod (dest, destination.st_mode);
                    strcpy (dest, argv[argc-1]);
                    if (options.kill == 1) remove (argv[optind]);
                } else
                {
                    realpath (argv[argc-1], dest);
                    bwritten = do_copy_total (argv[optind], dest, &options, true,
                                              &total_progress, &fs, &files_copied, &fc);
                    total_progress = bwritten;
                    files_copied += 1;
                }
            }
            else if (S_ISDIR (origtarget.st_mode)) 
            {
                if (S_ISDIR (destination.st_mode)) 
                {
                    printf ("%s: Skipping directory '%s'\n", NAME, argv[optind]);
                    continue;
                }
                else
                {
                    bwritten = do_copy_total (argv[optind], dest, &options, true,
                                              &total_progress, &fs, &files_copied, &fc);
                    total_progress = bwritten;
                    files_copied += 1;
                }
            }
            else if (!S_ISDIR (origtarget.st_mode))
            {
                bwritten = do_copy_total (argv[optind], dest, &options, false,
                                          &total_progress, &fs, &files_copied, &fc);
                total_progress = bwritten;
                files_copied += 1;
            }
        }
        free (dest);
        if (options.quiet != true)
            printf ("\n");
        
        return 0;
    }
    
    for(; optind < (argc-1); ++optind) 
    {
        stat (argv[optind], &destination);
        if (options.recursive == true) 
        {
            remove_trailing_slashes (argv[optind]);
            if (S_ISDIR (destination.st_mode)) 
            {
                apptodir (dest, base(argv[optind]));
                if (stat (dest, &target) == 0) 
                {
                    if (S_ISREG (target.st_mode)) 
                    {
                        printf ("%s: Target '%s' is a regular file ==> aborting\n", 
                                NAME, dest);
                        exit (EXIT_FAILURE);
                    }
                }
                mkdir (dest, 0777);
                copy_recursive (argv[optind], dest, 0, origtarget.st_ino, &options);
                chmod (dest, destination.st_mode);
                strcpy (dest, argv[argc-1]);
                if (options.kill == 1) remove (argv[optind]);
            } else
            {
                realpath (argv[argc-1], dest);
                do_copy (argv[optind], dest, &options, true);
            }
        }
        else if (S_ISDIR (origtarget.st_mode)) 
        {
            if (S_ISDIR (destination.st_mode)) 
            {
                printf ("%s: Skipping directory '%s'\n", NAME, argv[optind]);
                continue;
            }
            else
            {
                do_copy (argv[optind], dest, &options, true);
            }
        }
        else if (!S_ISDIR (origtarget.st_mode))
        {
            do_copy (argv[optind], dest, &options, false);
        }
    }
    free (dest);
    return 0;
}
