/*
  ecp is designed to copy files, and is named extended cp.
  Copyright (C) 2006  Andre Reffhaug <areffhaug@gmail.com>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  ecp homepage   : https://savannah.nongnu.org/projects/ecp/
  Contact author : areffhaug@gmail.com
*/

/* response from PASV is 76 bytes if we managed to grab it all. 
 * htons((132 << 8) + (186)) -> calculating the 16 bit int portnumber
 * received from PASV
 *
 *
 *
 */
//    address.sin_addr.s_addr = inet_addr ("64.111.115.19");



/*
  FTP COMMANDS

  PASV - indicate to server you want a passive transfer
  PORT - indicate active transfer, the server connects to client
  RETR <file> - retrieve file
  USER <name> - username
  PASS <pass> - password
  MKD <name> - make directory
  TYPE I or II - binary or ascii transfer

*/
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <errno.h>
#include <signal.h>
#include "ftp_conn.h"
#include "copyfun.h"
#include "ecp_info.h"
#include "dir.h"



int 
wait_server (ftp_t *ftp)
{
    int result;
    int counter = 0;
    while (1)
    {
        /* needs to be re-set every call, as linux modifies this value */
        ftp->timeout.tv_sec = 3;
        ftp->timeout.tv_usec = 0;

        result = select (10, &ftp->readfds, &ftp->writefds, (fd_set *) NULL,
                         &ftp->timeout);

        switch (result)
        {
        case 0:
            counter += ftp->timeout.tv_sec;
            printf ("%s: Timeout waiting for socket. %d seconds elapsed.\n",
                    NAME, counter);
            if (counter >= ftp->timeout.tv_sec * 10)
            {
                fprintf (stderr, "%s: Seems like the server is dead ==> aborting...\n",
                         NAME);
                exit (EXIT_FAILURE);
            }
            break;
        case -1:
            fprintf (stderr, "%s: select has returned -1\n", NAME);
            return -1;
            break;
        default:
            if (FD_ISSET (ftp->socket, &ftp->readfds)) 
            {
                return 0;
            }
            break;
        }
    }
}

ftp_t *
alloc_ftp (void)
{
    ftp_t *ftp;
  
    ftp = malloc (sizeof  (*ftp));

    if (ftp == NULL) 
    {
        fprintf(stderr, "%s: Could not malloc space for ftp_t structure\n", 
                NAME);
        exit(EXIT_FAILURE);
    }

    return ftp;
}

void
ftp_connect (ftp_t *ftp, int port)
{
    struct addrinfo *addr;
    const char *add;
    struct sockaddr_in *in;
    int result;
    char abuf[INET_ADDRSTRLEN];
  
    result = getaddrinfo(ftp->server, NULL, NULL, &addr);

    if (result != 0)
    {
        fprintf (stderr, "%s: An error occured while resolving host "
                 "==> aborting...\n", NAME);
        fprintf (stderr, "%s: %s\n", NAME, gai_strerror (result));
        exit (EXIT_FAILURE);
    }
      
    in = (struct sockaddr_in *) addr->ai_addr;
    add = inet_ntop (AF_INET, &in->sin_addr, abuf, INET_ADDRSTRLEN);

    if (add == NULL)
    {
        fprintf (stderr, "%s: An error occured while converting server address "
                 "==> aborting...\n", NAME);
        exit (EXIT_FAILURE);
    }
      

    ftp->usePasv = true;
    ftp->socket = socket (AF_INET, SOCK_STREAM, 0);
    ftp->address.sin_family = AF_INET;
    ftp->address.sin_addr.s_addr = inet_addr(add);
    ftp->address.sin_port = htons (port);
    ftp->len = (socklen_t) sizeof (ftp->address);

    /* establish connection */
    result = connect (ftp->socket, (struct sockaddr *) &ftp->address, ftp->len);

    if (result == -1) 
    {
        fprintf (stderr, "%s: Could not connect (%s) ==> aborting...\n",
                 NAME, strerror (errno));
        exit(EXIT_FAILURE);
    }

    FD_ZERO (&ftp->readfds);
    FD_SET (ftp->socket, &ftp->readfds);

}

int 
ftp_sendcmd (ftp_t *ftp, const char *cmd, const char *arg)
{
    char *buffer;
    int sendbytes;
    bool ok;
  
    buffer = malloc (20 + strlen (cmd) + strlen(arg) * sizeof (*buffer));

    if (buffer == NULL)
    {
        fprintf (stderr, "Malloc failed in ftp_sendcmd\n");
        exit (EXIT_FAILURE);
    }

    strcpy (buffer, cmd);

    if (arg[0] != '\0') 
    {
        strcat (buffer, " ");
        strcat (buffer, arg);
    }

    strcat (buffer, "\r\n");
    sendbytes = write (ftp->socket, buffer, strlen (buffer));

    if (sendbytes != strlen (buffer))
    {
        fprintf (stderr, "%s: Whole message (follows) might not have gotten through "
                 "to server.\n", NAME);
        fprintf (stderr, "%s: %s\n", NAME, buffer);
        ok = false;
    }
    else
        ok = true;

    /* wait for server before we return from function */
    wait_server (ftp);

    free (buffer);
    return ok != true;
}

char *
ftp_reader (ftp_t *ftp)
{
    static char returnbuf[FTP_BUFSIZ];

    while (1)
    {
        int bytes;
        char *readpt = ftp->buffer + strlen (ftp->buffer);
        char *eol;

        if((eol = strstr(ftp->buffer, "\r\n")))
        {
            strncpy(returnbuf, ftp->buffer, eol - ftp->buffer);
            returnbuf[eol - ftp->buffer] = '\0';
            strcpy(ftp->buffer, eol+2);
            return returnbuf;
        }

        readpt = ftp->buffer + strlen(ftp->buffer);

        bytes = read(ftp->socket, readpt, FTP_BUFSIZ - strlen (ftp->buffer) - 2);

        if (bytes == 0)
            return NULL;
      
        if(bytes == -1)
        {
            fprintf (stderr, "%s: There was an error while reading socket (%s)\n",
                     NAME, strerror (errno));
            exit (EXIT_FAILURE);
        }
        else
            readpt[bytes] = '\0';
   
    }
}

bool trapped = false;

void
trap_sigint (int sig)
{
    trapped = true;
    (void) signal (SIGINT, SIG_DFL);
}

int 
ftp_retr (ftp_t *ftp, ftp_t *dataftp, char *fname,
          char *toname, char *buffer, struct opt *options)
{
    FILE *fp;
    int bytes;
    int wrbytes;
    struct progress prog;
    char *bptr;
    off_t bsize;
    struct stat dest;
    bool restok = false;
    char resume_size[20];

    /* add a signal handler for Ctrl-C, so we have a chance at flushing the buffers */
    /* and behaving nicely, thus making resume possible. */
    (void) signal (SIGINT, trap_sigint);

    if (options->ftp_resume == true)
    {
        stat (toname, &dest);
        sprintf (resume_size, "%lu", dest.st_size);
        ftp_sendcmd (ftp, "REST", resume_size);
        wait_server (ftp);
        buffer = ftp_reader (ftp);
        if (options->verbose == true) printf ("%s\n", buffer);

        ftp_getresp (ftp, buffer);
        if (ftp->response != 350)
        {
            printf ("%s: Server does not support resuming downloads\n", NAME);
        } else
            restok = true;
        
        
    }        

    ftp_sendcmd (ftp, "RETR", fname);

    wait_server (ftp);
    buffer = ftp_reader (ftp);
    if (options->verbose == true) printf ("%s\n", buffer);

    ftp_getresp (ftp, buffer);

    if (ftp->response != FTP_RETROK) 
    {
        if ((ftp->response == FTP_NOFILE) || (ftp->response == FTP_FNOEXIST)
            || (ftp->response == FTP_NOFILE2))
            fprintf (stderr, "%s: Server responded that the requested file does not "
                     "exist ==> aborting...\n", NAME);
        else
            fprintf (stderr, "%s: could not retrieve file => aborting...\n", NAME);

        exit (EXIT_FAILURE);
    }

    bptr = malloc (20 * sizeof (*bptr));

    if (bptr == NULL)
    {
        fprintf (stderr, "%s: Could not allocate memory ==> aborting...\n", NAME);
        exit (EXIT_FAILURE);
    }

    bptr = strchr (buffer, '(');

    if (bptr == NULL)
    {
        fprintf (stderr, "%s: Couldn't find size of remote file, approximating "
                 "(this is bullshit)\n", NAME);
        bsize = 1000000;
    } else  sscanf (bptr+1, "%lu", &bsize);

    if ((options->ftp_resume == true) && (restok == true))
    {
        bsize += dest.st_size;
        fp = fopen (toname, "ab");
    } else
        fp = fopen (toname, "wb");

    if (fp == NULL) 
    {
        fprintf (stderr, "Couldn't create local file => aborting...\n");
        exit (EXIT_FAILURE);
    }

    wait_server (dataftp);

    init_progress (&prog, &bsize, toname);
        
    if ((options->ftp_resume == true) && (restok == true))
        prog.total.bytes = dest.st_size;
    
    while ((bytes = read (dataftp->socket, dataftp->buffer, FTP_BUFSIZ)) != 0) 
    {
        if (trapped == true)
        {
            fflush (fp);
            fprintf (stderr, "\n%s: Interrupted ==> flushing to disk and "
                     "aborting...\n", NAME);
            exit (EXIT_FAILURE);
        }
        if (bytes == -1) 
        {
            fprintf (stderr, "Error while reading from socket => aborting...\n");
            exit (EXIT_FAILURE);
        }

        prog.total.bytes += bytes;
        update_progress (&prog, options);
        do_progress (&prog, options);

        if ((wrbytes = fwrite (dataftp->buffer, bytes, sizeof (*dataftp->buffer),
                               fp)) != sizeof (*dataftp->buffer)) 
        {
            printf("wrote strange bytenumber %d\n", wrbytes);
        }

    }
    fclose (fp);
    return 0;

}

ftp_t *
ftp_mode (ftp_t *ftp, char *messagebuf)
{

    ftp_t *temp;
    int  h1, h2, h3, h4, p1, p2;

    temp = alloc_ftp ();
    if (ftp->usePasv) 
    {
        ftp_sendcmd (ftp, "PASV", "");
        do
        {
            messagebuf = ftp_reader (ftp);
            ftp_getresp (ftp, messagebuf);
        } while (ftp->multiline == true);


        ftp_getresp (ftp, messagebuf);
        if (ftp->response != FTP_PASV)
        {
            fprintf (stderr, "Response was: %d\n", ftp->response);
            fprintf (stderr, "Passive mode not supported\n");
            fprintf (stderr, "No PORT support yet\n");
            ftp->usePort = true;
            /* do something PORT related here */
            exit (EXIT_FAILURE);

        }
        sscanf(messagebuf+27, "%d,%d,%d,%d,%d,%d",
               &h1, &h2, &h3, &h4, &p1, &p2);
        temp->address.sin_family = AF_INET;
        temp->address.sin_addr.s_addr = htonl ((h1 << 24) + (h2 << 16) +
                                               (h3 << 8) + h4);
        temp->address.sin_port = htons ((p1 << 8) + p2);


    }
    return temp; 

}


void
ftp_getresp (ftp_t *ftp, char *response)
{
    char resp[5];
    if (response == NULL)
    {
        ftp->multiline = false;
        return;
    }
    strncpy (resp, response, 4);
    resp[4] = '\0';
    ftp->response = atoi (resp);

    if (resp[3] == '-')
        ftp->multiline = true;
    else if (resp[3] == ' ' && isdigit(resp[0]))
        ftp->multiline = false;
    else
        ftp->multiline = true;

}

void
ftp_parsecmd (ftp_t *ftp, char *cmdline)
{
    char *pass;
    char *user;
    char *server;

    bool username;
    bool password;
  

    if (strlen (cmdline) > 1024)
    {
        fprintf (stderr, "%s: Can't parse arguments, they are too long => "
                 "aborting...\n", NAME);
        exit (EXIT_FAILURE);
    }
  
    /* if this function is invoked at all, we know that cmdline starts with
     * "ftp://", so we just skip past that */

    cmdline += 6;

    /* find where in the string our username starts, and copy from start to
     * character before : */
    user = strchr (cmdline, ':');

    if (user == NULL) 
    {
        username = false;
        strcpy (ftp->username, "anonymous");
    }
  
    else
    {
        username = true;
        strncpy (ftp->username, cmdline, strlen (cmdline) - strlen (user));
        cmdline += strlen (cmdline) - strlen (user) + 1;
    }
  
    /* we have updated cmdline so that the username is out, lets now see where we 
     * find the '@', which should follow the password directly 
     */

    pass = strchr (cmdline, '@');

    if (pass == NULL)
    {
        password = false;
        strcpy (ftp->password, "some@mail.com");
    }
    else
    {
        password = true;
        if ((strlen (cmdline) - strlen (pass)) < 1)
        {
            fprintf (stderr, "%s: Blank password entered.\n", NAME);
        }
        strncpy (ftp->password, cmdline, strlen (cmdline) - strlen (pass));
        cmdline += strlen (cmdline) - strlen (pass) + 1;
    }
    /* finally, look for the server, which should follow the @ directly, and end 
     * at the first'/' we find. everything after that, the remaining cmdline, 
     * should be some sort of path to a file on the server
     */
    server = strchr (cmdline, '/');

    if (server == NULL)
    {
        version ();
        fprintf (stderr, "%s: not seeing / in ftp address => aborting...\n", NAME);
        fprintf (stderr, "Please use option --help for more information on ftp syntax\n");
        exit (EXIT_FAILURE);
    }

    strncpy (ftp->server, cmdline, strlen (cmdline) - strlen (server));
    cmdline += strlen (cmdline) - strlen (server) + 1;
    if (strlen (cmdline) == 0)
        strcpy (ftp->path, "/");
    else
        strncpy (ftp->path, cmdline, strlen (cmdline));
}

int
ftp_store (ftp_t *ftp, ftp_t *dataftp, char *fname,
           char *messagebuf, struct opt *options, off_t *fsize)
{
    int fd;
    int bytes;
    int wrbytes;
    struct progress prog;
    char buf[1024];

    ftp_sendcmd (ftp, "STOR", fname);

    wait_server (ftp);
    messagebuf =ftp_reader (ftp);

    if (options->verbose == true) printf ("%s\n", messagebuf);

    ftp_getresp (ftp, messagebuf);

    if (ftp->response != FTP_RETROK) 
    {
        fprintf (stderr, "%s: could not store file => aborting...\n", NAME);

        exit (EXIT_FAILURE);
    }
    init_progress (&prog, fsize, fname);
  
    fd = open (fname, O_RDONLY);

    if (fd < 0) 
    {
        fprintf (stderr, "Couldn't open local file => aborting...\n");
        exit (EXIT_FAILURE);
    }

    while ((bytes = read (fd, buf, 1024)) != 0)
    {
        if (bytes == -1)
        {
            fprintf (stderr, "%s: An error occured while reading from file ==> "
                     "aborting...\n", NAME);
            exit (EXIT_FAILURE);
        }

        wrbytes = write (dataftp->socket, buf, bytes);

        if (wrbytes == -1)
        {
            fprintf (stderr, "%s: An error occured while writing to socket ==> "
                     "aborting...\n", NAME);
            exit (EXIT_FAILURE);
        }

        prog.total.bytes += bytes;
        if (options->quiet != true)
        {
            update_progress (&prog, options);
            do_progress (&prog, options);
        }
    }
  
    close (fd);
    free (dataftp);
    return 0;

}


void
ftp_login (ftp_t *ftp, int port, struct opt *options, char *messagebuf)
{
  
    wait_server (ftp);
    if (options->quiet != true)
        printf ("%s: Connection to %s (port %d) made...\n", NAME, ftp->server, port);
    do
    {
        messagebuf = ftp_reader (ftp);
        ftp_getresp (ftp, messagebuf);
        if (options->verbose == true) printf ("%d %s\n", ftp->multiline, messagebuf);
    } while (ftp->multiline == true);

    ftp_sendcmd (ftp, "USER", ftp->username);
    do
    {
        messagebuf = ftp_reader (ftp);
        ftp_getresp (ftp, messagebuf);
        if (options->verbose == true) printf ("%d %s\n", ftp->multiline,messagebuf);
    } while (ftp->multiline == true);

    ftp_sendcmd (ftp, "PASS", ftp->password);
    do
    {
        messagebuf = ftp_reader (ftp);
        ftp_getresp (ftp, messagebuf);
        if (messagebuf == NULL)
        {
            fprintf (stderr, "%s: Server error ==> aborting...\n", NAME);
            exit (EXIT_FAILURE);
        }
        if (options->verbose == true) printf ("%d %s\n", ftp->multiline,messagebuf);
    } while (ftp->multiline == true);

    if (ftp->response != FTP_LOGIN)
    {
        fprintf (stderr, "%s: Server rejected us, code %d\n", NAME, ftp->response);
        exit (EXIT_FAILURE);
    }
      
    ftp_sendcmd (ftp, "TYPE", "I");
    do
    {
        messagebuf = ftp_reader (ftp);
        ftp_getresp (ftp, messagebuf);
        if (options->verbose == true) printf ("%d %s\n", ftp->multiline,messagebuf);
    } while (ftp->multiline == true);
}

inline void
ftp_usage (void)
{
    printf ("Valid syntax for ftp download/upload:\n");
    printf ("\n\tecp ftp://user:password@ftp.someserver.com/directory/file [destination]\n\n");
    printf ("\tThis also works the other way around:\n");
    printf ("\tecp SOURCE ftp://user:password@ftp.someserver.com/directory/\n\n");
    printf ("\tIf you are logging into an anonymous ftp server, user and password "
            "may be \n\tomitted, and will default to anonymous:some@mail.com\n\n");
    printf ("\tIf you do not specify a destination when downloading, ecp \n\twill place the file in the "
            "current directory, with the same name as the \n\tremote file.\n\n");
    printf ("\tThe --resume option will try to resume a broken ftp download. Not all servers \n"
            "\tsupport this.\n");
}
