/*
ecp is designed to copy files, and is named extended cp.
Copyright (C) 2006  Andre Reffhaug <areffhaug@areffhaug.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

ecp homepage   : https://savannah.nongnu.org/projects/ecp/
Contact author : areffhaug@gmail.com
*/




/*
FIXME:

Own realpath function would also be good, but thats in the 
works.

include functionality for mounting, ie copying right into
devices? holy crap, batman, that'll be strenous

have a look at size of CHUNK

if recursively copying into a directory directly beneath the one
we are copying, how to prevent it from forever recursing into 
itself? one thought is to check if it does, and recursively copy
it first, then add its inode to the checking mechanism. seems a 
bit dense, though.

what about human readable outputs? halfdecent hack thrown together

hidden files and shell globbing: * doesnt include hidden files, which
makes that whole shebang a little more crazy


*/

#include <syscall.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <dirent.h>
#include <utime.h>
#include <errno.h>
#include <sys/mman.h>
#include "copyfun.h"
#include "dir.h"

void init_progress(struct progress *prog, off_t *bsize, char *dstname) 
{
  prog->percent = 0;
  prog->in_total.bytes = *bsize;
  prog->in_total.kbytes = prog->in_total.bytes / 1000;
  prog->in_total.mbytes = prog->in_total.kbytes / 1000;
  prog->total.bytes = 0;
  prog->chunk = (prog->in_total.bytes / CHUNK);
  strcpy(prog->fname, dstname);
}

void update_progress(struct progress *prog)
{
  prog->percent = ((float) prog->total.bytes * 100) 
    / (float) prog->in_total.bytes;
  prog->chunk += (prog->in_total.bytes / CHUNK);

}

void do_progress(struct progress *prog, struct opt *options)
{

  if (options->hreadable == true) {
    if ((prog->in_total.bytes < 100000))
      printf("\r%6.2f%%  %3lu / %3lu kB ==> %s", prog->percent, 
             prog->total.bytes / 1000, prog->in_total.kbytes, prog->fname);
    if (prog->in_total.bytes > 100000)
      printf("\r%6.2f%%  %3lu / %3lu mB ==> %s", prog->percent, 
             prog->total.bytes / 1000000, prog->in_total.mbytes, prog->fname);
  }
  else
    if ((prog->in_total.bytes < 100000) && options->hreadable != true)
      printf("\r%6.2f%%  %6lu / %6lu  B ==> %s", prog->percent, 
             prog->total.bytes, prog->in_total.bytes, prog->fname);
    else printf("\r%6.2f%%  %6lu / %6lu kB ==> %s", prog->percent, 
                prog->total.bytes / 1000, prog->in_total.kbytes, prog->fname);

  (void)fflush(stdout);
}


bool do_copy(char infile[path_max], 
             char outfile[path_max],
             struct opt *options, 
             bool todir)
{
  ssize_t bytes, buffersize;
  char * __restrict__ buf;

  char temp[path_max]; /* for readlink */

  char srcname[path_max];
  int srcdesc;
  struct stat srcstat;

  char dstname[path_max];
  int dstdesc;
  struct stat dststat;


  struct progress prog;
  struct utimbuf utimdef;

  /* set up our local copies of the filenames */
  strcpy(srcname, infile);
  strcpy(dstname, outfile);
  if ((srcdesc = open(srcname, O_RDONLY, 0)) == -1) {
    fprintf(stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
            NAME, srcname, strerror(errno));
    return 0;
  }
  /* get hold of file information */
  lstat(srcname, &srcstat);


  if (S_ISLNK(srcstat.st_mode) && options->symlinks == true) {
    readlink(srcname, srcname, path_max);
    lstat(srcname, &srcstat);
  } 

  if (options->no_zero_bytes == true)
    if (srcstat.st_size == 0) {
      close(srcdesc);
      return 1;
    }

  if (options->no_hidden_files == true)
    if (srcname[0] == '.') {
      close(srcdesc);
      return 1;
    }


  if (todir == true) {
    apptodir(dstname, base(srcname));
    if (S_ISLNK(srcstat.st_mode) && options->symlinks != true) {
      memset(temp, '\0', sizeof(temp));
      readlink(srcname, temp, path_max);
      symlink(temp, dstname);
      close(srcdesc);
      if (options->kill == true) {
        if (remove(srcname) == -1)
          fprintf(stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                  NAME, srcname, strerror(errno));
      }
      return 1;
    } 
  }

  /* atleast it responds, but it really does need to be improved upon */
  if (options->update == true) {
    stat(dstname, &dststat);
    if (srcstat.st_ctime < dststat.st_ctime) {
      printf("%s: target is newer or the same as '%s'\n", NAME, dstname);
      return 1;
    }
  }



  if ((dstdesc = open(dstname, O_CREAT | O_TRUNC | O_WRONLY, 
                      srcstat.st_mode)) == -1) { 
    fprintf(stderr, "%s: failed opening '%s' (%s) ==> aborting\n", 
            NAME, dstname, strerror(errno));
    return 0;
  }


  /* initializing the progress struct for our current file */
  init_progress(&prog, &srcstat.st_size, dstname);

  if ((srcstat.st_size == 0) && options->quiet != true) {
    prog.percent = 100;
    do_progress(&prog, options);
    close(srcdesc);
    close(dstdesc);
    chmod(dstname, srcstat.st_mode&
          (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));
    if (options->kill == true) {
      chmod(srcname, 0777);
      if (remove(srcname) == -1)
        fprintf(stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                NAME, srcname, strerror(errno));
    }
    printf("\n");

    return 1;
  } else if ((srcstat.st_size == 0) && options->quiet == true) {
    close(srcdesc);
    close(dstdesc);
    if (options->kill == true) {
      chmod(srcname, 0777);
      if (remove(srcname) == -1)
        fprintf(stderr, "%s: couldn't remove source file '%s' (%s)\n", 
                NAME, srcname, strerror(errno));
    }

    return 0;
  }

    


  /* set up buffersize, according to disks blocksize */
  buffersize = srcstat.st_blksize;  

/* if the file is actually smaller than the chunk we would like to mmap, we
   map the whole thing to memory, and write it back in one big swoop */

  if (srcstat.st_size < buffersize) {
    buf = mmap(0, srcstat.st_size, PROT_READ, MAP_PRIVATE, srcdesc, 0);
    if (write(dstdesc, buf, srcstat.st_size) == -1) {
      fprintf(stderr, "%s: writing to file '%s' failed (%s)\n",
              NAME, dstname, strerror(errno));
      close(srcdesc);
      close(dstdesc);
      return 0;
    }

    if (options->quiet != true) {
      prog.percent = 100;
      prog.total.bytes = srcstat.st_size;
      do_progress(&prog, options);
    }
    munmap(buf, srcstat.st_size);
  } /* this is where we go if the file is going to be split up into multiple
       chunks in memory */
  else {
    while (prog.total.bytes < srcstat.st_size) {
      buf = mmap(0, buffersize, PROT_READ, MAP_PRIVATE, srcdesc, 
                 prog.total.bytes);
      if ((bytes = write(dstdesc, buf, buffersize)) == -1) {
        fprintf(stderr, "%s: writing to file '%s' failed (%s)\n",
                NAME, dstname, strerror(errno));
        close(srcdesc);
        close(dstdesc);
        return 0;
      }

      prog.total.bytes += buffersize;
      if ((srcstat.st_size - prog.total.bytes) < buffersize) {
        buffersize = srcstat.st_size - prog.total.bytes;
      }
      if (options->quiet != true) {
        if ((prog.total.bytes >= prog.chunk) || (bytes < buffersize)) {
          update_progress(&prog);
          do_progress(&prog, options);
        }
      }
      /* unmap the memory again, so it doesn't grow extremely large */
      munmap(buf, buffersize);

    }
  }
    

  if (options->quiet != true) printf("\n");

  if (options->preserve == true) {
    utimdef.actime = srcstat.st_atime;
    utimdef.modtime = srcstat.st_mtime;
    if (!utime(dstname, &utimdef) == 0)
      fprintf(stderr, "%s: error setting utime of file '%s' (%s)\n", 
              NAME, dstname, strerror(errno));
  }

  /* FIXME: still not working right, even though it now retains
     file permissions. Feels like some sort of cheap hack. what about
     S_ISUID (set-used-ID) and S_ISVTX (sticky bit) and S_ISGID? 
     use chmod here too? 
  */

  chmod(dstname, srcstat.st_mode&
        (S_IRWXU | S_IRWXG | S_IRWXO | S_ISUID | S_ISGID));

  /* clean up */
  if (options->kill == true) {
    chmod(srcname, 0777);
    if (remove(srcname) == -1)
      fprintf(stderr, "%s: couldn't remove source file '%s' (%s)\n", 
              NAME, srcname, strerror(errno));
  }

  close(srcdesc);
  close(dstdesc);

  return 1;
}

void copy_recursive(char real_src_dir[path_max], 
                    char real_target_dir[path_max],
                    int depth, ino_t orig_inode, 
                    struct opt *options) 
{
    char copy_realsrc[path_max], copy_targetdir[path_max], new_dir[path_max];
    char mkd[path_max];

    DIR *directory;
    struct dirent *current;
    struct stat statfile;

    realpath(real_src_dir, copy_realsrc);
    realpath(real_target_dir, copy_targetdir);


    lstat(real_src_dir, &statfile);
    if (statfile.st_ino == orig_inode) {
        printf("%s: skipping target directory '%s'\n", NAME, real_src_dir);
        return;
    }
    remove_trailing_slashes(copy_targetdir);

    if ((directory = opendir(copy_realsrc)) == NULL) {
        fprintf(stderr, "%s: error opening directory '%s' (%s)\n", 
                NAME, copy_realsrc, strerror(errno));
        return;
    }

    chdir(copy_realsrc);

    while ((current = readdir(directory)) != NULL) {
        lstat(current->d_name, &statfile);
        if (S_ISDIR(statfile.st_mode)) {

            /* if we are seeing ., .., continue */
            if ((strcmp(".", current->d_name) == 0) || 
                (strcmp("..", current->d_name) == 0)) continue;

            /* check for hidden files */ 
            if (options->no_hidden_files == true)
                if (current->d_name[0] == '.')
                    continue;

            realpath(current->d_name, new_dir);
            strcpy(mkd, copy_targetdir);
            apptodir(mkd, current->d_name);
            mkdir(mkd, 0777);
            chmod(mkd, statfile.st_mode);
            copy_recursive(new_dir, mkd, depth+4, orig_inode,
                           options);
            if (options->kill == true) { 
                chmod(new_dir, 0777); /* precaution against readonly dirs */
                if (remove(new_dir) != 0)
                    fprintf(stderr, "%s: removing '%s' failed (%s)\n", 
                            NAME, new_dir, strerror(errno));
            }

        }
        else {
            do_copy(current->d_name, copy_targetdir, options, true);
        }

    }
    chdir("..");
    closedir(directory);
}



