/* Eliot                                                                     */
/* Copyright (C) 2005  Antoine Fraboulet                                     */
/*                                                                           */
/* This file is part of Eliot.                                               */
/*                                                                           */
/* Eliot is free software; you can redistribute it and/or modify             */
/* it under the terms of the GNU General Public License as published by      */
/* the Free Software Foundation; either version 2 of the License, or         */
/* (at your option) any later version.                                       */
/*                                                                           */
/* Eliot is distributed in the hope that it will be useful,                  */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of            */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             */
/* GNU General Public License for more details.                              */
/*                                                                           */
/* You should have received a copy of the GNU General Public License         */
/* along with this program; if not, write to the Free Software               */
/* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */

/**
 *  \file   alist.h
 *  \brief  List type used by automaton
 *  \author Antoine Fraboulet
 *  \date   2005
 */

#ifndef _ALIST_H_
#define _ALIST_H_
#if defined(__cplusplus)
extern "C"
  {
#endif

    /**
     * untyped list type element
     */
    typedef struct alist_elt_t* alist_elt;

    /**
     * extract the value from an alist element
     * result is untyped si the user should know
     * what the value type is.
     * @returns untyped value
     */
    void* alist_elt_get_value(alist_elt);

    /**
     * untyped list type
     */
    typedef struct alist_t* alist;

    /**
     * list creation
     * @returns empty list
     */
    alist     alist_create     ();

    /**
     * list clone, data are shared between the two list
     */
    alist     alist_clone      (alist);

    /**
     * function to use on data during list deletion.
     */
    void      alist_set_delete (alist,void (*f)(void*));

    /**
     * delete a complete list.
     */
    void      alist_delete     (alist);

    /**
     * add a value to the list
     */
    void      alist_add        (alist, void*);
    
    /**
     * insert the second list in the first one. Each element is 
     * checked using alist_is_in() to avoid duplicate entries.
     */
    void      alist_insert     (alist, alist);

    /**
     * check if a void* value is already in the current list.
     * This function does not dereference the pointer.
     * @returns 0 or 1 (present)
     */
    int       alist_is_in      (alist, void*);

    /**
     * Verify if two lists are equal using the alist_is_in()
     * function.
     * @returns 0 or 1 (equal)
     */
    int       alist_equal      (alist, alist);

    /**
     * get the first alist_elt from the list. List element
     * is not removed from the list.
     * @returns first alist element
     */
    alist_elt alist_get_first  (alist);

    /**
     * get next element from current
     * @returns next alist element or NULL
     */
    alist_elt alist_get_next   (alist, alist_elt);

    /**
     * @returns 0 or 1
     */
    int       alist_is_empty   (alist);

    /**
     * @returns current size of the list
     */
    int       alist_get_size   (alist);

    /**
     * pop first element in the list, the list is modified.
     * @returns list element value
     */
    void*     alist_pop_first_value  (alist);

#if defined(__cplusplus)
  }
#endif
#endif /* _ALIST_H_ */
