/*  -*-comment-start: "//";comment-end:""-*-

   This example program demonstrates how to embed Emacsy into a
   minimal WebKit GTK browser with multiple buffers.
*/

/*
 * Copyright (C) 2006, 2007 Apple Inc.
 * Copyright (C) 2007 Alp Toker <alp@atoker.com>
 * Copyright (C) 2011 Lukasz Slachciak
 * Copyright (C) 2011 Bob Murphy
 * Copyright (C) 2013 Shane Celis
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef SCM_MAGIC_SNARFER
#include <libgen.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <webkit2/webkit2.h>
#include <JavaScriptCore/JavaScript.h>
#include <emacsy.h>
#endif
#include <libguile.h>

/* Event Handlers */
static void destroy_window(GtkWidget* widget, GtkWidget* window);
static gboolean close_window(WebKitWebView* webView, GtkWidget* window);
static gboolean key_press(GtkWidget* widget, GdkEventKey* event, gpointer user_data);
static gboolean process_and_update_emacsy(void *user_data);

/* Registers the Scheme primitive procedures */
static void init_primitives(void); 

/* Scheme Primitives */
SCM scm_webkit_load_url(SCM url);
SCM scm_webkit_forward();
SCM scm_webkit_backward();
SCM scm_webkit_reload();
SCM scm_webkit_find_next(SCM text);
SCM scm_webkit_find_previous(SCM text);
SCM scm_webkit_find_finish();
SCM scm_webkit_zoom_in();
SCM scm_webkit_zoom_out();

SCM scm_current_buffer ();
//GtkWidget *scm_c_current_buffer ();

SCM scm_current_window ();
GtkWidget *scm_c_current_window ();

SCM scm_current_web_view();
WebKitWebView *scm_c_current_web_view();

SCM scm_minibuffer ();
SCM scm_current_modeline ();
GtkWidget *scm_c_current_modeline ();

SCM scm_get_gtk_widget(SCM);
GtkWidget* scm_c_get_gtk_widget(SCM);

void update_modeline (GtkWidget *modeline, char const* string);

//SCM scm_webkit_eval_javascript(SCM script, SCM when_finished_proc);

/* Global state */
GtkWidget *echo_area;
GtkWidget *content_vbox;
GtkWidget *content;

char *
try_load_startup (char const* prefix, char const* dir, char const* startup_script)
{
  static char file_name[PATH_MAX];
  if (prefix)
    strcpy (file_name, prefix);
  if (dir)
    strcat (file_name, dir);
  strcat (file_name, startup_script);
  
  if (access (file_name, R_OK) != -1)
    {
      fprintf (stderr, "Loading '%s'.\n", file_name);
#if 0
      // We could load the file like this:
      scm_c_primitive_load (file_name);
#else
      // But this will drop us into a REPL if anything goes wrong.
      scm_call_1 (scm_c_private_ref ("guile-user", "safe-load"),
                  scm_from_locale_string (file_name));
#endif
      return file_name;
    }
  else
    fprintf (stderr, "no such file '%s'.\n", file_name);

  return 0;
}

/*
  Create a minimal web browser that has Emacsy integrated into it.
 */

int
main (int argc, char* argv[])
{
  int err;
  // Initialize GNU Guile.
  scm_init_guile();
  // Initialize Emacsy.
  err = emacsy_initialize (EMACSY_INTERACTIVE);
  if (err) 
    return err;

  // Register the primitive procedures that control the browser.
  init_primitives();  

  // You can evaluate S-expressions here.
  scm_c_eval_string("(use-modules (system repl error-handling))"
                    "(define (safe-load filename)              "
                    "  (call-with-error-handling               "
                    "    (lambda () (load filename))))         ");
  
  scm_c_eval_string("(use-modules (emacsy window))");

  // But to make the application easy to mold, it's best to load the
  // Scheme code from a file.
  char const *startup_script = "emacsy-webkit-gtk-w-windows.scm";
  char prefix[PATH_MAX];
  strcpy (prefix, argv[0]);
  if (getenv ("_"))
    strcpy (prefix, getenv ("_"));
  dirname (dirname (prefix));

  if (!try_load_startup (0, 0, startup_script)
      &&!try_load_startup (prefix, "/", startup_script)
      &&!try_load_startup (prefix, "/etc/emacsy/", startup_script))
    fprintf (stderr, "error: failed to find '%s'.\n", startup_script);
    
  
  // Initialize GTK+.
  gtk_init(&argc, &argv);

  // Create an 800x600 window that will contain the browser instance.
  GtkWidget *main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  fprintf (stderr, "main window=0x%x\n", main_window);
  gtk_window_set_default_size(GTK_WINDOW(main_window), 800, 600);
  
  // Handle key press and release events.
  g_signal_connect (main_window, "key_press_event", G_CALLBACK (key_press), NULL);
  g_signal_connect (main_window, "key_release_event", G_CALLBACK (key_press), NULL);
  
  GdkGeometry geom_struct;
  geom_struct.max_width = 800;
  geom_struct.max_height = 600;
  gtk_window_set_geometry_hints(GTK_WINDOW(main_window),
                                NULL,
                                &geom_struct,
                                GDK_HINT_MAX_SIZE);

  // Set up callbacks so that if either the main window or the browser
  // instance is closed, the program will exit.
  g_signal_connect(main_window, "destroy", G_CALLBACK(destroy_window), NULL);

  echo_area = gtk_label_new ("echo area");
  fprintf (stderr, "echo_area=0x%x\n", echo_area);

  gtk_misc_set_alignment (GTK_MISC (echo_area), 0.0f, 0.0f);
  gtk_label_set_use_underline (GTK_LABEL (echo_area), FALSE);
  gtk_label_set_line_wrap (GTK_LABEL (echo_area), TRUE);
  gtk_label_set_single_line_mode (GTK_LABEL (echo_area), TRUE);
  gtk_label_set_max_width_chars (GTK_LABEL (echo_area), 160);

  // While idle, process events in Emacsy and upate the echo-area.
  g_timeout_add (100, (GSourceFunc) process_and_update_emacsy, NULL);

  GtkWidget *vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 1);
  fprintf (stderr, "vbox=0x%x\n", vbox);
  content_vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 1);
  fprintf (stderr, "content_vbox=0x%x\n", content_vbox);

  SCM widget_pointer = scm_call_0 (scm_c_public_ref ("guile-user", "instantiate-root-window"));
  content = scm_c_get_gtk_widget (widget_pointer);
  gtk_box_pack_start (GTK_BOX (content_vbox), content, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), content_vbox, TRUE, TRUE, 0);

  // Add the echo area.
  gtk_container_add (GTK_CONTAINER (vbox), echo_area);

  // Put the scrollable area into the main window.
  gtk_container_add (GTK_CONTAINER (main_window), vbox);

  // Make sure the main window and all its contents are visible.
  gtk_widget_show_all (main_window);
  gtk_window_set_resizable (GTK_WINDOW (main_window), FALSE);

  // Run the main GTK+ event loop.
  gtk_main ();

  return 0;
}


/*
  Event Handlers
  ==============
*/

static void destroy_window(GtkWidget* widget, GtkWidget* window)
{
  gtk_main_quit();
}

static gboolean close_window(WebKitWebView* web_view, GtkWidget* window)
{
  gtk_widget_destroy(window);
  return TRUE;
}

static int scm_c_char_to_int(const char *char_name) {
  /* I should put a regex in here to validate it's a char */
  return scm_to_int(scm_char_to_integer(scm_c_eval_string(char_name)));
}

static gboolean
key_press
(GtkWidget* widget, GdkEventKey* event, gpointer user_data)
{
  static guint32 last_unichar = 0;
  guint32 unichar;
  GdkModifierType modifiers;
  int mod_flags = 0;
  
  modifiers = gtk_accelerator_get_default_mod_mask ();
  printf ("Key press MASK=%d, type=%d, keyval=%d\n", modifiers, event->type, event->keyval);

  if (event->state & modifiers & GDK_CONTROL_MASK)
      mod_flags |= EMACSY_MODKEY_CONTROL;

  if (event->state & modifiers & GDK_SHIFT_MASK) 
    mod_flags |= EMACSY_MODKEY_SHIFT;

  if (event->state & modifiers & GDK_SUPER_MASK) 
    mod_flags |= EMACSY_MODKEY_SUPER;

  if (event->state & modifiers & GDK_MOD1_MASK)
    mod_flags |= EMACSY_MODKEY_META;

  unichar = gdk_keyval_to_unicode (event->keyval);

  // Fix up any key values that don't translate perfectly.
  if (event->keyval == GDK_KEY_BackSpace)
    unichar = scm_c_char_to_int ("#\\del");

  // If unichar is 0 then it's not a regular key, e.g., Control, Meta, etc.
  if (event->type == GDK_KEY_PRESS)
    {
      printf("Key press %d %s (unicode %d last_unichar %d)\n", 
             event->keyval, event->string, unichar, last_unichar);
      // Fix up some keys.
      if (unichar)
        {
          // Register the key event with Emacsy.
          emacsy_key_event(unichar, mod_flags);
          /* 
             One can do the event handling and the actual processing
             separately in Emacsy.  However, in this case, it's convenient
             to do some processing in the event handling here so we know
             whether or not to pass the event on to the browser.
          */
          int flags = emacsy_tick();
          
          printf("flags = %d\n", flags);
          if (flags & EMACSY_RAN_UNDEFINED_COMMAND_P)
            {
              printf("Passing to browser.\n");
              return FALSE; // Pass the event through to the web browser.
            }
          else
            {
              printf("Emacsy handled it.\n");
              last_unichar = unichar;
              return TRUE; // Emacsy handled it. Don't pass the event through.
            }
        }
    }
  else if (event->type == GDK_KEY_RELEASE)
    {
      /* 
         We receive both key presses and key releases.  If we decide not
         to pass a key event when pressed, then we remember it
         (last_unichar) such that we squelch the key release event too.
      */
      printf("Key release %d %s (unicode %d last_unichar %d)\n", 
           event->keyval, event->string, unichar, last_unichar);
      if (last_unichar && last_unichar == unichar)
        {
          last_unichar = 0;
          return TRUE; // Don't pass event to the browser.
        }
    }
  return FALSE; // Pass the event to the browser.
}

void
update_modeline (GtkWidget *modeline, char const* string)
{
  gchar *markup = g_markup_printf_escaped ("<span foreground=\"white\" background=\"black\" underline=\"none\"><tt>%s </tt></span>", string);
  gtk_label_set_markup (GTK_LABEL (modeline), markup);
  g_free (markup);
}

/*
  Process events in Emacsy then update the echo area at the bottom of the
  screen.
 */
static gboolean process_and_update_emacsy(void *user_data)
{
  // fprintf (stderr, "process_and_update_emacsy\n");

  // Process events and any background coroutines.
  int flags = emacsy_tick();

  // If there's been a request to quit, quit.
  if (flags & EMACSY_QUIT_APPLICATION_P)
    gtk_main_quit();

  // Update the status line. 
  char const *status = emacsy_message_or_echo_area();
  // Use markup to style the status line.
  gchar *markup = g_markup_printf_escaped ("<span foreground=\"white\" background=\"black\" underline=\"single\"><tt>%s </tt></span>", status);
  gtk_label_set_markup (GTK_LABEL (echo_area), markup);
  g_free (markup);

  GtkWidget *modeline = scm_c_current_modeline ();
  if (modeline && scm_current_buffer () != scm_minibuffer ())
    update_modeline (modeline, emacsy_mode_line ());

  // Show the cursor.  Exercise for the reader: Make it blink.
  char message[255];
  memset(message, ' ', 254);
  message[255] = 0;
  message[emacsy_minibuffer_point () - 1] = '_';
  gtk_label_set_pattern (GTK_LABEL (echo_area), message);

  return TRUE;                  
}

/*
  Scheme Primitives
  =================
  
  These C functions are exposed as callable procedures in Scheme.
*/

SCM_DEFINE(scm_update_label_x, "update-label!", 3, 0, 0,
           (SCM scm_label, SCM string, SCM selected_p),
           "Update a GTK label to the given string.")
{
  char const *text = scm_to_locale_string (string);

  // Use markup to style the status line.
  char *markup;
  if (scm_is_true(selected_p)) {
    markup = g_markup_printf_escaped ("<span foreground=\"white\" background=\"grey\" underline=\"none\"><tt>%s </tt></span>", text);
  } else {
    markup = g_markup_printf_escaped ("<span foreground=\"white\" background=\"black\" underline=\"none\"><tt>%s </tt></span>", text);
  }

  GtkWidget *label = (GtkWidget *) scm_to_pointer(scm_label);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  
  return SCM_UNSPECIFIED;
}

SCM_DEFINE(scm_set_window_content_x, "set-window-content!", 1, 0, 0,
           (SCM widget),
           "Delete contents and update a content vbox to WIDGET.")
{
  gtk_container_remove (GTK_CONTAINER(content_vbox), content);
  content = scm_c_get_gtk_widget (widget);
  gtk_box_pack_start (GTK_BOX (content_vbox), content, TRUE, TRUE, 0);
  gtk_widget_show_all (content_vbox);
  return SCM_UNSPECIFIED;
}

SCM_DEFINE (scm_gtk_get_parent, "gtk-get-parent", 1, 0, 0,
           (SCM widget),
           "Return parent.")
{
  GtkWidget *w = GTK_WIDGET (scm_to_pointer (widget));
  GtkWidget *parent = GTK_WIDGET (gtk_widget_get_parent (w));
  if (parent)
    return scm_from_pointer (parent, /*g_free*/ NULL);
  return SCM_BOOL_F;
}

SCM_DEFINE (scm_gtk_add_x, "gtk-add!", 2, 0, 0,
           (SCM window, SCM widget),
           "Add WIDGET to CONTAINER.")
{
  GtkWidget *container = GTK_WIDGET (scm_to_pointer (window));
  gtk_container_add (GTK_CONTAINER (container), GTK_WIDGET (scm_to_pointer (widget)));
  //gtk_box_pack_start (GTK_BOX (container), widget, TRUE, TRUE, 0);
  gtk_widget_show_all (GTK_WIDGET (container));
  return widget;
}

SCM_DEFINE (scm_gtk_show_all, "gtk-show-all", 1, 0, 0,
           (SCM widget),
           "Show WIDGET.")
{
  gtk_widget_show_all (GTK_WIDGET (scm_to_pointer (widget)));
  return SCM_UNSPECIFIED;
}

SCM_DEFINE (scm_gtk_remove_x, "gtk-remove!", 2, 0, 0,
           (SCM container, SCM widget),
           "Delete WIDGET from CONTAINER.")
{
  gtk_container_remove (GTK_CONTAINER (scm_to_pointer (container)),
                        GTK_WIDGET (scm_to_pointer (widget)));
  return widget;
}

SCM_DEFINE (scm_make_text_view, "make-text-view", 0, 0, 0,
            (),
            "Return a pointer to a newly created new text view.")
{
  GtkTextView *view = GTK_TEXT_VIEW (gtk_text_view_new ());
  gtk_text_view_set_cursor_visible (view, false);
  gtk_text_view_set_monospace (view, true);
  fprintf (stderr, "MAKE text_view=0x%x\n", view);
  view = g_object_ref (view);
  return scm_from_pointer (view, /*g_free*/ NULL);
}

SCM_DEFINE (scm_text_view_set_text, "text-view-set-text", 2, 0, 0,
           (SCM widget, SCM text),
           "Set text.")
{
  GtkTextView *view = (GtkTextView *) scm_to_pointer (widget);
  GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
  gtk_text_buffer_set_text (buffer, scm_to_locale_string (text), -1);
  return SCM_UNSPECIFIED;
}

SCM_DEFINE (scm_text_view_set_point, "text-view-set-point", 3, 0, 0,
            (SCM widget, SCM text, SCM offset),
            "Set point.")
{
  fprintf (stderr, "hiero\n");
  GtkTextView *view = (GtkTextView *) scm_to_pointer (widget);
  GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
  GtkTextIter end;
  gtk_text_buffer_set_text (buffer, scm_to_locale_string (text), -1);

  gtk_text_buffer_get_end_iter (buffer, &end);
  int cend = gtk_text_iter_get_offset (&end);
  GtkTextIter point;
  int coffset = scm_to_int (offset) - 1;
  if (coffset > cend)
    coffset = cend;
  gtk_text_buffer_get_iter_at_offset (buffer, &point, coffset);
  gtk_text_buffer_place_cursor (buffer, &point);
  char c = ' ';
  fprintf (stderr, "offset=%d, end=%d\n", coffset, cend);
  if (coffset < cend)
    {
      fprintf (stderr, "getting char at=%d\n", coffset);
      c = gtk_text_iter_get_char (&point);
      gtk_text_buffer_get_iter_at_offset (buffer, &end, coffset+1);
      gtk_text_buffer_delete (buffer, &point, &end);
    }
  char markup[50];
  sprintf (markup, "<span color=\"white\" background=\"black\">%c</span>", c);
  fprintf (stderr, "markup: %s\n", markup);
  gtk_text_buffer_insert_markup (buffer, &point, markup, -1);

  return SCM_UNSPECIFIED;
}
    
SCM_DEFINE (scm_text_view_ping, "text-view-ping", 1, 0, 0,
           (SCM widget),
           "ping.")
{
  GtkTextView *view = (GtkTextView *) scm_to_pointer (widget);
  GtkTextBuffer *buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
  g_signal_emit_by_name (buffer, "changed");
  return SCM_UNSPECIFIED;
}


SCM_DEFINE (scm_make_web_view, "make-web-view", 0, 0, 0,
            (),
            "Return the pointer to a newly created new webkit view.")
{
  WebKitWebView *view = WEBKIT_WEB_VIEW (webkit_web_view_new ());
  view = g_object_ref (view);
  return scm_from_pointer (view, /*g_free*/ NULL);
}

SCM_DEFINE(scm_webkit_load_url, "webkit-load-url", 1, 0, 0,
           (SCM scm_url),
           "Loads a given URL into the WebView.")
{
  char const *c_url = scm_to_locale_string (scm_url);
  fprintf (stderr, "webkit-load-url...\n");
  webkit_web_view_load_uri (scm_c_current_web_view (), c_url);
  return SCM_UNSPECIFIED;
}

SCM_DEFINE(scm_webkit_get_url, "webkit-get-url", 0, 0, 0,
           (),
           "Returns the current URL to the WebView.")
{
  gchar const *url = webkit_web_view_get_uri (scm_c_current_web_view());
  return scm_from_locale_string (url ? url : "");
}

SCM_DEFINE(scm_webkit_get_title, "webkit-get-title", 0, 0, 0,
           (),
           "Returns the current Title to the WebView.")
{
  gchar const *title = webkit_web_view_get_title(scm_c_current_web_view());
  return scm_from_locale_string(title ? title : "");
}

// SCM_DEFINE(scm_webkit_get_content, "webkit-get-content", 0, 0, 0,
//            (),
//            "Returns the current content of the WebView.")
// {
//   gchar *content = webkit_get_content (scm_c_current_web_view ());
//   return scm_from_locale_string (content ? content : "");
// }


SCM_DEFINE(scm_webkit_reload, "webkit-reload", 0, 0, 0,
           (),
           "Reload browser.")
{
  WebKitWebView *v = scm_c_current_web_view ();
  if (v)
    webkit_web_view_reload (v);
  return SCM_UNSPECIFIED;
}

SCM
scm_minibuffer ()
{
  return scm_c_eval_string ("minibuffer");
}

SCM
scm_current_buffer ()
{
  return scm_c_eval_string ("(current-buffer)");
}

SCM
scm_current_window ()
{
  return scm_c_eval_string ("(current-web-view)");
}

GtkWidget *
scm_c_current_window ()
{
  SCM window = scm_current_window ();
  if (!scm_is_false (window))
    return (GtkWidget *) scm_to_pointer (window);
  return NULL;
}

SCM
scm_current_web_view ()
{
  return scm_c_eval_string ("(current-web-view)");
}

WebKitWebView *
scm_c_current_web_view ()
{
  SCM web_view = scm_current_web_view ();
  if (!scm_is_false (web_view))
    fprintf (stderr, "scm_c_current_web_view=0x%x\n", scm_to_pointer (web_view));
  if (!scm_is_false (web_view))
    return (WebKitWebView *) scm_to_pointer (web_view);
  return NULL;
}

SCM
scm_current_modeline ()
{
  return scm_c_eval_string ("(current-modeline)");
}

GtkWidget *
scm_c_current_modeline ()
{
  SCM modeline = scm_current_modeline ();
  if (!scm_is_false (modeline))
    return (GtkWidget *) scm_to_pointer (modeline);
  return NULL;
}

SCM_DEFINE(scm_web_view_load_string, "web-view-load-string", 2, 0, 0,
           (SCM scm_web_view, SCM string),
           "Loads the plaintext string into the given web view.")
{
 WebKitWebView *web_view = (WebKitWebView *) scm_to_pointer(scm_web_view);
 
 webkit_web_view_load_html (web_view,
                            scm_to_locale_string (string),
                            "buffer://?");
 return SCM_UNSPECIFIED;
}

SCM_DEFINE (scm_create_emacs_window, "create-emacs-window", 2, 0, 0,
            (SCM window, SCM view),
            "Return a Box with a scrolled window, and view with a modeline.")
{
  GtkWidget *scrolled_window = scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), GTK_WIDGET (scm_to_pointer (view)));

  GtkWidget *modeline = gtk_label_new ("modeline");
  gtk_misc_set_alignment (GTK_MISC (modeline), 0.0f, 0.0f);
  gtk_label_set_use_underline (GTK_LABEL (modeline), FALSE);
  gtk_label_set_line_wrap (GTK_LABEL (modeline), TRUE);
  gtk_label_set_single_line_mode (GTK_LABEL (modeline), TRUE);
  gtk_label_set_max_width_chars (GTK_LABEL (modeline), 160);

  GtkWidget *vbox = gtk_box_new (GTK_ORIENTATION_VERTICAL, 1);
  gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);

  gtk_container_add (GTK_CONTAINER (vbox), modeline);
  gtk_widget_show_all (GTK_WIDGET (scm_to_pointer (view)));
  gtk_widget_show_all (GTK_WIDGET (vbox));
  gtk_widget_show_all (GTK_WIDGET (scrolled_window));

  update_modeline (modeline, "boo");

  fprintf (stderr, "create-emacs-window=0x%x\n", view);
  SCM box = scm_from_pointer (vbox, NULL);
  fprintf (stderr, "create-emacs-window=0x%x\n", view);
  SCM window_user_data = scm_c_eval_string ("<window-user-data>");
  fprintf (stderr, "create-emacs-window=0x%x\n", view);
  SCM scm_user_data = scm_make_struct_no_tail (window_user_data,
                                               scm_list_3 (box,
                                                           view,
                                                           scm_from_pointer (modeline, NULL)));
  printf ("Finished create_text_view_window\n");
  return scm_user_data;
}

SCM_DEFINE (scm_get_gtk_widget, "get-gtk-widget", 1, 0, 0,
            (SCM pointer),
            "Returns a pointer to a GtkWidget from a pointer or a window-user-data object.")
{
  SCM window_user_data = scm_c_eval_string ("<window-user-data>");
  if (scm_is_true (scm_struct_p (pointer))
      && scm_is_true (scm_equal_p (scm_struct_vtable (pointer), window_user_data)))
    return scm_struct_ref (pointer, scm_from_int (0));
  else if (SCM_POINTER_P (pointer))
    return pointer;
  return SCM_BOOL_F;
}

GtkWidget *
scm_c_get_gtk_widget (SCM pointer)
{
  SCM widget = scm_get_gtk_widget (pointer);
  if (scm_is_true(widget))
    return GTK_WIDGET (scm_to_pointer (widget));
  return NULL;
}

SCM_DEFINE(scm_create_gtk_window, "create-gtk-window", 2, 0, 0,
           (SCM list, SCM vertical_p),
           "Returns a pointer to a GtkWidget* that contains a vertical or "
"horizontal window with the list of other widgets as its children.")
{
  GtkWidget *vbox = gtk_box_new (scm_is_true(vertical_p) ? GTK_ORIENTATION_VERTICAL
                                 : GTK_ORIENTATION_HORIZONTAL, 1);

  for (; ! scm_is_null (list); list = scm_cdr (list))
    {
      SCM pointer = scm_car (list);
      GtkWidget *widget = GTK_WIDGET (scm_c_get_gtk_widget (pointer));
      GtkContainer *parent = GTK_CONTAINER (gtk_widget_get_parent (widget));
      if (parent)
        gtk_container_remove (parent, widget);
      gtk_box_pack_start (GTK_BOX (vbox), widget, TRUE, TRUE, 0);
    }
  gtk_widget_show_all (GTK_WIDGET (vbox));
  return scm_from_pointer (vbox, NULL);
}

static void init_primitives(void)
{
/*
  We use guile-snarf to generate main.c.x that helps us register the C
  functions as Scheme procedures.
*/
#ifndef SCM_MAGIC_SNARFER
#include "emacsy-webkit-gtk-w-windows.c.x"
#endif
}
