(define-module (emacsy command)
  #:use-module (string completion)
  #:use-module (oop goops)
  #:use-module (ice-9 optargs)
  #:use-module (emacsy util)
  #:use-module (emacsy self-doc)
  #:use-module (emacsy coroutine)
  
  #:export-syntax (export-command))
(define-public (module-command-interface mod)
  (unless (module-variable mod '%module-command-interface)
    (module-define! mod '%module-command-interface 
                    (let ((iface (make-module)))
                      (set-module-name! iface (module-name mod))
                      (set-module-version! iface (module-version mod))
                      (set-module-kind! iface 'command)
                      ;(module-use! iface (resolve-interface '(guile)))
                      iface)))
  (module-ref mod '%module-command-interface))



(define-public (module-export-command! m names)
  (let ((public-i (module-command-interface m)))
    ;; Add them to this module.
    (for-each (lambda (name)
                (let* ((internal-name (if (pair? name) (car name) name))
                       (external-name (if (pair? name) (cdr name) name))
                       (var (module-ensure-local-variable! m internal-name)))
                  (module-add! public-i external-name var)))
              names)))

(define-syntax-rule (export-command name ...)
  (eval-when (eval load compile expand)
    (call-with-deferred-observers
     (lambda ()
       (module-export-command! (current-module) '(name ...))))))

(define-syntax-public define-interactive
  (syntax-rules ()
    ((define-interactive (name . args) . body)
     (begin (define-cmd global-cmdset (name . args)
              . body)
            (export-command name)))
    ((define-interactive name value)
     (begin (define-cmd global-cmdset name value)
            (export-command name))
     )))
(define-syntax-public define-cmd
  (lambda (x)
    (syntax-case x ()
      ((define-cmd (name . args) e0)
       #'(begin 
           (define* (name . args)
             (with-fluids ((in-what-command 'name))
               e0))
           (export name)
           (emacsy-kind-set! 
            (module-variable (current-module) 'name) 
            'command)
           (set-command-properties! name 'name)))
      ((define-cmd (name . args) e0 e1 . body)
       (string? (syntax->datum #'e0))
       ;; Handle the case where there is a documentation string.
       #'(begin 
           (define* (name . args)
                      e0
                      (with-fluids ((in-what-command 'name))
                        (let ()
                          e1 . body)))
           (export name)
           (emacsy-kind-set! 
            (module-variable (current-module) 'name) 
            'command)
           (set-command-properties! name 'name)))
      ((define-cmd (name . args) e0 e1 . body)
       #'(begin 
           (define* (name . args)
                      (with-fluids ((in-what-command 'name))
                        (let ()
                          e0 e1 . body)))
           (export name)
           (emacsy-kind-set! 
            (module-variable (current-module) 'name) 
            'command)
           (set-command-properties! name 'name)))
      ((define-cmd name value)
       #'(begin 
           (define name #f)
           (let ((v value))
             (set! name (colambda args
                          (with-fluids ((in-what-command 'name))
                            (apply v args))))
             (export name)
             (emacsy-kind-set! 
              (module-variable (current-module) 'name) 
              'command)
             (set-command-properties! name 'name))))
      ((define-cmd cmap (name . args) . body)
       #'(begin 
           (define-cmd (name . args) . body)
           (command-add! cmap 'name)))
      ((define-cmd cmap name value)
       #'(begin
           (define-cmd name value)
           (command-add! cmap 'name))))))
(define-syntax-public lambda-cmd
  (syntax-rules ()
    ((lambda-cmd args . body)
     (let ((proc (lambda* args
                          (with-fluids ((in-what-command #f))
                            . body))))
       (set-command-properties! proc)
       proc))))
(define-class-public <command-set> ()
  (commands #:getter commands #:init-form (list))
  (completer #:getter completer #:init-form (make <string-completer>))
  (parent #:accessor parent #:init-keyword #:parent #:init-value #f))
(export commands completer)
(define-public global-cmdset (make <command-set>))
(define in-what-command (make-fluid #f))
(define-public this-command #f)
(define-public last-command #f)
(define-public kill-rogue-coroutine? #f)
(define-public seconds-to-wait-for-yield 2)
(define this-interactive-command (make-fluid))
(define-method-public (command-contains? (cmap <command-set>) command-symbol)
  (or (memq command-symbol (commands cmap))
      (and (parent cmap) (command-contains? (parent cmap) command-symbol))))

(define-method-public (command-add! (cmap <command-set>) command-symbol)
  (when (not (command-contains? cmap command-symbol)) 
      (add-strings! (completer cmap) (list (symbol->string command-symbol)))
      (slot-set! cmap 'commands (cons command-symbol (commands cmap)))))

(define-method-public (command-remove! (cmap <command-set>) command-symbol)
  (when (command-contains? cmap command-symbol) 
    (slot-set! cmap 'commands (delq! command-symbol (commands cmap)))
    ;; Must rebuild the completer.
    (let ((c (make <string-completer>)))
      (add-strings! c (map symbol->string (commands cmap)))
      (slot-set! cmap 'completer c))))
(define-public (register-interactive name proc)
  (command-add! global-cmdset name)
  (set-command-properties! proc name))
(define-public (command->proc command)
  (cond
   ((thunk? command)
    command)
   (else
    (warn "command->proc not given a command: ~a" command)
    #f)))
(define-public (command-name command)
  (procedure-name command))
(define-public (command? object)
  (thunk? object))
(define* (set-command-properties! proc #:optional (name #f))
  (let ((cname (or name (procedure-name proc) #f)))
    (set-procedure-property! proc 'command-name 
                             (if (eq? cname 'proc) 
                                 #f 
                                 cname))))
(define-public (what-command-am-i?)
  (fluid-ref in-what-command))
(define-public (command-execute command . args)
  (if (command? command)
      (let ((cmd-proc (command->proc command))
            (cmd-name (command-name command)))
        (emacsy-log-info "Running command: ~a" cmd-name)
        (set! last-command this-command)
        (set! this-command cmd-name)
        (apply cmd-proc args))
      (error (emacsy-log-warning "command-execute not given a command: ~a" command))))
(define-public (call-interactively command . args)
  (dynamic-wind
   (lambda () (if kill-rogue-coroutine?
                  (alarm seconds-to-wait-for-yield))) 
   (lambda () (with-fluids ((this-interactive-command (command-name command)))
                (apply command-execute command args)))
   (lambda () (if kill-rogue-coroutine?
                  (alarm 0)))))
(define*-public (called-interactively? #:optional (kind 'any))
  (eq? (fluid-ref in-what-command) (fluid-ref this-interactive-command)))

