(define-module (emacsy event)
  #:use-module (ice-9 q)
  #:use-module (ice-9 regex)
  #:use-module (ice-9 optargs)
;  #:use-module (srfi srfi-1)
  #:use-module (ice-9 match)
  #:use-module (oop goops)
  #:use-module (emacsy util)
)
(define-syntax-public define-kbd-converter
  (syntax-rules ()
    ((define-kbd-converter (name args ...) expr ...)
     (begin (define* (name args ...)
              expr ...)
            (register-kbd-converter 'name name)))
    ((define-kbd-converter name value)
     (begin (define* name value)
            (register-kbd-converter 'name name)))))
(define-class-public <event> ()
  (time #:getter time #:init-thunk (lambda () (emacsy-time)))
  (discrete-event? #:getter discrete-event? #:init-keyword #:discrete-event? #:init-value #t))
(export time discrete-event?)
(define-class-public <modifier-key-event> (<event>)
  (modifier-keys #:getter modifier-keys 
                 #:init-keyword #:modifier-keys 
                 #:init-value '()))

(define-class-public <key-event> (<modifier-key-event>)
  (command-char #:getter command-char 
                #:init-keyword #:command-char))
(export modifier-keys command-char)
(define-class-public <mouse-event> (<event>)
  (modifier-keys #:getter modifier-keys #:init-keyword #:modifier-keys #:init-value '())
  (position #:getter position #:init-keyword #:position)
  (button #:getter button #:init-keyword #:button)
  (state #:getter state #:init-keyword #:state))
(export modifier-keys position button state)
(define-class-public <drag-mouse-event> (<mouse-event>)
  (rect #:getter rect #:init-keyword #:rect))
(define-class-public <dummy-event> (<event>))
(define kbd-converter-functions '())
(define char-set:requires-shift-key (char-set-union
                                     char-set:symbol
                                     char-set:upper-case
                                     (char-set-delete char-set:punctuation
                                        ;punctuation = !"#%&'()*,-./:;?@[\\]_{}
                                         #\. #\; #\[ #\] #\, #\' #\\)))
(define (kbd-entry->key-event kbd-entry)
  (match (strip-off-modifier-keys kbd-entry)
    ((mod-keys kbd-entry)
     (let ((regex "^([^ ]|RET|DEL|ESC|TAB|SPC)$"))
       (let ((match (string-match regex kbd-entry)))
         (if match
             (let* ((char (string->command-char (match:substring match 1))))
               (make <key-event> #:command-char char #:modifier-keys mod-keys))
             #f))))))

(define (get-modifier-keys str)
      (if str
          (map modifier-char->symbol 
               (filter (lambda (x) (not (char=? x #\-))) (string->list str)))
          '()))

(define-public (strip-off-modifier-keys kbd-entry)
  "Parse the kbd-entry and strip off the modifier-keys and return the kbd-entry
and a list of modifier keys."
  (let ((regex "^(([ACHMsS]-)*)(.*)$"))
    (let ((match (string-match regex kbd-entry)))
      (if match
          (let ((mod-keys (get-modifier-keys (match:substring match 1))))
            (list mod-keys (match:substring match 3)))
          (list '() kbd-entry)))))
(define-public (modifier-char->symbol char)
  (case char 
    ((#\A) 'alt)
    ((#\C) 'control)
    ((#\H) 'hyper)
    ((#\M) 'meta)
    ((#\s) 'super)
    ((#\S) 'shift)
    (else (warn (format #f "Invalid character for modifier key: ~a" char))
          #f)))
(define (string->command-char str)
  (if (= (string-length str) 1)
      ;; One character string, return first character; simple!
      (string-ref str 0)
      (string-case str
                   ("RET" #\cr)
                   ("DEL" #\del)
                   ("ESC" #\esc)
                   ("TAB" #\tab)
                   ("SPC" #\space)
                   (else (warn (format #f "Invalid command character: ~a" str)) ))))
(define-public (register-kbd-converter function-name function)
  (set! kbd-converter-functions
        (assq-set! kbd-converter-functions function-name function)))
(define*-public (kbd key-string #:optional (canonical? #t))
  (if canonical?
      (map event->kbd (map canonize-event! (kbd->events key-string)))
      (map event->kbd (kbd->events key-string))))
(define (kbd-entry->event kbd-entry)
    (or (find-first (lambda (f) (f kbd-entry)) 
                 (alist-values kbd-converter-functions))
        (throw 'invalid-kbd-entry kbd-entry)))
(define-public (kbd->events kbd-string)
  (let ((kbd-entries (string-tokenize kbd-string)))
        (map kbd-entry->event kbd-entries)))
(define-method-public (canonize-event! (event <key-event>))
  (if (memq 'shift (modifier-keys event))
      (if (char-set-contains? char-set:requires-shift-key (command-char event))
          ;; Remove extraneous shift.
          (slot-set! event 'modifier-keys (delq 'shift (modifier-keys event)))
          ;; No shift required, but there is a shift in the kbd-entry.
          (if (char-lower-case? (command-char event))
              (begin
                ;; Change the character to uppercase.
                (slot-set! event 'command-char (char-upcase (command-char event)))
                ;; Get rid of the shift.
                (slot-set! event 'modifier-keys (delq 'shift (modifier-keys event)))))))
  (let ((mod-keys (modifier-keys event)))
    ;; Put them in alphabetical order: ACHMsS.
    (slot-set! event 'modifier-keys 
               (intersect-order mod-keys 
                                '(alt control hyper meta super shift))))
  event)
(define-method-public (event->kbd (event <key-event>))
  (let ((mods (next-method))
        (cmd-char (command-char->string (command-char event))))
    (format #f "~a~a" mods cmd-char)))

(define-method-public (event->kbd (event <modifier-key-event>))
  (let ((mods (map string (map modifier-symbol->char (modifier-keys event)))))
    (string-join `(,@mods "") "-")))
(define-public (modifier-symbol->char sym)
    (case sym
      ((alt) #\A)
      ((control) #\C)
      ((hyper) #\H)
      ((meta) #\M)
      ((super) #\s)
      ((shift) #\S)
      (else (error "Bad modifier symbol " sym))))
(define (command-char->string c)
    (case c
      ((#\cr #\newline) "RET")
      ((#\del) "DEL")
      ((#\esc) "ESC")
      ((#\tab) "TAB")
      ((#\space) "SPC")
      (else (string c))))
(define-method (write (obj <key-event>) port)
  (display "#<key-event " port)
  (display (event->kbd obj) port)
  (display ">" port))
(define-public (event? obj)
  (is-a? obj <event>))

(define-public (key-event? obj)
  (is-a? obj <key-event>))
(define-method (canonize-event! (event <mouse-event>))
  (let ((mod-keys (modifier-keys event)))
    ;; Put them in alphabetical order: ACHMsS.
    (slot-set! event 'modifier-keys 
               (intersect-order mod-keys 
                                '(alt control hyper meta super shift)))
    event))
(define-kbd-converter (kbd-entry->mouse-event kbd-entry)
  (let* ((regex "^(([ACHMsS]-)*)((up-|down-|drag-)?mouse-([123]))$")
         (match (string-match regex kbd-entry)))
      (if match
          (let* ((symbol (string->symbol (match:substring match 3)))
                 (modifier-keys (get-modifier-keys (match:substring match 1))))
            ;; Warning that symbol is not used; squelch with this noop ref.
            symbol
            (make <mouse-event> #:position #f 
                  #:button (string->number (match:substring match 5))
                  #:state (let ((state-string (match:substring match 4))) 
                            (if state-string
                                (string->symbol 
                                 (string-trim-right state-string #\-))
                                'click))
                  #:modifier-keys modifier-keys))
          ;; It doesn't specify a mouse event; return false.
          #f)))
(define-method (event->kbd (event <mouse-event>))
  (define (state->list state)
    (case state
      ((up down drag)
       (list (symbol->string state)))
      ((click)
       '())
      (else
       (error "Bad state state for mouse event " state))))
  (let ((mods (map string (map modifier-symbol->char (modifier-keys event))))
        (state-list (state->list (state event))))
    (string-join 
     `(,@mods ,@state-list "mouse" ,(number->string (button event)))
     "-")))
(define*-public (mouse-event? obj #:optional (of-state #f))
  (and (is-a? obj <mouse-event>)
       (if of-state 
           (eq? of-state (state obj))
           #t)))

(define-public (up-mouse-event? e)
  (mouse-event? e 'up))

(define-public (down-mouse-event? e)
  (mouse-event? e 'down))

(define-public (drag-mouse-event? e)
  (mouse-event? e 'drag))

(define-public (click-mouse-event? e)
  (mouse-event? e 'click))

(define-public (motion-mouse-event? e)
  (mouse-event? e 'motion))
(define-method (canonize-event! (event <event>))
  event)

(define-method (event->kbd (event <event>))
  #f)
(register-kbd-converter 'kbd-entry->key-event kbd-entry->key-event)
