(define-module (emacsy keymap)
  #:use-module (ice-9 regex)
  #:use-module (ice-9 optargs)
  #:use-module (oop goops)
  #:use-module (emacsy util)
  #:use-module (emacsy event))

(define-class-public <keymap> ()
  (entries #:getter entries #:init-thunk (lambda () (make-hash-table)))
  (parent #:accessor parent #:init-keyword #:parent #:init-value #f))

(define*-public (lookup-key keymap keys #:optional (follow-parent? #t))
  (define* (lookup-key* keymap keys #:optional (follow-parent? #t))
    (if (null? keys)
        keymap
        (let ((entry (hash-ref (entries keymap) (car keys))))
          (if entry
              (if (keymap? entry)
                  ;; Recurse into the next keymap.
                  (1+if-number (lookup-key* entry (cdr keys) follow-parent?))
                  ;; Entry exists.
                  (if (null? (cdr keys))
                      ;; Specifies the right number of keys; return
                      ;; entry.
                      entry
                      ;; Entry exists but there are more keys; return a
                      ;; number.
                      1))
              ;; No entry; try the parent.
              (if (and follow-parent? (parent keymap))
                  (lookup-key* (parent keymap) keys follow-parent?)
                  ;; No entry; no parent.
                  #f)))))
  (lookup-key* keymap (if (string? keys)
                          (kbd keys)
                          keys) follow-parent?))
(define (1+if-number x)
  (if (number? x)
      (1+ x)
      x))
(define*-public (lookup-key? keymap keyspec #:optional (keymap-ok? #f))
   (let* ((keys (if (string? keyspec) 
                    (kbd keyspec)
                    keyspec))
          (result (lookup-key keymap keys)))
     (if keymap-ok?
         (and (not (boolean? result)) 
              (not (number? result)))
         (and (not (keymap? result)) 
              (not (boolean? result)) 
              (not (number? result))))))
(define (make-trampoline module name)
  "Creates a trampoline out of a symbol in a given module, e.g. (lambda () (name))"
  (let ((var (module-variable module name)))
    (unless var
      (scm-error 'no-such-variable "make-trampoline" "Can't make a trampoline for variable named '~a that does not exist in module ~a." (list name module) #f))
    (let ((proc (lambda () ((variable-ref var)))))
      (set-procedure-property! proc 'name 
                               (string->symbol (format #f "~a-trampoline" name)))
      proc)))
(define-public (define-key keymap key-list-or-string symbol-or-procedure-or-keymap)
  (let* ((keys (if (string? key-list-or-string)
                   (kbd key-list-or-string)
                   key-list-or-string))
         (entry (lookup-key keymap (list (car keys)) #f))
         (procedure-or-keymap
          (if (symbol? symbol-or-procedure-or-keymap)
              (make-trampoline (current-module) symbol-or-procedure-or-keymap)
              symbol-or-procedure-or-keymap)))
    (cond
     ;; Error
     ((number? entry)
      (error "Terminal key binding already found for ~a keys." entry))
     ;; Keymap available for the first key; recurse!
     ((keymap? entry)
      (define-key entry (cdr keys) procedure-or-keymap))
     (else
      (if (= 1 (length keys))
          ;; This is our last key, just add it to our keymap.
          (begin 
            (hash-set! (entries keymap) (car keys) procedure-or-keymap)
            keymap)
          ;; We've got a lot of keys left that need to be hung on some
          ;; keymap.
          (define-key keymap (rcdr keys)
            (define-key (make-keymap) (list (rcar keys)) procedure-or-keymap)))))))
(define-public (keymap? obj)
  (is-a? obj <keymap>))
(define*-public (make-keymap #:optional (parent #f))
  (make <keymap> #:parent parent))
(define-method (write (obj <keymap>) port)
  (write-keymap obj port))

(define* (write-keymap obj port #:optional (keymap-print-prefix 0))
  (display  "#<keymap " port)
  (hash-for-each (lambda (key value)
                   (do ((i 1 (1+ i)))
                       ((> i keymap-print-prefix))
                     (display " " port))
                   (display "\n" port)
                   (display key port)
                   (display " " port)
                   (if (keymap? value)
                       (write-keymap value port (+ 2 keymap-print-prefix))
                       (display value port))) 
                 (entries obj))
  (if (parent obj)
      (write-keymap (parent obj) port (+ 2 keymap-print-prefix)))
  (display ">" port))
(define-public (lookup-key-entry? result)
  (and (not (boolean? result)) (not (number? result))))

