(define-module (emacsy klecl)
  #:use-module (ice-9 optargs)
  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)
  #:use-module (ice-9 q)
  #:use-module (srfi srfi-1)
  #:use-module (rnrs io ports)
  #:use-module (oop goops)
  #:use-module (system repl error-handling)
  #:use-module (emacsy util)
  #:use-module (emacsy keymap)
  #:use-module (emacsy event)
  #:use-module (emacsy command)
  #:use-module (emacsy block)
  #:use-module (emacsy coroutine)
  #:use-module (emacsy agenda)
  #:export (command-loop)
  )


(define-public event-queue (make-q))
(define-public read-event-hook (make-hook 1))
(define-public emacsy-interactive? #f)
(define-public this-command-event #f)
(define-public last-command-event #f)

(define-public pre-command-hook (make-hook))
(define-public post-command-hook (make-hook))
(define-public emacsy-ran-undefined-command? #f)
(define command-loop-count 0)
(define-public (emacsy-event event)
  (enq! event-queue event))
(define*-public (emacsy-key-event char #:optional (modifier-keys '()))
  (emacsy-event (make <key-event> 
                  #:modifier-keys modifier-keys 
                  #:command-char char)))
(define*-public (emacsy-mouse-event position button state 
                                    #:optional (modifier-keys '()))
  (emacsy-event 
   (make <mouse-event> 
     #:position position #:button button 
     #:state state #:modifier-keys modifier-keys)))
(define-public (emacsy-discard-input!)
  (while (not (q-empty? event-queue))
    (deq! event-queue)))

(define-public (emacsy-event-unread event)
  (q-push! event-queue event))
;; (define*-public (read-event #:optional (prompt #f))
;;   (if prompt
;;       (message prompt))
;;   ;;;;(if (q-empty? event-queue)
;;   (block-while (lambda () (q-empty? event-queue)) #t)
;;   (let ((event (deq! event-queue)))
;;     (run-hook read-event-hook event)
;;     event))

(define (raw-read-event prompt)
  (if prompt
      (message prompt))
  (let ((event (deq! event-queue)))
    (run-hook read-event-hook event)
    (emacsy-log-debug "RAW-READ-EVENT ~a~%" event)
    event))

(define*-public (read-event #:optional (prompt #f))
  (if emacsy-interactive?
      (if (q-empty? event-queue)
          (yield (lambda (resume)
                   (block-read-event prompt resume)))
          (raw-read-event prompt))
      ;; We're non-interactive. I need to read from stdin.
      
      (let ((input-string (read-line)))
        (if (and (eof-object? input-string)
                 (q-empty? event-queue))
            (throw 'read-event-eof)
            (begin
              (unless (eof-object? input-string)
                (map emacsy-event (kbd->events input-string)))
              (raw-read-event prompt))))))

(define reader-request-queue (make-q))

(define (block-read-event prompt resume)
  (format #t "block-read-event ~a~%" (list prompt resume))
  (enq! reader-request-queue (list prompt resume)))

(codefine (fulfill-read-requests)
   (while #t
     (when (getenv "EMACSY_DEBUG") (format #t "fulfill-read-requests CHECK~%"))
     (when (and (not (q-empty? event-queue)) 
                (not (q-empty? reader-request-queue)))
       (format #t "fulfill-read-requests DO~%")
       (match (deq! reader-request-queue)
         ((prompt resume)
          ;; Do I need to schedule this with the agenda to make it
          ;; behave properly?
          (resume (raw-read-event prompt)))))
     (wait)))

(agenda-schedule fulfill-read-requests)
(define last-down-mouse-event #f)

(define*-public (read-key #:optional (prompt #f))
  (define* (new-mouse-state new-state event #:optional (event2 #f))
    (let ((e (make (if event2 <drag-mouse-event> <mouse-event>) 
             #:modifier-keys (modifier-keys event)
             #:position (position event)
             #:button (button event)
             #:state new-state)))
      (if event2
          (slot-set! e 'rect (list (position event) (position event2))))
      e))
  ;; XXX Can this be refashioned into a nice handle-event method?
  ;; That would split up all these mouse key concerns.
  (let loop ((event (read-event prompt)))
    (if (is-a? event <dummy-event>)
        ;; Ignore it.
        (loop (read-event prompt))
        (if (down-mouse-event? event)
            (begin
              (set! last-down-mouse-event event)
              (loop (read-event prompt)))
            (if (and last-down-mouse-event
                     (down-mouse-event? last-down-mouse-event)
                     (up-mouse-event? event))
                (let ((new-event
                       (if (vector= (position last-down-mouse-event) (position event))
                           ;; Is it a click?
                           (new-mouse-state 'click event)
                           ;; Or a drag?
                           (new-mouse-state 'drag last-down-mouse-event event ))))
                  (set! last-down-mouse-event #f)
                  new-event)
                event)))))
(define*-public (read-key-sequence 
                 #:optional 
                 (prompt #f)
                 #:key
                 (keymaps (default-klecl-maps)))
  (define (read-discrete-key)
    (let mini-loop ((event (read-key prompt)))
      (emacsy-log-trace "EVENT ~a~%" event)
      (if (discrete-event? event)
          event
          (mini-loop (read-key prompt)))))
  (let loop ((events (list (read-key prompt))))
    (let* ((keys (reverse (map (compose event->kbd canonize-event!) events)))
           (last-key (rcar keys)))
     ;; Do we have enough keys?
     (if (or
          ;; Does one of the keymaps points to a command (or is it the
          ;; quit key)?
          (or (quit-key? last-key keymaps)
              (any (lambda (keymap)
                     (lookup-key? keymap keys))
                   keymaps))
          ;; OR does none of the keymaps point to a command or keymap (or
          ;; is the quit key)?
          (not (or (quit-key? last-key keymaps)
                   (any (lambda (keymap)
                          (lookup-key? keymap keys #t))
                        keymaps))))
         ;; Yes. We have enough keys.
         (reverse events)
         ;; No.  Let's get some more.
         (loop (cons (read-discrete-key) events))))))
(define-public (quit-key? aKey keymaps)
  (define (quit-key?* key keymap)
   (let ((result (lookup-key keymap (list key))))
     (and (not (keymap? result)) (lookup-key-entry? result)
          (eq? 'keyboard-quit result))))
  (any (lambda (keymap) (quit-key?* aKey keymap)) keymaps))
(define-public (default-klecl-maps)
  (list))
(define-public (message . args) 
  (apply format #t args))
(define call-with-sigalrm
  (if (not (provided? 'posix))
      (lambda (thunk) (thunk))
      (lambda (thunk)
        (let ((handler #f))
          (dynamic-wind
            (lambda ()
              (set! handler
                    (sigaction SIGALRM
                      (lambda (sig)
                        ;;(block-yield)
                        (scm-error 'signal #f "Alarm interrupt" '()
                                   (list sig))))))
            thunk
            (lambda ()
              (if handler
                  ;; restore Scheme handler, SIG_IGN or SIG_DFL.
                  (sigaction SIGALRM (car handler) (cdr handler))
                  ;; restore original C handler.
                  (sigaction SIGALRM #f))))))))
(define*-public (primitive-command-tick #:optional
                                        (prompt #f)
                                        #:key 
                                        (keymaps (default-klecl-maps))
                                        (undefined-command undefined-command))
  "We do one iteration of the command-loop without any error handling."
  (call-with-sigalrm
   (lambda ()
     ((@@ (ice-9 top-repl) call-with-sigint) 
      (lambda ()
        (let* ((events (read-key-sequence prompt #:keymaps keymaps))
               (key-sequence (map event->kbd events))
               (keymap (find (lambda (k) (lookup-key? k key-sequence)) keymaps)))
          (set! emacsy-ran-undefined-command? #f)
          (if keymap
              (begin
                (set! last-command-event this-command-event)
                (set! this-command-event (rcar events))
                ;; The command hooks might need to go into the command module.
                (in-out 
                 (run-hook pre-command-hook)
                 (call-interactively (lookup-key keymap key-sequence))
                 (run-hook post-command-hook)))
              ;; Maybe this should be done by an undefined-command command?
              ;; I doubt we want this command to be executed by the user, so
              ;; we're going to leave it as a procedure.
              (undefined-command key-sequence events))))))))
(define* (undefined-command key-sequence events)
  (message "~a is undefined."
           (string-join key-sequence " "))
  (set! emacsy-ran-undefined-command? #t)
  (values #f 'no-such-command))
(define*-public (command-tick #:key (keymaps (default-klecl-maps)))
  "We do one iteration of command-tick and handle errors."
  
  (catch #t
    (lambda () 
      (if debug-on-error?
          (call-with-error-handling
           (lambda ()
             (primitive-command-tick #:keymaps keymaps))
           #:pass-keys 
           ;; XXX what the hell is the story with all these quits?
           '(silent-quit quit-command-loop quit-command keyboard-quit))
          
          (with-backtrace* (lambda ()
                             (primitive-command-tick #:keymaps keymaps))
                           '(silent-quit quit-command-loop))))
    (lambda (key . args)
      (case key
        ((silent-quit)
         (emacsy-log-warning "GOT SILENT QUIT in command-tick\n"))
        ((quit-command-loop)
         (emacsy-log-warning "GOT QUIT-COMMAND-LOOP in command-tick\n")
         (apply throw key args))
        ((encoding-error)
         (emacsy-log-warning "ENCODING-ERROR '~a'" (pp-string event-queue)))
        (else
         (emacsy-log-error
                 "command-tick: Uncaught throw to '~a: ~a\n" key args))))))
(define*-public (primitive-command-loop #:optional (continue-pred (const #t)))
  "We iterate with command-tick but we do not handle any errors."
  (with-fluids ((continue-command-loop? #t))
    (let loop ((continue? (call-with-values 
                              primitive-command-tick
                            continue-pred)))
      (if (and (fluid-ref continue-command-loop?) continue?)
          (loop (call-with-values 
                    primitive-command-tick
                  continue-pred))
          (decr! command-loop-count)))))
(define* (command-loop #:optional (continue-pred (const #t)))
  "We iterate with command-tick and handle errors."
  (catch #t
    (lambda () 
      (if debug-on-error?
          (call-with-error-handling
           (lambda ()
             (primitive-command-loop continue-pred))
           #:pass-keys 
           '(silent-quit quit-command-loop quit-command keyboard-quit))
          
          (with-backtrace* (lambda ()
                             (primitive-command-loop continue-pred))
                           '(silent-quit quit-command-loop))))
    (lambda (key . args)
      (case key
        ((silent-quit)
         (emacsy-log-warning "GOT SILENT QUIT in command-loop"))
        ((quit-command-loop)
         (emacsy-log-warning "GOT QUIT-COMMAND-LOOP in command-loop")
         (apply throw key args))
        ((encoding-error)
         (emacsy-log-error "ENCODING-ERROR '~a'" (pp-string event-queue)))
        (else
         (emacsy-log-error
                 "command-loop: Uncaught throw to '~a: ~a\n" key args))))))
(define-interactive (keyboard-quit)
  (message "Quit!")
  (throw 'quit-command))

