
(define-module (emacsy window)
  #:use-module (oop goops)
  #:use-module (emacsy emacsy)
    #:use-module (ice-9 match)
  #:export (<window>
            <internal-window>
            window?
            split-window
            window-buffer
            window-children
            orientation 
            user-data
            window-list
            root-window
            current-window
            window-configuration-change-hook
            window-clone)
  #:export-syntax (  ) 
)
(define-class <window> ()
  (window-parent #:accessor window-parent #:init-value #f)
  (user-data #:accessor user-data #:init-keyword #:user-data #:init-value #f)
  (window-buffer #:accessor window-buffer #:init-keyword #:window-buffer #:init-value #f)
  (window-dedicated? #:accessor window-dedicated? #:init-value #f))
(define-class <internal-window> ()
  (window-parent #:accessor window-parent #:init-value #f)
  (user-data #:accessor user-data #:init-keyword #:user-data #:init-value #f)
  (window-children #:accessor window-children #:init-keyword #:window-children #:init-value '()) 
  (orientation #:accessor orientation #:init-keyword #:orientation #:init-value 'vertical) ; or 'horizontal
  (size #:accessor size #:init-keyword #:size #:init-value .5))
(define-public root-window (make <window>))
(define-variable window-configuration-change-hook (make-hook 1) "This hook is called when a window is split.")
(define current-window #f)
(define-method (initialize (obj <internal-window>) initargs)
  (next-method)
  (for-each (lambda (window)
    (set! (window-parent window) obj)) (window-children obj)))

(define (window? o)
  (or (is-a? o <window>) (is-a? o <internal-window>)))
(define (window-live? o)
  (is-a? o <window>))
(define (frame-root-window)
  root-window)
(define (edges->bcoords edges)
  (match edges
   ((left top right bottom)
    (list left bottom (- right left) (- top bottom)))))
(define (bcoords->edges coords)
  (match coords
  ((x y w h)
    (list x (+ y h) (+ x w) y))))
(let ((old-func switch-to-buffer))
  (set! switch-to-buffer
        (lambda-cmd args
                    (let ((result (apply old-func args)))
                      (format #t "Setting current window to buffer ~a~%" (current-buffer))
                      (set! (window-buffer current-window) (current-buffer))
                      result))))
(define-method (window-clone (window <window>))
  (shallow-clone window))
(define-public (selected-window)
  current-window)
(define-method (update-window (window <internal-window>))
 #f
 (when #f
   (let ((children (window-children window)))
      (if (eq? (orientation window) 'vertical)
      #t
      #t))))
(define-method (window-tree (w <internal-window>))
  (map window-tree (window-children w)))

(define-method (window-tree (w <window>))
  w)

(define (flatten x)
    (cond ((null? x) '())
          ((not (pair? x)) (list x))
          (else (append (flatten (car x))
                        (flatten (cdr x))))))

(define* (window-list #:optional (w root-window))
  (flatten (window-tree w)))
(define-interactive (split-window #:optional 
                     (window (selected-window))
                     (size 0.5)
                     (side 'below))
  (define (substitute x y)
    "Returns a function that will substitute x for y when given x."
    (lambda (z)
     (if (eq? z x)
         y
         z)))
  (let* ((original-parent (window-parent window))
         (new-child (window-clone window))
         (internal-window (make <internal-window> 
                                #:window-children (list window new-child)
                                #:size size
                                #:orientation (if (memq side '(below above))
                                                  'vertical
                                                  'horizontal))))
    (set! (window-parent internal-window) original-parent)
    (set! (window-parent window)    internal-window)
    (set! (window-parent new-child) internal-window)
    (when original-parent
     (set! (window-children original-parent)
           (map (substitute window internal-window) 
                (window-children original-parent))))
    (run-hook window-configuration-change-hook original-parent)
    (update-window internal-window)
  internal-window))
(define-interactive (split-window-below #:optional (size .5))
  (split-window (selected-window) size 'below))
(define-interactive (split-window-right #:optional (size .5))
  (split-window (selected-window) size 'right))
(define-interactive (delete-window #:optional (window (selected-window)))
  (let ((p (window-parent window)))
    ;; Only delete if it has a parent.
    (when p
      (let* ((children (window-children p))
             (new-children (delq window children)))
        (set! (window-children p) new-children)
        (set! current-window (car new-children))
        (run-hook window-configuration-change-hook p)
        ;; XXX We will want to divest ourselves of any internal-windows
        ;; that only contain one child. Not sure if we want to do that here
        ;; or in another method though.
        ;;(if (= 1 (length new-children))
        ;;    (car new-children)
        ;;    (begin
        ;;      #t))
))))

(define-interactive (delete-other-windows #:optional (window (selected-window)))
  (set! root-window (make <internal-window> #:window-children (list window)))
  (set! current-window window)
  (run-hook window-configuration-change-hook root-window))

(define-interactive (other-window #:optional (count 1))
  (let* ((lst (window-list root-window))
         (index (member-ref current-window lst)))
    (set! current-window (list-ref lst (modulo (+ index count) (length lst))))))
(define-key global-map "C-x 0" 'delete-window)
(define-key global-map "C-x 1" 'delete-other-windows)
(define-key global-map "C-x 2" 'split-window-below)
(define-key global-map "C-x 3" 'split-window-right)

(define-key global-map "C-x o" 'other-window)
