/*
 * Copyright (C) 2002,2003,2004 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: video.hh,v 1.9 2004/03/21 17:31:47 dheck Exp $
 */

#ifndef VIDEO_HH_INCLUDED
#define VIDEO_HH_INCLUDED

#include <string>
#include <memory>
#include "SDL.h"
#include "px/pxfwd.hh"
#include "px/geom.hh"

namespace video
{

    enum VideoModes {
        VM_None     = -1,
        VM_640x480  = 0,
        VM_640x512  = 1,
        VM_800x600  = 2,
        VM_1024x768 = 3,
        VM_COUNT
    };

    struct VMInfo {
        int width, height;      // Screen width and height in pixels
        int tile_size;          // Tile size in pixels
        const char *name;       // Menu text 
        const char *gfxdir;     // Directory that contains the graphics
        int thumbw, thumbh;     // Width and height of thumbnails
        const char *thumbsdir;  // Directory that contains the thumbnails
        px::Rect gamearea;
        px::Rect statusbararea;
        px::Rect sb_timearea;
        px::Rect sb_movesarea;
        px::Rect sb_itemarea;
        px::Rect sb_textarea;
        VideoModes  fallback_videomode;
        bool     available;     // Is this video mode available?
    };


    void Init();
    void Shutdown();

    void ChangeVideoMode();

    px::Screen *GetScreen();

    /*! Return information about arbitrary video mode. */
    const VMInfo *GetInfo (VideoModes vm);

    /*! Return information about current video mode. */
    const VMInfo *GetInfo ();

    //! Return the current video mode
    VideoModes GetVideoMode();

    /*! The backbuffer is surface that has the same size and pixel
      format as the screen.  This surface is used by ShowScreen() and
      FX_* functions below. */
    px::Surface *BackBuffer();

    bool SetFullscreen(bool onoff);

    /*! Switch between windowed and fullscreen mode. Return true if
       fullscreen mode is active afterwards. */
    bool ToggleFullscreen();


    void SetCaption(const char *str);
    const std::string& GetCaption();

    /*! Return the number of bits per pixel in the current video
      mode. [currently always 16] */
    int GetColorDepth();

    /*! Take a screenshot and store it as FNAME. */
    void Screenshot (const std::string &fname);

/* -------------------- Mouse cursor -------------------- */

    void SetMouseCursor(px::Surface *s, int hotx, int hoty);
    void HideMouse();
    void ShowMouse();
    int Mousex();
    int Mousey();

/* -------------------- Visual effects -------------------- */

    class TransitionEffect {
    public:
        virtual ~TransitionEffect() {}
        virtual void tick (double dtime) = 0;
        virtual bool finished() const = 0;
    };

    enum FadeMode { FADEIN, FADEOUT };
    void FX_Fade(FadeMode mode);
    void FX_Fly (px::Surface *newscr, int originx, int originy);

    enum TransitionModes
    {
	TM_RANDOM,
	TM_FADEOUTIN,
	TM_SQUARES,
	TM_FLY_N, TM_FLY_S, TM_FLY_W, TM_FLY_E,
	TM_FLY_NW, TM_FLY_NE, TM_FLY_SE, TM_FLY_SW,
        TM_PUSH_RANDOM, TM_PUSH_N, TM_PUSH_S, TM_PUSH_W, TM_PUSH_E
    };
    void ShowScreen (TransitionModes tm, px::Surface *newscr);

    TransitionEffect *MakeEffect (TransitionModes tm, px::Surface *newscr);
}

#endif
